#------------------------------------------------------------------------------
#    59Tracker, weblog software for personal publisher.
#    Copyright (C) 2004-2010 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

package Lib::Trackback;

use strict;
use warnings;

use constant {
    LOCK_SH => 1,
    LOCK_EX => 2,
    LOCK_NB => 4,
    LOCK_UN => 8,
};

use Lib::DateTime;
use Lib::Conf;
use Lib::Language;

# コンストラクタ
sub new {
    my $self = {};

    my $conf = Lib::Conf->new();

    $self->{system_dir} = $conf->get_system_dir();
    $self->{data_dir} = $conf->get_data_dir();
    $self->{docroot_dir} = $conf->get_docroot_dir();
    $self->{site_title} = $conf->get_site_title();
    $self->{sub_title} = $conf->get_sub_title();
    $self->{trackback_num} = $conf->get_trackback_num();
    $self->{normal_point} = $conf->get_normal_point();

    bless($self);
    return $self;
}

# 最近のトラックバックに追加
sub add {
    my $self = shift;
    my ($topicid, $trackbackid, $title, $recvdate, $excerpt) = @_;
    my @newdata = ();

    # 新規作成のデータを先頭に追加
    my $data = "$topicid,$trackbackid,$title,$recvdate,$excerpt";
    push @newdata, $data;

    # 最近のトラックバックファイルを読み込み配列に保存する
    my $datafile = "$self->{data_dir}/newtb.data";
    if (-f $datafile) { # ファイルが存在する場合
        open my $infh, '<', $datafile;
        while (my $data = <$infh>) {
            chomp $data;
            if ($data ne '') {
                push @newdata, $data;
            }
        }
        close $infh;
    }

    my $trackback_num = $self->{trackback_num};
    if (scalar(@newdata) > $trackback_num) {
        $trackback_num--;
        @newdata = @newdata[0 .. $trackback_num];
    }

    open my $outfh, '>', $datafile;
    flock $outfh, LOCK_EX;
    foreach $data (@newdata) {
        print {$outfh} "$data\n";
    }
    flock $outfh, LOCK_UN;
    close $outfh;

    # ファイルのパーミッションを変更
    chmod 0766, $datafile;

    $self->outputrss2();
}

# 最近のトラックバックから削除
sub delete {
    my $self = shift;
    my ($topicid, $trackbackid) = @_;
    my @newdata = ();

    # 最近のトラックバックファイルを読み込む
    my $datafile = "$self->{data_dir}/newtb.data";
    open my $infh, '<', $datafile;
    while (my $data = <$infh>) {
        chomp $data;
        if ($data ne '') {
            my ($newtopicid, $newtrackbackid) = (split(/\,/, $data))[0,1];
            if (!(($newtopicid eq $topicid) && ($trackbackid eq $newtrackbackid))) {
                push @newdata, $data;
            }
        }
    }
    close $infh;

    # 最近のトラックバックファイルを更新
    open my $outfh, '>', $datafile;
    flock $outfh, LOCK_EX;
    foreach my $data (@newdata) {
        print {$outfh} "$data\n";
    }
    flock $outfh, LOCK_UN;
    close $outfh;

    $self->outputrss2();
}

# 最近のトラックバック一覧の取得(HTML)
sub new_trackback {
    my $self = shift;
    my $maxnum = shift;
    my $list = '';

    # 最近のトラックバックファイルを読み込み一覧を作成する
    my $datafile = "$self->{data_dir}/newtb.data";
    if (-f $datafile) { # ファイルが存在する場合
        open my $infh, '<', $datafile;
        my $count = 0;
        while (my $data = <$infh>) {
            last if ($count >= $maxnum); 
            chomp $data;
            if ($data ne '') {
                my ($topicid, $trackbackid, $title, $recvdate, $excerpt) = split(/\,/, $data);
                $title =~ s/enc_conma/,/g;
                my $topicurl = "./?func=detail&id=$topicid";
                $list .= <<"END_LIST";
<a href="$topicurl#trackback$trackbackid">$title</a><br>
END_LIST
            }
            $count++;
        }
        close $infh;
    }

    if ($list ne '') {
        my $language = Lib::Language->new();
        my $recenttrackback = $language->get_statictext('recent-trackback');
       $list = <<"END_LIST";
<h3>$recenttrackback</h3>
$list
<br />
END_LIST
    }

    return $list;
}

# RSS 2.0 ファイルを出力する
sub outputrss2 {
    my $self = shift;
    my @datalist = ();

    # 最近のトラックバックファイルを読み込み配列に保存する
    my $datafile = "$self->{data_dir}/newtb.data";
    if (-f $datafile) {
        open my $infh, '<', $datafile;
        while (my $data = <$infh>) {
            chomp $data;
            if ($data ne '') {
                push @datalist, $data;
            }
        }
        close $infh;
    }

    my $siteurl = $self->{docroot_dir} . "/";
    my $sitetitle = $self->{site_title};
    my $subtitle = $self->{sub_title};

    my $dt = Lib::DateTime->new();
    my $last_builddate = $dt->local_datetime(3);

    my $itemlist = '';
    foreach my $data (@datalist) {
        my ($topicid, $trackbackid, $title, $recvdate, $excerpt) = split(/\,/, $data);

        # コンマをデコード
        $title =~ s/enc_conma/,/g;
        $excerpt =~ s/enc_conma/,/g;
        $excerpt =~ s/enc_crlf//g;

        my $topicurl = $siteurl . "?func=detail&amp;id=$topicid#trackback$trackbackid";

        $title =~ s/&/&amp;/g;
        $title =~ s/"/&quot;/g;
        $title =~ s/</&lt;/g;
        $title =~ s/>/&gt;/g;

        $excerpt =~ s/&/&amp;/g;
        $excerpt =~ s/"/&quot;/g;
        $excerpt =~ s/</&lt;/g;
        $excerpt =~ s/>/&gt;/g;

        $dt->set_datetime($recvdate);
        my $pubdate = $dt->format_datetime(3);

        $itemlist .= <<"END_ITEM";
  <item>
    <title>$title</title>
    <description>$excerpt</description>
    <link>$topicurl</link>
    <pubDate>$pubdate</pubDate>
  </item>
END_ITEM
    }

    # RSSファイルの生成
    open my $templatefh, '<', "$self->{system_dir}/tmpl/rss2.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$SITEURL\$/$siteurl/g;
    $template =~ s/\$SITETITLE\$/$sitetitle/g;
    $template =~ s/\$SUBTITLE\$/$subtitle/g;
    $template =~ s/\$PUBDATE\$/$last_builddate/g;
    $template =~ s/\$LASTBUILDDATE\$/$last_builddate/g;
    $template =~ s/\$ITEMLIST\$/$itemlist/g;

    # RSS2.0ファイルに出力
    my $rssfile = "trackback.xml";
    open my $rssfh, '>', $rssfile;
    flock $rssfh, LOCK_EX;
    print {$rssfh} $template;
    flock $rssfh, LOCK_UN;
    close $rssfh;
}

1;
# End of Trackback.pm
