#------------------------------------------------------------------------------
#    59Tracker, weblog software for personal publisher.
#    Copyright (C) 2004-2010 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

package Lib::Comment;

use strict;
use warnings;

use constant {
    LOCK_SH => 1,
    LOCK_EX => 2,
    LOCK_NB => 4,
    LOCK_UN => 8,
};

use Lib::DateTime;
use Lib::String;
use Lib::Conf;
use Lib::Language;

# コンストラクタ
sub new {
    my $self = {};

    my $conf = Lib::Conf->new();

    $self->{system_dir} = $conf->get_system_dir();
    $self->{data_dir} = $conf->get_data_dir();
    $self->{docroot_dir} = $conf->get_docroot_dir();
    $self->{site_title} = $conf->get_site_title();
    $self->{sub_title} = $conf->get_sub_title();
    $self->{comment_num} = $conf->get_comment_num();

    bless($self);
    return $self;
}

# 最近のコメントデータファイルに追加
sub add {
    my $self = shift;
    my ($topicid, $commentid, $title, $postdate, $text) = @_;
    my @newdata = ();

    # 新規作成データを先頭に追加
    my $data = "$topicid,$commentid,$title,$postdate,$text";
    push @newdata, $data;

    # 最近のコメントファイルを読み込み配列に保存する
    my $datafile = "$self->{data_dir}/newcomment.data";
    if (-f $datafile) {
        open my $infh, '<', $datafile;
        while (my $data = <$infh>) {
            chomp $data;
            if ($data ne '') {
                push @newdata, $data;
            }
        }
        close $infh;
    }

    my $comment_num = $self->{comment_num};
    if (scalar(@newdata) > $comment_num) {
        $comment_num--;
        @newdata = @newdata[0 .. $comment_num];
    }

    open my $outfh, '>', $datafile;
    flock $outfh, LOCK_EX;
    foreach $data (@newdata) {
        print {$outfh} "$data\n";
    }
    flock $outfh, LOCK_UN;
    close $outfh;

    # ファイルのパーミッションを変更
    chmod 0766, $datafile;

    $self->outputrss2();
}

# 最近のコメントデータファイルから削除
sub delete {
    my $self = shift;
    my ($topicid, $commentid) = @_;
    my @newdata = ();

    # 最近のコメントファイルを読み込む
    my $datafile = "$self->{data_dir}/newcomment.data";
    open my $infh, '<', $datafile;
    while (my $data = <$infh>) {
        chomp $data;
        if ($data ne '') {
            my ($newtopicid, $newcommentid) = (split(/\,/, $data))[0,1];
            if (!(($newtopicid eq $topicid) && ($commentid eq $newcommentid))) {
                push @newdata, $data;
            }
        }
    }
    close $infh;

    # 最近のコメントファイルを更新
    open my $outfh, '>', $datafile;
    flock $outfh, LOCK_EX;
    foreach my $data (@newdata) {
        print {$outfh} "$data\n";
    }
    flock $outfh, LOCK_UN;
    close $outfh;

    $self->outputrss2();
}

# 最近のコメント一覧の取得(HTML)
sub new_comment {
    my $self = shift;
    my $maxnum = shift;
    my $list = '';

    # 最近のコメントファイルを読み込みコメントを作成する
    my $datafile = "$self->{data_dir}/newcomment.data";
    if (-f $datafile) {
        open my $infh, '<', $datafile;

        my $string = Lib::String->new();
        my $count = 0;
        while (my $data = <$infh>) {
            last if ($count >= $maxnum); 
            chomp $data ;
            if ($data ne '') {
                my ($topicid, $commentid, $title, $postdate, $text) = split(/\,/, $data);
#                $title =~ s/enc_conma/,/g;
                $text =~ s/enc_conma/,/g;
                $text =~ s/enc_crlf//g;
                $text = $string->cut($text, 100);
                my $topicurl = "./?func=detail&id=$topicid";

                $list .= <<"END_LIST";
<a href="$topicurl#comment$commentid">$text</a><br>
END_LIST
            }
            $count++;
        }
        close $infh;
    }

    if ($list ne '') {
        my $language = Lib::Language->new();
        my $recentcomment = $language->get_statictext('recent-comment');
       $list = <<"END_LIST";
<h3>$recentcomment</h3>
$list
<br />
END_LIST
    }

    return $list;
}

# RSS 2.0 ファイルを出力する
sub outputrss2 {
    my $self = shift;
    my @datalist = ();

    # 最近のコメントファイルを読み込み配列に保存する
    my $datafile = "$self->{data_dir}/newcomment.data";
    if (-f $datafile) {    # ファイルが存在する場合
        open my $infh, '<', $datafile;
        while (my $data = <$infh>) {
            chomp $data;
            if ($data ne '') {
                push @datalist, $data;
            }
        }
        close $infh;
    }
    else {
        return;
    }

    my $siteurl = $self->{docroot_dir} . "/";
    my $sitetitle = $self->{site_title};
    my $subtitle = $self->{sub_title};

    my $dt = Lib::DateTime->new();
    my $last_builddate = $dt->local_datetime(3);

    my $string = Lib::String->new();
    my $itemlist = '';
    foreach my $data (@datalist) {
        my ($topicid, $commentid, $title, $postdate, $text) = split(/\,/, $data);

        # コンマをデコード
        $title =~ s/enc_conma/,/g;
        $text =~ s/enc_conma/,/g;
        $text =~ s/enc_crlf//g;

        # コメントの長さを調整
        $text = $string->cut($text, 100);

        my $topicurl = $siteurl . "?func=detail&amp;id=$topicid#comment$commentid";

        $title =~ s/&/&amp;/g;
        $title =~ s/"/&quot;/g;
        $title =~ s/</&lt;/g;
        $title =~ s/>/&gt;/g;

        $text =~ s/&/&amp;/g;
        $text =~ s/"/&quot;/g;
        $text =~ s/</&lt;/g;
        $text =~ s/>/&gt;/g;

        $dt->set_datetime($postdate);
        my $pubdate = $dt->format_datetime(3);

        $itemlist .= <<"END_ITEM";
  <item>
    <title>$title</title>
    <description>$text</description>
    <link>$topicurl</link>
    <pubDate>$pubdate</pubDate>
  </item>
END_ITEM
    }

    # RSSファイルの生成
    open my $templatefh, '<', "$self->{system_dir}/tmpl/rss2.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$SITEURL\$/$siteurl/g;
    $template =~ s/\$SITETITLE\$/$sitetitle/g;
    $template =~ s/\$SUBTITLE\$/$subtitle/g;
    $template =~ s/\$PUBDATE\$/$last_builddate/g;
    $template =~ s/\$LASTBUILDDATE\$/$last_builddate/g;
    $template =~ s/\$ITEMLIST\$/$itemlist/g;

    # RSS2.0ファイルに出力
    my $rssfile = "comment.xml";
    open my $rssfh, '>', $rssfile;
    flock $rssfh, LOCK_EX;
    print {$rssfh} $template;
    flock $rssfh, LOCK_UN;
    close $rssfh;
}

1;
# End of Comment.pm
