#!/usr/bin/perl
#------------------------------------------------------------------------------
#    59bbs, bulletin board system for anonymous publisher.
#    Copyright (C) 2004-2009 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

use strict;
use warnings;

use Lib::Logger;
use Lib::Error;
use Lib::Calendar;
use Lib::DateTime;
use Lib::User;
use Lib::String;
use Lib::App::Ads;
use Lib::App::Archive;
use Lib::App::Trackback;
use Lib::App::Comment;
use Lib::App::Bookmark;
use Lib::App::Keyword;
use Lib::App::Search;
use Lib::App::Conf;

my $logger = Lib::Logger->new();
my $error = Lib::Error->new();
my $user = Lib::User->new();
my $conf = Lib::App::Conf->new();

sub act() {
    my ($act, $cgi) = @_;
    my ($operation, $object) = split(/_/, $act);

    if ($act eq 'input_session') {
        _input_session($cgi);
    }
    elsif ($act eq 'login_session') {
        _login_session($cgi);
    }
    elsif ($act eq 'logout_session') {
        _logout_session($cgi);
    }
    elsif ($act eq 'view_session') {
        _view_list('', $cgi);
    }
    else {
        $error->print_error('00002e');
    }
}

sub _input_session() {
    my ($cgi) = @_;
    my $msg = $cgi->param('msg');
    _show_login($msg);
}

sub _login_session() {
    my ($cgi) = @_;

    # 入力チェック
    my $userid = $cgi->param('userid');
    if ($userid eq '') {
        _show_login('ユーザーIDが未入力です');
    }

    my $password = $cgi->param('password');
    if ($userid eq '') {
        _show_login('パスワードが未入力です');
    }

    my $ipaddr = $ENV{'REMOTE_ADDR'};
    my $sessionid = $user->login($userid, $password, $ipaddr);
    if (!$sessionid) {
        _show_login('ユーザーIDまたはパスワードに誤りがあります');
    }

    _view_list($sessionid, undef);
}

sub _show_login() {
    my ($msg) = @_;

    my $systemdir = $conf->get_system_dir();
    my $sitetitle = $conf->get_site_title();

    $msg = "<p><strong><font class=\"message\" size=\"+1\">$msg</font></strong></p>";

    my $caption = "$sitetitleへログイン";

    my $path = <<"END_PATH";
<p><strong>
<a href="./" class="path">ホーム</a>&nbsp;
&gt;&nbsp;
$caption
</strong></p>
END_PATH

    open my $templatefh, '<', "$systemdir/tmpl/login.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$SITETITLE\$/$sitetitle/g;
    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$MSG\$/$msg/g;

    print "Content-Type: text/html\n\n";
    print $template;

    exit;
}

# ログアウト
sub _logout_session() {
    my ($cgi) = @_;

    # クッキーからセッションIDを取得
    my $sessionid = $cgi->cookie('cookie59t');
    if ($sessionid ne '') {
        $user->logout($sessionid);
    }

    print "Location: ./", "\n\n";
}

# リストページ表示
sub _view_list() {
    my ($sessionid, $cgi) = @_;

    my $session_userid = '';
    my $cookie = '';
    my $date = '';
    my $q = '';
    my $page = '';
    my $target = '';
    if ($sessionid eq '') {
        # セッションのチェック
        $sessionid = $cgi->cookie('cookie59t');
        if ($sessionid ne  '') {
            $session_userid = $user->check_session($sessionid);
        }
        $date = $cgi->param('date');
        $q = $cgi->param('q');
        $page = $cgi->param('page');
        $target = $cgi->param('target');

        my $string = Lib::String->new();
        $q = $string->tag_invalidate($q);
        # 入力チェック
        if ($date ne '' && $date !~ /\A[0-9]+\z/) {
            $error->print_error('00002e');
        }
        if ($page ne '' && $page !~ /\A[0-9]+\z/) {
            $error->print_error('00002e');
        }
    }
    else {
        $session_userid = $user->check_session($sessionid);
        # クッキー文字列を生成
        $cookie = "Set-Cookie: cookie59t=$sessionid\n";
    }
    if ($target eq '') {
        $target = $conf->get_default_target();
    }

    $page = ($page eq '') ? 1 : $page;

     my @items = ();
     my $format = 0;
     my ($year, $month, $day) = ('', '', '');
     if (!$target) { # トピック
        my $archive = Lib::App::Archive->new();

        if ($date ne '') {
            $format = 1;
            $year = substr($date, 0, 4);
            $month = substr($date, 4, 2);
            if (length($date) > 6) {
                $day = substr($date, 6, 2);
                $format = 2;
            }
            @items = $archive->get_archivetopics($year, $month, $day);
        }
        else {
            # 日時を取得
            my $dt = Lib::DateTime->new();
            $dt->local_datetime(0);
            $year = $dt->get_year();
            $month = $dt->get_month();
            if ($q eq '') {
                # 更新情報
                @items = $archive->get_newtopics();
            }
            else {
                # 検索
                if ($q =~ /^[0-9\,]+$/) { # 数字とコンマだけの場合はID指定
                    my $datadir = $conf->get_data_dir();
                    @items = split(/\,/, $q);
                    # データファイルが存在するかチェック
                    my @checkeditems = ();
                    foreach my $item (@items) {
                        my $filename = "$datadir/$item" . "tp.data";
                        if (-f $filename) {
                            push @checkeditems, $item;
                        }
                    }
                    @items = @checkeditems;
                }
                else {
                    my $search = Lib::App::Search->new();
                    @items = $search->search($target, $q);
                }
                $format = 3;
                @items = reverse (sort { $a <=> $b; } @items); # 降順にソート
            }
        }

#        if (scalar(@items) == 1) {
#            my $url = "./?act=view_topic&id=$items[0]";
#            print "Location: $url", "\n\n";
#            exit;
#        }
    }
    else { # ブックマーク
        if ($q eq '') {
            if ($session_userid eq '') { # ログインされていない
                # 新着ブックマーク
                @items = _new_bookmark();
                $format = 11;
            }
            else {
                # ユーザーが登録したブックマーク
                @items = _user_bookmarks($session_userid);
                $format = 12;
            }
        }
        else {
            # 検索
            if ($q =~ /^[0-9\,]+$/) { # 数字とコンマだけの場合はID指定
                @items = split(/\,/, $q);
                my $datadir = $conf->get_data_dir();
                # データファイルが存在するかチェック
                my @checkeditems = ();
                foreach my $item (@items) {
                    my $filename = "$datadir/bookmark/$item" . "bm.data";
                    if (-f $filename) {
                        push @checkeditems, $item;
                    }
                }
                @items = @checkeditems;
            }
            else {
                my $search = Lib::App::Search->new();
                @items = $search->search($target, $q);
            }
            $format = 13;
            @items = reverse (sort { $a <=> $b; } @items); # 降順にソート
        }

#        if (scalar(@items) == 1) {
#            my $url = "./?act=view_bookmark&id=$items[0]";
#            print "Location: $url", "\n\n";
#            exit;
#        }
    }

    my $listhtml = _list_html($session_userid, $format, "$year/$month/$day", $page, $q, $target, @items);;

    print "Content-Type: text/html\n";
    print "$cookie\n";
    print $listhtml;
}

# listページのhtmlソース作成
sub _list_html {
    my $session_userid = shift;
    my $format = shift;
    my $date = shift;
    my $page = shift;
    my $q = shift;
    my $target = shift;
    my @result = @_;

    my $docrootdir = $conf->get_docroot_dir();
    my $sitetitle = $conf->get_site_title();
    my $subtitle = $conf->get_sub_title();

    my $caption = "$sitetitle";
    my $listtitle = '更新情報';
    my $metadisc = "<meta name=\"description\" content=\"$subtitle\">";
    my $metakeywords = "<meta name=\"keywords\" content=\"ブログ,blog,ソフトウェア,ソーシャル,ブックマーク,メディア\">";

    my $ymd = ''; 
    my $currentq = ''; 
    my ($year, $month, $day) = split(/\//, $date);
    my $currentdate = ''; 
    if ($format == 1) {
        $listtitle = "$year年$month月";
        $caption = "$sitetitle：$listtitle";
        $ymd = "$year年$month";
        $currentdate = "$year$month";
        $metadisc = "<meta name=\"description\" content=\"$listtitle\">";
    }
    elsif ($format == 2) {
        $listtitle = "$year年$month月$day日";
        $caption = "$sitetitle：$listtitle";
        $currentdate = "$year$month$day";
        $metadisc = "<meta name=\"description\" content=\"$listtitle\">";
    }
    elsif ($format == 11) {
        $listtitle = "新着ブックマーク";
        $caption = "$sitetitle：$listtitle";
        $metadisc = "<meta name=\"description\" content=\"$listtitle\">";
    }
    elsif ($format == 12) {
        $listtitle = "$session_useridのブックマーク";
        $caption = "$sitetitle：$listtitle";
        $metadisc = "<meta name=\"description\" content=\"$listtitle\">";
    }
    elsif ($format == 3 || $format == 13) {
        $listtitle = "$qの検索結果";
        $caption = "$sitetitle：$listtitle";
        $metadisc = "<meta name=\"description\" content=\"$listtitle\">";
        $metakeywords = "<meta name=\"keywords\" content=\"$q\">";
        $currentq = $q; 
        $currentq =~ s/(\W)/'%' . unpack('H2', $1)/eg;	# uriパック
    }

    $sitetitle = "<a class=\"sitetitle\" href=\"$docrootdir/\">$sitetitle</a>";

    my $systemdir = $conf->get_system_dir();
    my $rssurl = "$docrootdir/rss.xml";

    my $menu = <<"END_MENU";
<a href="$rssurl"><img src="feed-icon-16x16.jpg" border="0"></a> <a href="$rssurl">更新情報の購読</a>&nbsp;&nbsp;
<a href="./?act=input_register">ユーザー登録</a>&nbsp;&nbsp;
<a href="./?act=input_session">ログイン</a>
END_MENU

    if ($session_userid ne '') { # ログイン中
        $menu = _get_menu($session_userid);
    }

    my $resultlist = '該当するトピックがありません';
    if ($target) {
        $resultlist = '該当するブックマークがありません';
    }
    my $pagenavi = '';
    if (scalar(@result) > 0) { # ページナビゲーションを作成

        # 総ページ数の算出
        my $itemnum = $conf->get_result_count();
        my $pagenum = int(scalar(@result) / $itemnum); 
        if (scalar(@result) / $itemnum > $pagenum) {
            $pagenum++;
        }

        # 該当ページのユーザーIDを取り出す
        my @items = ();
        for my $i (0 .. $#result) {
            if (($i >= ($page - 1) * $itemnum) && ($i < $page * $itemnum)) {
                push @items, $result[$i];
            }
        }

        if (!$target) {
            # トピック一覧を取得
            $resultlist = _get_topiclist($session_userid, @items);
        }
        else {
            # ブックマーク一覧を取得
            $resultlist = _get_bookmark_list(@items);
        }

        # ページナビゲーションの作成
        my $base = 1;
        my $currentnum = ($pagenum < 5) ? $pagenum : 5;
        if ($page > 5) {
            $base = $page - 4;
        }
        my @pagelist = ();
        for my $pagecount ($base .. ($base + $currentnum - 1)) {
            push @pagelist, $pagecount;
        }
        my $current = '';
        foreach my $pagecount (@pagelist) {
            if ($pagecount != $page) {
                $current .= "<a href=\"./?target=$target&date=$currentdate&q=$currentq&page=$pagecount\">$pagecount</a>&nbsp;";
            }
            else {
                $current .= "$pagecount&nbsp;";
            }
        }
        my $prev = ''; 
        if ($page > 1) {
            my $prevpage = $page - 1;
            $prev = "<a href=\"./?target=$target&date=$currentdate&q=$currentq&page=1\">&lt;&lt; 先頭へ</a>&nbsp;&nbsp;";
            $prev .= "<a href=\"./?target=$target&date=$currentdate&q=$currentq&page=$prevpage\">&lt; 前へ</a>";
        }
        my $next = ''; 
        if ($page < $pagenum) {
            my $nextpage = $page + 1;
            $next = "<a href=\"./?target=$target&date=$currentdate&q=$currentq&page=$nextpage\">次へ &gt;</a>";
        }
        $pagenavi = <<"END_PAGENAVI";
<p><strong>$prev&nbsp;&nbsp;$current&nbsp;$next</strong></p>
END_PAGENAVI
    }

    # 更新情報を取得
    my $archive = Lib::App::Archive->new();
    my $newtopiclist = $archive->new_topiclist();

    # 新着ブックマーク一覧を取得
    my $bookmark = Lib::App::Bookmark->new();
    my $bookmarknum = $conf->get_bookmark_num();
    my $newbookmarklist = $bookmark->new_bookmark_list($bookmarknum);

    my $calendar = '';
    my $archivelist = '';
    my $newtrackback = '';
    my $newcomment = '';
    my $usedbookmarklist = '';
    if (!$target) {
        # カレンダーを取得
        my $cal = Lib::Calendar->new();
        $calendar = $cal->get_calendar($year, $month);

        $archivelist = $archive->archive_list();

        # 最近のトラックバック一覧を取得
        my $trackback = Lib::App::Trackback->new();
        $newtrackback = $trackback->new_trackback();

        # 最近のコメント一覧を取得
        my $comment = Lib::App::Comment->new();
        $newcomment = $comment->new_comment();
    }
    else {
        # 最近使用されたブックマーク一覧を取得
        $usedbookmarklist = $bookmark->used_bookmark_list($bookmarknum);
    }

    # 広告の取得
    my $ads = Lib::App::Ads->new();
    my $ad1 = $ads->get_ad1();
    my $ad2 = $ads->get_ad2();
    my $ad3 = $ads->get_ad3();
    my $adlist = $ads->get_adlist_table();

    # クラウドを取得
    my $keywordobj = Lib::App::Keyword->new();
    my $clouds = $keywordobj->clouds(1, $target);

    # 検索ボックスの取得
    my $searchbox = _search_box();
    $searchbox =~ s/\$PREVQ\$/$q/g;
    if ($target) {
        $searchbox =~ s/\$TOPICCHECK\$//g;
        $searchbox =~ s/\$BMCHECK\$/checked/g;
    }
    else {
        $searchbox =~ s/\$TOPICCHECK\$/checked/g;
        $searchbox =~ s/\$BMCHECK\$//g;
    }

    # テンプレートの読み込み
    open my $templatefh, '<', "$systemdir/tmpl/list.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$METADISC\$/$metadisc/g;
    $template =~ s/\$METAKEYWORDS\$/$metakeywords/g;
    $template =~ s/\$RSSURL\$/$rssurl/g;
    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$SITETITLE\$/$sitetitle/g;
    $template =~ s/\$LISTTITLE\$/$listtitle/g;
    $template =~ s/\$SUBTITLE\$/$subtitle/g;
    $template =~ s/\$MENU\$/$menu/g;
    $template =~ s/\$RESULTLIST\$/$resultlist/g;
    $template =~ s/\$PAGENAVI\$/$pagenavi/g;
    $template =~ s/\$AD1\$/$ad1/g;
    $template =~ s/\$AD2\$/$ad2/g;
    $template =~ s/\$AD3\$/$ad3/g;
    $template =~ s/\$ADLIST\$/$adlist/g;
    $template =~ s/\$SEARCHBOX\$/$searchbox/g;
    $template =~ s/\$NEWTOPICLIST\$/$newtopiclist/g;
    $template =~ s/\$CALENDAR\$/$calendar/g;
    $template =~ s/\$ARCHIVELIST\$/$archivelist/g;
    $template =~ s/\$CLOUDS\$/$clouds/g;
    $template =~ s/\$NEWTRACKBACK\$/$newtrackback/g;
    $template =~ s/\$NEWCOMMENT\$/$newcomment/g;
    $template =~ s/\$NEWBOOKMARKLIST\$/$newbookmarklist/g;
    $template =~ s/\$USEDBOOKMARKLIST\$/$usedbookmarklist/g;

    return $template;
}

# 指定されたトピックの一覧を作成する
sub _get_topiclist {
    my $session_userid = shift;
    my @topics = @_;

    my $topiclist = '';
    my $topic = Lib::App::Topic->new();

    for my $i (0 .. $#topics) {
        my $topicid = $topics[$i];
        $topic->load($topicid, 0);
        my $open = ($i < $conf->get_open_item_num()) ? 1 : 0;
        $topiclist .= _topic_outline($topic, $open, $session_userid);
    }

    return $topiclist;
}

# トピック概要のHTMLを作成して返す、loadが前の処理で呼び出されていることが前提
sub _topic_outline {
    my ($topic, $open, $session_userid) = @_;
    my $topicid = $topic->get_topicid();
    my $attribute;
    my $status = $topic->get_status();
    my $outline = $topic->get_outline();
    my $title = $topic->get_title();
    my $text = $topic->get_text();
    my $keywords = $topic->get_keyword();
    my $point = $topic->get_point();
    my $commentnum = $topic->get_commentnum();
    my $trackbacknum = $topic->get_trackbacknum();
    my $psnum = $topic->get_psnum();
    my $lastdate = $topic->get_lastdate();

    if ($status == 9) {
        $title = "削除されました";
        $outline = '';
        $text = "削除されました";
    }

    my $topichtml = "./?act=view_topic&id=$topicid";
    if ($conf->get_static_html() && $session_userid eq '') {
        $topichtml = "detail$topicid.html";
    }

    my $editor = $topic->get_editor();
    my $user = Lib::User->new();
    $user->load($editor);
    my $mailaddr = $user->get_mailaddr();
    my $hpurl = $user->get_hpurl();
    my $username = $user->get_username();
    if ($username ne '') {
        $editor = $username;
    }
    if ($hpurl ne '') {
        $editor = "<a href=\"$hpurl\">$username</a>";
    }
    else {
        if ($mailaddr ne '') {
            $editor = "<a href=\"mailto:$mailaddr\">$username</a>";
        }
    }

    my $string = Lib::String->new();
    my $summary = $string->cut($outline, $conf->get_summary_length());
    if ($summary eq '') {
        $summary = $string->cut($text, $conf->get_summary_length());
    }
    # キーワードをリンク
    my $keyword = Lib::App::Keyword->new();
    $summary = $keyword->link_keyword($keywords, $summary, 0);

    my $psstr = '';
    if ($psnum > 0) {
        $psstr = "<div>(<a href=\"$topichtml#ps\">追記あり</a>)</div>";
    }

    my $showblock = $open ? "display:none"  : "display:block";
    my $hideblock = $open ? "display:block" : "display:none";

    my $item = <<"END_ITEM";
<div id="toggleShow$topicid" style="$showblock">
  <a href="javascript:;" onmousedown="toggleText('toggleText$topicid','toggleSummary$topicid','toggleShow$topicid','toggleHide$topicid');">＋</a>&nbsp;
  <strong>[$topicid] <a href="$topichtml">$title</a></strong>
</div>
<div id="toggleHide$topicid" style="$hideblock">
  <a href="javascript:;" onmousedown="toggleText('toggleText$topicid','toggleSummary$topicid','toggleShow$topicid','toggleHide$topicid');">－</a>&nbsp;
  <strong>[$topicid] <a href="$topichtml">$title</a></strong>
</div>
<div id="toggleSummary$topicid" style="$showblock">$summary</div>
<div id="toggleText$topicid" style="$hideblock">$text</div>
$psstr
<div>
<b><font class="point" size="+2">$point</font> point</b> |
 <a href="$topichtml#comment">Comment ($commentnum)</a> |
 <a href="$topichtml#trackback">Trackback ($trackbacknum)</a> |
 posted by $editor at <font class="datetime">$lastdate</font>
</div>
<br />
END_ITEM

    return $item;
}

# メニューを取得
sub _get_menu {
    my $session_userid = shift;

    my $menu = '';
    if ($user->is_admin($session_userid)) {
        $menu = <<"END_MENU";
<a href="./?act=input_topic">新規投稿</a>&nbsp;&nbsp;
<a href="./?act=input_bookmark">ブックマーク</a>&nbsp;&nbsp;
<a href="./?act=view_userlist">ユーザー管理</a>&nbsp;&nbsp;
<a href="./?act=rebuild_content">再構築</a>&nbsp;&nbsp;
<a href="./?act=logout_session">ログアウト</a><br />
<a href="./?act=input_password">パスワード</a>&nbsp;&nbsp;
<a href="./?act=edit_ads">広告設定</a>&nbsp;&nbsp;
<a href="./?act=edit_template&filecode=1">テンプレート</a>&nbsp;&nbsp;
<a href="./?act=edit_pinglist">Ping設定</a>&nbsp;&nbsp;
<a href="./?act=edit_filter">フィルター</a>
END_MENU
    }
    elsif ($user->is_editor($session_userid)) {
        $menu = <<"END_MENU";
<a href="./?act=input_topic">新規投稿</a>&nbsp;&nbsp;
<a href="./?act=input_bookmark">ブックマーク</a>&nbsp;&nbsp;
<a href="./?act=logout_session">ログアウト</a><br />
<a href="./?act=input_invite">招待</a>&nbsp;&nbsp;
<a href="./?act=edit_user&userid=$session_userid">ユーザー情報</a>&nbsp;&nbsp;
<a href="./?act=confirm_user">退会</a>&nbsp;&nbsp;
<a href="./?act=input_password">パスワード</a>
END_MENU
    }
    else {
        $menu = <<"END_MENU";
<a href="./?act=input_bookmark">ブックマーク</a>&nbsp;&nbsp;
<a href="./?act=logout_session">ログアウト</a><br />
<a href="./?act=edit_user&userid=$session_userid">ユーザー情報</a>&nbsp;&nbsp;
<a href="./?act=confirm_user">退会</a>&nbsp;&nbsp;
<a href="./?act=input_password">パスワード</a>
END_MENU
    }

    return $menu;
}

# リストページ用検索ボックスの作成
sub _search_box {

    my $form = <<"END_FORM";
<div class="searchbox">
<form method="post" action="./">
  <input type="radio" name="target" value="0" \$TOPICCHECK\$>トピック&nbsp;&nbsp;
  <input type="radio" name="target" value="1" \$BMCHECK\$>ブックマーク<br />
  <input type="text" name="q" size="60" value="\$PREVQ\$">&nbsp;
  <input type="submit" value="検索">
</form>
</div>
END_FORM

    return $form;
}

# 新着ブックマークを取得する
sub _new_bookmark {

    my $bookmark = Lib::App::Bookmark->new();
    my $counter = $bookmark->get_bookmark_counter();
    my @bookmarks = ();

    # ブックマークデータをチェックする
    for (my $i = $counter; $i > 0; $i--) {
        my $bookmarkid = $i;
        my $ret = $bookmark->load($bookmarkid, 0);
        if ($ret) {
            push @bookmarks, $bookmarkid;
        }
    }

    return @bookmarks;
}

# 指定のユーザーが登録者となっているブックマークのIDリストを返す
sub _user_bookmarks {
    my $userid = shift;

    my $bookmark = Lib::App::Bookmark->new();
    my $counter = $bookmark->get_bookmark_counter();
    my $list = '';
    my @bookmarks = ();

    for (my $i = $counter; $i > 0; $i--) {
        my $bookmarkid = $i;
        my $ret = $bookmark->load($bookmarkid, 0);
        next if (!$ret);

        my $registrant = $bookmark->get_registrant();
        my @userlist = split(/:/, $registrant);
        foreach my $user (@userlist) {
            if ($user eq $userid) {
                push @bookmarks, $bookmarkid;
                last;
            }
        }
    }

    return @bookmarks;
}

# 指定されたブックマークの一覧を作成
sub _get_bookmark_list {
    my @items = @_;
    my $list = '';
    my $count = 0;

    my $bookmark = Lib::App::Bookmark->new();

    # 既存のブックマークデータをチェックする
    foreach my $bookmarkid (@items) {
        my $ret = $bookmark->load($bookmarkid, 1);
        next if (!$ret);

        my $title = $bookmark->get_title();
        my $url = $bookmark->get_url();
        my $text = $bookmark->get_summary();
        my $registrant = $bookmark->get_registrant();
        my $datetime = $bookmark->get_datetime();
        my $keywords = $bookmark->get_keyword();
        my $point = $bookmark->get_point();

        $text =~ s/\n/<br \/>/g;

        my $userid = $registrant;
        my @userlist = split(/:/, $registrant);
        if (@userlist > 1) {
            $userid = $userlist[0];
        }

        my @comments = $bookmark->get_comments();
        my $commentnum = scalar(@comments);

        my $string = Lib::String->new();
        my $summary = $string->cut($text, $conf->get_summary_length());

        # キーワードをリンク
        my $keyword = Lib::App::Keyword->new();
        $text = $keyword->link_keyword($keywords, $text, 1);
        $summary = $keyword->link_keyword($keywords, $summary, 1);

        my $showblock = "display:block";
        my $hideblock = "display:none";
        if ($count < $conf->get_open_item_num()) {
            $showblock = "display:none";
            $hideblock = "display:block";
        }
        my $bookmarkhtml = "./?act=view_bookmark&id=$bookmarkid";

        $list .= <<"END_LIST";
<div id="toggleShow$bookmarkid" style="$showblock">
  <a href="javascript:;" onmousedown="toggleText('toggleText$bookmarkid','toggleSummary$bookmarkid','toggleShow$bookmarkid','toggleHide$bookmarkid');">＋</a>&nbsp;
  <b>[$bookmarkid]</b> <strong><a href="$bookmarkhtml">$title</a></strong>
</div>
<div id="toggleHide$bookmarkid" style="$hideblock">
  <a href="javascript:;" onmousedown="toggleText('toggleText$bookmarkid','toggleSummary$bookmarkid','toggleShow$bookmarkid','toggleHide$bookmarkid');">－</a>&nbsp;
  <b>[$bookmarkid]</b> <strong><a href="$bookmarkhtml">$title</a></strong></div>
<div id="toggleSummary$bookmarkid" style="$showblock">$summary</div>
<div id="toggleText$bookmarkid" style="$hideblock">$text</div>
<div>
 <b><font class="point" size="+2">$point</font> point</b> |
 <a href="$bookmarkhtml#comment">Comment ($commentnum)</a> |
 registered by $userid at <font class="datetime">$datetime</font>
</div>
<br />
END_LIST

        $count++;
        last if ($count >= $conf->get_result_count());
    }

    return $list;
}

1;
