#!/usr/bin/perl
#------------------------------------------------------------------------------
#    59Tracker, weblog software for personal publisher.
#    Copyright (C) 2004-2009 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

use strict;
use warnings;

use Lib::Logger;
use Lib::Error;
use Lib::User;
use Lib::App::Topic;
use Lib::App::Archive;
use Lib::App::Search;
use Lib::App::Conf;

my $logger = Lib::Logger->new();
my $error = Lib::Error->new();
my $user = Lib::User->new();
my $topic = Lib::App::Topic->new();
my $conf = Lib::App::Conf->new();

sub act() {
    my ($act, $cgi) = @_;
    my ($operation, $object) = split(/_/, $act);

    my $sessionid = $cgi->cookie('cookie59t');
    if ($sessionid eq '') {
        $error->print_error('00003e');
    }
    my $session_userid = $user->check_session($sessionid);
    if ($session_userid eq '') {
        $error->print_error('00003e');
    }

    my $topicid = $cgi->param('topicid');
    if ($topicid eq '') {
        $error->print_error('00003e');
    }
    $topic->load($topicid, 0);
    my $editor = $topic->get_editor();

    if (($editor ne $session_userid) && (!$user->is_admin($session_userid))) {
        $error->print_error('00004e');
    }

    if ($act eq 'input_ps') {
        _input_ps($operation, $cgi);
    }
    elsif ($act eq 'add_ps') {
        _add_ps($session_userid, $operation, $cgi);
    }
    elsif ($act eq 'edit_ps') {
        _edit_ps($operation, $cgi);
    }
    elsif ($act eq 'update_ps') {
        _update_ps($session_userid, $operation, $cgi);
    }
    elsif ($act eq 'confirm_ps') {
        _confirm_ps($operation, $cgi);
    }
    elsif ($act eq 'delete_ps') {
        _delete_ps($operation, $cgi);
    }
    else {
        $error->print_error('00002e');
    }
}

# 追記入力画面の表示
sub _input_ps() {
    my ($operation, $cgi) = @_;

    my $psdata;
    $psdata->{topicid} = $cgi->param('topicid');
    $psdata->{psid}    = '';
    $psdata->{title}   = '';
    $psdata->{text}    = '';

    # 画面表示
    _show_page($operation, $psdata, '');
}

# 追記の追加
sub _add_ps() {
    my ($session_userid, $operation, $cgi) = @_;

    my $psdata;
    $psdata->{topicid} = $cgi->param('topicid');
    $psdata->{psid}    = $cgi->param('psid');
    $psdata->{title}   = $cgi->param('title');
    $psdata->{text}    = $cgi->param('text');

    # データ読み込み
    $topic->load($psdata->{topicid}, 1);

    # 追記IDの設定
    my @pss = $topic->get_pss();
    my $psid = 0;
    if (scalar(@pss) > 0) {
        $psid = (split(/\,/, $pss[$#pss]))[0];
    }
    $psid++;

    # 入力チェック
    _check_input($operation, $psdata);

    my $dt = Lib::DateTime->new();
    $psdata->{lastdate} = $dt->local_datetime(0);
    $psdata->{editor}   = $session_userid;

    $psdata->{title} =~ s/\r?\n//g;
    $psdata->{text} =~ s/\r?\n/<br \/>enc_crlf/g;

    $psdata->{title} =~ s/,/enc_conma/g; 
    $psdata->{text} =~ s/,/enc_conma/g;

    my $ps = "$psid,$psdata->{lastdate},$psdata->{editor},$psdata->{title},$psdata->{text}";
    push @pss, $ps;
    $topic->set_pss(@pss);

    # データ更新
    $topic->update();

    # 更新情報データに追加
    my $archive = Lib::App::Archive->new();
    $archive->add($psdata->{topicid});

    # 検索用索引に追加
    my $str = "$psdata->{title} $psdata->{text} $psdata->{editor}";
    my $search = Lib::App::Search->new();
    $search->addindex(0, $psdata->{topicid}, $str);

    $topic->get_html($conf->get_static_html(), 0);

    my $nextpage = "./?act=view_topic&id=$psdata->{topicid}";
    print "Location: $nextpage", "\n\n";
}

# 追記編集画面の表示
sub _edit_ps() {
    my ($operation, $cgi) = @_;
    _show_ps($operation, $cgi);
}

# 追記データの更新
sub _update_ps() {
    my ($session_userid, $operation, $cgi) = @_;

    my $psdata;
    $psdata->{topicid} = $cgi->param('topicid');
    $psdata->{psid}    = $cgi->param('psid');
    $psdata->{title}   = $cgi->param('title');
    $psdata->{text}    = $cgi->param('text');

    if ($psdata->{psid} eq '') {
        $error->print_error('00002e');
    }

    my $dt = Lib::DateTime->new();
    $psdata->{lastdate} = $dt->local_datetime(0);
    $psdata->{editor}   = $session_userid;

    # データ読み込み
    $topic->load($psdata->{topicid}, 1);

    # 入力チェック
    _check_input($operation, $psdata);

    # 追記を置き換え
    my @pss = $topic->get_pss();
    my @new_pss = ();
    foreach my $ps (@pss) {
        my ($psid, $lastdate, $editor, $title, $text) = split(/\,/, $ps);
        if ($psid eq $psdata->{psid}) {
            $psdata->{title} =~ s/\r?\n//g;
            $psdata->{text} =~ s/\r?\n/enc_crlf/g;
            $psdata->{title} =~ s/,/enc_conma/g; 
            $psdata->{text} =~ s/,/enc_conma/g;
            $ps = "$psid,$psdata->{lastdate},$psdata->{editor},$psdata->{title},$psdata->{text}";
        }
        push @new_pss, $ps;
    }
    $topic->set_pss(@new_pss);
    $topic->update();

    $topic->get_html($conf->get_static_html(), 0);

    my $nextpage = "./?act=view_topic&id=$psdata->{topicid}";
    print "Location: $nextpage", "\n\n";
}

# 入力チェック
sub _check_input() {
    my ($operation, $psdata) = @_;

    if ($psdata->{title} eq '') {
        _show_page($operation, $psdata, 'タイトルが未入力です');
    }
    if ($psdata->{text} eq '') {
        _show_page($operation, $psdata, '追記が未入力です');
    }
}

# リンク削除画面の表示
sub _confirm_ps() {
    my ($operation, $cgi) = @_;
    _show_ps($operation, $cgi);
}

# 追記の削除
sub _delete_ps() {
    my ($operation, $cgi) = @_;

    my $psdata;
    $psdata->{topicid} = $cgi->param('topicid');
    $psdata->{psid}    = $cgi->param('psid');

    if ($psdata->{psid} eq '') {
        $error->print_error('00002e');
    }

    # データ読み込み
    $topic->load($psdata->{topicid}, 1);

    # 追記を削除
    my @pss = $topic->get_pss();
    my @new_pss = ();
    foreach my $ps (@pss) {
        my $psid = (split(/\,/, $ps))[0];
        next if ($psid eq $psdata->{psid});
        push @new_pss, $ps;
    }
    $topic->set_pss(@new_pss);
    $topic->update();

    $topic->get_html($conf->get_static_html(), 0);

    my $nextpage = "./?act=view_topic&id=$psdata->{topicid}";
    print "Location: $nextpage", "\n\n";
}

# 追記編集画面の表示
sub _show_ps() {
    my ($operation, $cgi) = @_;

    my $psdata;
    $psdata->{topicid} = $cgi->param('topicid');
    $psdata->{psid}    = $cgi->param('psid');
    $psdata->{title}   = '';
    $psdata->{text}    = '';
    if ($psdata->{psid} eq '') {
        $error->print_error('00002e');
    }

    # 編集対象の追記を取得
    $topic->load($psdata->{topicid}, 1);
    my @pss = $topic->get_pss();
    foreach my $ps (@pss) {
        my ($psid, $lastdate, $editor, $title, $text) = split(/\,/, $ps);
        if ($psid eq $psdata->{psid}) {
            $psdata->{title} = $title;
            $psdata->{text}  = $text;
            $psdata->{text} =~ s/enc_crlf/\n/g;
            $psdata->{title} =~ s/enc_conma/,/g; 
            $psdata->{text} =~ s/enc_conma/,/g;
            last;
        }
    }

    # 画面表示
    _show_page($operation, $psdata, '');
}

# 追記入力画面表示
sub _show_page() {
    my ($operation, $psdata, $msg) = @_;

    my $caption = '追記の追加';
    my $action = 'add_ps';
    my $button = '追記';
    my $guide = '強調文字の項目は入力必須です';
    my $readonly = '';
    if (($operation eq 'edit') || ($operation eq 'update')) {
        $caption = '追記の編集';
        $action = 'update_ps';
        $button = '編集';
        $readonly = '';
    }
    elsif (($operation eq 'confirm') || ($operation eq 'delete')) {
        $caption = '追記の削除';
        $action = 'delete_ps';
        $button = '削除';
        $readonly = 'readonly';
        $guide = '';
        $msg = "この追記を削除しますか？";
    }

    # パスの作成
    my $topictitle = $topic->get_title();
    my $path = <<"END_PATH";
<p><strong>
<a href="./?target=0" class="path">ホーム</a>&nbsp;
&gt;&nbsp;
<a href="./?act=view_topic&id=$psdata->{topicid}" class="path">$topictitle</a>
&gt;&nbsp;
$caption
</strong></p>
END_PATH

    $msg = "<p><strong><font class=\"message\" size=\"+1\">$msg</font></strong></p>";

    my $systemdir = $conf->get_system_dir();

    open my $templatefh, '<', "$systemdir/tmpl/ps.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$MSG\$/$msg/g;
    $template =~ s/\$GUIDE\$/$guide/g;
    $template =~ s/\$ACT\$/$action/g;
    $template =~ s/\$READONLY\$/$readonly/g;
    $template =~ s/\$BUTTON\$/$button/g;
    $template =~ s/\$TOPICID\$/$psdata->{topicid}/g;
    $template =~ s/\$PSID\$/$psdata->{psid}/g;
    $template =~ s/\$PSTITLE\$/$psdata->{title}/g;
    $template =~ s/\$TEXT\$/$psdata->{text}/g;

    print "Content-Type: text/html\n\n";
    print $template;

    exit;
}

1;
