#!/usr/bin/perl
#------------------------------------------------------------------------------
#    59Tracker, weblog software for personal publisher.
#    Copyright (C) 2004-2009 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

use strict;
use warnings;

use Lib::Logger;
use Lib::Error;
use Lib::User;
use Lib::App::Topic;
use Lib::App::Conf;

my $logger = Lib::Logger->new();
my $error = Lib::Error->new();
my $user = Lib::User->new();
my $topic = Lib::App::Topic->new();
my $conf = Lib::App::Conf->new();

sub act() {
    my ($act, $cgi) = @_;
    my ($operation, $object) = split(/_/, $act);

    my $sessionid = $cgi->cookie('cookie59t');
    if ($sessionid eq '') {
        $error->print_error('00003e');
    }
    my $session_userid = $user->check_session($sessionid);
    if ($session_userid eq '') {
        $error->print_error('00003e');
    }

    my $topicid = $cgi->param('topicid');
    if ($topicid eq '') {
        $error->print_error('00003e');
    }
    $topic->load($topicid, 0);
    my $editor = $topic->get_editor();

    if (($editor ne $session_userid) && (!$user->is_admin($session_userid))) {
        $error->print_error('00004e');
    }

    if ($act eq 'input_link') {
        _input_link($operation, $cgi);
    }
    elsif ($act eq 'add_link') {
        _add_link($operation, $cgi);
    }
    elsif ($act eq 'edit_link') {
        _edit_link($operation, $cgi);
    }
    elsif ($act eq 'update_link') {
        _update_link($operation, $cgi);
    }
    elsif ($act eq 'confirm_link') {
        _confirm_link($operation, $cgi);
    }
    elsif ($act eq 'delete_link') {
        _delete_link($operation, $cgi);
    }
    else {
        $error->print_error('00002e');
    }
}

# 関連リンク入力画面の表示
sub _input_link() {
    my ($operation, $cgi) = @_;

    my $linkdata;
    $linkdata->{topicid}     = $cgi->param('topicid');
    $linkdata->{linkid}      = '';
    $linkdata->{linktitle}   = '';
    $linkdata->{linkurl}     = '';
    $linkdata->{linksummary} = '';

    # 画面表示
    _show_page($operation, $linkdata, '');
}

# 関連リンクの追加
sub _add_link() {
    my ($operation, $cgi) = @_;

    my $linkdata;
    $linkdata->{topicid}     = $cgi->param('topicid');
    $linkdata->{linkid}      = $cgi->param('linkid');
    $linkdata->{linktitle}   = $cgi->param('linktitle');
    $linkdata->{linkurl}     = $cgi->param('linkurl');
    $linkdata->{linksummary} = $cgi->param('linksummary');

    # データ読み込み
    $topic->load($linkdata->{topicid}, 1);

    # 関連リンクIDの取得
    my @links = $topic->get_links();
    my $linkid = 0;
    if (scalar(@links) > 0) {
        $linkid = (split(/\,/, $links[$#links]))[0];
    }
    $linkid++;

    # 入力チェック
    _check_input($operation, $linkdata);

    $linkdata->{linktitle} =~ s/\r?\n//g;
    $linkdata->{linkurl} =~ s/\r?\n//g;
    $linkdata->{linksummary} =~ s/\r?\n/enc_crlf/g;

    $linkdata->{linktitle} =~ s/,/enc_conma/g; 
    $linkdata->{linkurl} =~ s/,/enc_conma/g;
    $linkdata->{linksummary} =~ s/,/enc_conma/g;

    my $link = "$linkid,$linkdata->{linktitle},$linkdata->{linkurl},$linkdata->{linksummary}";
    push @links, $link;
    $topic->set_links(@links);
    $topic->update();

    $topic->get_html($conf->get_static_html(), 0);

    my $nextpage = "./?act=view_topic&id=$linkdata->{topicid}";
    print "Location: $nextpage", "\n\n";
}

# 関連リンク編集画面の表示
sub _edit_link() {
    my ($operation, $cgi) = @_;
    _show_link($operation, $cgi);
}

# 関連リンクデータの更新
sub _update_link() {
    my ($operation, $cgi) = @_;

    my $linkdata;
    $linkdata->{topicid}     = $cgi->param('topicid');
    $linkdata->{linkid}      = $cgi->param('linkid');
    $linkdata->{linktitle}   = $cgi->param('linktitle');
    $linkdata->{linkurl}     = $cgi->param('linkurl');
    $linkdata->{linksummary} = $cgi->param('linksummary');

    if ($linkdata->{linkid} eq '') {
        $error->print_error('00002e');
    }

    # データ読み込み
    $topic->load($linkdata->{topicid}, 1);

    # 入力チェック
    _check_input($operation, $linkdata);

    # 関連リンクを置き換え
    my @links = $topic->get_links();
    my @new_links = ();
    foreach my $link (@links) {
        my ($linkid, $title, $url, $summary) = split(/\,/, $link);
        if ($linkid eq $linkdata->{linkid}) {

            $linkdata->{linktitle} =~ s/\r?\n//g;
            $linkdata->{linkurl} =~ s/\r?\n//g;
            $linkdata->{linksummary} =~ s/\r?\n/enc_crlf/g;

            $linkdata->{linktitle} =~ s/,/enc_conma/g; 
            $linkdata->{linkurl} =~ s/,/enc_conma/g;
            $linkdata->{linksummary} =~ s/,/enc_conma/g;

            $link = "$linkdata->{linkid},$linkdata->{linktitle},$linkdata->{linkurl},$linkdata->{linksummary}";
        }
        push @new_links, $link;
    }
    $topic->set_links(@new_links);
    $topic->update();

    $topic->get_html($conf->get_static_html(), 0);

    my $nextpage = "./?act=view_topic&id=$linkdata->{topicid}";
    print "Location: $nextpage", "\n\n";
}

# 入力チェック
sub _check_input() {
    my ($operation, $linkdata) = @_;

    if ($linkdata->{linktitle} eq '') {
        _show_page($operation, $linkdata, 'タイトルが未入力です');
    }
    if ($linkdata->{linkurl} eq '') {
        _show_page($operation, $linkdata, 'URLが未入力です');
    }
}

# リンク削除画面の表示
sub _confirm_link() {
    my ($operation, $cgi) = @_;
    _show_link($operation, $cgi);
}

# 関連リンクの削除
sub _delete_link() {
    my ($operation, $cgi) = @_;

    my $linkdata;
    $linkdata->{topicid} = $cgi->param('topicid');
    $linkdata->{linkid}  = $cgi->param('linkid');

    if ($linkdata->{linkid} eq '') {
        $error->print_error('00002e');
    }

    # データ読み込み
    $topic->load($linkdata->{topicid}, 1);

    # 関連リンクを削除
    my @links = $topic->get_links();
    my @new_links = ();
    foreach my $link (@links) {
        my $linkid = (split(/\,/, $link))[0];
        next if ($linkid eq $linkdata->{linkid});
        push @new_links, $link;
    }
    $topic->set_links(@new_links);
    $topic->update();

    $topic->get_html($conf->get_static_html(), 0);

    my $nextpage = "./?act=view_topic&id=$linkdata->{topicid}";
    print "Location: $nextpage", "\n\n";
}

# 関連リンク編集画面の表示
sub _show_link() {
    my ($operation, $cgi) = @_;

    my $linkdata;
    $linkdata->{topicid}     = $cgi->param('topicid');
    $linkdata->{linkid}      = $cgi->param('linkid');
    $linkdata->{linktitle}   = '';
    $linkdata->{linkurl}     = '';
    $linkdata->{linksummary} = '';
    if ($linkdata->{linkid} eq '') {
        $error->print_error('00002e');
    }

    # 編集対象の関連リンクを取得
    $topic->load($linkdata->{topicid}, 1);
    my @links = $topic->get_links();
    foreach my $link (@links) {
        my ($linkid, $title, $url, $summary) = split(/\,/, $link);
        if ($linkid eq $linkdata->{linkid}) {
            $linkdata->{linktitle}   = $title;
            $linkdata->{linkurl}     = $url;
            $linkdata->{linksummary} = $summary;
            $linkdata->{linksummary} =~ s/enc_crlf/\n/g;
            $linkdata->{linktitle} =~ s/enc_conma/,/g; 
            $linkdata->{linkurl} =~ s/enc_conma/,/g;
            $linkdata->{linksummary} =~ s/enc_conma/,/g;
            last;
        }
    }

    # 画面表示
    _show_page($operation, $linkdata, '');
}

# 関連リンク入力画面表示
sub _show_page() {
    my ($operation, $linkdata, $msg) = @_;

    my $caption = '関連リンクの追加';
    my $action = 'add_link';
    my $button = '追加';
    my $guide = '強調文字の項目は入力必須です';
    my $readonly = '';
    if (($operation eq 'edit') || ($operation eq 'update')) {
        $caption = '関連リンクの編集';
        $action = 'update_link';
        $button = '編集';
        $readonly = '';
    }
    elsif (($operation eq 'confirm') || ($operation eq 'delete')) {
        $caption = '関連リンクの削除';
        $action = 'delete_link';
        $button = '削除';
        $readonly = 'readonly';
        $guide = '';
        $msg = "この関連リンクを削除しますか？";
    }

    # パスの作成
    my $topictitle = $topic->get_title();
    my $path = <<"END_PATH";
<p><strong>
<a href="./?target=0" class="path">ホーム</a>&nbsp;
&gt;&nbsp;
<a href="./?act=view_topic&id=$linkdata->{topicid}" class="path">$topictitle</a>
&gt;&nbsp;
$caption
</strong></p>
END_PATH

    $msg = "<p><strong><font class=\"message\" size=\"+1\">$msg</font></strong></p>";

    my $systemdir = $conf->get_system_dir();

    open my $templatefh, '<', "$systemdir/tmpl/link.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$MSG\$/$msg/g;
    $template =~ s/\$GUIDE\$/$guide/g;
    $template =~ s/\$ACT\$/$action/g;
    $template =~ s/\$READONLY\$/$readonly/g;
    $template =~ s/\$BUTTON\$/$button/g;
    $template =~ s/\$TOPICID\$/$linkdata->{topicid}/g;
    $template =~ s/\$LINKID\$/$linkdata->{linkid}/g;
    $template =~ s/\$LINKTITLE\$/$linkdata->{linktitle}/g;
    $template =~ s/\$LINKURL\$/$linkdata->{linkurl}/g;
    $template =~ s/\$LINKSUMMARY\$/$linkdata->{linksummary}/g;

    print "Content-Type: text/html\n\n";
    print $template;

    exit;
}

1;
