#!/usr/bin/perl
#------------------------------------------------------------------------------
#    59Tracker, weblog software for personal publisher.
#    Copyright (C) 2004-2009 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

use strict;
use warnings;

use Lib::Logger;
use Lib::Error;
use Lib::User;
use Lib::String;
use Lib::App::Conf;
use Lib::App::Bookmark;
use Lib::App::Keyword;
use Lib::App::Search;
use Lib::App::Ads;

my $logger = Lib::Logger->new();
my $error = Lib::Error->new();
my $user = Lib::User->new();
my $conf = Lib::App::Conf->new();
my $bookmark = Lib::App::Bookmark->new();

sub act() {
    my ($act, $cgi) = @_;
    my ($operation, $object) = split(/_/, $act);

    my $session_userid = '';
    my $sessionid = $cgi->cookie('cookie59t');
    if ($sessionid ne '') {
        $session_userid = $user->check_session($sessionid);
    }
    if (($session_userid eq '') && (($act ne 'view_bookmark') &&  ($act ne 'jump_bookmark'))) {
        $error->print_error('00003e');
    }

    if ($act eq 'view_bookmark') {
        _view_bookmark($session_userid, $cgi);
    }
    elsif ($act eq 'jump_bookmark') {
        _jump_bookmark($cgi);
    }
    elsif ($act eq 'input_bookmark') {
        _input_bookmark($operation, $session_userid, $cgi);
    }
    elsif ($act eq 'add_bookmark') {
        _add_bookmark($operation, $session_userid, $cgi);
    }
    elsif ($act eq 'edit_bookmark') {
        _edit_bookmark($operation, $session_userid, $cgi);
    }
    elsif ($act eq 'update_bookmark') {
        _update_bookmark($operation, $session_userid, $cgi);
    }
    elsif ($act eq 'confirm_bookmark') {
        _confirm_bookmark($operation, $session_userid, $cgi);
    }
    elsif ($act eq 'delete_bookmark') {
        _delete_bookmark($operation, $session_userid, $cgi);
    }
    else {
        $error->print_error('00002e');
    }
}

# ブックマークの表示
sub _view_bookmark() {
    my ($session_userid, $cgi) = @_;

    my $bookmarkid = $cgi->param('id');
    my $html = _bookmark_html($session_userid, $bookmarkid);

    print "Content-Type: text/html\n\n";
    print $html;
}

# ブックマークページ表示用htmlソースの作成
sub _bookmark_html {
    my ($session_userid, $bookmarkid) = @_;

    my $systemdir = $conf->get_system_dir();
    my $docrootdir = $conf->get_docroot_dir();
    my $sitetitle = $conf->get_site_title();
    my $subtitle = $conf->get_sub_title();
    my $bookmarktarget = $conf->get_bookmark_target();

    $bookmark->load($bookmarkid, 1);

    my @userlist = split(/:/, $bookmark->get_registrant());
    my $userid;
    if (@userlist > 1) {
        $userid = $userlist[0];
    }
    else {
        $userid = $bookmark->get_registrant();
    }

    my $summary = $bookmark->get_summary();
    my $keywords = $bookmark->get_keyword();
    my $datetime = $bookmark->get_datetime();
    my @comments = $bookmark->get_comments();
    my $commentnum = scalar(@comments);
    my $point = $bookmark->get_point();
    my $bookmarktitle = $bookmark->get_title();

    $summary =~ s/\n/<br \/>/g;
    $summary =~ s/(http:\/\/[^()<>\[\]{}"'\x7F-\xFF\x00-\x20]+)/<a href="$1" $bookmarktarget>$1<\/a>/g;

    my $keyword = Lib::App::Keyword->new();
    $summary = $keyword->link_keyword($keywords, $summary, 1);

    my $caption = "$bookmarktitle - $sitetitle";
    my $title = "<a href=\"./?act=jump_bookmark&bookmarkid=$bookmarkid\" $bookmarktarget>$bookmarktitle</a>";

    my $information = <<"END_INFORMATION";
<b>[$bookmarkid]</b> registered by $userid at <font class="datetime">$datetime</font> |
 <a href="./?act=view_bookmark&id=$bookmarkid#comment">Comment ($commentnum)</a> |
 <b><font class="point" size="+1">$point</font> point</b>
END_INFORMATION

    $sitetitle = "<a class=\"sitetitle\" href=\"$docrootdir\">$sitetitle</a>";
    my $rssurl = 'rss.xml';

    my $login = "<a href=\"./?act=input_register\">ユーザー登録</a>&nbsp;&nbsp;<a href=\"./?act=login_session\">ログイン</a>";
    if ($session_userid ne '') {
        $login = "<a href=\"./?act=logout_session\">ログアウト</a>";
    }

    my $menulist = <<"END_MENU";
<a href="$rssurl"><img src="feed-icon-16x16.jpg" border="0"></a> <a href="$rssurl">更新情報の購読</a>&nbsp;&nbsp;
$login
END_MENU
    if ($session_userid ne '') {
        if ($userid eq $session_userid || $user->is_admin($session_userid)) {
            $menulist .= "<br />";
            $menulist .= "<a href=\"./?act=edit_bookmark&bookmarkid=$bookmarkid\">編集</a>&nbsp;&nbsp;";
            $menulist .= "<a href=\"./?act=confirm_bookmark&bookmarkid=$bookmarkid\">削除</a>";
        }
    }

    my @keyworditems = split(/ /, $keywords);
    my $keywordlist = $keyword->list(1, @keyworditems);

    $user->load($session_userid);
    my $hpurl = $user->get_hpurl();

    # ユーザー広告の取得
    $user->load($userid);
    my $ad1 = $user->get_ad1();
    my $ad2 = $user->get_ad2();
    my $ad3 = $user->get_ad3();
    my $blogparts = $user->get_parts();

    # デフォルト広告の取得
    my $ads = Lib::App::Ads->new();
    if ($ad1 eq '' && $ad2 eq '' && $ad3 eq '') {
        $ad1 = $ads->get_ad1();
        $ad2 = $ads->get_ad2();
        $ad3 = $ads->get_ad3();
    }
    my $adlist = $ads->get_adlist_table();

    my $path = <<"END_PATH";
<strong>
<a href="./?target=1" class="path">ホーム</a>&nbsp;
&gt;&nbsp;
$bookmarktitle
</strong>
END_PATH

    # コメント一覧を取得
    my $commentlist = $bookmark->comment_list($bookmarkid, $session_userid);

    my $commentform = <<"END_FORM";
<div id="toggleHide" style="display:block">
  <form name="hideform" method="post" action="./">
  <input type="hidden" name="act" value="add_bmcomment">
  <input type="hidden" name="bookmarkid" value="$bookmarkid">
  <a href="javascript:;" onmousedown="toggleInput('toggleShow','toggleHide');">＋</a>&nbsp;をクリックすると名前、メールアドレス、ホームページURLの入力フィールドが表示されます
  <br /><br />
  <input type="hidden" name="author" value="$session_userid">
  <input type="hidden" name="mailaddr" value="">
  <input type="hidden" name="hpurl" value="$hpurl">
  <input type="checkbox" name="evaluate" value="1"><strong>おすすめ</strong><br>
  <b>コメント：</b><br />
  <textarea name="text" rows="10" cols="60"></textarea><br />
  <input type="submit" value="投稿">
  </form>
</div>
<div id="toggleShow" style="display:none">
  <form name="showform" method="post" action="./">
  <input type="hidden" name="act" value="add_bmcomment">
  <input type="hidden" name="bookmarkid" value="$bookmarkid">
  <a href="javascript:;" onmousedown="toggleInput('toggleShow','toggleHide');">－</a>&nbsp;をクリックすると名前、メールアドレス、ホームページURLの入力フィールドが非表示になります
  <br /><br />
  名前：<br />
  <input type="text" name="author" size="60" value="$session_userid"><br />
  メールアドレス：<br />
  <input type="text" name="mailaddr" size="60"><br>
  ホームページURL：<br />
  <input type="text" name="hpurl" size="60" value="$hpurl"><br>
  <input type="checkbox" name="evaluate" value="1"><strong>おすすめ</strong><br />
  <b>コメント：<br />
  <textarea name="text" rows="10" cols="60"></textarea><br>
  <input type="submit" value="投稿">
  </form>
</div>
END_FORM

    my $bookmarknum = $conf->get_bookmark_num();
    my $newbookmarklist = $bookmark->new_bookmark_list($bookmarknum);
    my $usedbookmarklist = $bookmark->used_bookmark_list($bookmarknum);

    my $searchbox = _search_box();
    $searchbox =~ s/\$PREVQ\$//g;

    open my $templatefh, '<', "$systemdir/tmpl/summary.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$RSSURL\$/$rssurl/g;
    $template =~ s/\$DOCROOT\$/$docrootdir/g;
    $template =~ s/\$SITETITLE\$/$sitetitle/g;
    $template =~ s/\$SUBTITLE\$/$subtitle/g;
    $template =~ s/\$MENULIST\$/$menulist/g;
    $template =~ s/\$TITLE\$/$title/g;
    $template =~ s/\$SUMMARY\$/$summary/g;
    $template =~ s/\$INFORMATION\$/$information/g;
    $template =~ s/\$COMMENTLIST\$/$commentlist/g;
    $template =~ s/\$COMMENTFORM\$/$commentform/g;
    $template =~ s/\$SEARCHBOX\$/$searchbox/g;
    $template =~ s/\$KEYWORDLIST\$/$keywordlist/g;
    $template =~ s/\$AD1\$/$ad1/g;
    $template =~ s/\$AD2\$/$ad2/g;
    $template =~ s/\$AD3\$/$ad3/g;
    $template =~ s/\$ADLIST\$/$adlist/g;
    $template =~ s/\$NEWBOOKMARKLIST\$/$newbookmarklist/g;
    $template =~ s/\$USEDBOOKMARKLIST\$/$usedbookmarklist/g;
    $template =~ s/\$BLOGPARTS\$/$blogparts/g;

    return $template;
}

# ブックマークページ用検索ボックスの作成
sub _search_box {

    my $form = <<"END_FORM";
<div class="searchbox">
<form method="post" action="./">
  <input type="radio" name="target" value="0">トピック&nbsp;&nbsp;
  <input type="radio" name="target" value="1" checked>ブックマーク<br />
  <input type="text" name="q" size="60" value="\$PREVQ\$">&nbsp;
  <input type="submit" value="検索">
</form>
</div>
END_FORM

    return $form;
}

# ブックマークページへジャンプ
sub _jump_bookmark() {
    my ($cgi) = @_;

    my $bookmarkid = $cgi->param('bookmarkid');
    my $ret = $bookmark->load($bookmarkid, 1);
    if (!$ret) {
        $error->print_error('00002e');
    }

    my $url = $bookmark->get_url();
    my $point = $bookmark->get_point();
    $point++;
    $bookmark->set_point($point);
    $bookmark->update();

    # 最近使用されたブックマークを更新
    $bookmark->update_used_bookmark($bookmarkid, $point);

    # ジャンプ
    print "Status: 301 Moved Permanently\n";
    print "Location: $url", "\n\n";
}

# ブックマーク情報の入力
sub _input_bookmark() {
    my ($operation, $session_userid, $cgi) = @_;

    my $url = $cgi->param('url');
    my $title = '';
    my $blockquote = '';
    if ($url ne '') {
        use Lib::Util;
        my $util = Lib::Util->new();
        ($title, $blockquote) = $util->get_blockquote($url);
    }

    my $dt = Lib::DateTime->new();
    my $bookmarkdata;
    $bookmarkdata->{bookmarkid} = 0;
    $bookmarkdata->{title}      = $title;
    $bookmarkdata->{url}        = $url;
    $bookmarkdata->{summary}    = $blockquote;
    $bookmarkdata->{registrant} = $session_userid;
    $bookmarkdata->{datetime}   = $dt->local_datetime(0);
    $bookmarkdata->{keyword}    = '';
    $bookmarkdata->{point}      = 0;

    # 入力画面の表示
    _show_page($operation, $bookmarkdata, '');
}

# ブックマークの追加
sub _add_bookmark() {
    my ($operation, $session_userid, $cgi) = @_;

    my $bookmarkid = $cgi->param('bookmarkid');

    my $bookmarkdata;
    $bookmarkdata->{bookmarkid} = $bookmarkid;
    $bookmarkdata->{title}      = $cgi->param('title');
    $bookmarkdata->{url}        = $cgi->param('url');
    $bookmarkdata->{summary}    = $cgi->param('summary');
    $bookmarkdata->{registrant} = $cgi->param('registrant');
    $bookmarkdata->{datetime}   = $cgi->param('datetime');
    $bookmarkdata->{keyword}    = $cgi->param('keyword');
    $bookmarkdata->{point}      = $cgi->param('point');

    $bookmarkdata->{keyword} =~ s/\A[ ]+//g;
    $bookmarkdata->{keyword} =~ s/[ ]+\z//g;
    $bookmarkdata->{keyword} =~ s/[ ]+/ /g;

    # 入力チェック
    _check_input($operation, $bookmarkdata);

    my $string = Lib::String->new();
    $bookmarkdata->{summary} = $string->tag_invalidate($bookmarkdata->{summary});

    _add_boomark_data($session_userid, $bookmarkdata);

    print "Location: $bookmarkdata->{url}", "\n\n";
}

# ブックマークデータを追加
sub _add_boomark_data {
    my ($session_userid, $bookmarkdata) = @_;

    my $result = 0;
    my $hit = 0;
    my $counter = $bookmark->get_bookmark_counter();
    my $ipaddr = $ENV{'REMOTE_ADDR'};

    # ブックマークデータを新しい順にチェック
    for (my $i = 1; $i <= $counter; $i++) {
        my $bookmarkid = $i;
        $bookmark->load($bookmarkid, 1);

        # 既存データの取得
        my $url = $bookmark->get_url();
        my $registrant = $bookmark->get_registrant();
        my $point = $bookmark->get_point();

        if ($url eq $bookmarkdata->{url}) {
            my @users = split(/:/, $registrant);
            my $registered = 0;
            foreach my $user (@users) {
                if ($session_userid eq $user) {
                    $registered = 1;
                }
            }
            if (!$registered) {

                # 他のユーザーが登録済みの場合はコメントに追加
                $bookmark->add_comment($bookmarkdata->{summary}, $bookmarkdata->{registrant}, '', '', 1, $ipaddr);

                $registrant .= ":$bookmarkdata->{registrant}";
                $point += $conf->get_good_point();
                # ブックマークデータ更新
                $bookmark->set_registrant($registrant);
                $bookmark->set_point($point);

                $bookmark->update();
                $result = 2;
            }
            $hit = 1;
            last;
        }
    }

    # 未登録に場合はデータ追加
    if (!$hit) {
        my $bookmarkid = $counter + 1;
        $bookmark->update_bookmark_counter($bookmarkid);

        my $str = "$bookmarkdata->{title} $bookmarkdata->{summary} $bookmarkdata->{registrant} $bookmarkdata->{keyword}";

        $bookmark->set_bookmarkid($bookmarkid);
        $bookmark->set_title($bookmarkdata->{title});
        $bookmark->set_url($bookmarkdata->{url});
        $bookmark->set_summary($bookmarkdata->{summary});
        $bookmark->set_registrant($bookmarkdata->{registrant});
        $bookmark->set_datetime($bookmarkdata->{datetime});
        $bookmark->set_keyword($bookmarkdata->{keyword});
        $bookmark->set_point($conf->get_good_point());
        my @comments = ();
        $bookmark->set_comments(@comments);

        # ブックマークデータ更新
        $bookmark->update();

        # キーワードデータに追加
        my $keywordobj = Lib::App::Keyword->new();
        my @addarray = split(/ /, $bookmarkdata->{keyword});
        foreach my $item (@addarray) {
            if ($item ne '') {
                $keywordobj->add(1, $item, $bookmarkid);  # 追加
            }
        }

        # RSSを更新
        my $archive = Lib::App::Archive->new();
        $archive->outputrss2();

        # 検索用インデックスに追加
        my $search = Lib::App::Search->new();
        $search->addindex(1, $bookmarkid, $str);

        $result = 2;
    }

    return $result;
}

# ブックマークの編集
sub _edit_bookmark() {
    my ($operation, $session_userid, $cgi) = @_;

    my $bookmarkid = $cgi->param('bookmarkid');
    $bookmark->load($bookmarkid, 0);

    my $bookmarkdata;
    $bookmarkdata->{bookmarkid} = $bookmarkid;
    $bookmarkdata->{title}      = $bookmark->get_title();
    $bookmarkdata->{url}        = $bookmark->get_url();
    $bookmarkdata->{summary}    = $bookmark->get_summary();
    $bookmarkdata->{registrant} = $bookmark->get_registrant();
    $bookmarkdata->{datetime}   = $bookmark->get_datetime();
    $bookmarkdata->{keyword}    = $bookmark->get_keyword();
    $bookmarkdata->{point}      = $bookmark->get_point();

    # 入力画面の表示
    _show_page($operation, $bookmarkdata, '');
}

# ブックマークの更新
sub _update_bookmark() {
    my ($operation, $session_userid, $cgi) = @_;

    my $bookmarkid = $cgi->param('bookmarkid');
    $bookmark->load($bookmarkid, 1);

#    $logger->write("_update_bookmark - bookmarkid:$bookmarkid");   # for debug

    my $oldkeyword = $bookmark->get_keyword();

    my $bookmarkdata;
    $bookmarkdata->{bookmarkid} = $bookmarkid;
    $bookmarkdata->{title}      = $cgi->param('title');
    $bookmarkdata->{url}        = $cgi->param('url');
    $bookmarkdata->{summary}    = $cgi->param('summary');
    $bookmarkdata->{registrant} = $cgi->param('registrant');
    $bookmarkdata->{datetime}   = $cgi->param('datetime');
    $bookmarkdata->{keyword}    = $cgi->param('keyword');
    $bookmarkdata->{point}      = $cgi->param('point');

    $bookmarkdata->{keyword} =~ s/\A[ ]+//g;
    $bookmarkdata->{keyword} =~ s/[ ]+\z//g;
    $bookmarkdata->{keyword} =~ s/[ ]+/ /g;

    # 入力チェック
    _check_input($operation, $bookmarkdata);

    my $string = Lib::String->new();
    $bookmarkdata->{summary} = $string->tag_invalidate($bookmarkdata->{summary});

    $bookmark->set_bookmarkid($bookmarkdata->{bookmarkid});
    $bookmark->set_title($bookmarkdata->{title});
    $bookmark->set_url($bookmarkdata->{url});
    $bookmark->set_summary($bookmarkdata->{summary});
    $bookmark->set_registrant($bookmarkdata->{registrant});
    $bookmark->set_datetime($bookmarkdata->{datetime});
    $bookmark->set_keyword($bookmarkdata->{keyword});
    $bookmark->set_point($bookmarkdata->{point});

    $bookmark->update();

    # 入力されたキーワードと編集前のキーワードが違っていた場合
    if ($bookmarkdata->{keyword} ne $oldkeyword) {
        my @addarray = ();
        my @delarray = ();
        my @newarray = split(/ /, $bookmarkdata->{keyword});
        my @oldarray = split(/ /, $oldkeyword);
        foreach my $new (@newarray) {
            my $found = 0;
            foreach my $old (@oldarray) {
                if ($new eq $old) {
                    $found = 1;
                    last;
                }
            }
            if (!$found) {
                push @addarray, $new;
            }
        }

        foreach my $old (@oldarray) {
            my $found = 0;
            foreach my $new (@newarray) {
                if ($old eq $new) {
                    $found = 1;
                    last;
                }
            }
            if (!$found) {
                push @delarray, $old;
            }
        }

        my $keywordobj = Lib::App::Keyword->new();
        foreach my $item (@addarray) {
            $keywordobj->add(1, $item, $bookmarkid);  # 追加
        }
        foreach my $item (@delarray) {
            $keywordobj->delete(1, $item, $bookmarkid); # 削除
        }
    }

    print "Location: ./?act=view_bookmark&id=$bookmarkid", "\n\n";
}

# 入力チェック
sub _check_input() {
    my ($operation, $bookmarkdata) = @_;

    if ($bookmarkdata->{title} eq '') {
        _show_page($operation, $bookmarkdata, 'タイトルが未入力です');
    }
    if ($bookmarkdata->{url} eq '') {
        _show_page($operation, $bookmarkdata, 'URLが未入力です');
    }
    if ($bookmarkdata->{summary} =~ m/<a.+>.+?<\/a>/i) {
        _show_page($operation, $bookmarkdata, '概要にアンカータグを使用することはできません');
    }

    # キーワードのチェック
    my $keywordobj = Lib::App::Keyword->new();
    if ($keywordobj->check($bookmarkdata->{keyword})) {
        _show_page($operation, $bookmarkdata, 'キーワードは半角英数または全角文字で入力してください');
    }
}

# ブックマーク入力画面の表示
sub _show_page() {
    my ($operation, $bookmarkdata, $msg) = @_;

    $msg = "<p><strong><font size=\"+1\">$msg</font></strong></p>";

    my $caption = 'ブックマークの追加';
    my $action = 'add_bookmark';
    my $readonly = '';
    my $button = '追加';
    if (($operation eq 'edit') || ($operation eq 'update')) {
        $caption = 'ブックマークの編集';
        $action = 'update_bookmark';
        $readonly = 'readonly';
        $button = '更新';
    }

    my $path = <<"END_PATH";
<p><strong>
<a href="./?target=1" class="path">ホーム</a>&nbsp;
&gt;&nbsp;
$caption
</strong></p>
END_PATH

    my $systemdir = $conf->get_system_dir();

    open my $templatefh, '<', "$systemdir/tmpl/bookmark.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$MSG\$/$msg/g;
    $template =~ s/\$ACT\$/$action/g;
    $template =~ s/\$READONLY\$/$readonly/g;
    $template =~ s/\$BUTTON\$/$button/g;
    $template =~ s/\$BOOKMARKID\$/$bookmarkdata->{bookmarkid}/g;
    $template =~ s/\$BOOKMARKTITLE\$/$bookmarkdata->{title}/g;
    $template =~ s/\$URL\$/$bookmarkdata->{url}/g;
    $template =~ s/\$KEYWORD\$/$bookmarkdata->{keyword}/g;
    $template =~ s/\$REGISTRANT\$/$bookmarkdata->{registrant}/g;
    $template =~ s/\$DATETIME\$/$bookmarkdata->{datetime}/g;
    $template =~ s/\$POINT\$/$bookmarkdata->{point}/g;
    $template =~ s/\$SUMMARY\$/$bookmarkdata->{summary}/g;

    print "Content-Type: text/html\n\n";
    print $template;

    exit;
}

# ブックマーク削除画面の表示
sub _confirm_bookmark() {
    my ($operation, $session_userid, $cgi) = @_;

    my $bookmarkdata;
    $bookmarkdata->{bookmarkid} = $cgi->param('bookmarkid');
    if ($bookmarkdata->{bookmarkid} eq '') {
        $error->print_error('00002e');
    }

    # 削除対象のコメントを取得
    $bookmark->load($bookmarkdata->{bookmarkid}, 0);

    $bookmarkdata->{title}      = $bookmark->get_title();
    $bookmarkdata->{url}        = $bookmark->get_url();
    $bookmarkdata->{summary}    = $bookmark->get_summary();
    $bookmarkdata->{registrant} = $bookmark->get_registrant();
    $bookmarkdata->{datetime}   = $bookmark->get_datetime();
    $bookmarkdata->{keyword}    = $bookmark->get_keyword();
    $bookmarkdata->{point}      = $bookmark->get_point();

    # 登録ユーザーチェック
    my @users = split(/:/, $bookmarkdata->{registrant});
    my $hit = 0;
    foreach my $user (@users) {
        if ($session_userid eq $user) {
            $hit = 1;
            last;
        }
    }
    if (!$hit) {
        $error->print_error('00004e');
    }

    _show_confirm($bookmarkdata, 'このブックマークを削除しますか？（共有されている場合は登録ユーザーリストから削除されます）');
}

# ブックマークの削除
sub _delete_bookmark() {
    my ($operation, $session_userid, $cgi) = @_;

    my $bookmarkdata;
    $bookmarkdata->{bookmarkid} = $cgi->param('bookmarkid');

    if ($bookmarkdata->{bookmarkid} eq '') {
        $error->print_error('00002e');
    }

    # データ読み込み
    $bookmark->load($bookmarkdata->{bookmarkid}, 1);
    $bookmarkdata->{registrant} = $bookmark->get_registrant();

    # 他に登録ユーザーがいない場合はデータファイルを削除
    if ($bookmarkdata->{registrant} eq $session_userid) {
        $bookmark->delete_bookmark($bookmarkdata->{bookmarkid});
    }
    else {
        # 登録ユーザーから削除
        my @users = split(/:/, $bookmarkdata->{registrant});
        my $new_users = '';
        foreach my $user (@users) {
            if ($session_userid ne $user) {
                if ($new_users ne '') {
                    $new_users .= ':';
                }
                $new_users .= $user;
            }
        }
        $bookmark->set_registrant($new_users);

        my $point = $bookmark->get_point();
        $point -= $conf->get_good_point();
        $bookmark->set_point($point);

        $bookmark->update();
    }

    print "Location: ./?target=1", "\n\n";
}

# 削除確認画面の表示
sub _show_confirm() {
    my ($bookmarkdata, $msg) = @_;

    my $caption = 'ブックマーク削除の確認';
    my $action = 'delete_bookmark';

    # パスの作成
    my $bookmarktitle = $bookmarkdata->{title};
    my $path = <<"END_PATH";
<p><strong>
<a href="./?target=1" class="path">ホーム</a>&nbsp;
&gt;&nbsp;
<a href="./?act=view_bookmark&id=$bookmarkdata->{bookmarkid}" class="path">$bookmarktitle</a>
&gt;&nbsp;
$caption
</strong></p>
END_PATH

    $msg = "<p><strong><font size=\"+1\">$msg</font></strong></p>";

    # 確認内容を作成
    $bookmarkdata->{registrant} =~ s/:/,/g;
    $bookmarkdata->{summary} =~ s/\n/<br \/>/g;
    my $content = <<"END_CONTENT";
<div>
  <strong><a href="$bookmarkdata->{url}" target="_blank">$bookmarkdata->{title}</a></strong>
</div>
<div>
  $bookmarkdata->{summary}
</div>
<div>
  <strong>登録日時：</strong>$bookmarkdata->{datetime}&nbsp;<strong>登録ユーザー：</strong>$bookmarkdata->{registrant}
</div>
END_CONTENT

    my $systemdir = $conf->get_system_dir();

    open my $templatefh, '<', "$systemdir/tmpl/confirm.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$MSG\$/$msg/g;
    $template =~ s/\$ACT\$/$action/g;
    $template =~ s/\$CONTENT\$/$content/g;
    $template =~ s/\$TOPICID\$//g;
    $template =~ s/\$BOOKMARKID\$/$bookmarkdata->{bookmarkid}/g;
    $template =~ s/\$ID\$//g;

    print "Content-Type: text/html\n\n";
    print $template;

    exit;
}

1;
