#!/usr/bin/perl
#------------------------------------------------------------------------------
#    59Tracker, weblog software for personal publisher.
#    Copyright (C) 2004-2009 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

use strict;
use warnings;

use Lib::Logger;
use Lib::Error;
use Lib::User;
use Lib::DateTime;
use Lib::String;
use Lib::Filter;
use Lib::Mail;
use Lib::App::Conf;
use Lib::App::Bookmark;

my $logger = Lib::Logger->new();
my $error = Lib::Error->new();
my $conf = Lib::App::Conf->new();
my $bookmark = Lib::App::Bookmark->new();

sub act() {
    my ($act, $cgi) = @_;
    my ($operation, $object) = split(/_/, $act);

    my $bookmarkid = $cgi->param('bookmarkid');
    if ($bookmarkid eq '') {
        $error->print_error('00002e');
    }
    $bookmark->load($bookmarkid, 0);

    if ($act ne 'add_bmcomment') {

        my $user = Lib::User->new();
        my $sessionid = $cgi->cookie('cookie59t');
        if ($sessionid eq '') {
            $error->print_error('00003e');
        }
        my $session_userid = $user->check_session($sessionid);
        if ($session_userid eq '') {
            $error->print_error('00003e');
        }

        my $owner = $bookmark->get_owner();
        if (($owner ne $session_userid) && (!$user->is_admin($session_userid))) {
            $error->print_error('00004e');
        }
    }

    if ($act eq 'add_bmcomment') {
        _add_comment($cgi);
    }
    elsif ($act eq 'confirm_bmcomment') {
        _confirm_comment($cgi);
    }
    elsif ($act eq 'delete_bmcomment') {
        _delete_comment($cgi);
    }
    else {
        $error->print_error('00002e');
    }
}

# コメントの追加
sub _add_comment() {
    my ($cgi) = @_;

    my $string = Lib::String->new();
    my $dt = Lib::DateTime->new();

    my $commentdata;
    $commentdata->{bookmarkid} = $cgi->param('bookmarkid');
    $commentdata->{status}   = 1;
    $commentdata->{postdate} = $dt->local_datetime(0);
    $commentdata->{author}   = $cgi->param('author');
    $commentdata->{mailaddr} = $cgi->param('mailaddr');
    $commentdata->{hpurl}    = $cgi->param('hpurl');
    $commentdata->{text}     = $cgi->param('text');
    $commentdata->{evaluate} = $cgi->param('evaluate');
    $commentdata->{ipaddr}   = $ENV{'REMOTE_ADDR'};
    if ($commentdata->{evaluate} eq '') {
        $commentdata->{evaluate} = 0;
    }

    # 入力チェック
    _check_input($commentdata);

    # データ読み込み
    $bookmark->load($commentdata->{bookmarkid}, 1);

    $bookmark->add_comment($commentdata->{text}, $commentdata->{author}, $commentdata->{mailaddr}, $commentdata->{hpurl}, $commentdata->{evaluate}, $commentdata->{ipaddr});

    my $point = $bookmark->get_point() ;
    if ($commentdata->{evaluate} > 0) {
        $point += $conf->get_good_point();
    }
    else {
        $point += $conf->get_normal_point();
    }
    $bookmark->set_point($point);

    # データ更新
    $bookmark->update();

    my $registrant = $bookmark->get_registrant($point);
    my @users = split(/:/, $registrant);

    my $bookmarkurl = $conf->get_docroot_dir() . "/?act=view_bookmark&id=$commentdata->{bookmarkid}";
    my $bookmarktitle = $bookmark->get_title();

    $commentdata->{author} =~ s/enc_conma/,/g;
    $commentdata->{mailaddr} =~ s/enc_conma/,/g;
    $commentdata->{hpurl} =~ s/enc_conma/,/g;
    $commentdata->{text} =~ s/enc_conma/,/g;
    $commentdata->{text} =~ s/enc_crlf//g;

    my $subject = "[$bookmarktitle] にコメントがつきました";

    my $body = <<"END_BODY";
[$bookmarktitle] にコメントがつきました。
$bookmarkurl

----------

IPアドレス  ：$commentdata->{ipaddr}
投稿者      ：$commentdata->{author}
メール      ：$commentdata->{mailaddr}
ホームページ：$commentdata->{hpurl}
コメント：
$commentdata->{text}

----------
END_BODY

    my $mail = Lib::Mail->new();
    my $mailaddr = '';

    # ブックマーク登録者へのメール通知
    if (scalar(@users) > 0) {
        my $user = Lib::User->new();
        foreach my $userid (@users) {
            $user->load($userid);
            $mailaddr = $user->get_mailaddr();
            $mail->send($subject, $body, $mailaddr);
        }
    }

    # サイト管理者へのメール通知
    my $adminaddr = $conf->get_admin_address();
    if ($conf->get_notify_mail() && $adminaddr ne $mailaddr) {
        $mail->send($subject, $body, $adminaddr);
    }

    print "Location: $bookmarkurl", "\n\n";
}

# 入力チェック
sub _check_input() {
    my ($commentdata) = @_;

    my $filter = Lib::Filter->new();
    my $docrootdir = $conf->get_docroot_dir();

    my $referer = $ENV{'HTTP_REFERER'};# 呼び出し元を取得
    if ($referer !~ /$docrootdir/) {
        $error->print_error('10001e');
    }

    if ($filter->check_filter($commentdata->{ipaddr})) {
        $error->print_error('10001e');
    }

    # 名前
    if ($commentdata->{author} ne '') {
        $commentdata->{author} =~ s/[\r\n]//g;
        if ($filter->check_filter($commentdata->{author})) {
            $error->print_error('10001e');
        }
    }

    # メールアドレス
    if ($commentdata->{mailaddr} ne '') {
        $commentdata->{mailaddr} =~ s/[\r\n]//g;
        if ($commentdata->{mailaddr} !~ /.+\@.+[.].+/) {
            $error->print_error('10002e');
        }
        if ($filter->check_filter($commentdata->{mailaddr})) {
            $error->print_error('10001e');
        }
    }

    # ホームページURL
    if ($commentdata->{hpurl} ne '') {
        $commentdata->{hpurl} =~ s/[\r\n]//g;
        if ($commentdata->{hpurl} !~ /(http:\/\/[^()<>\[\]{}"'\x7F-\xFF\x00-\x20]+)/) {
            $error->print_error('10003e');
        }
        if ($filter->check_filter($commentdata->{hpurl})) {
            $error->print_error('10001e');
        }
    }

    # コメント
    if ($commentdata->{text} eq '') {
        $error->print_error('10000e');
    }
    else {
        if ($commentdata->{text} =~ /\A[\x20-\x7E\r\n]+\z/) {
            $error->print_error('10001e');
        }
        if ($commentdata->{text} =~ /<[aA] .+>.*<\/[aA]>/) {
            $error->print_error('10001e');
        }
        if ($filter->check_filter($commentdata->{text})) {
            $error->print_error('10001e');
        }
    }
}

# コメント削除画面の表示
sub _confirm_comment() {
    my ($cgi) = @_;

    my $commentdata;
    $commentdata->{bookmarkid} = $cgi->param('bookmarkid');
    $commentdata->{commentid}  = $cgi->param('id');
    $commentdata->{status}     = 0;
    $commentdata->{postdate}   = '';
    $commentdata->{author}     = '';
    $commentdata->{mailaddr}   = '';
    $commentdata->{hpurl}      = '';
    $commentdata->{text}       = '';
    $commentdata->{evaluate}   = '';
    $commentdata->{ipaddr}     = '';

    if ($commentdata->{commentid} eq '') {
        $error->print_error('00002e');
    }

    # 削除対象のコメントを取得
    $bookmark->load($commentdata->{bookmarkid}, 1);
    my @comments = $bookmark->get_comments();
    foreach my $comment (@comments) {
        my ($commentid, $status, $postdate, $author, $mailaddr, $hpurl, $text, $evaluate, $ipaddr) = split(/\,/, $comment);
        if ($commentid eq $commentdata->{commentid}) {
            $commentdata->{status}   = $status;
            $commentdata->{postdate} = $postdate;
            $commentdata->{author}   = $author;
            $commentdata->{mailaddr} = $mailaddr;
            $commentdata->{hpurl}    = $hpurl;
            $commentdata->{text}     = $text;
            $commentdata->{evaluate} = $evaluate;
            $commentdata->{ipaddr}   = $ipaddr;

            # デコード
            $commentdata->{author} =~ s/enc_conma/,/g;
            $commentdata->{mailaddr} =~ s/enc_conma/,/g;
            $commentdata->{hpurl} =~ s/enc_conma/,/g;
            $commentdata->{text} =~ s/enc_crlf/<br \/>/g;
            $commentdata->{text} =~ s/enc_conma/,/g;
            last;
        }
    }

    _show_page($commentdata, 'このコメントを削除しますか？');
}

# コメントの削除
sub _delete_comment() {
    my ($cgi) = @_;

    my $commentdata;
    $commentdata->{bookmarkid} = $cgi->param('bookmarkid');
    $commentdata->{commentid}  = $cgi->param('id');

    if ($commentdata->{commentid} eq '') {
        $error->print_error('00002e');
    }

    # データ読み込み
    $bookmark->load($commentdata->{bookmarkid}, 1);

    my $point = $bookmark->get_point();

    # コメントを削除
    my @comments = $bookmark->get_comments();
    my @new_comments = ();
    foreach my $comment (@comments) {
        my ($commentid, $evaluate) = (split(/\,/, $comment))[0,7];
        if ($commentid eq $commentdata->{commentid}) {
            if ($evaluate > 0) {
                $point -= $conf->get_good_point();
            }
            else {
                $point -= $conf->get_normal_point();
            }
            next;
        }
        push @new_comments, $comment;
    }
    $bookmark->set_comments(@new_comments);
    $bookmark->set_point($point) ;
    $bookmark->update();

    my $nextpage = "./?act=view_bookmark&id=$commentdata->{bookmarkid}";
    print "Location: $nextpage", "\n\n";
}

# 削除確認画面の表示
sub _show_page() {
    my ($commentdata, $msg) = @_;

    my $caption = 'コメント削除の確認';
    my $action = 'delete_bmcomment';

    # パスの作成
    my $bookmarktitle = $bookmark->get_title();
    my $path = <<"END_PATH";
<p><strong>
<a href="./?target=1" class="path">ホーム</a>&nbsp;
&gt;&nbsp;
<a href="./?act=view_bookmark&id=$commentdata->{bookmarkid}" class="path">$bookmarktitle</a>
&gt;&nbsp;
$caption
</strong></p>
END_PATH

    $msg = "<p><strong><font class=\"message\" size=\"+1\">$msg</font></strong></p>";

    my $status = '無効';
    if ($commentdata->{status} == 1) {
        $status = '有効';
    }

    my $evaluate = 'なし';
    if ($commentdata->{evaluate} == 1) {
        $evaluate = 'おすすめ';
    }

    my $content = <<"END_CONTENT";
<div>
  [$commentdata->{commentid}] &nbsp;ステータス：$status &nbsp;評価：$evaluate
  投稿者：$commentdata->{author}（$commentdata->{ipaddr}）&nbsp;投稿日時：$commentdata->{postdate}
</div>
<div>
  $commentdata->{text}
</div>
END_CONTENT

    my $systemdir = $conf->get_system_dir();

    open my $templatefh, '<', "$systemdir/tmpl/confirm.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$MSG\$/$msg/g;
    $template =~ s/\$ACT\$/$action/g;
    $template =~ s/\$CONTENT\$/$content/g;
    $template =~ s/\$TOPICID\$//g;
    $template =~ s/\$BOOKMARKID\$/$commentdata->{bookmarkid}/g;
    $template =~ s/\$ID\$/$commentdata->{commentid}/g;

    print "Content-Type: text/html\n\n";
    print $template;

    exit;
}

1;
