#------------------------------------------------------------------------------
#    59Tracker, weblog software for personal publisher.
#    Copyright (C) 2004-2009 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

package Lib::App::Keyword;

use strict;
use warnings;

use constant {
    LOCK_SH => 1,
    LOCK_EX => 2,
    LOCK_NB => 4,
    LOCK_UN => 8,
};

use Lib::Logger;
use Lib::String;

# コンストラクタ
sub new {
    my $self = {};

    use Lib::App::Conf;
    my $conf = Lib::App::Conf->new();

    $self->{system_dir} = $conf->get_system_dir();   # システムディレクトリ
    $self->{data_dir} = $conf->get_data_dir();       # データディレクトリ
    $self->{docroot_dir} = $conf->get_docroot_dir(); # ドキュメントルート

    bless($self);
    return $self;
}

# キーワードの入力チェック（全角文字と半角英数のみキーワードとして使用可能）
sub check {
    my $self = shift;
    my $str = shift;

    # UTF-8からシフトJISに変換
    my $string = Lib::String->new();
    $str = $string->convert($str, 'utf8', 'sjis');

    my $len = length($str);
    for (my $i = 0; $i < $len; $i++) {
        my $ch = substr($str, $i, 1);
        if ($ch =~ /[\x81-\x9F\xE0-\xEF]/) { # 全角文字
            $i++;
            next;
        }
        elsif ($ch =~ /[\x20\x30-\x39\x41-\x5a\x61-\x7a]/) { # 半角英数
            next;
        }
        else {
            return 1;
        }
    }

    return 0;
}

# キーワードを追加する
sub add {
    my $self = shift;
    my ($target, $item, $id) = @_;
    my $result = 0;
    my $notfound = 1;
    my @newdata = ();
    my $list;
    my $num;

    # キーワードデータの読み込み
    my $keywordfile = $self->{data_dir} . "/keyword/keyword.data";
    if (-f $keywordfile) {
        open my $infh, '<', $keywordfile;
        while (my $data = <$infh>) {
            chop $data;
            if ($data ne '') {
                my ($keywordid, $keyworditem, $topiclist, $topicnum, $bookmarklist, $bookmarknum, $refernum) = (split(/\,/, $data))[0..6];
                if (uc($keyworditem) eq uc($item)) {
                    if ($target == 0) { # トピック
                        $list = $topiclist;
                        $num = $topicnum;
                    }
                    else {
                        $list = $bookmarklist;
                        $num = $bookmarknum;
                    }

                    # 既に登録済かをチェック
                    my @idlist = split(/:/, $list);
                    my $hit = 0;
                    for my $i (0 .. $#idlist) {
                        if ($idlist[$i] == $id) {
                            $hit = 1;
                            last;
                        };
                    }
                    if ($hit) {
                        if ($num > 0) {
                            $list .= ':';
                        }
                        $list .= $id;
                        $num++;

                        if ($target == 0) {
                            $topiclist = $list;
                            $topicnum = $num;
                        }
                        else {
                            $bookmarklist = $list;
                            $bookmarknum = $num;
                        }
                    }

                    $data = "$keywordid,$keyworditem,$topiclist,$topicnum,$bookmarklist,$bookmarknum,$refernum,,,,,,";
                    $result = 1;
                    $notfound = 0;
                }
                push @newdata, $data;
           }
        }
        close $infh;
    }

    if ($notfound) {

        my ($topiclist, $topicnum, $bookmarklist, $bookmarknum) = ('', 0, '', 0);
        if ($target == 0) { # トピック
            $topiclist = $id;
            $topicnum = 1;
        }
        else {  # ブックマーク
            $bookmarklist = $id;
            $bookmarknum = 1;
        }

        # 新規キーワードIDを取得
        my $keywordid = 0;
        my $counterfile = $self->{system_dir} . "/keywordcounter.txt";
        open my $infh, '<', $counterfile;
        my $data = <$infh>; # 1行読み込み
        chop $data;      # 改行を取り除く
        if ($data ne '') {
            $keywordid = $data;
        }
        close $infh;
        $keywordid++;

        # キーワードカウンターを更新
        open my $outfh, '>', $counterfile;
        flock $outfh, LOCK_EX;
        print {$outfh} "$keywordid\n";
        flock $outfh, LOCK_UN;
        close $outfh;

        $data = "$keywordid,$item,$topiclist,$topicnum,$bookmarklist,$bookmarknum,0,,,,,,";
        push @newdata, $data;
        $result = 1;
    }

    # キーワードデータファイルを更新
    open my $keywordfh, '>', $keywordfile;
    flock $keywordfh, LOCK_EX;
    foreach my $data (@newdata) {
        print {$keywordfh} "$data\n";
    }
    flock $keywordfh, LOCK_UN;
    close $keywordfh;

    # ファイルのパーミッションを変更
    chmod 0766, $keywordfile;

    return $result;
}

# キーワードをキーに指定されたIDを削除する
sub delete {
    my $self = shift;
    my ($target, $item, $id) = @_;
    my $result = 0;
    my $notfound = 1;
    my @newdata = ();

    # キーワードデータの読み込み
    my $keywordfile = $self->{data_dir} . "/keyword/keyword.data";
    open my $infh, '<', $keywordfile;
    while (my $data = <$infh>) {
        chop $data;
        if ($data ne '') {
            my ($keywordid, $keyworditem, $topiclist, $topicnum, $bookmarklist, $bookmarknum, $refernum) = (split(/\,/, $data))[0..6];
            if ($keyworditem eq $item) {

                my $list;
                my $num;
                if ($target == 0) { # トピック
                    $list = $topiclist;
                    $num = $topicnum;
                }
                else {
                    $list = $bookmarklist;
                    $num = $bookmarknum;
                }

                # 登録されているかチェック
                my $newlist = '';
                my @idlist = split(/:/, $list);
                for my $i (0 .. $#idlist) {
                    if ($idlist[$i] != $id) {
                        if ($newlist ne '') {
                            $newlist .= ':';
                        }
                        $newlist .= $idlist[$i];
                    }
                    else {
                        $notfound = 0;
                    }
                }

                # 削除した場合は再編成
                if (!$notfound) {
                    $num = scalar(@idlist) - 1;
                    if ($target == 0) {   # トピック
                        $topiclist = $newlist;
                        $topicnum = $num;
                    }
                    else {
                        $bookmarklist = $newlist;
                        $bookmarknum = $num;
                    }
                    $data = "$keywordid,$item,$topiclist,$topicnum,$bookmarklist,$bookmarknum,$refernum,,,,,,";
                    $result = 1;
                }
            }
            push @newdata, $data;
        }
    }
    close $infh;

    if (!$notfound) {
        # キーワードデータファイルを更新
        open my $outfh ,'>', $keywordfile;
        flock $outfh, LOCK_EX;
        foreach my $data (@newdata) {
            print {$outfh} "$data\n";
        }
        flock $outfh, LOCK_UN;
        close $outfh;
    }

    return $result;
}

# キーワード項目配列からキーワードリストを作成する
sub list {
    my $self = shift;
    my $target = shift; # 0:トピック 1:ブックマーク
    my @items = @_;

    my $itemlist = '';
    foreach my $item (@items) {
        my $keyworditem = $item;
        $item =~ s/(\W)/'%' . unpack('H2', $1)/eg; # uriパック
        $itemlist .= "<a href=\"$self->{docroot_dir}/?target=$target&q=$item\" title=\"$keyworditem\">$keyworditem</a>&nbsp;";
    }

    my $result = '';
    if ($itemlist ne '') {
        $result = <<"END_TABLE";
<div class="keyword">
<table border="0" width="100%" cellpadding="3" cellspacing="0">
<tr>
  <td valign="top" width="80"><b>キーワード</b></td>
  <td valign="top">$itemlist</td>
</tr>
</table>
</div>
END_TABLE
    }

    return $result;
}

# キーワードをリンクする
sub link_keyword {
    my $self = shift;
    my $keywords = shift;
    my $text = shift;
    my $target = shift;

    my @keywordlist = split(/ /, $keywords);
    foreach my $keyword (@keywordlist) {
        my $enckeyword = $keyword;
        $enckeyword =~ s/(\W)/'%'.unpack('H2', $1)/eg; # uri encode
        my $link = "<a href=\"./?target=$target&q=$enckeyword\">$keyword</a>";
        $text =~ s/$keyword(?!.*?<\/a>)/$link/gi;
    }

    return $text;
}

# クラウドの取得
sub clouds {
    my $self = shift;
    my $format = shift; # 0:強調なし 1:強調あり
    my $target = shift; # 0:トピック 1:ブックマーク
    my $itemlist = '';
    my $max = 0;    # 最大値
    my $p1;         # 平均値 + ((最大値 - 平均値) / 2)
    my $avrg;       # 平均値
    my $m1;         # 平均値 - ((平均値 - 最小値) / 2)
    my $min = 0;    # 最小値
    my $num;
    my @dataarray = ();
    my $itemnum = 0;
    my $totalnum = 0;

    my $keywordfile = $self->{data_dir} . "/keyword/keyword.data";
    if (!(-f $keywordfile)) {
        return $itemlist;
    }

    open my $keywordfh, '<', $keywordfile;
    while (my $data = <$keywordfh>) {
        chop $data;
        if ($data ne '') {
            my ($keywordid, $keyworditem, $topiclist, $topicnum, $bookmarklist, $bookmarknum, $refernum) = (split(/\,/, $data))[0..6];
            if ($target == 0) { # 0:トピック
                $num = $topicnum;
            }
            else {  # 1:ブックマーク
                $num = $bookmarknum;
            }
            $num += $refernum;
            if ($num > $max) {
                $max = $num;
            }
            if ($num < $min) {
                $min = $num;
            }
            if ($num > 0) {
                $totalnum += $num;
                $itemnum++;
                push @dataarray, $data;
            }
        }
    }
    close $keywordfh;

    if ($itemnum > 0) {
        $avrg = $totalnum / $itemnum;
    }
    else {
        $avrg = 0;
    }
    $p1 = $avrg + (($max - $avrg) / 2);
    $m1 = $avrg - (($avrg - $min) / 2);

    foreach my $data (@dataarray) {
        my ($keywordid, $keyworditem, $topiclist, $topicnum, $bookmarklist, $bookmarknum, $refernum) = (split(/\,/, $data))[0..6];
        if ($target == 0) { # 0:トピック
            $num = $topicnum;
        }
        else {  # 1:ブックマーク
            $num = $bookmarknum;
        }
        $num += $refernum;

        if ($num > $p1) {
            my $item = $keyworditem;
            $item =~ s/(\W)/'%' . unpack('H2', $1)/eg;  # uriパック
            $itemlist .= <<"END_ITEM";
<a href="./?target=$target&q=$item" title="$keyworditem">$keyworditem</a>&nbsp;
END_ITEM
        }
    }

    if ($itemlist ne '') {
        $itemlist = "<div class=\"clouds\"><h5>注目キーワード</h5>$itemlist</div>";
    }

    return $itemlist;
}

1;
# End of Keyword.pm
