#------------------------------------------------------------------------------
#    59Tracker, weblog software for personal publisher.
#    Copyright (C) 2004-2009 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

package Lib::App::Bookmark;

use strict;
use warnings;

use constant {
    LOCK_SH => 1,
    LOCK_EX => 2,
    LOCK_NB => 4,
    LOCK_UN => 8,
};

use Lib::Logger;
use Lib::DateTime;
use Lib::User;
use Lib::String;
use Lib::App::Keyword;
use Lib::App::Conf;

my $logger = Lib::Logger->new();

my @comments = ();

# コンストラクタ
sub new {
    my $self = {};

    my $conf = Lib::App::Conf->new();

    $self->{system_dir} = $conf->get_system_dir();   # システムディレクトリ
    $self->{data_dir} = $conf->get_data_dir();       # データディレクトリ
    $self->{anonymous} = $conf->get_anonymous();     # 匿名の代替文字列
    $self->{bookmark_num} = $conf->get_bookmark_num();

    $self->{bookmarkid} = 0;       # ブックマークID
    $self->{title} = '';           # ブックマーク先のページタイトル
    $self->{url} = '';             # ブックマーク先のページURL
    $self->{summary} = '';         # ブックマーク先の概要（登録者）
    $self->{registrant} = '';      # 登録者
    $self->{datetime} = '';        # 登録日時
    $self->{keyword} = '';         # キーワード
    $self->{point} = 0;            # 累計ポイント

    bless($self);
    return $self;
}

# ブックマークIDの取得
sub get_bookmarkid {
    my $self = shift;
    return $self->{bookmarkid};
}

# ブックマークIDの設定
sub set_bookmarkid {
    my $self = shift;
    $self->{bookmarkid} = shift;
}

# ページタイトルの取得
sub get_title {
    my $self = shift;
    return $self->{title};
}

# ページタイトルの設定
sub set_title {
    my $self = shift;
    $self->{title} = shift;
}

# ページURLの取得
sub get_url {
    my $self = shift;
    return $self->{url};
}

# ページURLの設定
sub set_url {
    my $self = shift;
    $self->{url} = shift;
}

# 概要の取得
sub get_summary {
    my $self = shift;
    return $self->{summary};
}

# 概要の設定
sub set_summary {
    my $self = shift;
    $self->{summary} = shift;
}

# 登録者の取得
sub get_registrant {
    my $self = shift;
    return $self->{registrant};
}

# 登録者の設定
sub set_registrant {
    my $self = shift;
    $self->{registrant} = shift;
}

# 登録日時の取得
sub get_datetime {
    my $self = shift;
    return $self->{datetime};
}

# 登録日時の設定
sub set_datetime {
    my $self = shift;
    $self->{datetime} = shift;
}

# キーワードの取得
sub get_keyword {
    my $self = shift;
    return $self->{keyword};
}

# キーワードの設定
sub set_keyword {
    my $self = shift;
    $self->{keyword} = shift;
}

# 累計ポイントの取得
sub get_point {
    my $self = shift;
    return $self->{point};
}

# 累計ポイントの設定
sub set_point {
    my $self = shift;
    $self->{point} = shift;
}

# コメントの取得
sub get_comments {
    my $self = shift;
    return @comments;
}

# コメントの設定
sub set_comments {
    my $self = shift;
    @comments = @_;
}

# ブックマークIDをキーにブックマーク情報を読み込む
sub load {
    my $self = shift;
    my ($bookmarkid, $mode) = @_;
    my $result = 0;

    # コメントデータをクリア
    @comments = ();

    my $datafile = "$self->{data_dir}/bookmark/$bookmarkid" . "bm.data";
    if (-f $datafile) {    # ファイルが存在する場合
        open my $datafh, '<', $datafile;
        while (my $data = <$datafh>) {
            chop($data);   # 改行を取り除く
            if ($data ne '') {
                my $rectype = substr($data, 0, 1);
                if ($rectype == 0) {    # 現行ブックマークデータ
                    ($self->{bookmarkid},
                     $self->{title},
                     $self->{url},
                     $self->{summary},
                     $self->{registrant},
                     $self->{datetime},
                     $self->{keyword},
                     $self->{point}) = (split(/\,/, $data))[1..8];
                }
                elsif ($rectype == 2) { # コメント
                    push @comments, substr($data, 2);
                }
            }
            last if (!$mode);
        }
        close $datafh;
        $result = 1;

        # 改行を変換
        $self->{summary} =~ s/enc_crlf/\n/g;

        # コンマをデコード
        $self->{title} =~ s/enc_conma/,/g;
        $self->{url} =~ s/enc_conma/,/g;
        $self->{summary} =~ s/enc_conma/,/g;
    }


    return $result;
}

# ブックマークデータを更新する
sub update {
    my $self = shift;

    # 改行を変換
    $self->{title} =~ s/\r?\n//g;
    $self->{url} =~ s/\r?\n//g;
    $self->{summary} =~ s/\r?\n/enc_crlf/g;

    # コンマをエンコード
    $self->{title} =~ s/,/enc_conma/g;
    $self->{url} =~ s/,/enc_conma/g;
    $self->{summary} =~ s/,/enc_conma/g;

    my $bookmarkdata = "$self->{bookmarkid},$self->{title},$self->{url},$self->{summary},";
    $bookmarkdata .= "$self->{registrant},$self->{datetime},$self->{keyword},$self->{point}";

    my $datafile = "$self->{data_dir}/bookmark/$self->{bookmarkid}" . "bm.data";
    open my $outfh, '>', $datafile;
    flock $outfh, LOCK_EX;
    print {$outfh} "0,$bookmarkdata\n";
    if (scalar(@comments) > 0) { # コメント
        foreach my $comment (@comments) {
            print {$outfh} "2,$comment\n";
        }
    }
    flock $outfh, LOCK_UN;
    close $outfh;

    # ファイルのパーミッションを変更
    chmod 0766, $datafile;
}

# ブックマークIDをキーにブックマークを削除する
sub delete_bookmark {
    my $self = shift;
    my $bookmarkid = shift;

    my $datafile = "$self->{data_dir}/bookmark/$bookmarkid" . "bm.data";
    if (-f $datafile) {    # ファイルが存在する場合
        unlink $datafile;    # ファイルを削除
    }
}

# オーナーの取得
sub get_owner {
    my $self = shift;
    my @userlist = split(/:/, $self->{registrant});
    return $userlist[0];
}

# ブックマークカウンターの値を取得する
sub get_bookmark_counter {
    my $self = shift;
    my $counter = 0;

    # ブックマークカウンタを取得
    my $counterfile = "$self->{system_dir}/bmcounter.txt";
    if (-f $counterfile) {
        open my $counterfh, '<', $counterfile;
        $counter = <$counterfh>;
        chop($counter);    # 改行を取り除く
        close $counterfh;
    }

    return $counter;
}

# ブックマークカウンターを更新する
sub update_bookmark_counter {
    my $self = shift;
    my $counter = shift;

    # ブックマークカウンタを更新
    my $counterfile = "$self->{system_dir}/bmcounter.txt";
    open my $outfh, '>', $counterfile;
    flock $outfh, LOCK_EX;
    print {$outfh} "$counter\n";
    flock $outfh, LOCK_UN;
    close $outfh;
}

# ブックマークコメントの追加
sub add_comment {
    my $self = shift;
    my ($text, $author, $mailaddr, $hpurl, $evaluate, $ipaddr) = @_;
    my $status = 1;

    my $commentid = 0;
    if (scalar(@comments) > 0) {
        $commentid = (split(/\,/, $comments[$#comments]))[0];
    }
    $commentid++;

    my $dt = Lib::DateTime->new();
    my $postdate = $dt->local_datetime(0);

    my $string = Lib::String->new();
    $text = $string->tag_invalidate($text); # タグを無効化
    $text =~ s/\r?\n/enc_crlf/g;
    $text =~ s/,/enc_conma/g;
    $author = $string->tag_invalidate($author);
    $author =~ s/\r?\n//g;
    $author =~ s/,/enc_conma/g;
    $mailaddr = $string->tag_invalidate($mailaddr);
    $mailaddr =~ s/\r?\n//g;
    $mailaddr =~ s/,/enc_conma/g;
    $hpurl = $string->tag_invalidate($hpurl);
    $hpurl =~ s/\r?\n//g;
    $hpurl =~ s/,/enc_conma/g;

    my $comment = "$commentid,$status,$postdate,$author,$mailaddr,$hpurl,$text,$evaluate,$ipaddr";
    push @comments, $comment;
}

# ブックマークのコメント一覧を作成する
sub comment_list {
    my $self = shift;
    my ($bookmarkid, $session_userid) = @_;

    my $user = Lib::User->new();
    my $isadmin = 0;
    if ($session_userid ne '') {
        $isadmin = $user->is_admin($session_userid);
    }

    my $anonymous = $self->{anonymous};
    my $list = '';
    foreach my $comment (@comments) {
        my ($commentid, $status, $postdate, $author, $mailaddr, $hpurl, $text, $evaluate, $ipaddr) = split(/\,/, $comment);

        if ($status) {
            $text =~ s/enc_conma/,/g;
            $author =~ s/enc_conma/,/g;
            $hpurl =~ s/enc_conma/,/g;

            $text =~ s/enc_crlf/<br \/>/g;

            # >>$commentidに対してリンクを張る。
            $text =~ s/>>([0-9]+)/<a href=".\/?act=view_bookmark&id=$bookmarkid#comment$1">&gt;&gt;$1<\/a>/g;
            $text =~ s/&gt;&gt;([0-9]+)/<a href=".\/?act=view_bookmark&id=$bookmarkid#comment$1">&gt;&gt;$1<\/a>/g;

            $text =~ s/(http:\/\/[^()<>\[\]{}"'\x7F-\xFF\x00-\x20]+)/<a href="$1" target="_blank" rel="nofollow">$1<\/a>/g;

            my $keyword = Lib::App::Keyword->new();
            $text = $keyword->link_keyword($self->{keyword}, $text, 1);

            # ホームページURLやメールアドレスが入力されている場合はリンク
            if ($hpurl ne '') {
                $author = "<a href=\"$hpurl\" target=\"_blank\">$author</a>";
            }
            else {
                if ($mailaddr ne '') {
                    $author = "<a href=\"mailto:$mailaddr\">$author</a>";
                }
            }

            my @userlist = split(/:/, $self->{registrant});
            my $editmenu = '';
            if ($session_userid eq $userlist[0] || $isadmin) {
                $editmenu = "<a href=\"./?act=confirm_bmcomment&bookmarkid=$bookmarkid&id=$commentid\">削除</a>";
            }

            if ($author eq '') {
                $author = $anonymous;
            }
            $list .= <<"END_LIST";
<tr><td><a name="$commentid"></a><b>$commentid</b> posted by $author at $postdate </td><td width="40" alig="right" nowrap>$editmenu</td></tr>
<tr><td colspan="2">$text</td></tr>
END_LIST
        }
    }

    if ($list ne '') {
        $list = <<"END_LIST";
<table border="0" width="100%" cellpadding="3" cellspacing="0">
$list
</table>
END_LIST
    }

    return $list;
}

# 使用ブックマークデータを更新
sub update_used_bookmark {
    my $self = shift;
    my ($bookmarkid, $point) = @_;

    my @useddata = ();
    my $newdata = "$bookmarkid,$point";
    push @useddata, $newdata;

    # 使用ブックマークデータを取得
    my $usedfile = $self->{data_dir} . "/bookmark/used.data";
    if (-f $usedfile) {
        open my $infh, '<', $usedfile;
        while (my $data = <$infh>) {
            chop $data;
            my ($id, $point) = split(/\,/, $data);    # 1行読み込み
            if ($id ne $bookmarkid) {
                push @useddata, $data;
            }
        }
        close $infh;
    }

    my $bookmark_num = $self->{bookmark_num};
    my $counter = 0;

    open my $outfh, '>', $usedfile;
    flock $outfh, LOCK_EX;
    foreach my $data (@useddata) {
        print {$outfh} "$data\n";
        $counter++;
        last if ($counter >= $bookmark_num);
    }
    flock $outfh, LOCK_UN;
    close $outfh;

    # ファイルのパーミッションを変更
    chmod 0766, $usedfile;
}

# 新着ブックマークの一覧をIDの逆順で取得
sub new_bookmark_list {
    my $self = shift;
    my $num = shift;

    # ブックマークカウンタを取得
    my $bmcounter = $self->get_bookmark_counter();

    # 既存のブックマークデータをチェックする
    my $list = '';
    for (my $i = $bmcounter; $i && ($i > $bmcounter - $num); $i--) {
        my $bookmarkid = $i;
        my $ret = $self->load($bookmarkid, 0);
        next if (!$ret);

        my $title = $self->get_title();

        my $bookmarkurl = "./?act=view_bookmark&id=$bookmarkid";
        $list .= <<"END_LIST";
<a href="$bookmarkurl">$title</a><br />
END_LIST
    }

    if ($list ne '') {
        $list = <<"END_LIST";
<div class="list"><h5>新着ブックマーク</h5>
$list
<a href="./?target=1">もっと見る...</a><br />
</div><br />
END_LIST
    }

    return $list;
}

# 最近使用されたブックマークの一覧を取得
sub used_bookmark_list {
    my $self = shift;
    my $num = shift;

    my $list = '';
    my $usedfile = $self->{data_dir} . "/bookmark/used.data";
    if (-f $usedfile) {
        open my $infh, '<', $usedfile;
        while (my $data = <$infh>) {
            chop $data;
            if ($data ne '') {
                my ($bookmarkid, $point) = split(/\,/, $data);
                $self->load($bookmarkid, 0);

                my $title = $self->get_title();

                my $bookmarkurl = "./?act=view_bookmark&id=$bookmarkid";
                $list .= <<"END_LIST";
<a href="$bookmarkurl">$title</a> <strong><font class="point">$point</font> point</strong><br />
END_LIST
            }
        }
        close $infh;
    }

    if ($list ne '') {
        $list = "<div class=\"list\"><h5>最近使用されたブックマーク</h5>\n" . $list . "</div><br />";
    }

    return $list;
}

1;
# End of Bookmark.pm
