! This file is part of mstore.
! SPDX-Identifier: Apache-2.0
!
! Licensed under the Apache License, Version 2.0 (the "License");
! you may not use this file except in compliance with the License.
! You may obtain a copy of the License at
!
!     http://www.apache.org/licenses/LICENSE-2.0
!
! Unless required by applicable law or agreed to in writing, software
! distributed under the License is distributed on an "AS IS" BASIS,
! WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
! See the License for the specific language governing permissions and
! limitations under the License.

module mstore_x23
   use mctc_env_accuracy, only : wp
   use mctc_io_structure, only : structure_type, new
   use mstore_data_record, only : record_type, new_record, select_record
   implicit none
   private

   public :: get_x23_records

contains

subroutine get_x23_records(records)
   type(record_type), allocatable, intent(out) :: records(:)

   records = [ &
      new_record("acetic", acetic), &
      new_record("adaman", adaman), &
      new_record("ammonia", ammonia), &
      new_record("anthracene", anthracene), &
      new_record("benzene", benzene), &
      new_record("CO2", CO2), &
      new_record("cyanamide", cyanamide), &
      new_record("cytosine", cytosine), &
      new_record("ethcar", ethcar), &
      new_record("formamide", formamide), &
      new_record("hexamine", hexamine), &
      new_record("hexdio", hexdio), &
      new_record("imdazole", imdazole), &
      new_record("naph", naph), &
      new_record("oxaca", oxaca), &
      new_record("oxacb", oxacb), &
      new_record("pyrazine", pyrazine), &
      new_record("pyrazole", pyrazole), &
      new_record("succinic", succinic), &
      new_record("triazine", triazine), &
      new_record("trioxane", trioxane), &
      new_record("uracil", uracil), &
      new_record("urea", urea) &
      ]

end subroutine get_x23_records

subroutine acetic(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 32
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "C", "C", "C", "C", "C", "C", "C", "C", "O", "O", "O", "O", &
      & "O", "O", "O", "O"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  7.90377778530531_wp,  7.12936818903081_wp,  9.56314593193451_wp, &
      & 16.95915441875529_wp,  0.25435708066996_wp,  4.16571135682655_wp, &
      & 20.33533837362116_wp,  3.94612522923480_wp,  9.56314593193451_wp, &
      &  4.52759383043945_wp,  3.43741106789489_wp,  4.16571135682655_wp, &
      &  9.30463145470085_wp,  3.53964522884767_wp, -5.36209670431652_wp, &
      & 15.55848972193083_wp, -3.53964522884767_wp, 10.83020702100736_wp, &
      & 21.73600307044562_wp,  7.53584818941794_wp,  5.43277244589940_wp, &
      & 11.68606379541616_wp,  3.34481450806704_wp,  3.04226942177239_wp, &
      & 13.17705738121553_wp, -3.34481450806704_wp,  8.43970399688035_wp, &
      & 24.11762438373200_wp, 15.11421520732826_wp,  3.04226942177239_wp, &
      &  0.74549679289969_wp, -0.34714261306888_wp,  8.43970399688035_wp, &
      & 11.18925490605411_wp,  6.24818909009767_wp,  4.59108861432140_wp, &
      & 13.67386627057758_wp,  1.13553617960310_wp,  9.98833421685828_wp, &
      & 23.62081549436995_wp,  4.82730432816794_wp,  4.59108861432140_wp, &
      &  1.24230568226174_wp,  9.93995723866251_wp, -0.80634596078656_wp, &
      &  3.12692913361499_wp, -0.15212291971717_wp, 10.83020702100736_wp, &
      &  8.32726531708940_wp,  5.25608309194219_wp,  1.90276481817667_wp, &
      & 16.53585585954229_wp,  2.12764217775858_wp,  7.30019939328463_wp, &
      & 20.75863693283416_wp,  5.81941032632342_wp,  1.90276481817667_wp, &
      &  4.10429527122645_wp,  1.56412597080627_wp,  7.30019939328463_wp, &
      & 10.22719554669991_wp,  4.55253820982165_wp,  3.88773270477194_wp, &
      & 14.63592562993178_wp,  2.83099808730804_wp,  9.28516727987990_wp, &
      & 22.65875613501575_wp,  6.52276623587289_wp,  3.88773270477194_wp, &
      &  2.20436504161594_wp,  0.86077006125680_wp,  9.28516727987990_wp, &
      &  9.27212817247557_wp,  6.62140991797520_wp,  0.03533787079144_wp, &
      & 15.59099300415612_wp,  0.76212637915449_wp,  5.43277244589940_wp, &
      & 21.70368876079141_wp,  4.45389452771933_wp,  0.03533787079144_wp, &
      &  3.15943241584028_wp,  2.92964176941036_wp,  5.43277244589940_wp, &
      &  6.07055487328507_wp,  4.63171771710301_wp,  1.93885857925242_wp, &
      & 18.79237733077554_wp,  2.75181858002668_wp,  7.33629315436038_wp, &
      & 18.50211546160091_wp,  6.44377570116260_wp,  1.93885857925242_wp, &
      &  6.36100571503078_wp,  0.93994956853816_wp,  7.33629315436038_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      & 24.86304558760325_wp,  0.00000000000000_wp,  0.00000000000000_wp, &
      &  0.00000000000000_wp,  7.38360999643241_wp,  0.00000000000000_wp, &
      &  0.00000000000000_wp,  0.00000000000000_wp, 10.79482606446971_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine acetic

subroutine adaman(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 52
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", &
      & "C", "C", "C", "C", "C", "C", "C", "C", "C", "C"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  1.64614006665379_wp,  0.28175810347616_wp,  4.60261594115711_wp, &
      &  0.28175810347616_wp, 10.54807097238905_wp, 12.33046026278952_wp, &
      & 10.54807097238905_wp, 11.91245293556668_wp,  4.60261594115711_wp, &
      & 11.91245293556668_wp,  1.64614006665379_wp, 12.33046026278952_wp, &
      &  4.45087096658209_wp,  6.37895810928311_wp,  3.86392216081621_wp, &
      &  5.81525292975972_wp,  4.45087096658209_wp, 13.06915404313043_wp, &
      &  7.74334007246074_wp,  5.81525292975972_wp,  3.86392216081621_wp, &
      &  6.37895810928311_wp,  7.74334007246074_wp, 13.06915404313043_wp, &
      & 11.48858745864045_wp,  4.02398192851862_wp,  2.88787883120229_wp, &
      &  4.02398192851862_wp,  0.70562358040239_wp, 14.04519737274435_wp, &
      &  0.70562358040239_wp,  8.17022911052421_wp,  2.88787883120229_wp, &
      &  8.17022911052421_wp, 11.48858745864045_wp, 14.04519737274435_wp, &
      &  6.80282358620935_wp, 10.12118193432558_wp,  5.57865927077103_wp, &
      &  2.07302910471726_wp,  6.80282358620935_wp, 11.35441693317561_wp, &
      &  5.39138745283349_wp,  2.07302910471726_wp,  5.57865927077103_wp, &
      & 10.12118193432558_wp,  5.39138745283349_wp, 11.35441693317561_wp, &
      &  3.60030542416346_wp,  3.15281837585257_wp,  1.13761487788495_wp, &
      &  3.15281837585257_wp,  8.59390561487937_wp, 15.79546132606169_wp, &
      &  8.59390561487937_wp,  9.04139266319027_wp,  1.13761487788495_wp, &
      &  9.04139266319027_wp,  3.60030542416346_wp, 15.79546132606169_wp, &
      &  2.49670560907242_wp,  9.24982940908845_wp,  7.32892322408837_wp, &
      &  2.94438162995438_wp,  2.49670560907242_wp,  9.60415297985827_wp, &
      &  9.69750542997042_wp,  2.94438162995438_wp,  7.32892322408837_wp, &
      &  9.24982940908845_wp,  9.69750542997042_wp,  9.60415297985827_wp, &
      &  1.62289644041129_wp,  4.44539076202085_wp, 15.71004572393478_wp, &
      &  4.44539076202085_wp, 10.57131459863155_wp,  1.22321945258294_wp, &
      & 10.57131459863155_wp,  7.74863130445090_wp, 15.71004572393478_wp, &
      &  7.74863130445090_wp,  1.62289644041129_wp,  1.22321945258294_wp, &
      &  4.47430356539567_wp, 10.54259076782781_wp,  9.68975755455625_wp, &
      &  1.65162027121503_wp,  4.47430356539567_wp,  7.24331864939038_wp, &
      &  7.71990747364717_wp,  1.65162027121503_wp,  9.68975755455625_wp, &
      & 10.54259076782781_wp,  7.71990747364717_wp,  7.24331864939038_wp, &
      &  0.00000000000000_wp,  0.00000000000000_wp,  3.36201101203507_wp, &
      &  0.00000000000000_wp,  0.00000000000000_wp, 13.57106519191156_wp, &
      &  6.09701103323588_wp,  6.09701103323588_wp,  5.10452708993825_wp, &
      &  6.09701103323588_wp,  6.09701103323588_wp, 11.82873808657947_wp, &
      & 11.77979419067046_wp,  2.34798419563463_wp,  1.68412355344031_wp, &
      &  2.34798419563463_wp,  0.41441684837237_wp, 15.24895265050633_wp, &
      &  0.41441684837237_wp,  9.84603787083713_wp,  1.68412355344031_wp, &
      &  9.84603787083713_wp, 11.77979419067046_wp, 15.24895265050633_wp, &
      &  6.51142788160825_wp,  8.44518420144159_wp,  6.78241454853301_wp, &
      &  3.74902683760125_wp,  6.51142788160825_wp, 10.15085062798470_wp, &
      &  5.68278315743458_wp,  3.74902683760125_wp,  6.78241454853301_wp, &
      &  8.44518420144159_wp,  5.68278315743458_wp, 27.08392683193134_wp, &
      &  1.91542598043884_wp,  2.76977097427902_wp, -0.02040903767634_wp, &
      &  2.76977097427902_wp, 10.27878505860399_wp, 16.95348524162297_wp, &
      & 10.27878505860399_wp,  9.42444006476382_wp, -0.02040903767634_wp, &
      &  9.42444006476382_wp,  1.91542598043884_wp,  0.02040903767634_wp, &
      &  4.18158505279704_wp,  8.86697098008598_wp,  8.48694713964966_wp, &
      &  3.32724005895686_wp,  4.18158505279704_wp,  8.44612906429698_wp, &
      &  8.01262598624580_wp,  3.32724005895686_wp,  8.48694713964966_wp, &
      &  8.86697098008598_wp,  8.01262598624580_wp,  8.44612906429698_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      & 12.19419762199029_wp,  0.00000000000000_wp,  0.00000000000000_wp, &
      &  0.00000000000000_wp, 12.19419762199029_wp,  0.00000000000000_wp, &
      &  0.00000000000000_wp,  0.00000000000000_wp, 16.93311059695457_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine adaman

subroutine ammonia(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 16
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "N", "N", &
      & "N", "N"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  3.51413393175224_wp,  2.56209011866514_wp,  0.98870449187609_wp, &
      &  2.56209011866514_wp,  0.98870449187609_wp,  3.51413393175224_wp, &
      &  0.98870449187609_wp,  3.51413393175224_wp,  2.56209011866514_wp, &
      &  8.20273239274950_wp,  2.12650834233211_wp,  8.38849243011842_wp, &
      &  7.25049960709132_wp,  3.69989396912117_wp,  5.86287401767119_wp, &
      &  5.67730295287334_wp,  1.17427555667393_wp,  6.81510680332937_wp, &
      &  6.81510680332937_wp,  5.67730295287334_wp,  1.17427555667393_wp, &
      &  8.38849243011842_wp,  8.20273239274950_wp,  2.12650834233211_wp, &
      &  3.69989396912117_wp,  5.86287401767119_wp,  7.25049960709132_wp, &
      &  1.17427555667393_wp,  6.81510680332937_wp,  5.67730295287334_wp, &
      &  2.12650834233211_wp,  8.38849243011842_wp,  8.20273239274950_wp, &
      &  5.86287401767119_wp,  7.25049960709132_wp,  3.69989396912117_wp, &
      &  1.95435433008075_wp,  1.95435433008075_wp,  1.95435433008075_wp, &
      &  6.64276381850693_wp,  2.73424413091650_wp,  7.42284259191375_wp, &
      &  7.42284259191375_wp,  6.64276381850693_wp,  2.73424413091650_wp, &
      &  2.73424413091650_wp,  7.42284259191375_wp,  6.64276381850693_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      &  9.37712832495121_wp,  0.00000000000000_wp,  0.00000000000000_wp, &
      &  0.00000000000000_wp,  9.37712832495121_wp,  0.00000000000000_wp, &
      &  0.00000000000000_wp,  0.00000000000000_wp,  9.37712832495121_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine ammonia

subroutine anthracene(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 48
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "C", "C", "C", "C", "C", "C", "C", "C", &
      & "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", &
      & "C", "C", "C", "C", "C", "C"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -3.66228842747680_wp,  0.96035860621450_wp,  8.20178752989411_wp, &
      & -0.50644649048699_wp,  6.57586752834559_wp,  8.97052794903630_wp, &
      &  7.42888971418823_wp, 10.27084821061875_wp,  8.96429185419075_wp, &
      &  4.27323674976949_wp,  4.65515031591658_wp,  8.19574040761964_wp, &
      & -0.89043875491593_wp,  3.60219514987424_wp,  5.52140058173462_wp, &
      & -3.27829616304786_wp,  9.21770407200532_wp, 11.65091489719579_wp, &
      &  4.65722901419843_wp,  7.62901166695902_wp, 11.64467880235024_wp, &
      &  7.04489744975929_wp,  2.01350274482793_wp,  5.51516448688908_wp, &
      &  1.42220756992726_wp,  4.25377257494854_wp,  1.50176502235078_wp, &
      & -5.59094248789105_wp,  9.86928149707963_wp, 15.67055045657963_wp, &
      &  2.34439371678417_wp,  6.97743424188471_wp, 15.66431436173409_wp, &
      &  9.35773274717355_wp,  1.36173634718255_wp,  1.49571790007631_wp, &
      & 12.09556735694052_wp,  6.36535208416555_wp,  2.47270609254561_wp, &
      & -0.39344089298280_wp,  0.74984316203446_wp, 14.68732616926479_wp, &
      & -8.32877709765801_wp,  4.86566576009662_wp, 14.69337329153926_wp, &
      &  4.16004217969423_wp, 10.48117468222771_wp,  2.47894218739116_wp, &
      & 10.71852423150072_wp,  7.94780839436631_wp,  6.63822847680108_wp, &
      &  0.98360223245700_wp,  2.33229947223522_wp, 10.52161481243824_wp, &
      & -6.95192294478929_wp,  3.28320944989586_wp, 10.52785090728379_wp, &
      &  2.78318802682550_wp,  8.89871837202695_wp,  6.64446457164663_wp, &
      & -3.08044188112999_wp,  0.28175810347616_wp,  6.34569893677352_wp, &
      & -1.08848200940487_wp,  5.89726702560724_wp, 10.82642756958582_wp, &
      &  6.84704316784142_wp, 10.94944871335710_wp, 10.82038044731135_wp, &
      &  4.85508329611630_wp,  5.33375081865493_wp,  6.33965181449905_wp, &
      & -1.53256755143637_wp,  1.75838977387366_wp,  4.86320911667262_wp, &
      & -2.63616736652742_wp,  7.37389869600475_wp, 12.30910636225780_wp, &
      &  5.29916883814780_wp,  9.47281704295959_wp, 12.30305923998332_wp, &
      &  6.40295762580992_wp,  3.85711914825742_wp,  4.85697302182707_wp, &
      & -0.75003213460554_wp,  0.92350895485444_wp,  2.41525843093813_wp, &
      & -3.41889175592932_wp,  6.53920684955661_wp, 14.75705704799229_wp, &
      &  4.51663342131697_wp, 10.30750888940773_wp, 14.75100992571782_wp, &
      &  7.18549304264075_wp,  4.69199996727664_wp,  2.40902233609258_wp, &
      &  0.79519657909300_wp,  2.40297521381811_wp,  0.84527431042847_wp, &
      & -4.96393149705678_wp,  8.01867310852027_wp, 16.32704116850195_wp, &
      &  2.97159368018951_wp,  8.82804263044407_wp, 16.32080507365639_wp, &
      &  8.73053278376821_wp,  3.21253370831298_wp,  0.83922718815400_wp, &
      & 14.30333390483585_wp,  9.69108036255379_wp,  1.53218960629421_wp, &
      & -2.60120744087813_wp,  4.07557144042271_wp, 15.62784265551618_wp, &
      &-10.53673261812442_wp,  1.53993748170838_wp, 15.63388977779065_wp, &
      &  6.36799770016063_wp,  7.15544640383947_wp,  1.53823672856869_wp, &
      & 12.71294074664985_wp,  8.21350382930090_wp,  3.13392111874485_wp, &
      & -1.01081428269213_wp,  2.59799490716981_wp, 14.02592217049447_wp, &
      & -8.94615048736734_wp,  3.01751401496127_wp, 14.03215826534002_wp, &
      &  4.77741556940355_wp,  8.63302293709236_wp,  3.14015721359040_wp, &
      & 11.95497176405906_wp,  9.10224183107713_wp,  5.45903963327914_wp, &
      & -0.25265632753026_wp,  3.48673290894605_wp, 11.70099262853126_wp, &
      & -8.18818150477655_wp,  2.12896498575612_wp, 11.70703975080573_wp, &
      &  4.01944658681277_wp,  7.74447390788721_wp,  5.46508675555361_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      & 15.87093489225165_wp,  0.00000000000000_wp, -0.01227036698519_wp, &
      &  0.00000000000000_wp, 11.23112234609398_wp,  0.00000000000000_wp, &
      &-12.10426425260664_wp,  0.00000000000000_wp, 17.17840096263681_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine anthracene

subroutine benzene(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 48
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "C", "C", "C", "C", &
      & "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", &
      & "C", "C", "C", "C", "C", "C"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & 12.45574907741373_wp,  4.46617774483935_wp, 12.64906801762574_wp, &
      &  8.43082228603972_wp, 13.18272655834785_wp, 12.64906801762574_wp, &
      & 12.45574907741373_wp,  4.25037106866915_wp,  6.18715094963972_wp, &
      &  8.43082228603972_wp, 12.96691988217765_wp,  6.18715094963972_wp, &
      &  1.46869482241226_wp, 12.96691988217765_wp,  0.27476611834630_wp, &
      &  5.49362161378627_wp,  4.25037106866915_wp,  0.27476611834630_wp, &
      &  1.46869482241226_wp, 13.18272655834785_wp,  6.73668318633232_wp, &
      &  5.49362161378627_wp,  4.46617774483935_wp,  6.73668318633232_wp, &
      & 10.45566338513228_wp,  1.49571790007631_wp,  2.81266774791355_wp, &
      & 10.43090797832117_wp, 10.21226671358481_wp,  2.81266774791355_wp, &
      & 10.45566338513228_wp,  7.22064194086111_wp,  9.27458481589957_wp, &
      & 10.43090797832117_wp, 15.93719075436961_wp,  9.27458481589957_wp, &
      &  3.46878051469371_wp, 15.93719075436961_wp, 10.11116638805849_wp, &
      &  3.49334694893375_wp,  7.22064194086111_wp, 10.11116638805849_wp, &
      &  3.46878051469371_wp, 10.21226671358481_wp,  3.64924932007247_wp, &
      &  3.49334694893375_wp,  1.49571790007631_wp,  3.64924932007247_wp, &
      &  1.95322049465429_wp,  2.92510642770451_wp,  9.80540876805552_wp, &
      &  5.00890696897317_wp, 11.64165524121300_wp,  9.80540876805552_wp, &
      &  1.95322049465429_wp,  5.79144238580399_wp,  3.34349170006950_wp, &
      &  5.00890696897317_wp, 14.50799119931249_wp,  3.34349170006950_wp, &
      & 11.97122340517170_wp, 14.50799119931249_wp,  3.11842536791652_wp, &
      &  8.91534795828175_wp,  5.79144238580399_wp,  3.11842536791652_wp, &
      & 11.97122340517170_wp, 11.64165524121300_wp,  9.58034243590254_wp, &
      &  8.91534795828175_wp,  2.92510642770451_wp,  9.58034243590254_wp, &
      & 13.09296458708616_wp,  2.51012266161890_wp, 12.76906560025978_wp, &
      &  7.79360677636729_wp, 11.22667147512740_wp, 12.76906560025978_wp, &
      & 13.09296458708616_wp,  6.20642615188960_wp,  6.30714853227376_wp, &
      &  7.79360677636729_wp, 14.92297496539810_wp,  6.30714853227376_wp, &
      &  0.83129034016875_wp, 14.92297496539810_wp,  0.15476853571225_wp, &
      &  6.13083712345870_wp,  6.20642615188960_wp,  0.15476853571225_wp, &
      &  0.83129034016875_wp, 11.22667147512740_wp,  6.61668560369827_wp, &
      &  6.13083712345870_wp,  2.51012266161890_wp,  6.61668560369827_wp, &
      & 11.98067203372556_wp,  0.85358910355587_wp,  1.58566884390907_wp, &
      &  8.90589932972789_wp,  9.56994894449329_wp,  1.58566884390907_wp, &
      & 11.98067203372556_wp,  7.86295970995263_wp,  8.04758591189509_wp, &
      &  8.90589932972789_wp, 16.57950852346113_wp,  8.04758591189509_wp, &
      &  1.94358289352935_wp, 16.57950852346113_wp, 11.33816529206297_wp, &
      &  5.01854457009811_wp,  7.86295970995263_wp, 11.33816529206297_wp, &
      &  1.94358289352935_wp,  9.56994894449329_wp,  4.87624822407695_wp, &
      &  5.01854457009811_wp,  0.85358910355587_wp,  4.87624822407695_wp, &
      &  1.10397776023320_wp,  1.65180924378610_wp, 11.18056216778455_wp, &
      &  5.85814970339426_wp, 10.36835805729460_wp, 11.18056216778455_wp, &
      &  1.10397776023320_wp,  7.06473956972240_wp,  4.71864509979853_wp, &
      &  5.85814970339426_wp, 15.78128838323090_wp,  4.71864509979853_wp, &
      & 12.82027716702171_wp, 15.78128838323090_wp,  1.74327196818748_wp, &
      &  8.06610522386066_wp,  7.06473956972240_wp,  1.74327196818748_wp, &
      & 12.82027716702171_wp, 10.36835805729460_wp,  8.20518903617350_wp, &
      &  8.06610522386066_wp,  1.65180924378610_wp,  8.20518903617350_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      & 13.92433070525592_wp,  0.00000000000000_wp,  0.00000000000000_wp, &
      &  0.00000000000000_wp, 17.43307173777476_wp,  0.00000000000000_wp, &
      &  0.00000000000000_wp,  0.00000000000000_wp, 12.92384093898460_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine benzene

subroutine CO2(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 12
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "C", "C", "C", "C", "O", "O", "O", "O", "O", "O", "O", "O"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  0.00000000000000_wp,  0.00000000000000_wp,  0.00000000000000_wp, &
      &  5.47113387782808_wp,  0.00000000000000_wp,  5.47113387782808_wp, &
      &  5.47113387782808_wp,  5.47113387782808_wp,  0.00000000000000_wp, &
      &  0.00000000000000_wp,  5.47113387782808_wp,  5.47113387782808_wp, &
      &  1.27461999191594_wp,  1.27461999191594_wp,  1.27461999191594_wp, &
      &  6.74556489717295_wp,  1.27461999191594_wp,  4.19651388591214_wp, &
      &  6.74556489717295_wp,  4.19651388591214_wp,  9.66764776374022_wp, &
      &  9.66764776374022_wp,  6.74556489717295_wp,  4.19651388591214_wp, &
      &  9.66764776374022_wp,  9.66764776374022_wp,  9.66764776374022_wp, &
      &  4.19651388591214_wp,  9.66764776374022_wp,  6.74556489717295_wp, &
      &  4.19651388591214_wp,  6.74556489717295_wp,  1.27461999191594_wp, &
      &  1.27461999191594_wp,  4.19651388591214_wp,  6.74556489717295_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      & 10.94216438765978_wp,  0.00000000000000_wp,  0.00000000000000_wp, &
      &  0.00000000000000_wp, 10.94216438765978_wp,  0.00000000000000_wp, &
      &  0.00000000000000_wp,  0.00000000000000_wp, 10.94216438765978_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine CO2

subroutine cyanamide(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 40
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "C", "C", "C", "C", "C", "C", "C", "C", "N", "N", "N", "N", &
      & "N", "N", "N", "N", "N", "N", "N", "N", "N", "N", "N", "N"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  2.70098495840691_wp,  1.38120052200350_wp,  0.31558419369898_wp, &
      &  3.79834867865240_wp,  7.89924244359946_wp,  0.31558419369898_wp, &
      &  2.70098495840691_wp,  5.13665242702138_wp,  8.66647108217303_wp, &
      &  3.79834867865240_wp, 11.65469434861733_wp,  8.66647108217303_wp, &
      & 10.29787128828279_wp, 11.65469434861733_wp, 16.38618958324912_wp, &
      &  9.20031859546622_wp,  5.13665242702138_wp, 16.38618958324912_wp, &
      & 10.29787128828279_wp,  7.89924244359946_wp,  8.03530269477507_wp, &
      &  9.20031859546622_wp,  1.38120052200350_wp,  8.03530269477507_wp, &
      &  0.48773820595034_wp,  3.51488982203655_wp,  1.76991710070937_wp, &
      &  6.01178440368005_wp, 10.03293174363251_wp,  1.76991710070937_wp, &
      &  0.48773820595034_wp,  3.00296312698833_wp, 10.12080398918342_wp, &
      &  6.01178440368005_wp,  9.52081607601321_wp, 10.12080398918342_wp, &
      & 12.51111804073936_wp,  9.52081607601321_wp, 14.93185667623873_wp, &
      &  6.98707184300965_wp,  3.00296312698833_wp, 14.93185667623873_wp, &
      & 12.51111804073936_wp, 10.03293174363251_wp,  6.58096978776468_wp, &
      &  6.98707184300965_wp,  3.51488982203655_wp,  6.58096978776468_wp, &
      &  1.80619983435620_wp,  0.74530782032861_wp,  3.98751022230072_wp, &
      &  4.69313380270311_wp,  7.26316076935349_wp,  3.98751022230072_wp, &
      &  1.80619983435620_wp,  5.77273410126735_wp, 12.33858608334584_wp, &
      &  4.69313380270311_wp, 12.29058705029223_wp, 12.33858608334584_wp, &
      & 11.19265641233350_wp, 12.29058705029223_wp, 12.71426355464739_wp, &
      &  8.30553347141552_wp,  5.77273410126735_wp, 12.71426355464739_wp, &
      & 11.19265641233350_wp,  7.26316076935349_wp,  4.36337666617333_wp, &
      &  8.30553347141552_wp,  0.74530782032861_wp,  4.36337666617333_wp, &
      &  1.71435916481267_wp,  2.00405411627407_wp,  1.88273372564248_wp, &
      &  4.78497447224664_wp,  8.52209603787002_wp,  1.88273372564248_wp, &
      &  1.71435916481267_wp,  4.51379883275081_wp, 10.23362061411653_wp, &
      &  4.78497447224664_wp, 11.03165178177569_wp, 10.23362061411653_wp, &
      & 11.28449708187703_wp, 11.03165178177569_wp, 14.81922902387669_wp, &
      &  8.21369280187198_wp,  4.51379883275081_wp, 14.81922902387669_wp, &
      & 11.28449708187703_wp,  8.52209603787002_wp,  6.46834213540264_wp, &
      &  8.21369280187198_wp, 15.03994898689490_wp,  6.46834213540264_wp, &
      &  1.88367858849787_wp, -0.37246493759323_wp,  5.90577079130572_wp, &
      &  4.61584402113252_wp,  6.14538801143165_wp,  5.90577079130572_wp, &
      &  1.88367858849787_wp, -6.14538801143165_wp, 14.25665767977977_wp, &
      &  4.61584402113252_wp,  0.37246493759323_wp, 14.25665767977977_wp, &
      & 11.11517765819183_wp, 13.40835980821407_wp, 10.79619195821346_wp, &
      &  8.38301222555718_wp,  6.89050685918919_wp, 10.79619195821346_wp, &
      & 11.11517765819183_wp, 19.18128288205249_wp,  2.44530506973941_wp, &
      &  8.38301222555718_wp, -0.37246493759323_wp,  2.44530506973941_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      & 12.99879180704296_wp,  0.00000000000000_wp,  0.00000000000000_wp, &
      &  0.00000000000000_wp, 13.03587710719916_wp,  0.00000000000000_wp, &
      &  0.00000000000000_wp,  0.00000000000000_wp, 16.70183292536285_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine cyanamide

subroutine cytosine(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 52
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "C", "C", "C", "C", "C", "C", "C", "C", &
      & "C", "C", "C", "C", "C", "C", "C", "C", "N", "N", "N", "N", "N", "N", &
      & "N", "N", "N", "N", "N", "N", "O", "O", "O", "O"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & 24.24461395149590_wp, 16.58555564573560_wp,  2.37821980700699_wp, &
      & 11.99956929083329_wp, 10.24817150208948_wp,  4.73281804262933_wp, &
      &  0.24547536982933_wp,  7.64110591150796_wp,  1.17729911781117_wp, &
      & 12.49052003049194_wp,  1.30353279529076_wp,  5.93373873182515_wp, &
      &  3.78682135181669_wp,  8.51737172359309_wp,  3.65019418292785_wp, &
      & 16.03186601247931_wp,  0.42707801063455_wp,  3.46103263927954_wp, &
      & 20.70307899693746_wp, 17.46201043039181_wp,  7.01655156409770_wp, &
      &  8.45803433627484_wp,  9.37171671743327_wp,  0.09448628553862_wp, &
      &  6.33473852765104_wp,  6.70210120582518_wp,  4.85262665269229_wp, &
      & 18.57978318831366_wp,  2.24253750097354_wp,  2.25860016951510_wp, &
      & 18.15516182110311_wp, 15.64655094005283_wp,  5.81411909433326_wp, &
      &  5.91030613301157_wp, 11.18717620777226_wp,  1.29710772787413_wp, &
      &  5.99855632370464_wp,  2.24745078782154_wp,  5.72303431507403_wp, &
      & 18.24360098436725_wp,  6.69718791897718_wp,  1.38819250713336_wp, &
      & 18.49153299762058_wp, 11.19190052204919_wp,  4.94371143195152_wp, &
      &  6.24648833695797_wp, 15.64163765320482_wp,  2.16732641768480_wp, &
      &  3.32856286695440_wp, 16.20439796987282_wp,  4.85753993954030_wp, &
      & 15.57360752761701_wp, 10.62914020538118_wp,  2.25349791009602_wp, &
      & 21.16152645437083_wp,  7.25994823564518_wp,  5.80920580748525_wp, &
      &  8.91648179370821_wp,  1.68469047115354_wp,  1.30202101472214_wp, &
      & 24.06773562496761_wp,  2.69361502813489_wp,  2.26540318207388_wp, &
      & 11.82269096430499_wp,  6.25083470609275_wp,  4.84582364013351_wp, &
      &  0.42235369635762_wp, 11.63825373493361_wp,  1.29030471531535_wp, &
      & 12.66739835702023_wp, 15.19547341289147_wp,  5.82092210689204_wp, &
      &  3.20478583289881_wp,  4.68292928386494_wp,  3.86732366709560_wp, &
      & 15.44983049356143_wp,  4.26152045036271_wp,  3.24371418254072_wp, &
      & 21.28530348842641_wp, 13.62756799066366_wp,  6.79942207992996_wp, &
      &  9.04025882776380_wp, 13.20615915716143_wp,  0.31180474227744_wp, &
      &  4.20520662418169_wp,  2.32474056939213_wp,  4.73943208261703_wp, &
      & 16.45025128484430_wp,  6.61970916483551_wp,  2.37179473959036_wp, &
      & 20.28469372457246_wp, 11.26937927619085_wp,  5.92731366440852_wp, &
      &  8.03964906390985_wp, 15.56434787163423_wp,  1.18391315779887_wp, &
      &  2.78658953310489_wp,  0.22053099044713_wp,  4.28892147316890_wp, &
      & 15.03163419376751_wp,  8.72410771635159_wp,  2.82211637646741_wp, &
      & 21.70349978822033_wp,  9.16498072467478_wp,  6.37782427385665_wp, &
      &  9.45845512755772_wp, 17.66874642315031_wp,  0.73340254835075_wp, &
      &  0.54745353841075_wp,  0.39174013984311_wp,  3.06929249943644_wp, &
      & 12.79249819907336_wp,  8.55289856695561_wp,  4.04174535019988_wp, &
      & 23.94244681034340_wp,  9.33637884664183_wp,  0.48622642538172_wp, &
      & 11.69759112225187_wp, 17.49734830118325_wp,  6.62500039682567_wp, &
      &  0.94486285538617_wp,  4.82239104131994_wp,  2.70268571154660_wp, &
      & 13.18990751604878_wp,  4.12205869290771_wp,  4.40854111066079_wp, &
      & 23.54503749336798_wp, 13.76702974811866_wp,  0.85302218584263_wp, &
      & 11.29999283270537_wp, 13.06669739970643_wp,  6.25820463636476_wp, &
      &  4.51795622931451_wp,  6.81491783075829_wp,  4.21673395101740_wp, &
      & 16.76281191740605_wp,  2.12972087604043_wp,  2.89449287118999_wp, &
      & 19.97213309201071_wp, 15.75936756498593_wp,  6.45001179600815_wp, &
      &  7.72708843134810_wp, 11.07417061026807_wp,  0.66102605362816_wp, &
      & 21.98166741284602_wp,  2.73594488405619_wp,  1.12004042877477_wp, &
      &  9.73662275218341_wp,  6.20869382274253_wp,  5.99118639343263_wp, &
      &  2.50842190847920_wp, 11.68039461828384_wp,  2.43547849604339_wp, &
      & 14.75327759657074_wp, 15.15333252954125_wp,  4.67555935359293_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      & 24.49001524407736_wp,  0.00000000000000_wp,  0.00000000000000_wp, &
      &  0.00000000000000_wp, 17.88912037739087_wp,  0.00000000000000_wp, &
      &  0.00000000000000_wp,  0.00000000000000_wp,  7.11115822516409_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine cytosine

subroutine ethcar(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 26
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "C", "C", "C", "C", "C", "C", "N", "N", "O", "O", "O", "O"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  7.71688391250993_wp, 10.77219244168665_wp,  5.39800149282119_wp, &
      &  1.07336420371869_wp,  0.18122469566307_wp,  7.96972921261127_wp, &
      &  0.88930491948946_wp,  9.02211746094039_wp,  6.71041599895258_wp, &
      &  7.90094319673916_wp,  1.93148864898041_wp,  6.65731470647988_wp, &
      & 10.52974063299456_wp, 10.55865343636938_wp,  3.57649488020773_wp, &
      & -1.73949251676594_wp,  0.39495267355142_wp,  9.79123582522473_wp, &
      &  6.92282116884339_wp,  6.15181307884828_wp,  4.88645274291512_wp, &
      &  1.86761591995630_wp,  4.80179303107252_wp,  8.48146693508842_wp, &
      &  7.24672015566977_wp,  7.57987879847894_wp,  1.85098633370151_wp, &
      &  1.54371693312992_wp,  3.37372731144186_wp, 11.51674437173095_wp, &
      &  4.20142717276014_wp,  2.48461136452347_wp,  2.09457197782006_wp, &
      &  4.58900991603955_wp,  8.46899474539732_wp, 11.27315872761240_wp, &
      &  2.22534099700551_wp,  0.17479962824644_wp,  0.76722863857357_wp, &
      &  6.56490711922311_wp, 10.77861750910328_wp, 12.60050206685889_wp, &
      &  9.28195474617158_wp,  9.51817046001813_wp,  4.86377603438585_wp, &
      & -0.49151765737189_wp,  1.43543564990267_wp,  8.50395467104661_wp, &
      &  8.23599156525909_wp,  7.15676921183701_wp,  3.62524980354566_wp, &
      &  0.55444552354060_wp,  3.79683689808379_wp,  9.74248090188680_wp, &
      &  0.38833863356372_wp,  3.26752472649645_wp,  2.08172184298681_wp, &
      &  8.40209845523598_wp,  7.68608138342434_wp, 11.28600886244565_wp, &
      &  2.42943137376892_wp,  1.79127100124110_wp,  1.80846750520913_wp, &
      &  6.36100571503078_wp,  9.16214613610862_wp, 11.55926320022333_wp, &
      &  0.91802875029320_wp,  5.48209428695056_wp,  3.23067507513639_wp, &
      &  7.87240833850649_wp,  5.47132285039916_wp, 10.13724460286714_wp, &
      &  7.68419165771357_wp,  2.69644961670105_wp,  1.26649417135962_wp, &
      &  1.10624543108613_wp,  8.25696752064867_wp, 12.10123653407284_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      &  9.46493424112967_wp,  0.03496899633270_wp, -0.13268746872702_wp, &
      &  3.08071343471463_wp, 12.77401913826543_wp,  0.12194796854013_wp, &
      & -3.75526160963880_wp, -1.85547972349658_wp, 13.37850913396899_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine ethcar

subroutine formamide(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 24
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "C", "C", &
      & "C", "C", "N", "N", "N", "N", "O", "O", "O", "O"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  1.00930250212351_wp,  4.44028850260177_wp,  1.86326955082153_wp, &
      &  1.20488911318844_wp, 12.94499906393269_wp,  4.61527710341929_wp, &
      &  3.41926970107147_wp, 12.56894364748899_wp, 11.09382375766010_wp, &
      &  0.88552546806792_wp,  4.06423308615807_wp, 21.26999671016916_wp, &
      &  2.64599394022343_wp,  2.27315105748805_wp, -0.17158709453813_wp, &
      &  1.90635529702714_wp, 10.77786161881897_wp, -6.27804675632787_wp, &
      &  1.78257826297155_wp, 14.73627006517379_wp, 13.12868040301976_wp, &
      &  4.86037452810646_wp,  6.23155950384287_wp,  6.30695955970269_wp, &
      &  0.58090168349142_wp,  1.54371693312992_wp,  5.02515861008581_wp, &
      &  1.63328993182053_wp, 10.04842749446084_wp,  1.45338804415501_wp, &
      &  3.84767051970356_wp, 15.46570418953191_wp,  7.93193469839582_wp, &
      &  2.79528227137445_wp,  6.96099362820099_wp, 11.50370526432662_wp, &
      &  1.60664479929864_wp,  0.94051648625139_wp,  3.30475232299867_wp, &
      &  0.60773578858438_wp,  9.44522704758231_wp,  3.17379433124215_wp, &
      &  2.82192740389634_wp, 16.06890463641044_wp,  9.65234098548296_wp, &
      &  3.82083641461060_wp,  7.56419407507952_wp,  9.78311000466841_wp, &
      &  1.76235819786628_wp,  2.68530023500750_wp,  1.51348132175757_wp, &
      &  0.45202239001674_wp, 11.19001079633841_wp,  4.96506533248325_wp, &
      &  2.66621400532870_wp, 14.32412088765434_wp, 11.44342301415299_wp, &
      &  3.97654981317824_wp,  5.81941032632342_wp,  7.99202797599838_wp, &
      &  2.54149210841772_wp, 15.77580817866965_wp,  3.17776275523477_wp, &
      & -0.32730049310577_wp,  7.27109761733873_wp,  3.30078389900605_wp, &
      &  1.88708009477726_wp,  1.23361294399218_wp,  9.77933055324686_wp, &
      &  4.75587269630075_wp,  9.73813453275203_wp,  9.65630940947558_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      &  6.76677404471523_wp,  0.00000000000000_wp,  0.02876748346766_wp, &
      &  0.00000000000000_wp, 17.00934062034656_wp,  0.00000000000000_wp, &
      & -2.33817878686658_wp,  0.00000000000000_wp, 12.92822661441416_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine formamide

subroutine hexamine(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 22
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "C", "C", &
      & "C", "C", "C", "C", "N", "N", "N", "N"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  4.38170700556783_wp,  1.20583397604383_wp, -1.20583397604383_wp, &
      & -1.20583397604383_wp,  4.38170700556783_wp,  1.20583397604383_wp, &
      &  8.75679997114795_wp,  1.20583397604383_wp,  1.20583397604383_wp, &
      & -1.20583397604383_wp,  1.20583397604383_wp,  4.38170700556783_wp, &
      &  1.20583397604383_wp,  4.38170700556783_wp, -1.20583397604383_wp, &
      &  1.20583397604383_wp, -1.20583397604383_wp,  4.38170700556783_wp, &
      &  2.18754648279006_wp,  5.36341951231406_wp,  5.36341951231406_wp, &
      &  5.36341951231406_wp,  2.18754648279006_wp,  5.36341951231406_wp, &
      &  1.20583397604383_wp,  8.75679997114795_wp,  1.20583397604383_wp, &
      &  4.38170700556783_wp, -1.20583397604383_wp,  1.20583397604383_wp, &
      &  1.20583397604383_wp,  1.20583397604383_wp,  8.75679997114795_wp, &
      &  5.36341951231406_wp,  5.36341951231406_wp,  2.18754648279006_wp, &
      &  3.19325850606310_wp,  0.00000000000000_wp,  0.00000000000000_wp, &
      &  0.00000000000000_wp,  3.19325850606310_wp,  0.00000000000000_wp, &
      &  9.94543744322375_wp,  0.00000000000000_wp,  0.00000000000000_wp, &
      &  0.00000000000000_wp,  0.00000000000000_wp,  3.19325850606310_wp, &
      &  0.00000000000000_wp,  9.94543744322375_wp,  0.00000000000000_wp, &
      &  0.00000000000000_wp,  0.00000000000000_wp,  9.94543744322375_wp, &
      &  1.63933705409501_wp,  1.63933705409501_wp,  1.63933705409501_wp, &
      &  4.92991643426288_wp, -4.92991643426288_wp,  4.92991643426288_wp, &
      &  4.92991643426288_wp,  4.92991643426288_wp, -4.92991643426288_wp, &
      & -4.92991643426288_wp,  4.92991643426288_wp,  4.92991643426288_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      & -6.56930092047323_wp,  6.56930092047323_wp,  6.56930092047323_wp, &
      &  6.56930092047323_wp, -6.56930092047323_wp,  6.56930092047323_wp, &
      &  6.56930092047323_wp,  6.56930092047323_wp, -6.56930092047323_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine hexamine

subroutine hexdio(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 32
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", &
      & "O", "O", "O", "O"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  7.46101505127135_wp,  3.49958304377930_wp,  3.42512785077487_wp, &
      &  2.81285672048463_wp,  9.47565163152575_wp,  9.25020735423061_wp, &
      &  6.81510680332937_wp,  0.24056208298132_wp,  3.13562187188454_wp, &
      &  3.45895394099769_wp,  6.21681964329885_wp,  9.53971333312093_wp, &
      &  3.58178611219789_wp,  4.56047505780689_wp,  0.95695709993511_wp, &
      &  6.69227463212917_wp, 10.53673261812442_wp, 11.71837810507036_wp, &
      &  3.33574382465534_wp,  2.42054966292829_wp, 11.72839365133746_wp, &
      &  6.93812794710065_wp,  8.39680722324582_wp,  0.94694155366802_wp, &
      &  1.17503144695824_wp, 10.29390286429017_wp,  3.65983178405279_wp, &
      &  9.09884032479774_wp,  4.31764530397264_wp,  9.01550342095268_wp, &
      & 11.04809239545941_wp,  0.42632212035024_wp,  3.00277415441725_wp, &
      & -0.77422062370343_wp,  6.40239070809669_wp,  9.67256105058823_wp, &
      &  0.96659470106005_wp,  4.00716336969275_wp,  5.00626135297808_wp, &
      &  9.30727707069593_wp,  9.98323195743920_wp,  7.66907385202739_wp, &
      &  0.70335590954947_wp,  1.48740310694891_wp,  7.22139783114542_wp, &
      &  9.57051586220652_wp,  7.46347169469536_wp,  5.45393737386005_wp, &
      &  4.40457268666817_wp,  2.10420957894500_wp,  5.68845233456690_wp, &
      &  5.86929908508781_wp,  8.08027816669145_wp,  6.98688287043858_wp, &
      &  5.92032167927867_wp,  2.11800457663364_wp,  3.25051719509950_wp, &
      &  4.35355009247732_wp,  8.09426213695117_wp,  9.42481800990597_wp, &
      &  4.32747187766866_wp,  2.61197887742953_wp,  0.87569889437190_wp, &
      &  5.94639989408732_wp,  8.58804746517598_wp, 11.79963631063357_wp, &
      &  2.02106164767102_wp,  0.91670594229566_wp,  0.72716645350520_wp, &
      &  8.25281012408497_wp,  6.89277453004211_wp, 11.94816875150028_wp, &
      &  0.72282008437042_wp,  0.32937919138762_wp,  3.21876980315853_wp, &
      &  9.55105168738556_wp,  6.30563675170515_wp,  9.45656540184695_wp, &
      &  1.55864576624503_wp,  2.04543910933998_wp,  5.41482005164707_wp, &
      &  8.71522600551096_wp,  8.02150769708643_wp,  7.26051515335841_wp, &
      &  5.46130730413206_wp,  2.11687074120718_wp,  7.75411150901215_wp, &
      &  4.81256446762392_wp,  8.09312830152470_wp,  4.92122369599333_wp, &
      & -0.84735300871032_wp,  0.04384163648992_wp, 11.31889008981309_wp, &
      & 11.12122478046630_wp,  6.01991022423637_wp,  1.35644511519239_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      & 12.37386216010654_wp,  0.00000000000000_wp,  0.07882896316201_wp, &
      &  0.00000000000000_wp, 11.95225566129496_wp,  0.00000000000000_wp, &
      & -2.09997923896887_wp,  0.00000000000000_wp, 12.59652211553943_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine hexdio

subroutine imdazole(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 36
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", &
      & "N", "N", "N", "N", "N", "N", "N", "N"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -5.68467288314536_wp,  2.66999345675024_wp, 15.77599715124073_wp, &
      &  6.68074730529346_wp,  7.72463178792410_wp,  8.45992406198561_wp, &
      & 11.05810794172651_wp,  7.43947217816855_wp,  0.40591308267390_wp, &
      & -1.30750121928338_wp,  2.38483384699469_wp,  7.72217514450009_wp, &
      & -0.19898811734433_wp,  0.34638672278457_wp,  2.49935122506750_wp, &
      &  9.94959483978745_wp,  5.40121402652950_wp,  5.62892597467757_wp, &
      &  5.57242317592548_wp,  9.76307891213422_wp, 13.68274798141821_wp, &
      & -4.57615978120630_wp,  4.70825160838929_wp, 10.55317323180814_wp, &
      &  1.75064189845950_wp,  7.05661374916607_wp,  6.27502319519063_wp, &
      &  7.99996482398363_wp,  2.00178644542114_wp,  1.85306503198336_wp, &
      &  3.62260418755058_wp,  3.05285188575272_wp,  9.90688703872400_wp, &
      & -2.62652976540248_wp,  8.10767918949765_wp, 14.32903417450235_wp, &
      &  4.34580221706315_wp,  6.87784569692701_wp,  1.74516169389826_wp, &
      &  5.40480450537997_wp,  1.82320736575315_wp,  6.38292653327574_wp, &
      &  1.02763284151800_wp,  3.23161993799178_wp, 14.43674854001637_wp, &
      & -0.03136944679882_wp,  8.28625826916564_wp,  9.79917267320997_wp, &
      &  0.66442755990755_wp,  2.15504320056478_wp,  2.92756307112851_wp, &
      &  9.08617916253557_wp,  7.20968153173863_wp,  5.20052515604548_wp, &
      &  4.70900749867360_wp,  7.95442243435401_wp, 13.25453613535720_wp, &
      & -3.71293307652549_wp,  2.89978410318016_wp, 10.98157405044023_wp, &
      &  1.70906793282250_wp,  5.61494200441786_wp,  4.82182412360670_wp, &
      &  8.04153878962062_wp,  0.56011470067292_wp,  3.30626410356729_wp, &
      &  3.66436712575865_wp,  4.49452363050094_wp, 11.36027508287900_wp, &
      & -2.66810373103947_wp,  9.54935093424587_wp, 12.87564613034734_wp, &
      &  3.01656915210589_wp,  5.55692742509714_wp,  2.56681443294207_wp, &
      &  6.73403757033724_wp,  0.50228909392329_wp,  5.56127379423192_wp, &
      &  2.35667693390419_wp,  4.55253820982165_wp, 13.61528477354364_wp, &
      & -1.36060251175609_wp,  9.60717654099550_wp, 10.62063643968271_wp, &
      &  2.32474056939213_wp,  3.34991676748613_wp,  1.39518449226322_wp, &
      &  7.42586615305099_wp,  8.40474407123106_wp,  6.73290373491077_wp, &
      &  3.04869448918902_wp,  6.75954886743266_wp, 14.78691471422249_wp, &
      & -2.05262006704092_wp,  1.70472156368773_wp,  9.44919547157494_wp, &
      &  0.24018413783916_wp,  3.47539455468141_wp,  5.03026086950489_wp, &
      &  9.51042258460396_wp,  8.53003288585527_wp,  3.09782735766910_wp, &
      &  5.13306194817091_wp,  6.63407108023738_wp, 11.15164936440974_wp, &
      & -4.13698752602281_wp,  1.57943274906352_wp, 13.08427184881661_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      & 14.12785038485468_wp,  0.00000000000000_wp,  0.07428379488246_wp, &
      &  0.00000000000000_wp, 10.10946657978165_wp,  0.00000000000000_wp, &
      & -8.75447239598999_wp,  0.00000000000000_wp, 16.10778045167759_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine imdazole

subroutine naph(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 36
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", &
      & "C", "C", "C", "C", "C", "C", "C", "C"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -2.24575003468185_wp,  0.75022110717662_wp,  6.38972954583452_wp, &
      &  0.67085262732418_wp,  6.30960517569777_wp,  7.16035969068748_wp, &
      &  8.33255654907956_wp, 10.36854702986568_wp,  7.20174468375339_wp, &
      &  5.41595388707353_wp,  4.80916296134453_wp,  6.43092556632935_wp, &
      &  0.35961480275998_wp,  3.63961171894753_wp,  3.78568751639023_wp, &
      & -1.93451221011765_wp,  9.19899578746868_wp,  9.76440172013176_wp, &
      &  5.72719171163773_wp,  7.47915641809477_wp,  9.80559774062660_wp, &
      &  8.02131872451535_wp,  1.91977234957362_wp,  3.82707250945614_wp, &
      & -6.72723455777846_wp,  4.62567059482854_wp, 13.25510305307043_wp, &
      &  5.15233715042079_wp, 10.18505466334969_wp,  0.29498618345156_wp, &
      & 12.81404107217617_wp,  6.49309754221376_wp,  0.33618220394640_wp, &
      &  0.93446936397692_wp,  0.93371347369261_wp, 13.29648804613634_wp, &
      & 11.62030134068128_wp,  7.71329343365946_wp,  4.68916537871049_wp, &
      &  2.12820909547181_wp,  2.15390936513831_wp,  8.94369384394333_wp, &
      & -5.53349482628357_wp,  3.40547470338283_wp,  8.90230885087742_wp, &
      &  3.95859741892590_wp,  8.96485877190398_wp,  4.64778038564457_wp, &
      & -1.72645340936161_wp,  0.21202722474866_wp,  4.47014616883197_wp, &
      &  0.15155600200394_wp,  5.77141129326981_wp,  9.07994306769002_wp, &
      &  7.81325992375932_wp, 10.90674091229364_wp,  9.12113908818486_wp, &
      &  5.93525051239377_wp,  5.34735684377249_wp,  4.51153116189789_wp, &
      & -0.27193152978014_wp,  1.83246702173594_wp,  3.02639572580190_wp, &
      & -1.30296587757753_wp,  7.39185109025709_wp, 10.52369351072009_wp, &
      &  6.35873804417785_wp,  9.28630111530636_wp, 10.56507850378600_wp, &
      &  7.38977239197524_wp,  3.72691704678521_wp,  3.06778071886782_wp, &
      &  0.41328301294591_wp,  1.18958233493119_wp,  0.50852518876884_wp, &
      & -1.98818042030358_wp,  6.74896640345234_wp, 13.04156404775315_wp, &
      &  5.67352350145180_wp,  9.92918580211111_wp, 13.08294904081907_wp, &
      &  8.07479796213021_wp,  4.36980173358996_wp,  0.54991018183475_wp, &
      & -7.34668664576963_wp,  2.82343918446495_wp, 12.47861475851407_wp, &
      &  5.77178923841196_wp,  8.38282325298610_wp,  1.07147447800792_wp, &
      & 13.43349316016734_wp,  8.29532895257734_wp,  1.11267049850275_wp, &
      &  0.31501727598575_wp,  2.73594488405619_wp, 12.51999975157999_wp, &
      & 12.77738039338718_wp,  8.98205527587201_wp,  3.54720413169076_wp, &
      &  0.97113004276591_wp,  3.42267120735086_wp, 10.08546611839198_wp, &
      & -6.69057387898947_wp,  2.13671286117029_wp, 10.04427009789715_wp, &
      &  5.11567647163180_wp,  7.69609692969144_wp,  3.50581913862485_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      & 15.32338913522622_wp,  0.00000000000000_wp,  0.08265773642461_wp, &
      &  0.00000000000000_wp, 11.11875453101718_wp,  0.00000000000000_wp, &
      & -9.23662022637015_wp,  0.00000000000000_wp, 13.50875885652912_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine naph

subroutine oxaca(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 32
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "H", "H", "H", "H", "H", "H", "H", "C", "C", "C", "C", "C", "C", &
      & "C", "C", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", &
      & "O", "O", "O", "O"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  1.19940890862720_wp,  4.37244734958504_wp,  9.97642894488042_wp, &
      &  5.09715715966623_wp,  4.37244734958504_wp,  4.07632733070702_wp, &
      &  7.49578600434957_wp,  2.75503111373500_wp,  9.97642894488042_wp, &
      & 11.39372322795967_wp,  2.75503111373500_wp,  4.07632733070702_wp, &
      & 11.39372322795967_wp,  9.88250957705503_wp,  1.82396325603746_wp, &
      &  7.49578600434957_wp,  9.88250957705503_wp,  7.72406487021086_wp, &
      &  5.09715715966623_wp, 11.49992581290508_wp,  1.82396325603746_wp, &
      &  1.19940890862720_wp, 11.49992581290508_wp,  7.72406487021086_wp, &
      &  0.72074138608857_wp,  0.82108582133058_wp, 10.81981352959811_wp, &
      &  5.57582468220487_wp,  0.82108582133058_wp,  4.91971191542471_wp, &
      &  7.01711848181093_wp,  6.30639264198946_wp, 10.81981352959811_wp, &
      & 11.87239075049831_wp,  6.30639264198946_wp,  4.91971191542471_wp, &
      & 11.87239075049831_wp, 13.43387110530949_wp,  0.98057867131977_wp, &
      &  7.01711848181093_wp, 13.43387110530949_wp,  6.88068028549317_wp, &
      &  5.57582468220487_wp,  7.94875325722170_wp,  0.98057867131977_wp, &
      &  0.72074138608857_wp,  7.94875325722170_wp,  6.88068028549317_wp, &
      &  1.95397638493860_wp, 14.15291173825837_wp,  9.08769094310419_wp, &
      &  4.34258968335484_wp, 14.15291173825837_wp,  3.18758932893078_wp, &
      &  8.25054245323204_wp,  7.22952365170174_wp,  9.08769094310419_wp, &
      & 10.63896677907720_wp,  7.22952365170174_wp,  3.18758932893078_wp, &
      & 10.63896677907720_wp,  0.10204518838171_wp,  2.71270125781369_wp, &
      &  8.25054245323204_wp,  0.10204518838171_wp,  8.61299184455818_wp, &
      &  4.34258968335484_wp,  7.02543327493833_wp,  2.71270125781369_wp, &
      &  1.95397638493860_wp,  7.02543327493833_wp,  8.61299184455818_wp, &
      &  0.41384993065914_wp,  3.22198233686684_wp, 11.28676475272996_wp, &
      &  5.88271613763430_wp,  3.22198233686684_wp,  5.38666313855656_wp, &
      &  6.71041599895258_wp,  3.90549612645320_wp, 11.28676475272996_wp, &
      & 12.17928220592774_wp,  3.90549612645320_wp,  5.38666313855656_wp, &
      & 12.17928220592774_wp, 11.03316356234431_wp,  0.51362744818792_wp, &
      &  6.71041599895258_wp, 11.03316356234431_wp,  6.41391803493240_wp, &
      &  5.88271613763430_wp, 10.34946080018688_wp,  0.51362744818792_wp, &
      &  0.41384993065914_wp, 10.34946080018688_wp,  6.41391803493240_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      & 12.59303405982249_wp,  0.00000000000000_wp,  0.00000000000000_wp, &
      &  0.00000000000000_wp, 14.25501834272567_wp,  0.00000000000000_wp, &
      &  0.00000000000000_wp,  0.00000000000000_wp, 11.80042224755668_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine oxaca

subroutine oxacb(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 16
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "H", "H", "H", "C", "C", "C", "C", "O", "O", "O", "O", "O", "O", &
      & "O", "O"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  4.37188043187181_wp, 10.56507850378600_wp,  1.84078181486334_wp, &
      &  3.41284463365485_wp,  4.74056591804349_wp,  2.82646274560219_wp, &
      &  1.23758136798481_wp,  1.08394666769901_wp,  7.41131526507804_wp, &
      & -2.15372039256724_wp,  6.90845925344152_wp, 15.59571731843305_wp, &
      &  1.46038002928486_wp,  0.02683410509297_wp, -0.00321253370831_wp, &
      & 10.67468259501080_wp,  5.85134669083548_wp, -4.49943691734894_wp, &
      &  4.14908177057175_wp, 11.62219106639205_wp,  9.25549858622077_wp, &
      & -0.71488323638518_wp,  5.79767848064954_wp,  4.58163998576754_wp, &
      & -1.71794964366313_wp,  1.12646549619139_wp,  7.48388073237170_wp, &
      &  5.15233715042079_wp,  6.95097808193390_wp,  6.35325783961661_wp, &
      &  7.32741144351975_wp, 10.52255967529362_wp,  1.76821634756968_wp, &
      &  0.45712464943583_wp,  4.69804708955112_wp,  2.89883924032477_wp, &
      &  2.43604541375662_wp, 10.48854461249972_wp,  1.92279591071086_wp, &
      &  5.34867965177003_wp,  4.66403202675721_wp,  2.74425967718359_wp, &
      &  3.17341638609999_wp,  1.16048055898529_wp,  7.32930116923052_wp, &
      &  0.26078214808658_wp,  6.98499314472780_wp,  6.50783740275779_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      &  9.95985605039694_wp,  0.00000000000000_wp,  0.08219040725633_wp, &
      &  0.00000000000000_wp, 11.64905767476724_wp,  0.00000000000000_wp, &
      & -4.35038574677463_wp,  0.00000000000000_wp,  9.16996090781294_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine oxacb

subroutine pyrazine(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 20
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "H", "H", "H", "H", "H", "H", "H", "C", "C", "C", "C", "C", "C", &
      & "C", "C", "N", "N", "N", "N"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  2.40505391209996_wp,  3.54607029626430_wp,  1.66560424147474_wp, &
      & 15.15503328268094_wp,  3.54607029626430_wp,  1.66560424147474_wp, &
      & 11.18490853691933_wp,  1.85533270283628_wp,  5.14591208300416_wp, &
      &  6.37498968529049_wp,  1.85533270283628_wp,  5.14591208300416_wp, &
      & 15.15503328268094_wp,  7.25673570193687_wp,  5.29501144158410_wp, &
      &  2.40505391209996_wp,  7.25673570193687_wp,  5.29501144158410_wp, &
      &  6.37498968529049_wp,  8.94747329536488_wp,  1.81470360005468_wp, &
      & 11.18490853691933_wp,  8.94747329536488_wp,  1.81470360005468_wp, &
      &  1.32356388782495_wp,  1.95718891864691_wp,  0.90309991717810_wp, &
      & 16.23652330695595_wp,  1.95718891864691_wp,  0.90309991717810_wp, &
      & 10.10360748521540_wp,  3.44421408045367_wp,  4.38340775870752_wp, &
      &  7.45647970956550_wp,  3.44421408045367_wp,  4.38340775870752_wp, &
      & 16.23652330695595_wp,  8.84561707955425_wp,  6.05751576588074_wp, &
      &  1.32356388782495_wp,  8.84561707955425_wp,  6.05751576588074_wp, &
      &  7.45647970956550_wp,  7.35859191774749_wp,  2.57720792435132_wp, &
      & 10.10360748521540_wp,  7.35859191774749_wp,  2.57720792435132_wp, &
      &  2.65525359620622_wp,  0.00000000000000_wp,  0.00000000000000_wp, &
      & 14.90483359857468_wp,  0.00000000000000_wp,  0.00000000000000_wp, &
      & 11.43529719359667_wp,  5.40140299910058_wp,  3.48030784152942_wp, &
      &  6.12479000118423_wp,  5.40140299910058_wp,  3.48030784152942_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      & 17.56002842431129_wp,  0.00000000000000_wp,  0.00000000000000_wp, &
      &  0.00000000000000_wp, 10.80272568485845_wp,  0.00000000000000_wp, &
      &  0.00000000000000_wp,  0.00000000000000_wp,  6.96065234373763_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine pyrazine

subroutine pyrazole(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 72
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", &
      & "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", &
      & "N", "N", "N", "N", "N", "N", "N", "N", "N", "N", "N", "N", "N", "N", &
      & "N", "N"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  0.71450529124302_wp,  5.73626239504944_wp,  6.61328409741888_wp, &
      &  8.25942416407267_wp, 18.69448953895753_wp,  6.43961830459891_wp, &
      &  0.71450529124302_wp,  6.47911357195405_wp,  0.08692738269553_wp, &
      &  8.25942416407267_wp, 17.95182733462401_wp, 12.96616399189334_wp, &
      & 14.55296667122887_wp,  2.21532545073841_wp,  2.99597114185847_wp, &
      &  7.00804779839923_wp, 22.21542648326856_wp, 10.05693126015932_wp, &
      & 14.55296667122887_wp, 10.00005051626507_wp,  9.52251682915290_wp, &
      &  7.00804779839923_wp, 14.43089039031298_wp,  3.53038557286489_wp, &
      &  5.01986737809565_wp,  6.37612352071695_wp,  8.79894085449817_wp, &
      & 12.56459727835422_wp, 18.05462841329002_wp,  4.25415052009069_wp, &
      &  5.01986737809565_wp,  5.83925244628653_wp,  2.27239516720374_wp, &
      & 12.56459727835422_wp, 18.59168846029152_wp, 10.78050723481405_wp, &
      & 10.30202868484649_wp, 24.21381142241031_wp,  2.98803429387322_wp, &
      &  2.75710981201685_wp,  0.21712948416774_wp, 10.06505708071564_wp, &
      & 10.30202868484649_wp, 12.43250545117123_wp,  9.51457998116766_wp, &
      &  2.75710981201685_wp, 11.99843545540682_wp,  3.53851139342121_wp, &
      &  8.05684556787787_wp,  2.47629657139608_wp,  7.09062881195998_wp, &
      &  0.51211566761930_wp, 21.95464433518198_wp,  5.96246256262889_wp, &
      &  8.05684556787787_wp,  9.73926836817849_wp,  0.56408312466554_wp, &
      &  0.51211566761930_wp, 14.69167253839956_wp, 12.48881927735224_wp, &
      & 22.42159555831382_wp,  2.22364024386581_wp, 12.48428393564639_wp, &
      & -0.21278311503297_wp, 22.20730066271224_wp,  0.56880743894247_wp, &
      & 22.42159555831382_wp,  9.99173572313767_wp,  5.95773824835196_wp, &
      & -0.21278311503297_wp, 14.43901621086930_wp,  7.09535312623691_wp, &
      &  4.88720863319943_wp, 24.21059888870200_wp,  3.92647208184277_wp, &
      & 12.43212750602907_wp,  0.22015304530498_wp,  9.12661929274610_wp, &
      &  4.88720863319943_wp, 12.43571798487954_wp, 10.45301776913720_wp, &
      & 12.43212750602907_wp, 11.99522292169851_wp,  2.60007360545166_wp, &
      & 10.49213509135019_wp,  5.92863647240606_wp, 10.62006952196948_wp, &
      &  2.94721621852054_wp, 18.50230443417199_wp,  2.43302185261939_wp, &
      & 10.49213509135019_wp,  6.28673949459742_wp,  4.09352383467504_wp, &
      &  2.94721621852054_wp, 18.14401243940955_wp,  8.95956753991382_wp, &
      &  4.49546849335632_wp,  1.47058454812304_wp,  5.02137915866426_wp, &
      & 12.04019839361489_wp, 22.96035635845502_wp,  8.03152324335352_wp, &
      &  4.49546849335632_wp, 10.74479141888045_wp, 11.54792484595870_wp, &
      & 12.04019839361489_wp, 13.68596051512652_wp,  1.50497755605909_wp, &
      & 10.87234790435758_wp,  4.51890109216990_wp, 12.05758387015400_wp, &
      &  3.32742903152794_wp, 19.91185084183708_wp,  0.99531853186379_wp, &
      & 10.87234790435758_wp,  7.69647487483359_wp,  5.53122715543064_wp, &
      &  3.32742903152794_wp, 16.73446603174446_wp,  7.52186421915822_wp, &
      &  6.09927870408881_wp,  2.85329685069516_wp,  6.63898436708539_wp, &
      & 13.64419757691845_wp, 21.57764405588290_wp,  6.41410700750348_wp, &
      &  6.09927870408881_wp,  9.36226808887941_wp,  0.11243867979095_wp, &
      & 13.64419757691845_wp, 15.06867281769864_wp, 25.99355509681570_wp, &
      &  9.26570310505894_wp,  2.62199442369662_wp,-13.09126383394647_wp, &
      &  1.72078423222929_wp, 21.80875751031035_wp, 13.09126383394647_wp, &
      &  9.26570310505894_wp,  9.59338154330687_wp,  6.48818425536575_wp, &
      &  1.72078423222929_wp, 14.83755936327119_wp,  6.56490711922311_wp, &
      &  4.59789162688018_wp,  4.82258001389101_wp,  7.53301360085178_wp, &
      & 12.14281049970983_wp, 19.60836089268704_wp,  5.52007777373708_wp, &
      &  4.59789162688018_wp,  7.39298492568355_wp,  1.00646791355735_wp, &
      & 12.14281049970983_wp, 17.03795598089450_wp, 12.04643448846044_wp, &
      & 10.71890217664287_wp,  1.37156292087856_wp,  1.76462586871921_wp, &
      &  3.17398330381322_wp, 23.05937798569949_wp, 11.28846550586965_wp, &
      & 10.71890217664287_wp, 10.84400201869600_wp,  8.29098258344257_wp, &
      &  3.17398330381322_wp, 13.58693888788205_wp,  4.76191981857522_wp, &
      &  2.28959167117177_wp,  4.53496376071146_wp,  6.47060980625557_wp, &
      &  9.83432157143034_wp, 19.89578817329551_wp,  6.58248156833329_wp, &
      &  2.28959167117177_wp,  7.68041220629202_wp, 12.99696652097893_wp, &
      &  9.83432157143034_wp, 16.75033972771495_wp,  0.05593588103886_wp, &
      & 13.00301364325340_wp,  2.52278382388107_wp,  1.79826298637096_wp, &
      &  5.45828374299483_wp, 21.90796811012590_wp, 11.25463941564683_wp, &
      & 13.00301364325340_wp,  9.69259214312241_wp,  8.32480867366539_wp, &
      &  5.45828374299483_wp, 14.73834876345564_wp,  4.72828270092347_wp, &
      &  2.18641264736360_wp,  2.52939786386878_wp,  4.93766430967705_wp, &
      &  9.73133152019325_wp, 21.90154304270927_wp,  8.11523809234074_wp, &
      &  2.18641264736360_wp,  9.68616707570579_wp, 11.46420999697148_wp, &
      &  9.73133152019325_wp, 14.74477383087227_wp,  1.58888137761738_wp, &
      & 13.13850697671578_wp,  4.42932809347929_wp,  0.14872141343778_wp, &
      &  5.59358810388613_wp, 20.00142384052769_wp, 12.90436996115108_wp, &
      & 13.13850697671578_wp,  7.78604787352420_wp,  6.67526710073222_wp, &
      &  5.59358810388613_wp, 16.64489303305385_wp,  6.37782427385665_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      & 15.08965311945735_wp,  0.00000000000000_wp,  0.00000000000000_wp, &
      &  0.00000000000000_wp, 24.43086078220792_wp,  0.00000000000000_wp, &
      &  0.00000000000000_wp,  0.00000000000000_wp, 13.05302145473757_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine pyrazole

subroutine succinic(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 28
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "C", "C", &
      & "C", "C", "C", "C", "C", "C", "O", "O", "O", "O", "O", "O", "O", "O"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  9.80540876805552_wp, 13.75191194243248_wp,  8.69991922725370_wp, &
      &  0.24736509554010_wp,  2.76466871485993_wp,  0.93635908968769_wp, &
      &  9.94354771751298_wp, 11.02314801607722_wp,  3.87563846022299_wp, &
      &  0.10922614608264_wp,  5.49362161378627_wp,  5.76045088414733_wp, &
      &  1.88821393020372_wp, 14.74326205030365_wp,  1.37742107058196_wp, &
      &  8.16474890596298_wp,  1.77350757955984_wp,  8.25866827378836_wp, &
      &  1.75007498074626_wp, 10.03179790820605_wp,  6.20170183761267_wp, &
      &  8.30269888284936_wp,  6.48497172165744_wp,  3.43457647932873_wp, &
      &  5.51365270632046_wp, 14.60757974427019_wp,  5.71755411051279_wp, &
      &  4.53912115727516_wp,  1.90900091302222_wp,  3.91872420642860_wp, &
      &  5.65179165577792_wp, 10.16729124166842_wp,  0.89327334348209_wp, &
      &  4.40117118038878_wp,  6.34928941562399_wp,  8.74281600088823_wp, &
      &  0.58222449148896_wp, 15.40031967993919_wp,  9.34904000890400_wp, &
      &  9.47073834467774_wp,  1.11626097735322_wp,  0.28704933546632_wp, &
      &  0.72017446837534_wp,  9.37455130599943_wp,  4.52494821444437_wp, &
      &  9.33259939522028_wp,  7.14202935129298_wp,  5.11133010249703_wp, &
      &  2.50105197820719_wp, 15.96213513375181_wp,  7.31588411668404_wp, &
      &  7.55191085795951_wp,  0.55463449611168_wp,  2.32020522768628_wp, &
      &  2.63900195509357_wp,  8.81292482475789_wp,  2.49160334965333_wp, &
      &  7.41377190850205_wp,  7.70384480510560_wp,  7.14448599471699_wp, &
      &  2.41790404693321_wp,  1.34000450150867_wp,  5.94205352495255_wp, &
      &  7.63505878923349_wp, 15.17657615578375_wp,  3.69403581941777_wp, &
      &  2.55604299639067_wp,  6.91828582713754_wp,  1.11796173049292_wp, &
      &  7.49691983977603_wp,  9.59829483015487_wp,  8.51831658644848_wp, &
      &  4.26851243549256_wp, 14.20941453701046_wp,  7.14013962558221_wp, &
      &  5.78426142810306_wp,  2.30716612028195_wp,  2.49594971878811_wp, &
      &  4.40665138495002_wp, 10.56545644892816_wp,  2.31585885855150_wp, &
      &  5.64612247864560_wp,  5.95112420836425_wp,  7.32023048581882_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      & 10.32902927477458_wp,  0.00000000000000_wp, -0.01221556493957_wp, &
      &  0.00000000000000_wp, 16.51663942776202_wp,  0.00000000000000_wp, &
      & -0.27618057804081_wp,  0.00000000000000_wp,  9.64836538053264_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine succinic

subroutine triazine(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 54
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", &
      & "C", "C", "C", "C", "C", "C", "C", "C", "N", "N", "N", "N", "N", "N", &
      & "N", "N", "N", "N", "N", "N", "N", "N", "N", "N", "N", "N"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  2.26483626436065_wp,  3.92288160299230_wp,  3.27602849219493_wp, &
      &  6.82795693816262_wp, 11.82647041572654_wp,  9.82827444915587_wp, &
      &  4.56330964637305_wp,  5.24984699709664_wp, 12.01223045309546_wp, &
      &  6.82795693816262_wp,  9.17253962751787_wp,  5.46017346870560_wp, &
      &  2.26483626436065_wp, -3.92288160299230_wp,  3.27602849219493_wp, &
      &  4.52967252872130_wp,  0.00000000000000_wp,  9.82827444915587_wp, &
      &  6.82814591073370_wp,  1.32696539410434_wp,  5.46017346870560_wp, &
      & 11.35762946688392_wp,  9.17253962751787_wp, 12.01223045309546_wp, &
      &  4.56330964637305_wp, -5.24984699709664_wp,  7.64412947264519_wp, &
      &  6.82814591073370_wp, -1.32696539410434_wp,  1.09207248825534_wp, &
      & 11.35762946688392_wp,  1.32696539410434_wp, 12.01223045309546_wp, &
      & 13.62246573124457_wp,  5.24984699709664_wp,  5.46017346870560_wp, &
      &  6.82795693816262_wp, -9.17253962751787_wp,  1.09207248825534_wp, &
      & 11.35762946688392_wp, -1.32696539410434_wp,  7.64412947264519_wp, &
      &  6.82795693816262_wp,-11.82647041572654_wp,  9.82827444915587_wp, &
      & 13.65610284889632_wp,  0.00000000000000_wp,  3.27602849219493_wp, &
      & 11.35762946688392_wp, -9.17253962751787_wp,  7.64412947264519_wp, &
      & 13.62246573124457_wp, -5.24984699709664_wp,  1.09207248825534_wp, &
      &  1.23134527313926_wp,  2.13293340974874_wp,  3.27602849219493_wp, &
      &  7.86144792938401_wp, 13.61641860897010_wp,  9.82827444915587_wp, &
      &  1.23134527313926_wp, -2.13293340974874_wp,  3.27602849219493_wp, &
      &  2.46287951884959_wp,  0.00000000000000_wp,  9.82827444915587_wp, &
      &  6.63010265624476_wp,  5.24984699709664_wp, 12.01223045309546_wp, &
      &  7.86144792938401_wp,  7.38259143427430_wp,  5.46017346870560_wp, &
      &  7.86144792938401_wp,  3.11691358734790_wp,  5.46017346870560_wp, &
      & 10.32432744823361_wp,  7.38259143427430_wp, 12.01223045309546_wp, &
      & 10.32432744823361_wp,  3.11691358734790_wp, 12.01223045309546_wp, &
      & 11.55567272137286_wp,  5.24984699709664_wp,  5.46017346870560_wp, &
      &  6.63010265624476_wp, -5.24984699709664_wp,  7.64412947264519_wp, &
      &  7.86144792938401_wp, -3.11691358734790_wp,  1.09207248825534_wp, &
      &  7.86144792938401_wp, -7.38259143427430_wp,  1.09207248825534_wp, &
      & 10.32432744823361_wp, -3.11691358734790_wp,  7.64412947264519_wp, &
      & 10.32432744823361_wp, -7.38259143427430_wp,  7.64412947264519_wp, &
      & 11.55567272137286_wp, -5.24984699709664_wp,  1.09207248825534_wp, &
      &  7.86144792938401_wp,-13.61641860897010_wp,  9.82827444915587_wp, &
      & 15.72289585876803_wp,  0.00000000000000_wp,  3.27602849219493_wp, &
      &  1.29861950844275_wp,  2.24915154096124_wp,  9.82827444915587_wp, &
      &  7.79436266665160_wp, 13.50020047775760_wp,  3.27602849219493_wp, &
      &  1.29861950844275_wp, -2.24915154096124_wp,  9.82827444915587_wp, &
      &  2.59705004431443_wp,  0.00000000000000_wp,  3.27602849219493_wp, &
      &  6.49574315820884_wp,  5.24984699709664_wp,  5.46017346870560_wp, &
      &  7.79436266665160_wp,  7.49880956548680_wp, 12.01223045309546_wp, &
      &  7.79436266665160_wp,  3.00069545613540_wp, 12.01223045309546_wp, &
      & 10.39141271096602_wp,  7.49880956548680_wp,  5.46017346870560_wp, &
      & 10.39141271096602_wp,  3.00069545613540_wp,  5.46017346870560_wp, &
      & 11.68984324683770_wp,  5.24984699709664_wp, 12.01223045309546_wp, &
      &  6.49574315820884_wp, -5.24984699709664_wp,  1.09207248825534_wp, &
      &  7.79436266665160_wp, -3.00069545613540_wp,  7.64412947264519_wp, &
      &  7.79436266665160_wp, -7.49880956548680_wp,  7.64412947264519_wp, &
      & 10.39141271096602_wp, -3.00069545613540_wp,  1.09207248825534_wp, &
      & 10.39141271096602_wp, -7.49880956548680_wp,  1.09207248825534_wp, &
      & 11.68984324683770_wp, -5.24984699709664_wp,  7.64412947264519_wp, &
      &  7.79436266665160_wp,-13.50020047775760_wp,  3.27602849219493_wp, &
      & 15.58872533330319_wp,  0.00000000000000_wp,  9.82827444915587_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      &  9.09287729531740_wp,-15.74932537358632_wp,  0.00000000000000_wp, &
      &  9.09287729531740_wp, 15.74932537358632_wp,  0.00000000000000_wp, &
      &  0.00000000000000_wp,  0.00000000000000_wp, 13.10427100498628_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine triazine

subroutine trioxane(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 72
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "H", "H", "C", "C", "C", "C", "C", "C", &
      & "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "O", "O", &
      & "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", &
      & "O", "O"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  2.68322153672565_wp,  3.53114146314920_wp,  7.98239037487344_wp, &
      &  7.11784086219510_wp, 11.21193161458337_wp,  0.41951910779146_wp, &
      &  1.54409487827208_wp,  2.03504561793073_wp,  5.09205490024715_wp, &
      &  7.84387348027383_wp, 12.94651084450131_wp, 12.65492616732913_wp, &
      &  4.43443035289837_wp,  5.65859466833670_wp,  2.94041320596176_wp, &
      &  6.15086821599289_wp,  8.63151115652374_wp, 10.50347344561482_wp, &
      &  2.53450012328786_wp,  0.31974159026268_wp, 12.65492616732913_wp, &
      &  9.82449499773432_wp, 12.94651084450131_wp,  5.09205490024715_wp, &
      &  6.29958962943067_wp,  5.42011128363723_wp,  0.05007773133547_wp, &
      &  7.28999487444646_wp,  7.13560428387636_wp,  7.61294899841745_wp, &
      &  4.39965939982016_wp,  0.55822497496215_wp,  0.41951910779146_wp, &
      & 10.55071658838413_wp, 11.21193161458337_wp,  7.98239037487344_wp, &
      &  7.11784086219510_wp,  1.01100325526320_wp, 10.50347344561482_wp, &
      & 11.51750026201526_wp,  8.63151115652374_wp,  2.94041320596176_wp, &
      &  7.84387348027383_wp,  2.74558248518113_wp,  7.61294899841745_wp, &
      & 10.37837360356169_wp,  7.13560428387636_wp,  0.05007773133547_wp, &
      &  4.43443035289837_wp, -4.54214471841240_wp, 13.02436754378513_wp, &
      &  6.15086821599289_wp, -1.56922823022535_wp,  5.46149627670314_wp, &
      &  9.82449499773432_wp,  2.74558248518113_wp,  0.05007773133547_wp, &
      & 11.36877884857748_wp,  5.42011128363723_wp,  7.61294899841745_wp, &
      &  6.29958962943067_wp, -4.78062810311187_wp, 10.13403206915883_wp, &
      &  7.28999487444646_wp, -3.06532407544381_wp,  2.57116080207685_wp, &
      & 10.55071658838413_wp,  1.01100325526320_wp,  2.94041320596176_wp, &
      & 13.23393812510978_wp,  5.65859466833670_wp, 10.50347344561482_wp, &
      &  7.11784086219510_wp, -9.18973613148589_wp,  5.46149627670314_wp, &
      & 11.51750026201526_wp, -1.56922823022535_wp, 13.02436754378513_wp, &
      &  7.84387348027383_wp, -7.45534587413904_wp,  2.57116080207685_wp, &
      & 10.37837360356169_wp, -3.06532407544381_wp, 10.13403206915883_wp, &
      & 13.26870907818799_wp,  0.55822497496215_wp,  7.98239037487344_wp, &
      & 14.98514694128251_wp,  3.53114146314920_wp,  0.41951910779146_wp, &
      &  9.82449499773432_wp, -7.45534587413904_wp, 10.13403206915883_wp, &
      & 11.36877884857748_wp, -4.78062810311187_wp,  2.57116080207685_wp, &
      & 15.13386835472029_wp,  0.31974159026268_wp,  5.09205490024715_wp, &
      & 16.12427359973607_wp,  2.03504561793073_wp, 12.65492616732913_wp, &
      & 10.55071658838413_wp, -9.18973613148589_wp, 13.02436754378513_wp, &
      & 13.23393812510978_wp, -4.54214471841240_wp,  5.46149627670314_wp, &
      &  1.53105577086775_wp,  2.01104610140393_wp,  7.18435920721428_wp, &
      &  7.85804642310462_wp, 12.96994344331488_wp, 14.74723047429627_wp, &
      &  2.50709910048166_wp,  0.32030850797591_wp, 14.74723047429627_wp, &
      &  9.81032205490353_wp, 12.96994344331488_wp,  7.18435920721428_wp, &
      &  6.32699065223687_wp,  5.42086717392154_wp,  2.14238203830260_wp, &
      &  7.30322295442186_wp,  7.11160476734955_wp,  9.70525330538459_wp, &
      &  7.85804642310462_wp,  2.76901508399471_wp,  9.70525330538459_wp, &
      & 10.36514552358629_wp,  7.11160476734955_wp,  2.14238203830260_wp, &
      &  9.81032205490353_wp,  2.76901508399471_wp,  2.14238203830260_wp, &
      & 11.34137782577128_wp,  5.42086717392154_wp,  9.70525330538459_wp, &
      &  6.32699065223687_wp, -4.78006118539864_wp, 12.22633637612597_wp, &
      &  7.30322295442186_wp, -3.08932359197062_wp,  4.66346510904398_wp, &
      &  7.85804642310462_wp, -7.43191327532546_wp,  4.66346510904398_wp, &
      & 10.36533449615737_wp, -3.08932359197062_wp, 12.22633637612597_wp, &
      &  9.81032205490353_wp, -7.43191327532546_wp, 12.22633637612597_wp, &
      & 11.34137782577128_wp, -4.78006118539864_wp,  4.66346510904398_wp, &
      & 15.16126937752649_wp,  0.32030850797591_wp,  7.18435920721428_wp, &
      & 16.13750167971148_wp,  2.01104610140393_wp, 14.74723047429627_wp, &
      &  1.54428385084316_wp, -2.03107719393811_wp,  0.50890313391099_wp, &
      &  2.53109861700847_wp, -0.32200926111561_wp,  8.07177440099298_wp, &
      &  6.30299113571006_wp,  4.77854940483002_wp, 10.59266849916328_wp, &
      &  7.84746395912430_wp,  7.45345614842827_wp,  3.02979723208129_wp, &
      &  7.28980590187538_wp,  3.06929249943644_wp,  3.02979723208129_wp, &
      &  9.82109349145493_wp,  7.45345614842827_wp, 10.59285747173436_wp, &
      & 10.37856257613277_wp,  3.06929249943644_wp, 10.59266849916328_wp, &
      & 11.36537734229809_wp,  4.77854940483002_wp,  3.02979723208129_wp, &
      &  6.30299113571006_wp, -5.42237895449015_wp,  5.55088030282267_wp, &
      &  7.84746395912430_wp, -2.74747221089191_wp, 13.11375156990466_wp, &
      &  7.28980590187538_wp, -7.13144688731266_wp, 13.11375156990466_wp, &
      &  9.82109349145493_wp, -2.74747221089191_wp,  5.55088030282267_wp, &
      & 10.37856257613277_wp, -7.13144688731266_wp,  5.55088030282267_wp, &
      & 11.36537734229809_wp, -5.42237895449015_wp, 13.11375156990466_wp, &
      &  7.84746395912430_wp,-12.94821159764100_wp,  8.07177440099298_wp, &
      & 15.13726986099968_wp, -0.32200926111561_wp,  0.50890313391099_wp, &
      &  9.82109349145493_wp,-12.94821159764100_wp,  0.50890313391099_wp, &
      & 16.12408462716500_wp, -2.03107719393811_wp,  8.07177440099298_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      &  8.83421504153316_wp,-15.30130939104899_wp,  0.00000000000000_wp, &
      &  8.83421504153316_wp, 15.30130939104899_wp,  0.00000000000000_wp, &
      &  0.00000000000000_wp,  0.00000000000000_wp, 15.12583758736722_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine trioxane

subroutine uracil(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 48
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", "C", &
      & "C", "C", "C", "C", "N", "N", "N", "N", "N", "N", "N", "N", "O", "O", &
      & "O", "O", "O", "O", "O", "O"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  9.50210779147656_wp,  3.49315797636267_wp,  0.55784702982000_wp, &
      & -1.83851414401041_wp, 15.08322370567159_wp,  5.24417781996432_wp, &
      &  9.40119643852132_wp, 19.68697348225517_wp,  5.16537625782512_wp, &
      & 20.74162940143721_wp,  8.09690775294625_wp,  0.47904546768079_wp, &
      &  6.71627414865598_wp,  6.89334144775534_wp,  0.15665826142303_wp, &
      &  0.94731949881017_wp, 18.48340717706427_wp,  5.64536658836129_wp, &
      & 12.18703008134190_wp, 16.28697898343357_wp,  5.56656502622208_wp, &
      & 17.95579575861662_wp,  4.69672428155357_wp,  0.07785669928382_wp, &
      & -3.07326092342906_wp,  2.22685277757413_wp,  5.77575766240458_wp, &
      & 10.73685457089521_wp, 13.81691850688305_wp,  0.02645615995081_wp, &
      & 21.97637618085586_wp, 20.95327868104371_wp, -0.05253437475947_wp, &
      &  8.16626068653159_wp,  9.36321295173479_wp,  5.69676712769430_wp, &
      &  7.81590553975440_wp, 22.17649813362665_wp,  0.55746908467784_wp, &
      & -0.15231189228825_wp, 10.58643240431773_wp,  5.24474473767755_wp, &
      & 11.08720971767240_wp,  1.00363332499119_wp,  5.16575420296727_wp, &
      & 19.05561612228613_wp, 12.59369905430011_wp,  0.47847854996756_wp, &
      &  3.35142854805475_wp,  4.88021664806957_wp,  0.04592033477177_wp, &
      &  4.31216509941140_wp, 16.47028237737849_wp,  5.75610451501255_wp, &
      & 15.55187568194313_wp, 18.29991481054827_wp,  5.67730295287334_wp, &
      & 14.59095015801540_wp,  6.70984908123935_wp, -0.03288122736744_wp, &
      &  3.90360640074242_wp,  0.25492399838319_wp,  0.19615352877817_wp, &
      &  3.75998724672373_wp, 11.84498972769211_wp,  5.60606029357723_wp, &
      & 14.99950885668438_wp, 22.92520746023465_wp,  5.52706975886694_wp, &
      & 15.14312801070308_wp, 11.33514173092573_wp,  0.11716299406789_wp, &
      &  6.57737930891421_wp,  0.63249119539550_wp,  0.44843191116628_wp, &
      &  1.08621433855194_wp, 12.22255692470442_wp,  5.35378191118912_wp, &
      & 12.32573594851259_wp, 22.54782923579342_wp,  5.27498034904991_wp, &
      & 17.81708989144594_wp, 10.95776350648450_wp,  0.36944137645599_wp, &
      &  7.49408525120987_wp,  3.04642681833609_wp,  0.44597526774227_wp, &
      &  0.16950839625628_wp, 14.63649254764501_wp,  5.35623855461312_wp, &
      & 11.40903000621693_wp, 20.13370464028175_wp,  5.27743699247392_wp, &
      & 18.73360686117052_wp,  8.54363891097283_wp,  0.36698473303199_wp, &
      &  5.94280941523686_wp,  5.08922031168099_wp,  0.24887687610872_wp, &
      &  1.72078423222929_wp, 16.67928604098991_wp,  5.55314797367560_wp, &
      & 12.96049481476102_wp, 18.09110011950792_wp,  5.47434641153639_wp, &
      & 17.18233102519751_wp,  6.50103439019901_wp,  0.17007531396951_wp, &
      &  2.45475369829327_wp,  2.42697473034492_wp,  0.02891280337482_wp, &
      &  5.20883994917288_wp, 14.01704045965384_wp,  5.77330101898058_wp, &
      & 16.44836155913353_wp, 20.75315672827292_wp,  5.69431048427029_wp, &
      & 13.69427530825392_wp,  9.16309099896400_wp, -0.04988875876439_wp, &
      & -1.64160472494793_wp,  6.72213229835937_wp,  5.78180478467905_wp, &
      &  9.30519837241408_wp, 18.31219802766829_wp,  0.02040903767634_wp, &
      & 20.54490895494581_wp, 16.45818813282955_wp, -0.05858149703394_wp, &
      &  9.59791688501272_wp,  4.86812240352063_wp,  5.70300322253985_wp, &
      &  2.81380158334002_wp, 21.32045238664678_wp,  0.11035998150910_wp, &
      &  4.84979206412613_wp,  9.73038665733786_wp,  5.69166486827521_wp, &
      & 16.08931367408679_wp,  1.85986804454214_wp,  5.61286330613601_wp, &
      & 14.05332319330067_wp, 13.44993377385106_wp,  0.03155841936990_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      & 22.47917417791306_wp,  0.00000000000000_wp, -0.15767795741656_wp, &
      &  0.00000000000000_wp, 23.18016925313205_wp,  0.00000000000000_wp, &
      & -3.57598503221097_wp,  0.00000000000000_wp,  5.88091107163537_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine uracil

subroutine urea(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 16
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "H", "H", "H", "H", "H", "H", "H", "C", "C", "N", "N", "N", "N", &
      & "O", "O"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  2.73311029549004_wp,  7.98560290858176_wp,  2.52731916558693_wp, &
      &  7.98560290858176_wp,  7.77168595812233_wp,  6.32245531053102_wp, &
      &  7.77168595812233_wp,  2.51919334503061_wp,  2.52731916558693_wp, &
      &  2.51919334503061_wp,  2.73311029549004_wp,  6.32245531053102_wp, &
      &  1.52274097774035_wp,  6.77523359083207_wp,  8.53343439213466_wp, &
      &  6.77523359083207_wp,  8.98205527587201_wp,  0.31615111141221_wp, &
      &  8.98205527587201_wp,  3.72956266278029_wp,  8.53343439213466_wp, &
      &  3.72956266278029_wp,  1.52274097774035_wp,  0.31615111141221_wp, &
      &  0.00000000000000_wp,  5.25249261309172_wp,  2.92453950999127_wp, &
      &  5.25249261309172_wp,  0.00000000000000_wp,  5.92523496612667_wp, &
      &  1.54107131713484_wp,  6.79337495765549_wp,  1.60305432044818_wp, &
      &  6.79337495765549_wp,  8.96372493647752_wp,  7.24653118309870_wp, &
      &  8.96372493647752_wp,  3.71142129595688_wp,  1.60305432044818_wp, &
      &  3.71142129595688_wp,  1.54107131713484_wp,  7.24653118309870_wp, &
      &  0.00000000000000_wp,  5.25249261309172_wp,  5.32978239466231_wp, &
      &  5.25249261309172_wp,  0.00000000000000_wp,  3.51999208145564_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      & 10.50481250525348_wp,  0.00000000000000_wp,  0.00000000000000_wp, &
      &  0.00000000000000_wp, 10.50481250525348_wp,  0.00000000000000_wp, &
      &  0.00000000000000_wp,  0.00000000000000_wp,  8.84967734421642_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine urea

end module mstore_x23
