/* -*- C++ -*-
 * Cppcheck - A tool for static C/C++ code analysis
 * Copyright (C) 2007-2024 Cppcheck team.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#ifndef fixtureH
#define fixtureH

#include "check.h"
#include "color.h"
#include "config.h"
#include "errorlogger.h"
#include "platform.h"
#include "settings.h"
#include "standards.h"

#include <cstddef>
#include <cstdint>
#include <list>
#include <sstream>
#include <stdexcept>
#include <string>
#include <type_traits>
#include <utility>

class options;
class Tokenizer;
enum class Certainty : std::uint8_t;
enum class Severity : std::uint8_t;

class TestFixture : public ErrorLogger {
private:
    static std::ostringstream errmsg;
    static unsigned int countTests;
    static std::size_t fails_counter;
    static std::size_t todos_counter;
    static std::size_t succeeded_todos_counter;
    bool mVerbose{};
    std::string mTemplateFormat;
    std::string mTemplateLocation;
    std::string mTestname;

protected:
    std::string exename;
    std::string testToRun;
    bool quiet_tests{};
    bool dry_run{};

    virtual void run() = 0;

    bool prepareTest(const char testname[]);
    virtual void prepareTestInternal() {}
    void teardownTest();
    virtual void teardownTestInternal() {}
    std::string getLocationStr(const char * filename, unsigned int linenr) const;

    bool assert_(const char * filename, unsigned int linenr, bool condition) const;

    template<typename T>
    bool assertEquals(const char* const filename, const unsigned int linenr, const T& expected, const T& actual, const std::string& msg = emptyString) const {
        if (expected != actual) {
            std::ostringstream expectedStr;
            expectedStr << expected;
            std::ostringstream actualStr;
            actualStr << actual;

            assertEqualsFailed(filename, linenr, expectedStr.str(), actualStr.str(), msg);
        }
        return expected == actual;
    }

    template<typename T>
    bool assertEqualsEnum(const char* const filename, const unsigned int linenr, const T& expected, const T& actual, const std::string& msg = emptyString) const {
        if (std::is_unsigned<T>())
            return assertEquals(filename, linenr, static_cast<std::uint64_t>(expected), static_cast<std::uint64_t>(actual), msg);
        return assertEquals(filename, linenr, static_cast<std::int64_t>(expected), static_cast<std::int64_t>(actual), msg);
    }

    //Helper function to be called when an assertEquals assertion fails.
    //Writes the appropriate failure message to errmsg and increments fails_counter
    void assertEqualsFailed(const char* filename, unsigned int linenr, const std::string& expected, const std::string& actual, const std::string& msg) const;

    bool assertEquals(const char * filename, unsigned int linenr, const std::string &expected, const std::string &actual, const std::string &msg = emptyString) const;
    bool assertEqualsWithoutLineNumbers(const char * filename, unsigned int linenr, const std::string &expected, const std::string &actual, const std::string &msg = emptyString) const;
    bool assertEquals(const char * filename, unsigned int linenr, const char expected[], const std::string& actual, const std::string &msg = emptyString) const;
    bool assertEquals(const char * filename, unsigned int linenr, const char expected[], const char actual[], const std::string &msg = emptyString) const;
    bool assertEquals(const char * filename, unsigned int linenr, const std::string& expected, const char actual[], const std::string &msg = emptyString) const;
    bool assertEquals(const char * filename, unsigned int linenr, long long expected, long long actual, const std::string &msg = emptyString) const;
    bool assertEqualsDouble(const char * filename, unsigned int linenr, double expected, double actual, double tolerance, const std::string &msg = emptyString) const;

    void todoAssertEquals(const char * filename, unsigned int linenr, const std::string &wanted,
                          const std::string &current, const std::string &actual) const;
    void todoAssertEquals(const char * filename, unsigned int linenr, const char wanted[],
                          const char current[], const std::string &actual) const;
    void todoAssertEquals(const char * filename, unsigned int linenr, long long wanted,
                          long long current, long long actual) const;
    void assertThrow(const char * filename, unsigned int linenr) const;
    void assertThrowFail(const char * filename, unsigned int linenr) const;
    void assertNoThrowFail(const char * filename, unsigned int linenr) const;
    static std::string deleteLineNumber(const std::string &message);

    void setVerbose(bool v) {
        mVerbose = v;
    }

    void setTemplateFormat(const std::string &templateFormat);

    void setMultiline() {
        setTemplateFormat("multiline");
    }

    void processOptions(const options& args);

    template<typename T>
    static T& getCheck()
    {
        for (Check *check : Check::instances()) {
            //cppcheck-suppress useStlAlgorithm
            if (T* c = dynamic_cast<T*>(check))
                return *c;
        }
        throw std::runtime_error("instance not found");
    }

    template<typename T>
    static void runChecks(const Tokenizer &tokenizer, ErrorLogger *errorLogger)
    {
        Check& check = getCheck<T>();
        check.runChecks(tokenizer, errorLogger);
    }

    class SettingsBuilder
    {
    public:
        explicit SettingsBuilder(const TestFixture &fixture) : fixture(fixture) {}
        SettingsBuilder(const TestFixture &fixture, Settings settings) : fixture(fixture), settings(std::move(settings)) {}

        SettingsBuilder& severity(Severity sev, bool b = true) {
            if (REDUNDANT_CHECK && settings.severity.isEnabled(sev) == b)
                throw std::runtime_error("redundant setting: severity");
            settings.severity.setEnabled(sev, b);
            return *this;
        }

        SettingsBuilder& certainty(Certainty cert, bool b = true) {
            if (REDUNDANT_CHECK && settings.certainty.isEnabled(cert) == b)
                throw std::runtime_error("redundant setting: certainty");
            settings.certainty.setEnabled(cert, b);
            return *this;
        }

        SettingsBuilder& clang() {
            if (REDUNDANT_CHECK && settings.clang)
                throw std::runtime_error("redundant setting: clang");
            settings.clang = true;
            return *this;
        }

        SettingsBuilder& checkLibrary() {
            if (REDUNDANT_CHECK && settings.checkLibrary)
                throw std::runtime_error("redundant setting: checkLibrary");
            settings.checkLibrary = true;
            return *this;
        }

        SettingsBuilder& checkUnusedTemplates(bool b = true) {
            if (REDUNDANT_CHECK && settings.checkUnusedTemplates == b)
                throw std::runtime_error("redundant setting: checkUnusedTemplates");
            settings.checkUnusedTemplates = b;
            return *this;
        }

        SettingsBuilder& debugwarnings(bool b = true) {
            if (REDUNDANT_CHECK && settings.debugwarnings == b)
                throw std::runtime_error("redundant setting: debugwarnings");
            settings.debugwarnings = b;
            return *this;
        }

        SettingsBuilder& c(Standards::cstd_t std) {
            // TODO: CLatest and C23 are the same - handle differently?
            //if (REDUNDANT_CHECK && settings.standards.c == std)
            //    throw std::runtime_error("redundant setting: standards.c");
            settings.standards.c = std;
            return *this;
        }

        SettingsBuilder& cpp(Standards::cppstd_t std) {
            // TODO: CPPLatest and CPP26 are the same - handle differently?
            //if (REDUNDANT_CHECK && settings.standards.cpp == std)
            //    throw std::runtime_error("redundant setting: standards.cpp");
            settings.standards.cpp = std;
            return *this;
        }

        SettingsBuilder& checkLevel(Settings::CheckLevel level);

        SettingsBuilder& library(const char lib[]);

        SettingsBuilder& libraryxml(const char xmldata[], std::size_t len);

        SettingsBuilder& platform(Platform::Type type);

        SettingsBuilder& checkConfiguration() {
            if (REDUNDANT_CHECK && settings.checkConfiguration)
                throw std::runtime_error("redundant setting: checkConfiguration");
            settings.checkConfiguration = true;
            return *this;
        }

        SettingsBuilder& checkHeaders(bool b = true) {
            if (REDUNDANT_CHECK && settings.checkHeaders == b)
                throw std::runtime_error("redundant setting: checkHeaders");
            settings.checkHeaders = b;
            return *this;
        }

        Settings build() {
            return std::move(settings);
        }
    private:
        const TestFixture &fixture;
        Settings settings;

        const bool REDUNDANT_CHECK = false;
    };

    SettingsBuilder settingsBuilder() const {
        return SettingsBuilder(*this);
    }

    SettingsBuilder settingsBuilder(Settings settings) const {
        return SettingsBuilder(*this, std::move(settings));
    }

    std::string output_str() {
        std::string s = mOutput.str();
        mOutput.str("");
        return s;
    }

    std::string errout_str() {
        std::string s = mErrout.str();
        mErrout.str("");
        return s;
    }

    void ignore_errout() {
        if (errout_str().empty())
            throw std::runtime_error("no errout to ignore");
    }

    const Settings settingsDefault;

private:
    std::ostringstream mOutput;
    std::ostringstream mErrout;

    void reportOut(const std::string &outmsg, Color c = Color::Reset) override;
    void reportErr(const ErrorMessage &msg) override;
    void run(const std::string &str);

public:
    static void printHelp();
    const std::string classname;

    explicit TestFixture(const char * _name);

    static std::size_t runTests(const options& args);
};

class TestInstance {
public:
    explicit TestInstance(const char * _name);
    virtual ~TestInstance() = default;

    virtual TestFixture* create() = 0;

    const std::string classname;
protected:
    std::unique_ptr<TestFixture> impl;
};

// TODO: most asserts do not actually assert i.e. do not return
#define TEST_CASE( NAME ) do { if (prepareTest(#NAME)) { setVerbose(false); try { NAME(); teardownTest(); } catch (...) { assertNoThrowFail(__FILE__, __LINE__); } } } while (false)
#define ASSERT( CONDITION ) if (!assert_(__FILE__, __LINE__, (CONDITION))) return
#define ASSERT_LOC( CONDITION, FILE_, LINE_ ) (void)assert_(FILE_, LINE_, (CONDITION))
#define CHECK_EQUALS( EXPECTED, ACTUAL ) (void)assertEquals(__FILE__, __LINE__, (EXPECTED), (ACTUAL))
// *INDENT-OFF*
#define ASSERT_EQUALS( EXPECTED, ACTUAL ) do { try { if (!assertEquals(__FILE__, __LINE__, (EXPECTED), (ACTUAL))) return; } catch (...) { assertNoThrowFail(__FILE__, __LINE__); } } while (false)
// *INDENT-ON*
#define ASSERT_EQUALS_WITHOUT_LINENUMBERS( EXPECTED, ACTUAL ) (void)assertEqualsWithoutLineNumbers(__FILE__, __LINE__, EXPECTED, ACTUAL)
#define ASSERT_EQUALS_DOUBLE( EXPECTED, ACTUAL, TOLERANCE ) (void)assertEqualsDouble(__FILE__, __LINE__, EXPECTED, ACTUAL, TOLERANCE)
#define ASSERT_EQUALS_LOC_MSG( EXPECTED, ACTUAL, MSG, FILE_, LINE_ ) (void)assertEquals(FILE_, LINE_, EXPECTED, ACTUAL, MSG)
#define ASSERT_EQUALS_MSG( EXPECTED, ACTUAL, MSG ) (void)assertEquals(__FILE__, __LINE__, EXPECTED, ACTUAL, MSG)
#define ASSERT_EQUALS_ENUM( EXPECTED, ACTUAL ) if (!assertEqualsEnum(__FILE__, __LINE__, (EXPECTED), (ACTUAL))) return
#define ASSERT_THROW( CMD, EXCEPTION ) do { try { (void)(CMD); assertThrowFail(__FILE__, __LINE__); } catch (const EXCEPTION&) {} catch (...) { assertThrowFail(__FILE__, __LINE__); } } while (false)
#define ASSERT_THROW_EQUALS( CMD, EXCEPTION, EXPECTED ) do { try { (void)(CMD); assertThrowFail(__FILE__, __LINE__); } catch (const EXCEPTION&e) { (void)assertEquals(__FILE__, __LINE__, EXPECTED, e.errorMessage); } catch (...) { assertThrowFail(__FILE__, __LINE__); } } while (false)
#define ASSERT_THROW_EQUALS_2( CMD, EXCEPTION, EXPECTED ) do { try { (void)(CMD); assertThrowFail(__FILE__, __LINE__); } catch (const EXCEPTION&e) { (void)assertEquals(__FILE__, __LINE__, EXPECTED, e.what()); } catch (...) { assertThrowFail(__FILE__, __LINE__); } } while (false)
#define ASSERT_THROW_INTERNAL( CMD, TYPE ) do { try { (void)(CMD); assertThrowFail(__FILE__, __LINE__); } catch (const InternalError& e) { (void)assertEqualsEnum(__FILE__, __LINE__, InternalError::TYPE, e.type); } catch (...) { assertThrowFail(__FILE__, __LINE__); } } while (false)
#define ASSERT_THROW_INTERNAL_EQUALS( CMD, TYPE, EXPECTED ) do { try { (void)(CMD); assertThrowFail(__FILE__, __LINE__); } catch (const InternalError& e) { (void)assertEqualsEnum(__FILE__, __LINE__, InternalError::TYPE, e.type); (void)assertEquals(__FILE__, __LINE__, EXPECTED, e.errorMessage); } catch (...) { assertThrowFail(__FILE__, __LINE__); } } while (false)
#define ASSERT_NO_THROW( CMD ) do { try { (void)(CMD); } catch (...) { assertNoThrowFail(__FILE__, __LINE__); } } while (false)
#define TODO_ASSERT_THROW( CMD, EXCEPTION ) do { try { (void)(CMD); } catch (const EXCEPTION&) {} catch (...) { assertThrow(__FILE__, __LINE__); } } while (false)
#define TODO_ASSERT( CONDITION ) do { const bool condition=(CONDITION); todoAssertEquals(__FILE__, __LINE__, true, false, condition); } while (false)
#define TODO_ASSERT_EQUALS( WANTED, CURRENT, ACTUAL ) todoAssertEquals(__FILE__, __LINE__, WANTED, CURRENT, ACTUAL)
#define EXPECT_EQ( EXPECTED, ACTUAL ) (void)assertEquals(__FILE__, __LINE__, EXPECTED, ACTUAL)
#define REGISTER_TEST( CLASSNAME ) namespace { class CLASSNAME ## Instance : public TestInstance { public: CLASSNAME ## Instance() : TestInstance(#CLASSNAME) {} TestFixture* create() override { impl.reset(new CLASSNAME); return impl.get(); } }; CLASSNAME ## Instance instance_ ## CLASSNAME; }

#define PLATFORM( P, T ) do { std::string errstr; (void)assertEquals(__FILE__, __LINE__, true, P.set(Platform::toString(T), errstr, {exename}), errstr); } while (false)

#endif // fixtureH
