# !/usr/bin/env python
"""Define the unit tests for the :mod:`colour.colorimetry.spectrum` module."""

import colour
import numpy as np
import unittest

from colour.algebra import CubicSplineInterpolator
from colour.colorimetry.spectrum import SPECTRAL_SHAPE_DEFAULT
from colour.colorimetry.spectrum import (
    SpectralShape,
    SpectralDistribution,
    MultiSpectralDistributions,
    reshape_sd,
    reshape_msds,
    sds_and_msds_to_sds,
    sds_and_msds_to_msds,
)
from colour.utilities import tstack

__author__ = "Colour Developers"
__copyright__ = "Copyright 2013 Colour Developers"
__license__ = "BSD-3-Clause - https://opensource.org/licenses/BSD-3-Clause"
__maintainer__ = "Colour Developers"
__email__ = "colour-developers@colour-science.org"
__status__ = "Production"

__all__ = [
    "DATA_SAMPLE",
    "DATA_SAMPLE_NON_UNIFORM",
    "DATA_SAMPLE_INTERPOLATED",
    "DATA_SAMPLE_INTERPOLATED_NON_UNIFORM",
    "DATA_SAMPLE_NORMALISED",
    "DATA_STANDARD_OBSERVER_2_DEGREE_CIE1931",
    "DATA_CMFS",
    "DATA_SAMPLE_ABRIDGED",
    "DATA_MULTI_SAMPLE_ABRIDGED",
    "TestSpectralShape",
    "TestSpectralDistribution",
    "TestMultiSpectralDistributions",
    "TestReshapeSd",
    "TestSdsAndMdsToSds",
    "TestSdsAndMsdsToMsds",
]

DATA_SAMPLE: dict = {
    340: 0.0000,
    360: 0.0000,
    380: 0.0000,
    400: 0.0641,
    420: 0.0645,
    440: 0.0562,
    460: 0.0537,
    480: 0.0559,
    500: 0.0651,
    520: 0.0705,
    540: 0.0772,
    560: 0.0870,
    580: 0.1128,
    600: 0.1360,
    620: 0.1511,
    640: 0.1688,
    660: 0.1996,
    680: 0.2397,
    700: 0.2852,
    720: 0.0000,
    740: 0.0000,
    760: 0.0000,
    780: 0.0000,
    800: 0.0000,
    820: 0.0000,
}

DATA_SAMPLE_NON_UNIFORM: dict = {
    391.898: 16.331740,
    392.069: 16.333122,
    405.606: 40.197224,
    406.794: 39.923366,
    406.891: 39.924098,
    407.026: 39.925138,
    416.286: 40.064293,
    418.690: 40.009950,
    426.700: 18.045809,
    426.726: 18.045986,
    432.556: 38.435883,
    464.742: 29.534647,
    465.025: 29.534647,
    465.147: 29.534647,
    466.586: 38.226470,
    477.175: 7.487795,
    493.205: 7.684766,
    505.217: 7.684766,
    513.294: 20.701285,
    513.328: 20.704211,
    514.349: 20.704211,
    514.516: 20.709788,
    515.109: 20.709788,
    538.034: 7.684766,
    564.807: 20.704211,
    566.247: 20.709788,
    569.592: 32.103387,
    580.133: 37.548490,
    581.198: 37.548490,
    582.642: 40.197224,
    588.977: 18.045986,
    589.159: 18.045809,
    600.113: 8.643020,
    600.603: 8.647157,
    600.718: 8.640394,
    601.068: 8.640394,
    601.322: 8.647157,
    601.484: 8.643020,
    657.805: 14.448826,
    658.288: 14.448826,
    658.761: 8.537097,
    674.438: 38.22647,
    678.390: 20.709788,
    703.725: 38.435883,
    711.318: 8.647157,
    711.519: 8.640394,
    711.563: 22.532398,
    711.699: 8.647157,
    711.990: 22.536906,
    723.132: 16.33174,
    723.642: 16.333122,
    761.265: 41.342187,
    786.089: 8.850659,
    805.862: 8.850659,
}

DATA_SAMPLE_INTERPOLATED: tuple = (
    0.000000000000000,
    0.000230709627131,
    0.000384144814593,
    0.000507137093115,
    0.000632114832536,
    0.000778810112328,
    0.000955965592105,
    0.001163041382140,
    0.001391921913876,
    0.001628622810444,
    0.001854997757177,
    0.002050445372122,
    0.002193616076555,
    0.002264118965498,
    0.002244228678230,
    0.002120592268802,
    0.001885936076555,
    0.001540772596628,
    0.001095107350478,
    0.000570145756392,
    0.000000000000000,
    -0.000581325882457,
    -0.001183945630981,
    -0.001820338942229,
    -0.002492278660287,
    -0.003192248184715,
    -0.003904858878589,
    -0.004608267476544,
    -0.005275593492823,
    -0.005876336629317,
    -0.006377794183612,
    -0.006746478457031,
    -0.006949534162679,
    -0.006956155833489,
    -0.006739005230263,
    -0.006275628749720,
    -0.005549874832536,
    -0.004553311371393,
    -0.003286643119019,
    -0.001761129096236,
    0.000000000000000,
    0.001979832128906,
    0.004221516875000,
    0.006762744980469,
    0.009617760000000,
    0.012780285644531,
    0.016226453125000,
    0.019917728496094,
    0.023803840000000,
    0.027825705410156,
    0.031918359375000,
    0.036013880761719,
    0.040044320000000,
    0.043944626425781,
    0.047655575625000,
    0.051126696777344,
    0.054319200000000,
    0.057208903691406,
    0.059789161875000,
    0.062073791542969,
    0.064100000000000,
    0.065908998066406,
    0.067477265625000,
    0.068776060136719,
    0.069791040000000,
    0.070520520019531,
    0.070973726875000,
    0.071169054589844,
    0.071132320000000,
    0.070895018222656,
    0.070492578125000,
    0.069962617792969,
    0.069343200000000,
    0.068671087675781,
    0.067979999375000,
    0.067298864746094,
    0.066650080000000,
    0.066047763378906,
    0.065496010625000,
    0.064987150449219,
    0.064500000000000,
    0.064007805449219,
    0.063510900625000,
    0.063016138378906,
    0.062528640000000,
    0.062051989746094,
    0.061588429375000,
    0.061139052675781,
    0.060704000000000,
    0.060282652792969,
    0.059873828125000,
    0.059475973222656,
    0.059087360000000,
    0.058706279589844,
    0.058331236875000,
    0.057961145019531,
    0.057595520000000,
    0.057234675136719,
    0.056879915625000,
    0.056533733066406,
    0.056200000000000,
    0.055883811757812,
    0.055589516250000,
    0.055320009023437,
    0.055076960000000,
    0.054860961914063,
    0.054671678750000,
    0.054507994179687,
    0.054368160000000,
    0.054249944570312,
    0.054150781250000,
    0.054067916835938,
    0.053998560000000,
    0.053940029726562,
    0.053889903750000,
    0.053846166992187,
    0.053807360000000,
    0.053772727382813,
    0.053742366250000,
    0.053717374648438,
    0.053700000000000,
    0.053692483144531,
    0.053693175625000,
    0.053700270058594,
    0.053712960000000,
    0.053731311035156,
    0.053756131875000,
    0.053788845449219,
    0.053831360000000,
    0.053885940175781,
    0.053955078125000,
    0.054041364589844,
    0.054147360000000,
    0.054275465566406,
    0.054427794375000,
    0.054606042480469,
    0.054811360000000,
    0.055044222207031,
    0.055304300625000,
    0.055590334121094,
    0.055900000000000,
    0.056231407851562,
    0.056587597500000,
    0.056971279335937,
    0.057383360000000,
    0.057823168945313,
    0.058288685000000,
    0.058776762929688,
    0.059283360000000,
    0.059803762539063,
    0.060332812500000,
    0.060865134023438,
    0.061395360000000,
    0.061918358632813,
    0.062429460000000,
    0.062924682617188,
    0.063400960000000,
    0.063856367226563,
    0.064290347500000,
    0.064703938710938,
    0.065100000000000,
    0.065481647265625,
    0.065846786250000,
    0.066192982265625,
    0.066519040000000,
    0.066824853515625,
    0.067111256250000,
    0.067379871015625,
    0.067632960000000,
    0.067873274765625,
    0.068103906250000,
    0.068328134765625,
    0.068549280000000,
    0.068770551015625,
    0.068994896250000,
    0.069224853515625,
    0.069462400000000,
    0.069708802265625,
    0.069964466250000,
    0.070228787265625,
    0.070500000000000,
    0.070776270703125,
    0.071059446250000,
    0.071351597578125,
    0.071653920000000,
    0.071966845703125,
    0.072290156250000,
    0.072623095078125,
    0.072964480000000,
    0.073312815703125,
    0.073666406250000,
    0.074023467578125,
    0.074382240000000,
    0.074741100703125,
    0.075098676250000,
    0.075453955078125,
    0.075806400000000,
    0.076156060703125,
    0.076503686250000,
    0.076850837578125,
    0.077200000000000,
    0.077552701992188,
    0.077904262500000,
    0.078250762070313,
    0.078590880000000,
    0.078925561523438,
    0.079257685000000,
    0.079591729101562,
    0.079933440000000,
    0.080289498554688,
    0.080667187500000,
    0.081074058632813,
    0.081517600000000,
    0.082004903085938,
    0.082542330000000,
    0.083135180664063,
    0.083787360000000,
    0.084501045117188,
    0.085276352500000,
    0.086111005195313,
    0.087000000000000,
    0.087938453242188,
    0.088930890000000,
    0.089981833007812,
    0.091092960000000,
    0.092263452148438,
    0.093490342500000,
    0.094768864414062,
    0.096092800000000,
    0.097454828554687,
    0.098846875000000,
    0.100260458320312,
    0.101687040000000,
    0.103118372460937,
    0.104546847500000,
    0.105965844726563,
    0.107370080000000,
    0.108755953867188,
    0.110121900000000,
    0.111468733632812,
    0.112800000000000,
    0.114120657988281,
    0.115431176875000,
    0.116730532871094,
    0.118017600000000,
    0.119291174316406,
    0.120549998125000,
    0.121792784199219,
    0.123018240000000,
    0.124225091894531,
    0.125412109375000,
    0.126578129277344,
    0.127722080000000,
    0.128843005722656,
    0.129940090625000,
    0.131012683105469,
    0.132060320000000,
    0.133082750800781,
    0.134079961875000,
    0.135052200683594,
    0.136000000000000,
    0.136923531484375,
    0.137820920000000,
    0.138690739765625,
    0.139532640000000,
    0.140347216796875,
    0.141135885000000,
    0.141900750078125,
    0.142644480000000,
    0.143370177109375,
    0.144081250000000,
    0.144781285390625,
    0.145473920000000,
    0.146162712421875,
    0.146851015000000,
    0.147541845703125,
    0.148237760000000,
    0.148940722734375,
    0.149651980000000,
    0.150371931015625,
    0.151100000000000,
    0.151834687363281,
    0.152574745625000,
    0.153319862089844,
    0.154070560000000,
    0.154828088378906,
    0.155594311875000,
    0.156371600605469,
    0.157162720000000,
    0.157970720644531,
    0.158798828125000,
    0.159650332871094,
    0.160528480000000,
    0.161436359160156,
    0.162376794375000,
    0.163352233886719,
    0.164364640000000,
    0.165415378925781,
    0.166505110625000,
    0.167633678652344,
    0.168800000000000,
    0.170002988242187,
    0.171244585000000,
    0.172526722382812,
    0.173850400000000,
    0.175215795898437,
    0.176622377500000,
    0.178069012539063,
    0.179554080000000,
    0.181075581054688,
    0.182631250000000,
    0.184218665195313,
    0.185835360000000,
    0.187478933710938,
    0.189147162500000,
    0.190838110351563,
    0.192550240000000,
    0.194282523867188,
    0.196034555000000,
    0.197806658007813,
    0.199600000000000,
    0.201405046894531,
    0.203174116875000,
    0.204868198964844,
    0.206468000000000,
    0.207971350097656,
    0.209390608125000,
    0.210750067167969,
    0.212083360000000,
    0.213430864550781,
    0.214837109375000,
    0.216348179121094,
    0.218009120000000,
    0.219861345253906,
    0.221940040625000,
    0.224271569824219,
    0.226870880000000,
    0.229738907207031,
    0.232859981875000,
    0.236199234277344,
    0.239700000000000,
    0.243337282929688,
    0.247262056250000,
    0.251598942851562,
    0.256394240000000,
    0.261625952148438,
    0.267213823750000,
    0.273029372070313,
    0.278905920000000,
    0.284648628867188,
    0.290044531250000,
    0.294872563789062,
    0.298913600000000,
    0.301960483085938,
    0.303828058750000,
    0.304363208007813,
    0.303454880000000,
    0.301044124804688,
    0.297134126250000,
    0.291800234726562,
    0.285200000000000,
    0.277470396855469,
    0.268408756875000,
    0.257826504003906,
    0.245651040000000,
    0.231911267089844,
    0.216723110625000,
    0.200275041738281,
    0.182813600000000,
    0.164628916074219,
    0.146040234375000,
    0.127381435722656,
    0.108986560000000,
    0.091175328808594,
    0.074238668125000,
    0.058424230957031,
    0.043921920000000,
    0.030849410292969,
    0.019237671875000,
    0.009016492441406,
    0.000000000000000,
    -0.008014721386719,
    -0.014901410625000,
    -0.020510217441406,
    -0.024767360000000,
    -0.027665856933594,
    -0.029256259375000,
    -0.029637382988281,
    -0.028947040000000,
    -0.027352771230469,
    -0.025042578125000,
    -0.022215654785156,
    -0.019073120000000,
    -0.015808749277344,
    -0.012599706875000,
    -0.009597277832031,
    -0.006917600000000,
    -0.004632396074219,
    -0.002759705625000,
    -0.001254617128906,
    0.000000000000000,
    0.001146203203125,
    0.002165737500000,
    0.003010363984375,
    0.003650560000000,
    0.004073291015625,
    0.004279782500000,
    0.004283291796875,
    0.004106880000000,
    0.003781183828125,
    0.003342187500000,
    0.002828994609375,
    0.002281600000000,
    0.001738661640625,
    0.001235272500000,
    0.000800732421875,
    0.000456320000000,
    0.000213064453125,
    0.000069517500000,
    0.000009525234375,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
)

DATA_SAMPLE_INTERPOLATED_NON_UNIFORM: tuple = (
    16.329808636577400,
    16.722487609243078,
    17.780769796558388,
    19.388440919210822,
    21.429286697887836,
    23.787092853276910,
    26.345645106065515,
    28.988729176941121,
    31.600130786591194,
    34.063635655703216,
    36.263029504964656,
    38.082098055062993,
    39.404627026685688,
    40.114402140520198,
    40.111247588026082,
    39.925103635141149,
    39.890167058876870,
    39.799323872976096,
    39.680556569501256,
    39.561845869769570,
    39.471172495098209,
    39.436517166804364,
    39.485860606205222,
    39.647183534617994,
    39.948466673359860,
    40.374181475729280,
    40.480943048717741,
    39.587763146544347,
    37.399010971904012,
    34.259517350145224,
    30.574008069142096,
    26.747208916768866,
    23.183845680899680,
    20.288644149408725,
    18.466330110170176,
    18.131455040802454,
    19.618408431271092,
    22.601624593221480,
    26.621530082560952,
    31.218551455196845,
    35.933115267036499,
    40.312773209586069,
    44.121677838324345,
    47.374686358105983,
    50.100647613008405,
    52.328410447109043,
    54.086823704485290,
    55.404736229214613,
    56.310996865374428,
    56.834454457042114,
    57.003957848295137,
    56.848355883210893,
    56.396497405866839,
    55.677231260340363,
    54.719406290708918,
    53.551871341049903,
    52.203475255440743,
    50.703066877958868,
    49.079495052681693,
    47.361608623686649,
    45.578256435051173,
    43.758287330852653,
    41.930550155168532,
    40.123893752076235,
    38.367166965653183,
    36.689218639976801,
    35.118897619124496,
    33.685052747173714,
    32.416532868201863,
    31.342186826286373,
    30.490863465504646,
    29.891411629934137,
    29.572680163652254,
    29.538621481156376,
    33.594913456316092,
    41.048330705400552,
    45.551744552213606,
    47.142069193311166,
    46.264235269850914,
    43.363173422990592,
    38.883814293887831,
    33.271088523700378,
    26.969926753585888,
    20.425259624702068,
    14.082017778206605,
    8.385131855257200,
    3.728429425651282,
    0.156961054721180,
    -2.429909888236355,
    -4.133307786198728,
    -5.054357022143353,
    -5.294181979047651,
    -4.953907039889023,
    -4.134656587644884,
    -2.937555005292646,
    -1.463726675809728,
    0.185704017826468,
    1.909612692638526,
    3.606874965649032,
    5.176366453880572,
    6.516962774355742,
    7.527539544097118,
    8.122379803803623,
    8.331483144611147,
    8.237444938631239,
    7.923124732066758,
    7.471382071120567,
    6.965076501995521,
    6.487067570894484,
    6.120214824020312,
    5.947377807575871,
    6.051416067764015,
    6.515189150787608,
    7.421556602849507,
    8.833103319101975,
    10.654113909347958,
    12.714472051171430,
    14.843629856075644,
    16.871039435563851,
    18.626152901139314,
    19.938422364305275,
    20.637299936564997,
    20.706835013628812,
    20.713132276905259,
    20.620587314150111,
    20.396819412610807,
    20.054405097058748,
    19.605938071902443,
    19.064012041550384,
    18.441220710411045,
    17.750157782892924,
    17.003416963404511,
    16.213591956354293,
    15.393276466150766,
    14.555064197202412,
    13.711548853917725,
    12.875324140705192,
    12.058983761973309,
    11.275121422130562,
    10.536330825585436,
    9.855205676746426,
    9.244339680022023,
    8.716326539820709,
    8.283759960550988,
    7.959233646621334,
    7.755341302440248,
    7.684676632416213,
    7.756084346822568,
    7.961801724157814,
    8.289468950531905,
    8.726726048591537,
    9.261213040983417,
    9.880569950354237,
    10.572436799350699,
    11.324453610619505,
    12.124260406807357,
    12.959497210560945,
    13.817804044526978,
    14.686820931352152,
    15.554187893683173,
    16.407544954166731,
    17.234532135449530,
    18.022789460178267,
    18.759956950999651,
    19.433674630560375,
    20.031582521507140,
    20.541320646486646,
    20.950529028145592,
    21.246847689130675,
    21.417916652088600,
    21.451375939666065,
    21.334865574509770,
    21.056025579266414,
    20.604750276779001,
    20.494345115463702,
    22.221705466331883,
    25.661450384670388,
    29.775598875658655,
    33.531925166546948,
    36.383260993760757,
    38.399428506854193,
    39.690190939354423,
    40.365311524788659,
    40.534553496684069,
    40.307680088567857,
    39.794454533967205,
    39.104640066409281,
    38.347999919421312,
    37.634297326530451,
    37.374685121273394,
    39.119690201154867,
    40.253206513156250,
    38.299118522290833,
    34.130306343480264,
    28.883500017392159,
    23.695429584694160,
    19.702825086053920,
    18.042370406596557,
    17.942146149526579,
    17.437952042997388,
    16.603615436175311,
    15.527509997088327,
    14.298009393764401,
    13.003487294231499,
    11.732317366517600,
    10.572873278650665,
    9.613528698658669,
    8.942657294569587,
    8.648632734411382,
    8.638275005914387,
    8.608461770725761,
    8.468358277399256,
    8.236485894689265,
    7.918459394063972,
    7.519893546991550,
    7.046403124940173,
    6.503602899378026,
    5.897107641773280,
    5.232532123594114,
    4.515491116308703,
    3.751599391385226,
    2.946471720291860,
    2.105722874496783,
    1.234967625468170,
    0.339820744674198,
    -0.574102996416954,
    -1.501188826337108,
    -2.435821973618090,
    -3.372387666791723,
    -4.305271134389827,
    -5.228857604944231,
    -6.137532306986749,
    -7.025680469049206,
    -7.887687319663434,
    -8.717938087361247,
    -9.510818000674471,
    -10.260712288134929,
    -10.962006178274445,
    -11.609084899624833,
    -12.196333680717927,
    -12.718137750085543,
    -13.168882336259514,
    -13.542952667771653,
    -13.834733973153785,
    -14.038611480937735,
    -14.148970419655322,
    -14.160196017838375,
    -14.066673504018716,
    -13.862788106728164,
    -13.542925054498539,
    -13.101469575861673,
    -12.532806899349385,
    -11.831322253493493,
    -10.991400866825831,
    -10.007427967878208,
    -8.873788785182459,
    -7.584868547270405,
    -6.135052482673858,
    -4.518725819924658,
    -2.730273787554616,
    -0.764081614095559,
    1.385465471920696,
    3.723982241962316,
    6.257083467497486,
    8.990383919994379,
    11.929498370921175,
    14.976162260830137,
    5.084237255461949,
    -6.729230790271000,
    -14.622433089729542,
    -19.038838171389944,
    -20.421914563728482,
    -19.215130795221405,
    -15.861955394344996,
    -10.805856889575507,
    -4.490303809389212,
    2.641235317737622,
    10.145291963328749,
    17.578397598907866,
    24.497083695998736,
    30.457881726125084,
    35.017323160810655,
    37.731939471579153,
    38.169944675202451,
    36.103648065327157,
    31.478944095141092,
    24.247257545409393,
    14.377470469911161,
    2.089605892978410,
    -12.208009328653077,
    -28.102486071018522,
    -45.180935210153116,
    -63.030467622092019,
    -81.238194182870473,
    -99.391225768523640,
    -117.076673255086732,
    -133.881647518594917,
    -149.393259435083451,
    -163.198619880587444,
    -174.884839731142108,
    -184.039029862782684,
    -190.248301151544382,
    -193.099764473462415,
    -192.180530704571822,
    -187.077710720907902,
    -177.378415398505837,
    -162.669755613400923,
    -142.538842241628146,
    -116.572786159222886,
    -84.358698242220214,
    -45.483689366655454,
    0.465129591436323,
    53.867241377821607,
    111.906385750398641,
    165.969680931356891,
    206.832112792864763,
    225.268667207090687,
    212.054330046203120,
    157.964087182370434,
    53.772924487761195,
    24.128558965593012,
    155.948189345952358,
    238.667542187567193,
    279.823268952005947,
    286.952021100836475,
    267.590450095627091,
    229.275207397946076,
    179.542944469361430,
    125.930312771441450,
    75.973963765754249,
    37.210548913868003,
    17.176719677350921,
    17.152394375030820,
    19.379794324130732,
    21.517434431022508,
    23.565545988087592,
    25.524360287707406,
    27.394108622263396,
    29.175022284136993,
    30.867332565709628,
    32.471270759362753,
    33.987068157477793,
    35.414956052436175,
    36.755165736619354,
    38.007928502408760,
    39.173475642185821,
    40.252038448331973,
    41.243848213228674,
    42.149136229257330,
    42.968133788799399,
    43.701072184236310,
    44.348182707949491,
    44.909696652320392,
    45.385845309730442,
    45.776859972561077,
    46.082971933193733,
    46.304412484009845,
    46.441412917390849,
    46.494204525718175,
    46.463018601373285,
    46.348086436737596,
    46.149639324192542,
    45.867908556119552,
    45.503125424900077,
    45.055521222915552,
    44.525327242547419,
    43.912774776177095,
    43.218095116186007,
    42.441519554955640,
    41.583279384867382,
    40.643991169034841,
    39.627797988936805,
    38.540734852902020,
    37.388854826093969,
    36.178210973676094,
    34.914856360811882,
    33.604844052664760,
    32.254227114398248,
    30.869058611175774,
    29.455391608160820,
    28.019279170516839,
    26.566774363407301,
    25.103930251995678,
    23.636799901445418,
    22.171436376920010,
    20.713892743582907,
    19.270222066597562,
    17.846477411127466,
    16.448711842336071,
    15.082978425386836,
    13.755330225443233,
    12.471820307668732,
    11.238501737226787,
    10.061427579280879,
    8.946650898994456,
    7.900224761530997,
    6.928202232053956,
    6.036636375726809,
    5.231580257713018,
    4.519086943176044,
    3.905209497279358,
    3.396000985186418,
    2.997514472060695,
    2.715803023065654,
    2.556919703364760,
    2.526917578121476,
    2.631849712499274,
    2.877769171661610,
    3.270729020771954,
    3.816782324993773,
    4.521982149490528,
    5.392381559425689,
    6.434033619962720,
    7.652991396265083,
)

DATA_SAMPLE_NORMALISED: tuple = (
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    22.475455820476860,
    22.615708274894811,
    19.705469845722302,
    18.828892005610097,
    19.600280504908834,
    22.826086956521742,
    24.719495091164092,
    27.068723702664798,
    30.504908835904626,
    39.551192145862551,
    47.685834502103788,
    52.980364656381497,
    59.186535764375883,
    69.985974754558200,
    84.046283309957929,
    100.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
    0.000000000000000,
)

DATA_STANDARD_OBSERVER_2_DEGREE_CIE1931: dict = {
    380: (0.001368, 0.000039, 0.006450),
    385: (0.002236, 0.000064, 0.010550),
    390: (0.004243, 0.000120, 0.020050),
    395: (0.007650, 0.000217, 0.036210),
    400: (0.014310, 0.000396, 0.067850),
    405: (0.023190, 0.000640, 0.110200),
    410: (0.043510, 0.001210, 0.207400),
    415: (0.077630, 0.002180, 0.371300),
    420: (0.134380, 0.004000, 0.645600),
    425: (0.214770, 0.007300, 1.039050),
    430: (0.283900, 0.011600, 1.385600),
    435: (0.328500, 0.016840, 1.622960),
    440: (0.348280, 0.023000, 1.747060),
    445: (0.348060, 0.029800, 1.782600),
    450: (0.336200, 0.038000, 1.772110),
    455: (0.318700, 0.048000, 1.744100),
    460: (0.290800, 0.060000, 1.669200),
    465: (0.251100, 0.073900, 1.528100),
    470: (0.195360, 0.090980, 1.287640),
    475: (0.142100, 0.112600, 1.041900),
    480: (0.095640, 0.139020, 0.812950),
    485: (0.057950, 0.169300, 0.616200),
    490: (0.032010, 0.208020, 0.465180),
    495: (0.014700, 0.258600, 0.353300),
    500: (0.004900, 0.323000, 0.272000),
    505: (0.002400, 0.407300, 0.212300),
    510: (0.009300, 0.503000, 0.158200),
    515: (0.029100, 0.608200, 0.111700),
    520: (0.063270, 0.710000, 0.078250),
    525: (0.109600, 0.793200, 0.057250),
    530: (0.165500, 0.862000, 0.042160),
    535: (0.225750, 0.914850, 0.029840),
    540: (0.290400, 0.954000, 0.020300),
    545: (0.359700, 0.980300, 0.013400),
    550: (0.433450, 0.994950, 0.008750),
    555: (0.512050, 1.000000, 0.005750),
    560: (0.594500, 0.995000, 0.003900),
    565: (0.678400, 0.978600, 0.002750),
    570: (0.762100, 0.952000, 0.002100),
    575: (0.842500, 0.915400, 0.001800),
    580: (0.916300, 0.870000, 0.001650),
    585: (0.978600, 0.816300, 0.001400),
    590: (1.026300, 0.757000, 0.001100),
    595: (1.056700, 0.694900, 0.001000),
    600: (1.062200, 0.631000, 0.000800),
    605: (1.045600, 0.566800, 0.000600),
    610: (1.002600, 0.503000, 0.000340),
    615: (0.938400, 0.441200, 0.000240),
    620: (0.854450, 0.381000, 0.000190),
    625: (0.751400, 0.321000, 0.000100),
    630: (0.642400, 0.265000, 0.000050),
    635: (0.541900, 0.217000, 0.000030),
    640: (0.447900, 0.175000, 0.000020),
    645: (0.360800, 0.138200, 0.000010),
    650: (0.283500, 0.107000, 0.000000),
    655: (0.218700, 0.081600, 0.000000),
    660: (0.164900, 0.061000, 0.000000),
    665: (0.121200, 0.044580, 0.000000),
    670: (0.087400, 0.032000, 0.000000),
    675: (0.063600, 0.023200, 0.000000),
    680: (0.046770, 0.017000, 0.000000),
    685: (0.032900, 0.011920, 0.000000),
    690: (0.022700, 0.008210, 0.000000),
    695: (0.015840, 0.005723, 0.000000),
    700: (0.011359, 0.004102, 0.000000),
    705: (0.008111, 0.002929, 0.000000),
    710: (0.005790, 0.002091, 0.000000),
    715: (0.004109, 0.001484, 0.000000),
    720: (0.002899, 0.001047, 0.000000),
    725: (0.002049, 0.000740, 0.000000),
    730: (0.001440, 0.000520, 0.000000),
    735: (0.001000, 0.000361, 0.000000),
    740: (0.000690, 0.000249, 0.000000),
    745: (0.000476, 0.000172, 0.000000),
    750: (0.000332, 0.000120, 0.000000),
    755: (0.000235, 0.000085, 0.000000),
    760: (0.000166, 0.000060, 0.000000),
    765: (0.000117, 0.000042, 0.000000),
    770: (0.000083, 0.000030, 0.000000),
    775: (0.000059, 0.000021, 0.000000),
    780: (0.000042, 0.000015, 0.000000),
}

DATA_CMFS: dict = {
    380: np.array([0.001368, 3.90e-05, 0.006450]),
    385: np.array([0.002236, 6.40e-05, 0.010550]),
    390: np.array([0.004243, 0.000120, 0.020050]),
    395: np.array([0.007650, 0.000217, 0.036210]),
    400: np.array([0.014310, 0.000396, 0.067850]),
    405: np.array([0.023190, 0.000640, 0.110200]),
    410: np.array([0.043510, 0.001210, 0.207400]),
    415: np.array([0.077630, 0.002180, 0.371300]),
    420: np.array([0.134380, 0.004000, 0.645600]),
    425: np.array([0.214770, 0.007300, 1.039050]),
    430: np.array([0.283900, 0.011600, 1.385600]),
    435: np.array([0.328500, 0.016840, 1.622960]),
    440: np.array([0.348280, 0.023000, 1.747060]),
    445: np.array([0.348060, 0.029800, 1.782600]),
    450: np.array([0.336200, 0.038000, 1.772110]),
    455: np.array([0.318700, 0.048000, 1.744100]),
    460: np.array([0.290800, 0.060000, 1.669200]),
    465: np.array([0.251100, 0.073900, 1.528100]),
    470: np.array([0.195360, 0.090980, 1.287640]),
    475: np.array([0.142100, 0.112600, 1.041900]),
    480: np.array([0.095640, 0.139020, 0.812950]),
    485: np.array([0.057950, 0.169300, 0.616200]),
    490: np.array([0.032010, 0.208020, 0.465180]),
    495: np.array([0.014700, 0.258600, 0.353300]),
    500: np.array([0.004900, 0.323000, 0.272000]),
    505: np.array([0.002400, 0.407300, 0.212300]),
    510: np.array([0.009300, 0.503000, 0.158200]),
    515: np.array([0.029100, 0.608200, 0.111700]),
    520: np.array([0.063270, 0.710000, 0.078250]),
    525: np.array([0.109600, 0.793200, 0.057250]),
    530: np.array([0.165500, 0.862000, 0.042160]),
    535: np.array([0.225750, 0.914850, 0.029840]),
    540: np.array([0.290400, 0.954000, 0.020300]),
    545: np.array([0.359700, 0.980300, 0.013400]),
    550: np.array([0.433450, 0.994950, 0.008750]),
    555: np.array([0.512050, 1.000000, 0.005750]),
    560: np.array([0.594500, 0.995000, 0.003900]),
    565: np.array([0.678400, 0.978600, 0.002750]),
    570: np.array([0.762100, 0.952000, 0.002100]),
    575: np.array([0.842500, 0.915400, 0.001800]),
    580: np.array([0.916300, 0.870000, 0.001650]),
    585: np.array([0.978600, 0.816300, 0.001400]),
    590: np.array([1.026300, 0.757000, 0.001100]),
    595: np.array([1.056700, 0.694900, 0.001000]),
    600: np.array([1.062200, 0.631000, 0.000800]),
    605: np.array([1.045600, 0.566800, 0.000600]),
    610: np.array([1.002600, 0.503000, 0.000340]),
    615: np.array([0.938400, 0.441200, 0.000240]),
    620: np.array([0.854450, 0.381000, 0.000190]),
    625: np.array([0.751400, 0.321000, 0.000100]),
    630: np.array([0.642400, 0.265000, 5.00e-05]),
    635: np.array([0.541900, 0.217000, 3.00e-05]),
    640: np.array([0.447900, 0.175000, 2.00e-05]),
    645: np.array([0.360800, 0.138200, 1.00e-05]),
    650: np.array([0.283500, 0.107000, 0.000000]),
    655: np.array([0.218700, 0.081600, 0.000000]),
    660: np.array([0.164900, 0.061000, 0.000000]),
    665: np.array([0.121200, 0.044580, 0.000000]),
    670: np.array([0.087400, 0.032000, 0.000000]),
    675: np.array([0.063600, 0.023200, 0.000000]),
    680: np.array([0.046770, 0.017000, 0.000000]),
    685: np.array([0.032900, 0.011920, 0.000000]),
    690: np.array([0.022700, 0.008210, 0.000000]),
    695: np.array([0.015840, 0.005723, 0.000000]),
    700: np.array([0.011359, 0.004102, 0.000000]),
    705: np.array([0.008111, 0.002929, 0.000000]),
    710: np.array([0.005790, 0.002091, 0.000000]),
    715: np.array([0.004109, 0.001484, 0.000000]),
    720: np.array([0.002899, 0.001047, 0.000000]),
    725: np.array([0.002049, 0.000740, 0.000000]),
    730: np.array([0.001440, 0.000520, 0.000000]),
    735: np.array([0.001000, 0.000361, 0.000000]),
    740: np.array([0.000690, 0.000249, 0.000000]),
    745: np.array([0.000476, 0.000172, 0.000000]),
    750: np.array([0.000332, 0.000120, 0.000000]),
    755: np.array([0.000235, 8.50e-05, 0.000000]),
    760: np.array([0.000166, 6.00e-05, 0.000000]),
    765: np.array([0.000117, 4.20e-05, 0.000000]),
    770: np.array([8.30e-05, 3.00e-05, 0.000000]),
    775: np.array([5.90e-05, 2.10e-05, 0.000000]),
    780: np.array([4.20e-05, 1.50e-05, 0.000000]),
}

DATA_SAMPLE_ABRIDGED: dict = {
    500: 0.0651,
    520: 0.0705,
    540: 0.0772,
    560: 0.0870,
    580: 0.1128,
    600: 0.1360,
}

DATA_MULTI_SAMPLE_ABRIDGED: dict = {
    500: (0.004900, 0.323000, 0.272000),
    510: (0.009300, 0.503000, 0.158200),
    520: (0.063270, 0.710000, 0.078250),
    530: (0.165500, 0.862000, 0.042160),
    540: (0.290400, 0.954000, 0.020300),
    550: (0.433450, 0.994950, 0.008750),
    560: (0.594500, 0.995000, 0.003900),
}


class TestSpectralShape(unittest.TestCase):
    """
    Define :class:`colour.colorimetry.spectrum.SpectralShape` class unit tests
    methods.
    """

    def test_required_attributes(self):
        """Test the presence of required attributes."""

        required_attributes = (
            "start",
            "end",
            "interval",
            "boundaries",
            "wavelengths",
        )

        for attribute in required_attributes:
            self.assertIn(attribute, dir(SpectralShape))

    def test_required_methods(self):
        """Test the presence of required methods."""

        required_methods = (
            "__init__",
            "__str__",
            "__repr__",
            "__hash__",
            "__iter__",
            "__contains__",
            "__len__",
            "__eq__",
            "__ne__",
            "range",
        )

        for method in required_methods:
            self.assertIn(method, dir(SpectralShape))

    def test_start(self):
        """
        Test :attr:`colour.colorimetry.spectrum.SpectralShape.start`
        attribute.
        """

        self.assertEqual(SpectralShape(360, 830, 1).start, 360)

        self.assertRaises(AssertionError, lambda: SpectralShape(360, 360, 1))

        self.assertRaises(AssertionError, lambda: SpectralShape(360, 0, 1))

    def test_end(self):
        """Test :attr:`colour.colorimetry.spectrum.SpectralShape.end` property."""

        self.assertEqual(SpectralShape(360, 830, 1).end, 830)

        self.assertRaises(AssertionError, lambda: SpectralShape(830, 830, 1))

        self.assertRaises(AssertionError, lambda: SpectralShape(830, 0, 1))

    def test_interval(self):
        """
        Test :attr:`colour.colorimetry.spectrum.SpectralShape.interval`
        property.
        """

        self.assertEqual(SpectralShape(360, 830, 1).interval, 1)

    def test_boundaries(self):
        """
        Test :attr:`colour.colorimetry.spectrum.SpectralShape.boundaries`
        property.
        """

        shape = SpectralShape(400, 700, 1)
        shape.boundaries = (360, 830)

        self.assertEqual(shape.start, 360)
        self.assertEqual(shape.end, 830)

    def test_wavelengths(self):
        """
        Test :attr:`colour.colorimetry.spectrum.SpectralShape.wavelengths`
        property.
        """

        np.testing.assert_array_almost_equal(
            SpectralShape(0, 10, 0.1).wavelengths,
            np.arange(0, 10 + 0.1, 0.1),
        )

    def test__hash__(self):
        """
        Test :meth:`colour.colorimetry.spectrum.SpectralShape.__hash__`
        method.
        """

        self.assertIsInstance(hash(SpectralShape(0, 10, 0.1)), int)

    def test__iter__(self):
        """
        Test :meth:`colour.colorimetry.spectrum.SpectralShape.__iter__`
        method.
        """

        np.testing.assert_array_almost_equal(
            list(SpectralShape(0, 10, 0.1)),
            np.arange(0, 10 + 0.1, 0.1),
        )

    def test__contains__(self):
        """
        Test :meth:`colour.colorimetry.spectrum.SpectralShape.__contains__`
        method.
        """

        self.assertIn(360.1, SpectralShape(360, 830, 0.1))

        self.assertNotIn(360.11, SpectralShape(360, 830, 0.1))

        self.assertIn(np.array([0.5, 0.6]), SpectralShape(0, 10, 0.1))

        self.assertNotIn(np.array([0.5, 0.61]), SpectralShape(0, 10, 0.1))

    def test__len__(self):
        """
        Test :meth:`colour.colorimetry.spectrum.SpectralShape.__len__`
        method.
        """

        self.assertEqual(len(SpectralShape(0, 10, 0.1)), 101)

    def test__eq__(self):
        """
        Test :meth:`colour.colorimetry.spectrum.SpectralShape.__eq__`
        method.
        """

        self.assertEqual(SpectralShape(0, 10, 0.1), SpectralShape(0, 10, 0.1))
        self.assertNotEqual(SpectralShape(0, 10, 0.1), None)

    def test__ne__(self):
        """
        Test :meth:`colour.colorimetry.spectrum.SpectralShape.__ne__`
        method.
        """

        self.assertNotEqual(
            SpectralShape(0, 10, 0.1), SpectralShape(1, 10, 0.1)
        )

    def test_range(self):
        """Test :func:`colour.colorimetry.spectrum.SpectralShape.range` method."""

        np.testing.assert_array_almost_equal(
            list(SpectralShape(0, 10, 0.1)),
            np.arange(0, 10 + 0.1, 0.1),
        )


class TestSpectralDistribution(unittest.TestCase):
    """
    Define :class:`colour.colorimetry.spectrum.SpectralDistribution`
    class unit tests methods.
    """

    def setUp(self):
        """Initialise the common tests attributes."""

        self._sd = SpectralDistribution(DATA_SAMPLE, name="Sample")

        self._non_uniform_sd = SpectralDistribution(
            DATA_SAMPLE_NON_UNIFORM,
            name="Non Uniform Sample",
            display_name="Display Non Uniform Sample",
        )

        self._phi = (1 + np.sqrt(5)) / 2

    def test_required_attributes(self):
        """Test the presence of required attributes."""

        required_attributes = (
            "display_name",
            "wavelengths",
            "values",
            "shape",
        )

        for attribute in required_attributes:
            self.assertIn(attribute, dir(SpectralDistribution))

    def test_required_methods(self):
        """Test the presence of required methods."""

        required_methods = (
            "__init__",
            "interpolate",
            "extrapolate",
            "align",
            "trim",
            "normalise",
        )

        for method in required_methods:
            self.assertIn(method, dir(SpectralDistribution))

    def test_display_name(self):
        """
        Test :attr:`colour.colorimetry.spectrum.SpectralDistribution.display_name`
        property.
        """

        self.assertEqual(self._sd.display_name, "Sample")
        self.assertEqual(
            self._non_uniform_sd.display_name, "Display Non Uniform Sample"
        )

    def test_wavelengths(self):
        """
        Test :attr:`colour.colorimetry.spectrum.SpectralDistribution.wavelengths`
        property.
        """

        np.testing.assert_array_equal(self._sd.wavelengths, self._sd.domain)

        sd = self._sd.copy()
        sd.wavelengths = sd.wavelengths + 10
        np.testing.assert_array_equal(sd.wavelengths, sd.domain)

    def test_values(self):
        """
        Test :attr:`colour.colorimetry.spectrum.SpectralDistribution.values`
        property.
        """

        np.testing.assert_array_equal(self._sd.values, self._sd.range)

        sd = self._sd.copy()
        sd.values = sd.values + 10
        np.testing.assert_array_equal(sd.values, sd.range)

    def test_shape(self):
        """
        Test :attr:`colour.colorimetry.spectrum.SpectralDistribution.shape`
        property.
        """

        self.assertEqual(self._sd.shape, SpectralShape(340, 820, 20))

    def test__init__(self):
        """
        Test :meth:`colour.colorimetry.spectrum.SpectralDistribution.__init__`
        method.
        """

        np.testing.assert_array_almost_equal(
            SpectralDistribution(DATA_SAMPLE).wavelengths,
            SpectralDistribution(
                DATA_SAMPLE.values(),
                SpectralShape(340, 820, 20),
            ).wavelengths,
        )

    def test_interpolate(self):
        """
        Test :func:`colour.colorimetry.spectrum.\
SpectralDistribution.interpolate` method.
        """

        shape = SpectralShape(self._sd.shape.start, self._sd.shape.end, 1)
        sd = reshape_sd(self._sd, shape, "Interpolate")
        np.testing.assert_array_almost_equal(
            sd.values,
            DATA_SAMPLE_INTERPOLATED,
            decimal=7,
        )
        self.assertEqual(sd.shape, shape)

        shape = SpectralShape(
            self._non_uniform_sd.shape.start,
            self._non_uniform_sd.shape.end,
            1,
        )
        sd = reshape_sd(self._non_uniform_sd, shape, "Interpolate")
        np.testing.assert_allclose(
            sd.values,
            DATA_SAMPLE_INTERPOLATED_NON_UNIFORM,
            rtol=0.0000001,
            atol=0.0000001,
        )
        self.assertEqual(
            sd.shape,
            SpectralShape(
                np.ceil(self._non_uniform_sd.shape.start),
                np.floor(self._non_uniform_sd.shape.end),
                1,
            ),
        )

    def test_extrapolate(self):
        """
        Test :func:`colour.colorimetry.spectrum.\
SpectralDistribution.extrapolate` method.
        """

        data = dict(zip(range(25, 35), [0] * 5 + [1] * 5))
        sd = SpectralDistribution(data)
        sd.extrapolate(SpectralShape(10, 50, 5))

        self.assertAlmostEqual(sd[10], 0, places=7)
        self.assertAlmostEqual(sd[50], 1, places=7)

        sd = SpectralDistribution(
            np.linspace(0, 1, 10), np.linspace(25, 35, 10)
        )
        shape = SpectralShape(10, 50, 10)
        sd.extrapolate(
            shape,
            extrapolator_kwargs={
                "method": "Linear",
                "left": None,
                "right": None,
            },
        )

        self.assertAlmostEqual(sd[10], -1.5000000000000004, places=7)
        self.assertAlmostEqual(sd[50], 2.4999999999999964, places=7)

    def test_align(self):
        """
        Test :func:`colour.colorimetry.spectrum.\
SpectralDistribution.align` method.
        """

        shape = SpectralShape(100, 900, 5)
        self.assertEqual(self._sd.copy().align(shape).shape, shape)

        shape = SpectralShape(600, 650, 1)
        self.assertEqual(self._sd.copy().align(shape).shape, shape)

    def test_trim(self):
        """
        Test :func:`colour.colorimetry.spectrum.\
SpectralDistribution.trim` method.
        """

        shape = SpectralShape(400, 700, 20)
        self.assertEqual(self._sd.copy().trim(shape).shape, shape)

        shape = SpectralShape(200, 900, 1)
        self.assertEqual(self._sd.copy().trim(shape).shape, self._sd.shape)

    def test_normalise(self):
        """
        Test :func:`colour.colorimetry.spectrum.\
SpectralDistribution.normalise` method.
        """

        np.testing.assert_array_almost_equal(
            self._sd.copy().normalise(100).values, DATA_SAMPLE_NORMALISED
        )

    def test_callback_on_domain_changed(self):
        """
        Test :class:`colour.colorimetry.spectrum.\
SpectralDistribution` *on_domain_changed* callback.
        """

        sd = self._sd.copy()
        self.assertEqual(sd.shape, SpectralShape(340, 820, 20))
        sd[840] = 0
        self.assertEqual(sd.shape, SpectralShape(340, 840, 20))


class TestMultiSpectralDistributions(unittest.TestCase):
    """
    Define :class:`colour.colorimetry.spectrum.MultiSpectralDistributions`
    class unit tests methods.
    """

    def setUp(self):
        """Initialise the common tests attributes."""

        self._labels = ("x_bar", "y_bar", "z_bar")
        self._display_labels = (
            "Display x_bar",
            "Display y_bar",
            "Display z_bar",
        )

        self._msds = MultiSpectralDistributions(
            DATA_STANDARD_OBSERVER_2_DEGREE_CIE1931,
            name="Observer",
            labels=self._labels,
        )

        sd = SpectralDistribution(DATA_SAMPLE)
        domain = sd.domain
        range_ = tstack([sd.values, sd.values, sd.values])
        self._sample_msds = MultiSpectralDistributions(
            range_,
            domain,
            name="Sample Observer",
            labels=self._labels,
        )

        sd = SpectralDistribution(DATA_SAMPLE_NON_UNIFORM)
        domain = sd.domain
        range_ = tstack([sd.values, sd.values, sd.values])
        self._non_uniform_sample_msds = MultiSpectralDistributions(
            range_,
            domain,
            name="Non Uniform Sample Observer",
            display_name="Display Non Uniform Sample Observer",
            labels=self._labels,
            display_labels=("Display x_bar", "Display y_bar", "Display z_bar"),
        )

        self._phi = (1 + np.sqrt(5)) / 2

    def test_required_attributes(self):
        """Test the presence of required attributes."""

        required_attributes = (
            "display_name",
            "display_labels",
            "wavelengths",
            "values",
            "shape",
        )

        for attribute in required_attributes:
            self.assertIn(attribute, dir(MultiSpectralDistributions))

    def test_required_methods(self):
        """Test the presence of required methods."""

        required_methods = (
            "__init__",
            "interpolate",
            "extrapolate",
            "align",
            "trim",
            "normalise",
            "to_sds",
        )

        for method in required_methods:
            self.assertIn(method, dir(MultiSpectralDistributions))

    def test_display_name(self):
        """
        Test :attr:`colour.colorimetry.spectrum.MultiSpectralDistributions.display_name`
        property.
        """

        self.assertEqual(self._sample_msds.display_name, "Sample Observer")
        self.assertEqual(
            self._non_uniform_sample_msds.display_name,
            "Display Non Uniform Sample Observer",
        )

    def test_wavelengths(self):
        """
        Test :attr:`colour.colorimetry.spectrum.MultiSpectralDistributions.wavelengths`
        property.
        """

        np.testing.assert_array_equal(
            self._msds.wavelengths, self._msds.domain
        )

        msds = self._msds.copy()
        msds.wavelengths = msds.wavelengths + 10
        np.testing.assert_array_equal(msds.wavelengths, msds.domain)

    def test_values(self):
        """
        Test :attr:`colour.colorimetry.spectrum.MultiSpectralDistributions.values`
        property.
        """

        np.testing.assert_array_equal(self._msds.values, self._msds.range)

        msds = self._msds.copy()
        msds.values = msds.values + 10
        np.testing.assert_array_equal(msds.values, msds.range)

    def test_display_labels(self):
        """
        Test :attr:`colour.colorimetry.spectrum.MultiSpectralDistributions.\
display_labels` property.
        """

        self.assertTupleEqual(
            tuple(self._sample_msds.display_labels), self._labels
        )
        self.assertEqual(
            tuple(self._non_uniform_sample_msds.display_labels),
            ("Display x_bar", "Display y_bar", "Display z_bar"),
        )

    def test_shape(self):
        """
        Test :attr:`colour.colorimetry.spectrum.MultiSpectralDistributions.shape`
        property.
        """

        self.assertEqual(self._msds.shape, SpectralShape(380, 780, 5))

    def test__init__(self):
        """
        Test :func:`colour.colorimetry.spectrum.\
MultiSpectralDistributions.__init__` method.
        """

        np.testing.assert_array_almost_equal(
            MultiSpectralDistributions(DATA_CMFS).wavelengths,
            MultiSpectralDistributions(
                DATA_CMFS.values(),
                SpectralShape(380, 780, 5),
            ).wavelengths,
        )

    def test_interpolate(self):
        """
        Test :func:`colour.colorimetry.spectrum.\
MultiSpectralDistributions.interpolate` method.
        """

        shape = SpectralShape(
            self._sample_msds.shape.start, self._sample_msds.shape.end, 1
        )
        msds = reshape_msds(self._sample_msds, shape, "Interpolate")
        for signal in msds.signals.values():
            np.testing.assert_array_almost_equal(
                signal.values, DATA_SAMPLE_INTERPOLATED, decimal=7
            )
        self.assertEqual(msds.shape, shape)

        shape = SpectralShape(
            self._non_uniform_sample_msds.shape.start,
            self._non_uniform_sample_msds.shape.end,
            1,
        )
        msds = reshape_msds(
            self._non_uniform_sample_msds,
            shape,
            "Interpolate",
        )
        for signal in msds.signals.values():
            np.testing.assert_allclose(
                signal.values,
                DATA_SAMPLE_INTERPOLATED_NON_UNIFORM,
                rtol=0.0000001,
                atol=0.0000001,
            )
        self.assertEqual(
            msds.shape,
            SpectralShape(
                np.ceil(self._non_uniform_sample_msds.shape.start),
                np.floor(self._non_uniform_sample_msds.shape.end),
                1,
            ),
        )

    def test_extrapolate(self):
        """
        Test :func:`colour.colorimetry.spectrum.\
MultiSpectralDistributions.extrapolate` method.
        """

        data = dict(zip(range(25, 35), tstack([[0] * 5 + [1] * 5] * 3)))
        msds = MultiSpectralDistributions(data)
        msds.extrapolate(SpectralShape(10, 50, 5))

        np.testing.assert_array_almost_equal(
            msds[10], np.array([0.0, 0.0, 0.0]), decimal=7
        )
        np.testing.assert_array_almost_equal(
            msds[50], np.array([1.0, 1.0, 1.0]), decimal=7
        )

        msds = MultiSpectralDistributions(
            tstack([np.linspace(0, 1, 10)] * 3), np.linspace(25, 35, 10)
        )
        msds.extrapolate(
            SpectralShape(10, 50, 10),
            extrapolator_kwargs={
                "method": "Linear",
                "left": None,
                "right": None,
            },
        )
        np.testing.assert_array_almost_equal(
            msds[10], np.array([-1.5, -1.5, -1.5]), decimal=7
        )
        np.testing.assert_array_almost_equal(
            msds[50], np.array([2.5, 2.5, 2.5]), decimal=7
        )

    def test_align(self):
        """
        Test :func:`colour.colorimetry.spectrum.\
MultiSpectralDistributions.align` method.
        """

        msds = self._sample_msds.copy()

        shape = SpectralShape(100, 900, 5)
        self.assertEqual(msds.align(shape).shape, shape)

        shape = SpectralShape(600, 650, 1)
        self.assertEqual(msds.align(shape).shape, shape)

    def test_trim(self):
        """
        Test :func:`colour.colorimetry.spectrum.\
MultiSpectralDistributions.trim` method.
        """

        shape = SpectralShape(400, 700, 5)
        self.assertEqual(self._msds.copy().trim(shape).shape, shape)

        shape = SpectralShape(200, 900, 1)
        self.assertEqual(self._msds.copy().trim(shape).shape, self._msds.shape)

    def test_normalise(self):
        """
        Test :func:`colour.colorimetry.spectrum.
        MultiSpectralDistributions.normalise` method.
        """

        np.testing.assert_array_almost_equal(
            self._sample_msds.copy().normalise(100).values,
            tstack([DATA_SAMPLE_NORMALISED] * 3),
        )

    def test_to_sds(self):
        """
        Test :func:`colour.colorimetry.spectrum.\
MultiSpectralDistributions.to_sds` method.
        """

        sds = self._non_uniform_sample_msds.to_sds()
        self.assertEqual(len(sds), 3)

        for i, sd in enumerate(sds):
            self.assertEqual(sd.name, self._labels[i])
            self.assertEqual(sd.display_name, self._display_labels[i])

    def test_callback_on_domain_changed(self):
        """
        Test underlying :class:`colour.colorimetry.spectrum.\
SpectralDistribution` *on_domain_changed* callback when used with
        :class:`colour.colorimetry.spectrum.MultiSpectralDistributions` class.
        """

        msds = self._msds.copy()
        self.assertEqual(msds.shape, SpectralShape(380, 780, 5))
        msds[785] = 0
        self.assertEqual(msds.shape, SpectralShape(380, 785, 5))


class TestReshapeSd(unittest.TestCase):
    """
    Define :func:`colour.colorimetry.spectrum.reshape_sd` definition unit
    tests methods.
    """

    def test_reshape_sd(self):
        """Test :func:`colour.colorimetry.spectrum.reshape_sd` definition."""

        sd = SpectralDistribution(DATA_SAMPLE_ABRIDGED)
        sd_reshaped = reshape_sd(sd)
        self.assertEqual(sd_reshaped, sd.copy().align(SPECTRAL_SHAPE_DEFAULT))
        self.assertEqual(reshape_sd(sd), sd_reshaped)

        shape = colour.SpectralShape(100, 900, 1)
        extrapolator_kwargs = {
            "method": "Constant",
            "left": 0.05,
            "right": 0.15,
        }
        sd_reshaped = reshape_sd(
            sd,
            shape,
            method="Extrapolate",
            extrapolator_kwargs=extrapolator_kwargs,
        )
        self.assertEqual(
            sd_reshaped,
            sd.copy().extrapolate(
                shape, extrapolator_kwargs=extrapolator_kwargs
            ),
        )

        shape = colour.SpectralShape(400, 700, 1)
        interpolator_kwargs = {"fill_value": 0}
        sd_reshaped = reshape_sd(
            sd,
            shape,
            method="Interpolate",
            interpolator=CubicSplineInterpolator,
            interpolator_kwargs=interpolator_kwargs,
        )
        self.assertEqual(
            sd_reshaped,
            sd.copy().interpolate(
                shape,
                interpolator=CubicSplineInterpolator,
                interpolator_kwargs=interpolator_kwargs,
            ),
        )

        sd = SpectralDistribution(DATA_SAMPLE)
        shape = colour.SpectralShape(500, 600, 1)
        sd_reshaped = reshape_sd(sd, shape, method="Trim")
        self.assertEqual(sd_reshaped, sd.copy().trim(shape))

        self.assertIs(
            reshape_sd(sd, shape, method="Trim", copy=False), sd_reshaped
        )


class TestSdsAndMdsToSds(unittest.TestCase):
    """
    Define :func:`colour.colorimetry.spectrum.sds_and_msds_to_sds` definition
    unit tests methods.
    """

    def test_sds_and_msds_to_sds(self):
        """
        Test :func:`colour.colorimetry.spectrum.sds_and_msds_to_sds`
        definition.
        """

        sd_1 = SpectralDistribution(DATA_SAMPLE_ABRIDGED)
        sd_2 = SpectralDistribution(DATA_SAMPLE_ABRIDGED)
        multi_sds_1 = MultiSpectralDistributions(DATA_MULTI_SAMPLE_ABRIDGED)
        multi_sds_2 = MultiSpectralDistributions(DATA_MULTI_SAMPLE_ABRIDGED)

        self.assertEqual(sds_and_msds_to_sds(sd_1), [sd_1])
        self.assertEqual(
            len(
                sds_and_msds_to_sds(
                    [
                        sd_1,
                        sd_2,
                        multi_sds_1,
                        multi_sds_2,
                    ]
                )
            ),
            8,
        )

        self.assertEqual(len(sds_and_msds_to_sds(multi_sds_1)), 3)


class TestSdsAndMsdsToMsds(unittest.TestCase):
    """
    Define :func:`colour.colorimetry.spectrum.sds_and_msds_to_msds`
    definition unit tests methods.
    """

    def test_sds_and_msds_to_msds(self):
        """
        Test :func:`colour.colorimetry.spectrum.sds_and_msds_to_msds`
        definition.
        """

        sd_1 = SpectralDistribution(DATA_SAMPLE_ABRIDGED)
        sd_2 = SpectralDistribution(DATA_SAMPLE_ABRIDGED)
        multi_sds_1 = MultiSpectralDistributions(DATA_MULTI_SAMPLE_ABRIDGED)
        multi_sds_2 = MultiSpectralDistributions(DATA_MULTI_SAMPLE_ABRIDGED)

        self.assertEqual(len(sds_and_msds_to_msds(sd_1)), 6)

        self.assertEqual(sds_and_msds_to_msds(multi_sds_1), multi_sds_1)

        multi_sds_0 = sds_and_msds_to_msds([multi_sds_1])
        np.testing.assert_array_equal(multi_sds_0.range, multi_sds_1.range)
        self.assertEqual(sds_and_msds_to_msds([multi_sds_1]), multi_sds_1)

        shape = SpectralShape(500, 560, 10)
        self.assertEqual(
            sds_and_msds_to_msds([sd_1, sd_2, multi_sds_1, multi_sds_2]).shape,
            shape,
        )

        np.testing.assert_array_almost_equal(
            sds_and_msds_to_msds(
                [sd_1, sd_2, multi_sds_1, multi_sds_2]
            ).wavelengths,
            shape.wavelengths,
            decimal=7,
        )

        np.testing.assert_array_almost_equal(
            sds_and_msds_to_msds(
                [sd_1, sd_2, multi_sds_1, multi_sds_2]
            ).values,
            tstack(
                [sd_1.align(shape).values, sd_2.align(shape).values]
                + [
                    sd.values
                    for sd in sds_and_msds_to_sds(multi_sds_1.align(shape))
                ]
                + [
                    sd.values
                    for sd in sds_and_msds_to_sds(multi_sds_2.align(shape))
                ]
            ),
            decimal=7,
        )


if __name__ == "__main__":
    unittest.main()
