/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { BaseDeclarativeTool, BaseToolInvocation, Kind } from './tools.js';
import * as fs from 'fs/promises';
import * as fsSync from 'fs';
import * as path from 'path';
import * as process from 'process';
import { QWEN_DIR } from '../utils/paths.js';
const todoWriteToolSchemaData = {
    name: 'todo_write',
    description: 'Creates and manages a structured task list for your current coding session. This helps track progress, organize complex tasks, and demonstrate thoroughness.',
    parametersJsonSchema: {
        type: 'object',
        properties: {
            todos: {
                type: 'array',
                items: {
                    type: 'object',
                    properties: {
                        content: {
                            type: 'string',
                            minLength: 1,
                        },
                        status: {
                            type: 'string',
                            enum: ['pending', 'in_progress', 'completed'],
                        },
                        id: {
                            type: 'string',
                        },
                    },
                    required: ['content', 'status', 'id'],
                    additionalProperties: false,
                },
                description: 'The updated todo list',
            },
        },
        required: ['todos'],
        $schema: 'http://json-schema.org/draft-07/schema#',
    },
};
const todoWriteToolDescription = `
Use this tool to create and manage a structured task list for your current coding session. This helps you track progress, organize complex tasks, and demonstrate thoroughness to the user.
It also helps the user understand the progress of the task and overall progress of their requests.

## When to Use This Tool
Use this tool proactively in these scenarios:

1. Complex multi-step tasks - When a task requires 3 or more distinct steps or actions
2. Non-trivial and complex tasks - Tasks that require careful planning or multiple operations
3. User explicitly requests todo list - When the user directly asks you to use the todo list
4. User provides multiple tasks - When users provide a list of things to be done (numbered or comma-separated)
5. After receiving new instructions - Immediately capture user requirements as todos
6. When you start working on a task - Mark it as in_progress BEFORE beginning work. Ideally you should only have one todo as in_progress at a time
7. After completing a task - Mark it as completed and add any new follow-up tasks discovered during implementation

## When NOT to Use This Tool

Skip using this tool when:
1. There is only a single, straightforward task
2. The task is trivial and tracking it provides no organizational benefit
3. The task can be completed in less than 3 trivial steps
4. The task is purely conversational or informational

NOTE that you should not use this tool if there is only one trivial task to do. In this case you are better off just doing the task directly.

## Examples of When to Use the Todo List

<example>
User: I want to add a dark mode toggle to the application settings. Make sure you run the tests and build when you're done!
Assistant: I'll help add a dark mode toggle to your application settings. Let me create a todo list to track this implementation.
*Creates todo list with the following items:*
1. Create dark mode toggle component in Settings page
2. Add dark mode state management (context/store)
3. Implement CSS-in-JS styles for dark theme
4. Update existing components to support theme switching
5. Run tests and build process, addressing any failures or errors that occur
*Begins working on the first task*

<reasoning>
The assistant used the todo list because:
1. Adding dark mode is a multi-step feature requiring UI, state management, and styling changes
2. The user explicitly requested tests and build be run afterward
3. The assistant inferred that tests and build need to pass by adding "Ensure tests and build succeed" as the final task
</reasoning>
</example>

<example>
User: Help me rename the function getCwd to getCurrentWorkingDirectory across my project
Assistant: Let me first search through your codebase to find all occurrences of 'getCwd'.
*Uses grep or search tools to locate all instances of getCwd in the codebase*
Assistant: I've found 15 instances of 'getCwd' across 8 different files. Let me create a todo list to track these changes.
*Creates todo list with specific items for each file that needs updating*

<reasoning>
The assistant used the todo list because:
1. First, the assistant searched to understand the scope of the task
2. Upon finding multiple occurrences across different files, it determined this was a complex task with multiple steps
3. The todo list helps ensure every instance is tracked and updated systematically
4. This approach prevents missing any occurrences and maintains code consistency
</reasoning>
</example>


<example>
User: I need to implement these features for my e-commerce site: user registration, product catalog, shopping cart, and checkout flow.
Assistant: I'll help implement these features. First, let's add all the features to the todo list.
*Creates a todo list breaking down each feature into specific tasks based on the project architecture*
Assistant: Let's start with user registration. This will involve creating database models, API endpoints, and frontend forms that integrate with your existing authentication system.

<reasoning>
The assistant used the todo list because:
1. The user provided multiple complex features to implement in a comma separated list
2. The todo list helps organize these large features into manageable tasks
3. This approach allows for tracking progress across the entire implementation
</reasoning>
</example>

<example>
User: Can you help optimize my React application? It's rendering slowly and has performance issues.</user>
Assistant: I'll help optimize your React application. First, let me examine your codebase to identify potential performance bottlenecks.
*Reviews component structure, render patterns, state management, and data fetching*
Assistant: After analyzing your codebase, I've identified several performance issues. Let me create a todo list to track our optimization efforts.
*Creates todo list with items like: 1) Implement memoization for expensive calculations in ProductList, 2) Add virtualization for long lists in Dashboard, 3) Optimize image loading in Gallery component, 4) Fix state update loops in ShoppingCart, 5) Review bundle size and implement code splitting*
Let's start by implementing memoization for the expensive calculations in your ProductList component.</assistant>

<reasoning>
The assistant used the todo list because:
1. First, the assistant examined the codebase to identify specific performance issues
2. Based on this analysis, it identified multiple optimization opportunities
3. Performance optimization is a non-trivial task requiring multiple steps
4. The todo list helps methodically track improvements across different components
5. This systematic approach ensures all performance bottlenecks are addressed
</reasoning>
</example>

## Examples of When NOT to Use the Todo List

<example>
User: How do I print 'Hello World' in Python?
Assistant: In Python, you can print "Hello World" with this simple code:

python
print("Hello World")

This will output the text "Hello World" to the console when executed.</assistant>

<reasoning>
The assistant did not use the todo list because this is a single, trivial task that can be completed in one step. There's no need to track multiple tasks or steps for such a straightforward request.
</reasoning>
</example>

<example>
User: What does the git status command do?
Assistant: The git status command shows the current state of your working directory and staging area. It displays which changes have been staged, which haven't, and which files aren't being tracked by Git. This is a fundamental command for understanding the current state of your repository before making commits.

<reasoning>
The assistant did not use the todo list because this is an informational request with no actual coding task to complete. The user is simply asking for an explanation, not for the assistant to perform multiple steps or tasks.
</reasoning>
</example>

<example>
User: Can you add a comment to the calculateTotal function to explain what it does?
Assistant: Sure, let me add a comment to the calculateTotal function to explain what it does.
* Uses the Edit tool to add a comment to the calculateTotal function *

<reasoning>
The assistant did not use the todo list because this is a single, straightforward task confined to one location in the code. Adding a comment doesn't require tracking multiple steps or systematic organization.
</reasoning>
</example>

<example>
User: Run npm install for me and tell me what happens.
Assistant: I'll run the npm install command for you.

*Executes: npm install*

The command completed successfully. Here's the output:
[Output of npm install command]

All dependencies have been installed according to your package.json file.

<reasoning>
The assistant did not use the todo list because this is a single command execution with immediate results. There are no multiple steps to track or organize, making the todo list unnecessary for this straightforward task.
</reasoning>
</example>

## Task States and Management

1. **Task States**: Use these states to track progress:
   - pending: Task not yet started
   - in_progress: Currently working on (limit to ONE task at a time)
   - completed: Task finished successfully

2. **Task Management**:
   - Update task status in real-time as you work
   - Mark tasks complete IMMEDIATELY after finishing (don't batch completions)
   - Only have ONE task in_progress at any time
   - Complete current tasks before starting new ones
   - Remove tasks that are no longer relevant from the list entirely

3. **Task Completion Requirements**:
   - ONLY mark a task as completed when you have FULLY accomplished it
   - If you encounter errors, blockers, or cannot finish, keep the task as in_progress
   - When blocked, create a new task describing what needs to be resolved
   - Never mark a task as completed if:
     - Tests are failing
     - Implementation is partial
     - You encountered unresolved errors
     - You couldn't find necessary files or dependencies

4. **Task Breakdown**:
   - Create specific, actionable items
   - Break complex tasks into smaller, manageable steps
   - Use clear, descriptive task names

When in doubt, use this tool. Being proactive with task management demonstrates attentiveness and ensures you complete all requirements successfully.
`;
const TODO_SUBDIR = 'todos';
function getTodoFilePath(sessionId) {
    const homeDir = process.env['HOME'] || process.env['USERPROFILE'] || process.cwd();
    const todoDir = path.join(homeDir, QWEN_DIR, TODO_SUBDIR);
    // Use sessionId if provided, otherwise fall back to 'default'
    const filename = `${sessionId || 'default'}.json`;
    return path.join(todoDir, filename);
}
/**
 * Reads the current todos from the file system
 */
async function readTodosFromFile(sessionId) {
    try {
        const todoFilePath = getTodoFilePath(sessionId);
        const content = await fs.readFile(todoFilePath, 'utf-8');
        const data = JSON.parse(content);
        return Array.isArray(data.todos) ? data.todos : [];
    }
    catch (err) {
        const error = err;
        if (!(error instanceof Error) || error.code !== 'ENOENT') {
            throw err;
        }
        return [];
    }
}
/**
 * Writes todos to the file system
 */
async function writeTodosToFile(todos, sessionId) {
    const todoFilePath = getTodoFilePath(sessionId);
    const todoDir = path.dirname(todoFilePath);
    await fs.mkdir(todoDir, { recursive: true });
    const data = {
        todos,
        sessionId: sessionId || 'default',
    };
    await fs.writeFile(todoFilePath, JSON.stringify(data, null, 2), 'utf-8');
}
class TodoWriteToolInvocation extends BaseToolInvocation {
    config;
    operationType;
    constructor(config, params, operationType = 'update') {
        super(params);
        this.config = config;
        this.operationType = operationType;
    }
    getDescription() {
        return this.operationType === 'create' ? 'Create todos' : 'Update todos';
    }
    async shouldConfirmExecute(_abortSignal) {
        // Todo operations should execute automatically without user confirmation
        return false;
    }
    async execute(_signal) {
        const { todos, modified_by_user, modified_content } = this.params;
        const sessionId = this.config.getSessionId();
        try {
            let finalTodos;
            if (modified_by_user && modified_content !== undefined) {
                // User modified the content in external editor, parse it directly
                const data = JSON.parse(modified_content);
                finalTodos = Array.isArray(data.todos) ? data.todos : [];
            }
            else {
                // Use the normal todo logic - simply replace with new todos
                finalTodos = todos;
            }
            await writeTodosToFile(finalTodos, sessionId);
            // Create structured display object for rich UI rendering
            const todoResultDisplay = {
                type: 'todo_list',
                todos: finalTodos,
            };
            return {
                llmContent: JSON.stringify({
                    success: true,
                    todos: finalTodos,
                }),
                returnDisplay: todoResultDisplay,
            };
        }
        catch (error) {
            const errorMessage = error instanceof Error ? error.message : String(error);
            console.error(`[TodoWriteTool] Error executing todo_write: ${errorMessage}`);
            return {
                llmContent: JSON.stringify({
                    success: false,
                    error: `Failed to write todos. Detail: ${errorMessage}`,
                }),
                returnDisplay: `Error writing todos: ${errorMessage}`,
            };
        }
    }
}
/**
 * Utility function to read todos for a specific session (useful for session recovery)
 */
export async function readTodosForSession(sessionId) {
    return readTodosFromFile(sessionId);
}
/**
 * Utility function to list all todo files in the todos directory
 */
export async function listTodoSessions() {
    try {
        const homeDir = process.env['HOME'] || process.env['USERPROFILE'] || process.cwd();
        const todoDir = path.join(homeDir, QWEN_DIR, TODO_SUBDIR);
        const files = await fs.readdir(todoDir);
        return files
            .filter((file) => file.endsWith('.json'))
            .map((file) => file.replace('.json', ''));
    }
    catch (err) {
        const error = err;
        if (!(error instanceof Error) || error.code !== 'ENOENT') {
            throw err;
        }
        return [];
    }
}
export class TodoWriteTool extends BaseDeclarativeTool {
    config;
    static Name = todoWriteToolSchemaData.name;
    constructor(config) {
        super(TodoWriteTool.Name, 'TodoWrite', todoWriteToolDescription, Kind.Think, todoWriteToolSchemaData.parametersJsonSchema);
        this.config = config;
    }
    validateToolParams(params) {
        // Validate todos array
        if (!Array.isArray(params.todos)) {
            return 'Parameter "todos" must be an array.';
        }
        // Validate individual todos
        for (const todo of params.todos) {
            if (!todo.id || typeof todo.id !== 'string' || todo.id.trim() === '') {
                return 'Each todo must have a non-empty "id" string.';
            }
            if (!todo.content ||
                typeof todo.content !== 'string' ||
                todo.content.trim() === '') {
                return 'Each todo must have a non-empty "content" string.';
            }
            if (!['pending', 'in_progress', 'completed'].includes(todo.status)) {
                return 'Each todo must have a valid "status" (pending, in_progress, completed).';
            }
        }
        // Check for duplicate IDs
        const ids = params.todos.map((todo) => todo.id);
        const uniqueIds = new Set(ids);
        if (ids.length !== uniqueIds.size) {
            return 'Todo IDs must be unique within the array.';
        }
        return null;
    }
    createInvocation(params) {
        // Determine if this is a create or update operation by checking if todos file exists
        const sessionId = this.config.getSessionId();
        const todoFilePath = getTodoFilePath(sessionId);
        const operationType = fsSync.existsSync(todoFilePath) ? 'update' : 'create';
        return new TodoWriteToolInvocation(this.config, params, operationType);
    }
}
//# sourceMappingURL=todoWrite.js.map