%define OT_SimulationResult_doc
"Probability simulation result.

Notes
-----
Structure created by the method run() of a :class:`~openturns.EventSimulation`,
and obtained thanks to the method getResult().

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> limitState = ot.SymbolicFunction(['E', 'F', 'L', 'I'], ['-F*L^3/(3.*E*I)'])
>>> # Enable the history mechanism in order to use the getImportanceFactors method
>>> limitState = ot.MemoizeFunction(limitState)
>>> distribution = ot.Normal([50.0, 1.0, 10.0, 5.0], [1.0]*4, ot.IdentityMatrix(4))
>>> output = ot.CompositeRandomVector(limitState, ot.RandomVector(distribution))
>>> event = ot.ThresholdEvent(output, ot.Less(), -3.0)
>>> experiment = ot.MonteCarloExperiment()
>>> algo = ot.ProbabilitySimulationAlgorithm(event, experiment)
>>> algo.run()
>>> result = algo.getResult()
>>> importanceFactors = result.getImportanceFactors()"
%enddef
%feature("docstring") OT::ProbabilitySimulationResult
OT_SimulationResult_doc

// ---------------------------------------------------------------------

%define OT_SimulationResult_getBlockSize_doc
"Accessor to the block size.

Returns
-------
blockSize : int
    Number of terms in the probability simulation estimator grouped together."
%enddef
%feature("docstring") OT::ProbabilitySimulationResult::getBlockSize
OT_SimulationResult_getBlockSize_doc

// ---------------------------------------------------------------------

%define OT_SimulationResult_setBlockSize_doc
"Accessor to the block size.

Parameters
----------
blockSize : int, :math:`blockSize \\geq 0`
    Number of terms in the probability simulation estimator grouped together."
%enddef
%feature("docstring") OT::ProbabilitySimulationResult::setBlockSize
OT_SimulationResult_setBlockSize_doc

// ---------------------------------------------------------------------

%define OT_SimulationResult_setEvent_doc
"Accessor to the event.

Parameters
----------
event : :class:`~openturns.RandomVector`
    Event we want to evaluate the probability."
%enddef
%feature("docstring") OT::ProbabilitySimulationResult::setEvent
OT_SimulationResult_setEvent_doc

// ---------------------------------------------------------------------

%define OT_SimulationResult_getEvent_doc
"Accessor to the event.

Returns
-------
event : :class:`~openturns.RandomVector`
    Event we want to evaluate the probability."
%enddef
%feature("docstring") OT::ProbabilitySimulationResult::getEvent
OT_SimulationResult_getEvent_doc

// ---------------------------------------------------------------------

%define OT_SimulationResult_getImportanceFactors_doc
"Accessor to the importance factors.

Returns
-------
importanceFactors : :class:`~openturns.PointWithDescription`
    Sequence containing the importance factors with a description for each
    component.

Notes
-----
The importance factors :math:`\\alpha_i` are evaluated from the coordinates of
the mean point of event domain :math:`\\vect{X}^*_{event}`, mapped into the
standard space as follows:

.. math::

  \\alpha_i = \\displaystyle \\frac{\\left(U_{i}^*\\right)^2}{||\\vect{U}^*||^2}

where :math:`\\vect{U}^* = T(\\vect{X}^*_{event})`
with :math:`T` the iso-probabilistic transformation and the mean point
:math:`\\vect{X}^*_{event} = \\displaystyle \\frac{1}{n} \\sum_{i=1}^{n} \\vect{X}_i 1_{event}(\\vect{X}_i)`.

.. warning::
    This notion is only available if the history mechanism of the model is
    activated (see :class:`~openturns.MemoizeFunction`).

See also
--------
drawImportanceFactors"
%enddef
%feature("docstring") OT::ProbabilitySimulationResult::getImportanceFactors
OT_SimulationResult_getImportanceFactors_doc

// ---------------------------------------------------------------------

%define OT_SimulationResult_getMeanPointInEventDomain_doc
"Accessor to the mean point conditioned to the event realization.

Returns
-------
meanPoint : :class:`~openturns.Point`
    Mean point in the physical space of all the simulations generated by the
    :class:`~openturns.EventSimulation` algorithm that failed into the event domain.

Notes
-----
.. warning::
    This notion is only available if the history mechanism of the model is
    activated (see :class:`~openturns.MemoizeFunction`)."
%enddef
%feature("docstring") OT::ProbabilitySimulationResult::getMeanPointInEventDomain
OT_SimulationResult_getMeanPointInEventDomain_doc

// ---------------------------------------------------------------------

%define OT_SimulationResult_getCoefficientOfVariation_doc
"Accessor to the coefficient of variation.

Returns
-------
coefficient : float
    Coefficient of variation of the simulated sample which is equal to
    :math:`\\sqrt{Var_e} / P_e` with :math:`Var_e` the variance estimate and
    :math:`P_e` the probability estimate."
%enddef
%feature("docstring") OT::ProbabilitySimulationResult::getCoefficientOfVariation
OT_SimulationResult_getCoefficientOfVariation_doc

// ---------------------------------------------------------------------

%define OT_SimulationResult_getConfidenceLength_doc
"Accessor to the confidence length.

Parameters
----------
level : float, :math:`level \\in ]0, 1[`
    Confidence level. By default, it is :math:`0.95`.

Returns
-------
confidenceLength : float
    Length of the confidence interval at the confidence level *level*."
%enddef
%feature("docstring") OT::ProbabilitySimulationResult::getConfidenceLength
OT_SimulationResult_getConfidenceLength_doc

// ---------------------------------------------------------------------

%define OT_SimulationResult_getOuterSampling_doc
"Accessor to the outer sampling.

Returns
-------
outerSampling : int
    Number of groups of terms in the probability simulation estimator."
%enddef
%feature("docstring") OT::ProbabilitySimulationResult::getOuterSampling
OT_SimulationResult_getOuterSampling_doc

// ---------------------------------------------------------------------

%define OT_SimulationResult_setOuterSampling_doc
"Accessor to the outer sampling.

Parameters
----------
outerSampling : int, :math:`outerSampling \\geq 0`
    Number of groups of terms in the probability simulation estimator."
%enddef
%feature("docstring") OT::ProbabilitySimulationResult::setOuterSampling
OT_SimulationResult_setOuterSampling_doc

// ---------------------------------------------------------------------

%define OT_SimulationResult_getStandardDeviation_doc
"Accessor to the standard deviation.

Returns
-------
sigma : float
    Standard deviation of the estimator at the end of the simulation."
%enddef
%feature("docstring") OT::ProbabilitySimulationResult::getStandardDeviation
OT_SimulationResult_getStandardDeviation_doc

// ---------------------------------------------------------------------

%define OT_SimulationResult_getVarianceEstimate_doc
"Accessor to the variance estimate.

Returns
-------
varianceEstimate : float
    Variance estimate."
%enddef
%feature("docstring") OT::ProbabilitySimulationResult::getVarianceEstimate
OT_SimulationResult_getVarianceEstimate_doc

// ---------------------------------------------------------------------

%define OT_SimulationResult_setVarianceEstimate_doc
"Accessor to the variance estimate.

Parameters
----------
varianceEstimate : float, :math:`Var_e \\geq 0`
    Variance estimate."
%enddef
%feature("docstring") OT::ProbabilitySimulationResult::setVarianceEstimate
OT_SimulationResult_setVarianceEstimate_doc

// ---------------------------------------------------------------------

%define OT_SimulationResult_getProbabilityEstimate_doc
"Accessor to the probability estimate.

Returns
-------
probaEstimate : float
    Estimate of the event probability."
%enddef
%feature("docstring") OT::ProbabilitySimulationResult::getProbabilityEstimate
OT_SimulationResult_getProbabilityEstimate_doc

// ---------------------------------------------------------------------

%define OT_SimulationResult_setProbabilityEstimate_doc
"Accessor to the probability estimate.

Parameters
----------
probaEstimate : float, :math:`0 \\leq P_e \\leq 1`
    Estimate of the event probability."
%enddef
%feature("docstring") OT::ProbabilitySimulationResult::setProbabilityEstimate
OT_SimulationResult_setProbabilityEstimate_doc

// ---------------------------------------------------------------------

%define OT_SimulationResult_getProbabilityDistribution_doc
"Accessor to the asymptotic probability distribution.

Returns
-------
probaDistribution : :class:`~openturns.Normal`
    Asymptotic normal distribution of the event probability estimate."
%enddef
%feature("docstring") OT::ProbabilitySimulationResult::getProbabilityDistribution
OT_SimulationResult_getProbabilityDistribution_doc

// ---------------------------------------------------------------------

%define OT_SimulationResult_drawImportanceFactors_doc
"Draw the importance factors.

.. warning::
    It is necessary to enable the history of the model to perform this analysis
    (see :class:`~openturns.MemoizeFunction`).

Returns
-------
graph : :class:`~openturns.Graph`
    Importance factor graph.

See also
--------
getImportanceFactors"
%enddef
%feature("docstring") OT::ProbabilitySimulationResult::drawImportanceFactors
OT_SimulationResult_drawImportanceFactors_doc
