%feature("docstring") OT::SimulationAlgorithm
"Base class for simulation algorithms.

See also
--------
ProbabilitySimulationAlgorithm, ExpectationSimulationAlgorithm"

// ---------------------------------------------------------------------

%feature("docstring") OT::SimulationAlgorithm::getBlockSize
"Accessor to the block size.

Returns
-------
blockSize : int
    Number of terms in the probability simulation estimator grouped together.
    It is set by default to 1."

// ---------------------------------------------------------------------

%feature("docstring") OT::SimulationAlgorithm::setBlockSize
"Accessor to the block size.

Parameters
----------
blockSize : int, :math:`blockSize \\geq 1`
    Number of terms in the probability simulation estimator grouped together.
    It is set by default to 1.

Notes
-----
For Monte Carlo, LHS and Importance Sampling methods, this allows one to save space
while allowing multithreading, when available we recommend
to use the number of available CPUs; for the Directional Sampling, we recommend
to set it to 1."

// ---------------------------------------------------------------------

%feature("docstring") OT::SimulationAlgorithm::getMaximumCoefficientOfVariation
"Accessor to the maximum coefficient of variation.

Returns
-------
coefficient : float
    Maximum coefficient of variation of the simulated sample."

// ---------------------------------------------------------------------

%feature("docstring") OT::SimulationAlgorithm::setMaximumCoefficientOfVariation
"Accessor to the maximum coefficient of variation.

Parameters
----------
coefficient : float
    Maximum coefficient of variation of the simulated sample."

// ---------------------------------------------------------------------

%feature("docstring") OT::SimulationAlgorithm::getMaximumOuterSampling
"Accessor to the maximum sample size.

Returns
-------
outerSampling : int
    Maximum number of groups of terms in the probability simulation estimator."

// ---------------------------------------------------------------------

%feature("docstring") OT::SimulationAlgorithm::setMaximumOuterSampling
"Accessor to the maximum sample size.

Parameters
----------
outerSampling : int
    Maximum number of groups of terms in the probability simulation estimator."

// ---------------------------------------------------------------------

%feature("docstring") OT::SimulationAlgorithm::getMaximumStandardDeviation
"Accessor to the maximum standard deviation.

Returns
-------
sigma : float, :math:`\\sigma > 0`
    Maximum standard deviation of the estimator."

// ---------------------------------------------------------------------

%feature("docstring") OT::SimulationAlgorithm::setMaximumStandardDeviation
"Accessor to the maximum standard deviation.

Parameters
----------
sigma : float, :math:`\\sigma > 0`
    Maximum standard deviation of the estimator."

// ---------------------------------------------------------------------

%feature("docstring") OT::SimulationAlgorithm::run
"Launch simulation.

Notes
-----
It launches the simulation on a sample of size at most *outerSampling * blockSize*,
this sample being built by blocks of size *blockSize*. It allows one to use
efficiently the distribution of the computation as well as it allows one to deal
with a sample size :math:`> 2^{32}` by a combination of *blockSize* and
*outerSampling*.

see also
--------
setBlockSize, setMaximumOuterSampling, ResourceMap, SimulationResult"

// ---------------------------------------------------------------------

%feature("docstring") OT::SimulationAlgorithm::setProgressCallback
"Set up a progress callback.

Can be used to programmatically report the progress of a simulation.

Parameters
----------
callback : callable
    Takes a float as argument as percentage of progress.

Examples
--------
>>> import sys
>>> import openturns as ot
>>> experiment = ot.MonteCarloExperiment()
>>> X = ot.RandomVector(ot.Normal())
>>> Y = ot.CompositeRandomVector(ot.SymbolicFunction(['X'], ['1.1*X']), X)
>>> event = ot.ThresholdEvent(Y, ot.Less(), -2.0)
>>> algo = ot.ProbabilitySimulationAlgorithm(event, experiment)
>>> algo.setMaximumOuterSampling(100)
>>> algo.setMaximumCoefficientOfVariation(-1.0)
>>> def report_progress(progress):
...     sys.stderr.write('-- progress=' + str(progress) + '%\\n')
>>> algo.setProgressCallback(report_progress)
>>> algo.run()"

// ---------------------------------------------------------------------

%feature("docstring") OT::SimulationAlgorithm::setStopCallback
"Set up a stop callback.

Can be used to programmatically stop a simulation.

Parameters
----------
callback : callable
    Returns an int deciding whether to stop or continue.

Examples
--------
Stop a Monte Carlo simulation algorithm using a time limit

>>> import openturns as ot
>>> experiment = ot.MonteCarloExperiment()
>>> X = ot.RandomVector(ot.Normal())
>>> Y = ot.CompositeRandomVector(ot.SymbolicFunction(['X'], ['1.1*X']), X)
>>> event = ot.ThresholdEvent(Y, ot.Less(), -2.0)
>>> algo = ot.ProbabilitySimulationAlgorithm(event, experiment)
>>> algo.setMaximumOuterSampling(10000000)
>>> algo.setMaximumCoefficientOfVariation(-1.0)
>>> timer = ot.TimerCallback(0.1)
>>> algo.setStopCallback(timer)
>>> algo.run()"

// ---------------------------------------------------------------------

%feature("docstring") OT::SimulationAlgorithm::getConvergenceStrategy
"Accessor to the convergence strategy.

Returns
-------
storage_strategy : :class:`~openturns.HistoryStrategy`
    Storage strategy used to store the values of the probability estimator
    and its variance during the simulation algorithm."

// ---------------------------------------------------------------------

%feature("docstring") OT::SimulationAlgorithm::setConvergenceStrategy
"Accessor to the convergence strategy.

Parameters
----------
storage_strategy : :class:`~openturns.HistoryStrategy`
    Storage strategy used to store the values of the probability estimator
    and its variance during the simulation algorithm."
