export type NvimVersion = {
    /** Path to `nvim` executable. */
    readonly path: string;
    /** Nvim version, or undefined if there was an error. */
    readonly nvimVersion?: string;
    /** Nvim build type, or undefined if there was an error. */
    readonly buildType?: string;
    /** Nvim LuaJIT version, or undefined if there was an error. */
    readonly luaJitVersion?: string;
    /** Error caught while attempting to access or run Nvim at the given path. */
    readonly error?: Readonly<Error>;
};
export type FindNvimOptions = {
    /**
     * (Optional) Minimum `nvim` version (inclusive) to search for.
     *
     * - Example: `'0.5.0'`
     */
    readonly minVersion?: string;
    /**
     * (Optional) Sort order of list of `nvim` versions.
     *
     * - "desc" - (Default) Sort by version in descending order (highest to lowest).
     *   - Example: `['0.5.0', '0.4.4', '0.4.3']`
     * - "none" - Order is that of the searched `$PATH` components.
     *   - Example: `['0.4.4', '0.5.0', '0.4.3']`
     */
    readonly orderBy?: 'desc' | 'none';
    /**
     * (Optional) Stop searching after found a valid match
     */
    readonly firstMatch?: boolean;
    /**
     * (Optional) Additional specific file paths to check for Nvim executables.
     * These paths will be checked before searching `dirs`.
     * Useful for allowing users to specify exact Nvim executable locations.
     *
     * Example: ['/usr/local/bin/nvim', '/opt/homebrew/bin/nvim']
     */
    readonly paths?: string[];
    /**
     * (Optional) Additional directories to search for Nvim executables.
     * These directories will be searched after checking `paths`
     * but before searching `$PATH` and other default locations.
     * Useful for including non-standard installation directories.
     *
     * Example: ['/opt/neovim/bin', '/home/user/custom/bin']
     */
    readonly dirs?: string[];
};
export type FindNvimResult = {
    /**
     * List of satisfying `nvim` versions found (if any) on the current system, sorted in the order
     * specified by `orderBy`.
     */
    readonly matches: ReadonlyArray<NvimVersion>;
    /**
     * List of invalid or failed `nvim` versions found (if any), in order of searched `$PATH` components.
     */
    readonly invalid: ReadonlyArray<NvimVersion>;
};
/**
 * Tries to find a usable `nvim` binary on the current system.
 *
 * @param opt.minVersion See {@link FindNvimOptions.minVersion}
 * @param opt.orderBy See {@link FindNvimOptions.orderBy}
 * @param opt.firstMatch See {@link FindNvimOptions.firstMatch}
 * @param opt.paths See {@link FindNvimOptions.paths}
 * @param opt.dirs See {@link FindNvimOptions.dirs}
 */
export declare function findNvim(opt?: FindNvimOptions): Readonly<FindNvimResult>;
export declare let exportsForTesting: any;
