/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { getResponseText } from '../utils/partUtils.js';
import { reportError } from '../utils/errorReporting.js';
import { getErrorMessage } from '../utils/errors.js';
import { logMalformedJsonResponse } from '../telemetry/loggers.js';
import { MalformedJsonResponseEvent } from '../telemetry/types.js';
import { retryWithBackoff } from '../utils/retry.js';
/**
 * A client dedicated to stateless, utility-focused LLM calls.
 */
export class BaseLlmClient {
    contentGenerator;
    config;
    // Default configuration for utility tasks
    defaultUtilityConfig = {
        temperature: 0,
        topP: 1,
    };
    constructor(contentGenerator, config) {
        this.contentGenerator = contentGenerator;
        this.config = config;
    }
    async generateJson(options) {
        const { contents, schema, model, abortSignal, systemInstruction, promptId, } = options;
        const requestConfig = {
            abortSignal,
            ...this.defaultUtilityConfig,
            ...options.config,
            ...(systemInstruction && { systemInstruction }),
            responseJsonSchema: schema,
            responseMimeType: 'application/json',
        };
        try {
            const apiCall = () => this.contentGenerator.generateContent({
                model,
                config: requestConfig,
                contents,
            }, promptId);
            const result = await retryWithBackoff(apiCall);
            let text = getResponseText(result)?.trim();
            if (!text) {
                const error = new Error('API returned an empty response for generateJson.');
                await reportError(error, 'Error in generateJson: API returned an empty response.', contents, 'generateJson-empty-response');
                throw error;
            }
            text = this.cleanJsonResponse(text, model);
            try {
                return JSON.parse(text);
            }
            catch (parseError) {
                const error = new Error(`Failed to parse API response as JSON: ${getErrorMessage(parseError)}`);
                await reportError(parseError, 'Failed to parse JSON response from generateJson.', {
                    responseTextFailedToParse: text,
                    originalRequestContents: contents,
                }, 'generateJson-parse');
                throw error;
            }
        }
        catch (error) {
            if (abortSignal.aborted) {
                throw error;
            }
            if (error instanceof Error &&
                (error.message === 'API returned an empty response for generateJson.' ||
                    error.message.startsWith('Failed to parse API response as JSON:'))) {
                // We perform this check so that we don't report these again.
            }
            else {
                await reportError(error, 'Error generating JSON content via API.', contents, 'generateJson-api');
            }
            throw new Error(`Failed to generate JSON content: ${getErrorMessage(error)}`);
        }
    }
    cleanJsonResponse(text, model) {
        const prefix = '```json';
        const suffix = '```';
        if (text.startsWith(prefix) && text.endsWith(suffix)) {
            logMalformedJsonResponse(this.config, new MalformedJsonResponseEvent(model));
            return text.substring(prefix.length, text.length - suffix.length).trim();
        }
        return text;
    }
}
//# sourceMappingURL=baseLlmClient.js.map