/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
vi.mock('../ui/commands/aboutCommand.js', async () => {
    const { CommandKind } = await import('../ui/commands/types.js');
    return {
        aboutCommand: {
            name: 'about',
            description: 'About the CLI',
            kind: CommandKind.BUILT_IN,
        },
    };
});
vi.mock('../ui/commands/ideCommand.js', () => ({ ideCommand: vi.fn() }));
vi.mock('../ui/commands/restoreCommand.js', () => ({
    restoreCommand: vi.fn(),
}));
import { describe, it, expect, vi, beforeEach } from 'vitest';
import { BuiltinCommandLoader } from './BuiltinCommandLoader.js';
import { CommandKind } from '../ui/commands/types.js';
import { ideCommand } from '../ui/commands/ideCommand.js';
import { restoreCommand } from '../ui/commands/restoreCommand.js';
vi.mock('../ui/commands/authCommand.js', () => ({ authCommand: {} }));
vi.mock('../ui/commands/bugCommand.js', () => ({ bugCommand: {} }));
vi.mock('../ui/commands/chatCommand.js', () => ({ chatCommand: {} }));
vi.mock('../ui/commands/clearCommand.js', () => ({ clearCommand: {} }));
vi.mock('../ui/commands/compressCommand.js', () => ({ compressCommand: {} }));
vi.mock('../ui/commands/corgiCommand.js', () => ({ corgiCommand: {} }));
vi.mock('../ui/commands/docsCommand.js', () => ({ docsCommand: {} }));
vi.mock('../ui/commands/editorCommand.js', () => ({ editorCommand: {} }));
vi.mock('../ui/commands/extensionsCommand.js', () => ({
    extensionsCommand: {},
}));
vi.mock('../ui/commands/helpCommand.js', () => ({ helpCommand: {} }));
vi.mock('../ui/commands/memoryCommand.js', () => ({ memoryCommand: {} }));
vi.mock('../ui/commands/privacyCommand.js', () => ({ privacyCommand: {} }));
vi.mock('../ui/commands/quitCommand.js', () => ({ quitCommand: {} }));
vi.mock('../ui/commands/statsCommand.js', () => ({ statsCommand: {} }));
vi.mock('../ui/commands/themeCommand.js', () => ({ themeCommand: {} }));
vi.mock('../ui/commands/toolsCommand.js', () => ({ toolsCommand: {} }));
vi.mock('../ui/commands/mcpCommand.js', () => ({
    mcpCommand: {
        name: 'mcp',
        description: 'MCP command',
        kind: 'BUILT_IN',
    },
}));
describe('BuiltinCommandLoader', () => {
    let mockConfig;
    const ideCommandMock = ideCommand;
    const restoreCommandMock = restoreCommand;
    beforeEach(() => {
        vi.clearAllMocks();
        mockConfig = { some: 'config' };
        ideCommandMock.mockReturnValue({
            name: 'ide',
            description: 'IDE command',
            kind: CommandKind.BUILT_IN,
        });
        restoreCommandMock.mockReturnValue({
            name: 'restore',
            description: 'Restore command',
            kind: CommandKind.BUILT_IN,
        });
    });
    it('should correctly pass the config object to command factory functions', async () => {
        const loader = new BuiltinCommandLoader(mockConfig);
        await loader.loadCommands(new AbortController().signal);
        expect(ideCommandMock).toHaveBeenCalledTimes(1);
        expect(ideCommandMock).toHaveBeenCalledWith(mockConfig);
        expect(restoreCommandMock).toHaveBeenCalledTimes(1);
        expect(restoreCommandMock).toHaveBeenCalledWith(mockConfig);
    });
    it('should filter out null command definitions returned by factories', async () => {
        // Override the mock's behavior for this specific test.
        ideCommandMock.mockReturnValue(null);
        const loader = new BuiltinCommandLoader(mockConfig);
        const commands = await loader.loadCommands(new AbortController().signal);
        // The 'ide' command should be filtered out.
        const ideCmd = commands.find((c) => c.name === 'ide');
        expect(ideCmd).toBeUndefined();
        // Other commands should still be present.
        const aboutCmd = commands.find((c) => c.name === 'about');
        expect(aboutCmd).toBeDefined();
    });
    it('should handle a null config gracefully when calling factories', async () => {
        const loader = new BuiltinCommandLoader(null);
        await loader.loadCommands(new AbortController().signal);
        expect(ideCommandMock).toHaveBeenCalledTimes(1);
        expect(ideCommandMock).toHaveBeenCalledWith(null);
        expect(restoreCommandMock).toHaveBeenCalledTimes(1);
        expect(restoreCommandMock).toHaveBeenCalledWith(null);
    });
    it('should return a list of all loaded commands', async () => {
        const loader = new BuiltinCommandLoader(mockConfig);
        const commands = await loader.loadCommands(new AbortController().signal);
        const aboutCmd = commands.find((c) => c.name === 'about');
        expect(aboutCmd).toBeDefined();
        expect(aboutCmd?.kind).toBe(CommandKind.BUILT_IN);
        const ideCmd = commands.find((c) => c.name === 'ide');
        expect(ideCmd).toBeDefined();
        const mcpCmd = commands.find((c) => c.name === 'mcp');
        expect(mcpCmd).toBeDefined();
    });
});
//# sourceMappingURL=BuiltinCommandLoader.test.js.map