"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Window = void 0;
const Base_1 = require("./Base");
const types_1 = require("./types");
const createChainableApi_1 = require("./utils/createChainableApi");
const Tabpage_1 = require("./Tabpage");
const Buffer_1 = require("./Buffer");
class Window extends Base_1.BaseApi {
    constructor() {
        super(...arguments);
        this.prefix = types_1.Metadata[types_1.ExtType.Window].prefix;
    }
    /**
     * The windowid that not change within a Vim session
     */
    get id() {
        return this.data;
    }
    /** Get current buffer of window */
    get buffer() {
        return createChainableApi_1.createChainableApi.call(this, 'Buffer', Buffer_1.Buffer, () => this.request(`${this.prefix}get_buf`, [this]));
    }
    /** Get the Tabpage that contains the window */
    get tabpage() {
        return createChainableApi_1.createChainableApi.call(this, 'Tabpage', Tabpage_1.Tabpage, () => this.request(`${this.prefix}get_tabpage`, [this]));
    }
    /** Get cursor position */
    get cursor() {
        return this.request(`${this.prefix}get_cursor`, [this]);
    }
    /** Set cursor position */
    set cursor(pos) {
        this.request(`${this.prefix}set_cursor`, [this, pos]);
    }
    /** Get window height by number of rows */
    get height() {
        return this.request(`${this.prefix}get_height`, [this]);
    }
    /** Set window height by number of rows */
    set height(height) {
        this.request(`${this.prefix}set_height`, [this, height]);
    }
    /** Get window width by number of columns */
    get width() {
        return this.request(`${this.prefix}get_width`, [this]);
    }
    /** Set window width by number of columns  */
    set width(width) {
        this.request(`${this.prefix}set_width`, [this, width]);
    }
    /** Get window position */
    get position() {
        return this.request(`${this.prefix}get_position`, [this]);
    }
    /** 0-indexed, on-screen window position(row) in display cells. */
    get row() {
        return this.request(`${this.prefix}get_position`, [this]).then(position => position[0]);
    }
    /** 0-indexed, on-screen window position(col) in display cells. */
    get col() {
        return this.request(`${this.prefix}get_position`, [this]).then(position => position[1]);
    }
    /** Is window valid */
    get valid() {
        return this.request(`${this.prefix}is_valid`, [this]);
    }
    /** Get window number */
    get number() {
        return this.request(`${this.prefix}get_number`, [this]);
    }
    /**
     * Closes window
     *
     * @param {Boolean} force Force close window
     */
    close(force = false) {
        return this.request(`${this.prefix}close`, [this, force]);
    }
    /**
     * Configure window position. Currently this is only used to
     * configure floating and external windows (including changing a
     * split window to these types).
     *
     * See documentation at |nvim_open_win()|, for the meaning of
     * parameters. Pass in -1 for 'witdh' and 'height' to keep
     * exiting size.
     *
     * When reconfiguring a floating window, absent option keys will
     * not be changed. The following restriction apply: `row`, `col`
     * and `relative` must be reconfigured together. Only changing a
     * subset of these is an error.
     *
     * @param {Window}  window Window handle
     * @Param {Object}  options Options object
     */
    config(options = {}) {
        return this.request(`${this.prefix}set_config`, [this, options]);
    }
}
exports.Window = Window;
