%feature("docstring") OT::LinearModelAlgorithm
"Class used to create a linear model from numerical samples.

Parameters
----------
XSample : 2-d sequence of float
    The input samples of a model.

YSample : 2-d sequence of float
    The output samples of a model, must be of dimension 1.

basis : :class:`~openturns.Basis`
    Optional.
    The :math:`\\phi` basis .

See Also
--------
LinearModelResult

Notes
-----
This class is used in order to create a linear model from data samples. The
linear regression model between the scalar variable :math:`Y` and the :math:`n`
-dimensional vector :math:`\\vect{X} = (X_i)_{i \\leq n}` writes as follows:

.. math::

    \\tilde{Y} = \\sum_{i=0}^p a_i \\phi_i(X) + \\epsilon

where :math:`\\epsilon` is the residual, supposed to follow the standard Normal
distribution, :math:`\\phi` a functional basis.
The algorithm class enables to estimate the coefficients of the linear expansion.

If basis is not specified, the underlying model is :

.. math::

    \\tilde{Y} = a_0 + \\sum_{i=1}^n a_i X_i + \\epsilon

The coefficients :math:`a_i` are evaluated using a least squares method.
Default method is `QR`. User might also choose `SVD` or `Cholesky` (useful
if basis is orthogonal) and large dataset.

The evaluation of the coefficients is completed by some useful parameters that could
help the diagnostic of the linearity.

Examples
--------
>>> import openturns as ot
>>> distribution = ot.Normal()
>>> func = ot.SymbolicFunction(['x1','x2', 'x3'], ['x1 + x2 + sin(x2 * 2 * pi_)/5 + 1e-3 * x3^2'])
>>> dimension = 3
>>> distribution = ot.ComposedDistribution([ot.Normal()]*dimension)
>>> input_sample = distribution.getSample(20)
>>> output_sample = func(input_sample)
>>> algo = ot.LinearModelAlgorithm(input_sample, output_sample)
>>> algo.run()
>>> result = ot.LinearModelResult(algo.getResult())"

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAlgorithm::getInputSample
"Accessor to the input sample.

Returns
-------
inputSample : :class:`~openturns.Sample`
    The Xsample which had been passed to the constructor."

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAlgorithm::getBasis
"Accessor to the input basis.

Returns
-------
basis : :class:`~openturns.Basis`
    The basis which had been passed to the constructor."

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAlgorithm::getOutputSample
"Accessor to the output sample.

Returns
-------
outputSample : :class:`~openturns.Sample`
    The Ysample which had been passed to the constructor."

// ---------------------------------------------------------------------

%feature("docstring") OT::LinearModelAlgorithm::getResult
"Accessor to the computed linear model.

Returns
-------
result : :class:`~openturns.LinearModelResult`
    The linear model built from numerical samples, along with other useful information."

