%feature("docstring") OT::FisherSnedecorFactory
"Fisher-Snedecor factory.

Available constructor:
    FisherSnedecorFactory()

Notes
-----
Several estimators to build a FisherSnedecor distribution from a scalar sample
are proposed.

**Maximum likelihood estimator:**

The parameters are estimated by numerical maximum likelihood estimation. 
The starting point of the optimization algorithm is based on the moment based 
estimator. 

The optimization sets lower bounds for the :math:`d_1` and :math:`d_2` parameters 
in order to ensure that :math:`d_1>0` and :math:`d_2>0`. 
The default values for these lower bounds are from the :class:`~openturns.ResourceMap` 
keys `FisherSnedecorFactory-D1LowerBound` and `FisherSnedecorFactory-D2LowerBound`. 

**Moment based estimator:**

Lets denote:

- :math:`\\displaystyle \\overline{x}_n = \\frac{1}{n} \\sum_{i=1}^n x_i` the empirical
  mean of the sample, 
- :math:`\\displaystyle s_n^2 = \\frac{1}{n-1} \\sum_{i=1}^n (x_i - \\overline{x}_n)^2`
  its empirical variance,

We first compute :math:`d_2`:

.. math::

    d_2 = \\frac{2 \\overline{x}_n}{\\overline{x}_n-1}

if :math:`\\overline{x}_n>1` (otherwise, the moment based estimator fails). 

Then we compute :math:`d_1`:

.. math::

    d_1 = \\frac{2 d_2^2 (d_2-2)}{(d_2-2)^2 (d_2-4)s_n^2 - 2d_2^2}

if :math:`s_n^2>0` (otherwise, the moment based estimator fails). 


See also
--------
DistributionFactory, FisherSnedecor

Examples
--------

In the following example, the parameters of a :class:`~openturns.FisherSnedecor` 
are estimated from a sample. 

>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> size = 10000
>>> distribution = ot.FisherSnedecor(4.5, 8.4)
>>> sample = distribution.getSample(size)
>>> factory = ot.FisherSnedecorFactory()
>>> estimated = factory.build(sample)
>>> estimated = factory.buildMethodOfMoments(sample)
>>> estimated = factory.buildMethodOfLikelihoodMaximization(sample)"

// ----------------------------------------------------------------------------

%feature("docstring") OT::FisherSnedecorFactory::buildMethodOfLikelihoodMaximization
"Method of likelihood maximization.

Refer to :class:`~openturns.MaximumLikelihoodFactory`.

Parameters
----------
sample : :class:`~openturns.Sample`
    Data

Returns
-------
distribution : :class:`~openturns.FisherSnedecor`
    The estimated distribution"

// ----------------------------------------------------------------------------

%feature("docstring") OT::FisherSnedecorFactory::buildMethodOfMoments
"Method of moments estimator.


Parameters
----------
sample : :class:`~openturns.Sample`
    Data

Returns
-------
distribution : :class:`~openturns.FisherSnedecor`
    The estimated distribution"

// ----------------------------------------------------------------------------

%feature("docstring") OT::FisherSnedecorFactory::buildAsFisherSnedecor
"Estimate the distribution as native distribution.

Parameters
----------
sample : :class:`~openturns.Sample`
    Data

Returns
-------
distribution : :class:`~openturns.FisherSnedecor`
    The estimated distribution"

// ----------------------------------------------------------------------------

%feature("docstring") OT::FisherSnedecorFactory::build
"Estimate the distribution using the default strategy.

Parameters
----------
sample : :class:`~openturns.Sample`
    Data

Returns
-------
distribution : :class:`~openturns.Distribution`
    The estimated distribution

Notes
-----
The default strategy is using the maximum likelihood estimators."

// ---------------------------------------------------------------------

%define OT_DistributionFactory_buildAsFisherSnedecor_doc
"Estimate the distribution using the default strategy.

**Available usages**:

    buildAsFisherSnedecor(*sample*)

    buildAsFisherSnedecor(*param*)

Parameters
----------
sample : 2-d sequence of float
    Sample from which the distribution parameters are estimated.
param : Collection of :class:`~openturns.PointWithDescription`
    A vector of parameters of the distribution.

Returns
-------
dist : :class:`~openturns.FisherSnedecor`
    The built distribution.

Notes
-----
The default strategy is using the maximum likelihood estimators."
%enddef
%feature("docstring") OT::FisherSnedecorFactory::buildAsFisherSnedecor
OT_DistributionFactory_buildAsFisherSnedecor_doc

