/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 *  ORBit-C++: C++ bindings for ORBit.
 *
 *  Copyright (C) 2000 Andreas Kloeckner
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Author:	Andreas Kloeckner <ak@ixion.net>
 *
 *  Purpose:	IDL compiler type representation
 *
 *
 */

#include "IDLUnion.h"

#include "IDLTypedef.h"

IDLUnion::IDLUnion(const string                &id,
		   IDL_tree                     node,
		   const IDLUnionDiscriminator &discriminator,
		   IDLScope                    *parentscope) :
    IDLScope (id, node, parentscope),
    m_discriminator (discriminator)
{
}

const IDLUnionDiscriminator &
IDLUnion::get_discriminator () const
{
	return m_discriminator;
}

std::string
IDLUnion::get_discriminator_default_value () const
{
	std::set<std::string> explicit_values;
	for (const_iterator i = begin (); i != end (); ++i)
	{
		const IDLCaseStmt &case_stmt = static_cast<const IDLCaseStmt&> (**i);
		if (!case_stmt.isDefault ())
			explicit_values.insert (*(case_stmt.labelsBegin ()));
	}
	
	return m_discriminator.get_default_value (explicit_values);
}

bool
IDLUnion::is_fixed () const
{
	bool all_fixed = true;
	for (const_iterator i = begin (); all_fixed && i != end (); ++i)
	{
		const IDLCaseStmt &case_stmt = static_cast<const IDLCaseStmt&> (**i);
		const IDLMember   &member = case_stmt.get_member ();
		
		all_fixed = member.getType ()->is_fixed ();
	}
	
	return all_fixed;
}

void
IDLUnion::typedef_decl_write (ostream          &ostr,
			      Indent           &indent,
			      IDLCompilerState &state,
			      const IDLTypedef &target,
			      const IDLTypedef *active_typedef) const
{
#warning "WRITE ME"
}

string
IDLUnion::stub_decl_arg_get (const string     &cpp_id,
			     IDL_param_attr    direction,
			     const IDLTypedef *active_typedef) const
{
	const string cpp_typename = active_typedef ?
		active_typedef->get_cpp_typename () : get_cpp_typename ();
	string retval;
	
	switch (direction)
	{
	case IDL_PARAM_IN:
		retval = "const " + cpp_typename + " &" + cpp_id;
		break;
	case IDL_PARAM_INOUT:
		retval = cpp_typename + " &" + cpp_id;
		break;
	case IDL_PARAM_OUT:
		retval = cpp_typename + "_out " + cpp_id;
		break;
	}

	return retval;
}

void
IDLUnion::stub_impl_arg_pre (ostream          &ostr,
			     Indent           &indent,
			     const string     &cpp_id,
			     IDL_param_attr    direction,
			     const IDLTypedef *active_typedef) const
{
	const string c_type = active_typedef ?
		active_typedef->get_c_typename () : get_c_typename ();

	switch (direction)
	{
	case IDL_PARAM_IN:
	case IDL_PARAM_INOUT:
		ostr << indent << c_type << " *_c_" << cpp_id
		     << " = " << cpp_id << "._orbitcpp_pack ();" << endl;
		break;
	case IDL_PARAM_OUT:
		ostr << indent << c_type << " *_c_" << cpp_id << ";" << endl;
		break;
	}		
}
	
string
IDLUnion::stub_impl_arg_call (const string     &cpp_id,
			      IDL_param_attr    direction,
			      const IDLTypedef *active_typedef) const
{
	switch (direction)
	{
	case IDL_PARAM_IN:
	case IDL_PARAM_INOUT:
		return "_c_" + cpp_id;
		break;
	case IDL_PARAM_OUT:
		return "&_c_" + cpp_id;
		break;
	}
}
	
void
IDLUnion::stub_impl_arg_post (ostream          &ostr,
			      Indent           &indent,
			      const string     &cpp_id,
			      IDL_param_attr    direction,
			      const IDLTypedef *active_typedef) const
{
	const string cpp_type = active_typedef ?
		active_typedef->get_cpp_typename () : get_cpp_typename ();

	// Load back values
	switch (direction)
	{
	case IDL_PARAM_IN:
		// Do nothing
		break;
	case IDL_PARAM_INOUT:
		ostr << indent << cpp_id << "._orbitcpp_unpack "
		     << "(*_c_" << cpp_id << ");" << endl;
		break;
	case IDL_PARAM_OUT:
		ostr << indent << cpp_id << " = new " << cpp_type << ";" << endl;
		ostr << indent << cpp_id << "->_orbitcpp_unpack "
		     << "(*_c_" << cpp_id << ");" << endl;
	}
	
	ostr << indent << "CORBA_free (_c_" << cpp_id << ");" << endl;
}




string
IDLUnion::stub_decl_ret_get (const IDLTypedef *active_typedef) const
{
	string cpp_typename = active_typedef ?
		active_typedef->get_cpp_typename () : get_cpp_typename ();
	
	if (is_fixed ())
		return cpp_typename;
	else
		return cpp_typename + "*";
}
	
void
IDLUnion::stub_impl_ret_pre (ostream &ostr,
			     Indent  &indent,
			     const IDLTypedef *active_typedef) const
{
	// Do nothing
}

void
IDLUnion::stub_impl_ret_call (ostream          &ostr,
			      Indent           &indent,
			      const string     &c_call_expression,
			      const IDLTypedef *active_typedef) const
{
	string c_typename = active_typedef ?
		active_typedef->get_c_typename () : get_c_typename ();
	
	string ret_id = is_fixed () ? "_c_retval" : "*_c_retval";
	
	ostr << indent << c_typename << " " << ret_id << " = "
	     << c_call_expression << ";" << endl;
}

void
IDLUnion::stub_impl_ret_post (ostream          &ostr,
			      Indent           &indent,
			      const IDLTypedef *active_typedef) const
{
	string cpp_typename = active_typedef ?
		active_typedef->get_cpp_typename () : get_cpp_typename ();

	if (is_fixed ())
	{
		ostr << indent << cpp_typename << " _cpp_retval;" << endl;
		ostr << indent << "_cpp_retval._orbitcpp_unpack  (_c_retval);" << endl;
	} else {
		ostr << indent << cpp_typename << " *_cpp_retval = "
		     << "new " << cpp_typename << ";" << endl;
		ostr << indent << "_cpp_retval->_orbitcpp_unpack (*_c_retval);" << endl;
		ostr << indent << "CORBA_free (_c_retval);" << endl;
	}
		
	ostr << indent << "return _cpp_retval;" << endl;
}
	



string
IDLUnion::skel_decl_arg_get (const string     &c_id,
			     IDL_param_attr    direction,
			     const IDLTypedef *active_typedef) const
{
	string c_typename = active_typedef ?
		active_typedef->get_c_typename () : get_c_typename ();
	string retval;
	
	switch (direction)
	{
	case IDL_PARAM_IN:
		retval = "const " + c_typename + " *" + c_id;
		break;
	case IDL_PARAM_INOUT:
		retval = c_typename + " *" + c_id;
		break;
	case IDL_PARAM_OUT:
		if (is_fixed ())
			retval = c_typename + " *" + c_id;
		else
			retval = c_typename + " **" + c_id;
		break;
	}

	return retval;
}

void
IDLUnion::skel_impl_arg_pre (ostream          &ostr,
			     Indent           &indent,
			     const string     &c_id,
			     IDL_param_attr    direction,
			     const IDLTypedef *active_typedef) const
{
	const string cpp_type = active_typedef ?
		active_typedef->get_cpp_typename () : get_cpp_typename ();
	const string cpp_id = "_cpp_" + c_id;
	
	switch (direction)
	{
	case IDL_PARAM_IN:
	case IDL_PARAM_INOUT:
		ostr << indent << cpp_type << " " << cpp_id << " (*" << c_id << ")"
		     << ";" << endl;
		break;
	case IDL_PARAM_OUT:
		ostr << indent << cpp_type << "_var " << cpp_id
		     << ";" << endl;
		break;
	}
}
	
string
IDLUnion::skel_impl_arg_call (const string     &c_id,
			      IDL_param_attr    direction,
			      const IDLTypedef *active_typedef) const
{
	if (is_fixed ())
		return "*_cpp_" + c_id;
	
	if (direction == IDL_PARAM_OUT)
		return get_cpp_typename () + "_out (_cpp_" + c_id + ")";

	return "_cpp_" + c_id;
}
	
void
IDLUnion::skel_impl_arg_post (ostream          &ostr,
			      Indent           &indent,
			      const string     &c_id,
			      IDL_param_attr    direction,
			      const IDLTypedef *active_typedef) const
{
	const string cpp_id = "_cpp_" + c_id;
	
	switch (direction)
	{
	case IDL_PARAM_IN:
		// Do nothing
		break;
	case IDL_PARAM_INOUT:
		ostr << indent << cpp_id << "._orbitcpp_pack "
		     << "(*" << c_id << ");" << endl;
		break;
	case IDL_PARAM_OUT:
		ostr << indent << "*" << c_id << " = "
		     << cpp_id << "->_orbitcpp_pack ();" << endl;
		break;
	}
}




string
IDLUnion::skel_decl_ret_get (const IDLTypedef *active_typedef) const
{
	if (is_fixed ())
		return get_c_typename ();
	else
		return get_c_typename () + "*";
}

void
IDLUnion::skel_impl_ret_pre (ostream          &ostr,
			     Indent           &indent,
			     const IDLTypedef *active_typedef) const
{
	string cpp_type = active_typedef ?
		active_typedef->get_cpp_typename () : get_cpp_typename ();

	if (is_fixed ())
		ostr << indent << cpp_type << " _cpp_retval;" << endl;
	else
		ostr << indent << cpp_type << "_var _cpp_retval = 0;" << endl;
}

void
IDLUnion::skel_impl_ret_call (ostream          &ostr,
			      Indent           &indent,
			      const string     &cpp_call_expression,
			      const IDLTypedef *active_typedef) const
{
	ostr << indent << "_cpp_retval = " << cpp_call_expression
	     << ";" << endl;
}

void
IDLUnion::skel_impl_ret_post (ostream          &ostr,
			      Indent           &indent,
			      const IDLTypedef *active_typedef) const
{
	string c_typename = active_typedef ?
		active_typedef->get_c_typename () : get_c_typename ();
	
	if (is_fixed ())
	{
		ostr << indent << c_typename << " _c_retval;" << endl;
		ostr << indent << "_cpp_retval._orbitcpp_pack (_c_retval);" << endl;
		ostr << indent << "return _c_retval;" << endl;
	} else {
		ostr << indent << "return _cpp_retval->_orbitcpp_pack ();" << endl;
	}
}


string
IDLUnion::get_cpp_member_typename (const IDLTypedef *active_typedef) const
{
#warning "WRITE ME"
}

string
IDLUnion::get_c_member_typename (const IDLTypedef *active_typedef) const
{
#warning "WRITE ME"
}

string
IDLUnion::get_seq_typename (unsigned int      length,
			    const IDLTypedef *active_typedef) const
{
#warning "WRITE ME"
}

string
IDLUnion::member_decl_arg_get (const IDLTypedef *active_typedef) const
{
#warning "WRITE ME"
}

void
IDLUnion::member_impl_arg_copy (ostream          &ostr,
				Indent           &indent,
				const string     &cpp_id,
				const IDLTypedef *active_typedef) const
{
#warning "WRITE ME"
}

void
IDLUnion::member_init_cpp (ostream          &ostr,
			   Indent           &indent,
			   const string     &cpp_id,
			   const IDLTypedef *active_typedef) const
{
#warning "WRITE ME"
}

void
IDLUnion::member_init_c (ostream          &ostr,
			 Indent           &indent,
			 const string     &c_id,
			 const IDLTypedef *active_typedef) const
{
#warning "WRITE ME"
}

void
IDLUnion::member_pack_to_c (ostream          &ostr,
			    Indent           &indent,
			    const string     &cpp_id,
			    const string     &c_id,
			    const IDLTypedef *active_typedef) const
{
#warning "WRITE ME"
}

void
IDLUnion::member_unpack_from_c  (ostream          &ostr,
				 Indent           &indent,
				 const string     &cpp_id,
				 const string     &c_id,
				 const IDLTypedef *active_typedef) const
{
#warning "WRITE ME"
}
