package org.gnu.gtk;

/**
 * This is a widget which contains a list of strings. It is a wrapper class for
 * the more powerfull set of tree and list objects, described at {@link
 * TreeView}
 */
public class SimpleList extends Widget {

	private int size = 0;

	/** The view widget */
	private TreeView view;
	/** Data store */
	private ListStore store;
	/** Column for displaying the data */
	private TreeViewColumn column;
	/** renderer for the strings */
	private CellRendererText renderer;

	private DataBlockString dataBlock;

	/**
	 * Constructs a SimpleList object.
	 */
	public SimpleList() {
		noopInit();
	}

	/**
	 * Construct a SimpleList from a handle to a native resource.
	 */
	public SimpleList(int handle) {
		super(handle);
	}

	protected void noopInit() {
		if (-1 == handle) {
			dataBlock = new DataBlockString();
			store = new ListStore(new DataBlock[] { dataBlock });
			view = new TreeView(store);
			column = new TreeViewColumn();
			renderer = new CellRendererText();
			column.packStart(renderer, false);
			column.addAttributeMapping(renderer, CellRendererText.Attribute.TEXT, dataBlock);
			view.appendColumn(column);
			view.setHeadersVisible(false);
			this.size = 0;

			this.handle = view.getHandle();
		}
	}

	/**
	 * Adds an element to the start of the list
	 * @param text the text of the element
	 */
	public void addStart(String text) {
		TreeIter iter = store.prependRow();
		store.setValue(iter, dataBlock, text);
		size++;
	}

	/** 
	 * Adds an element to the end of the list
	 * @param text The text of the element
	 */
	public void addEnd(String text) {
		TreeIter iter = store.appendRow();
		store.setValue(iter, dataBlock, text);
		size++;
	}

	/**
	 * Adds a new element at the given position
	 * @param text The text of the new element
	 * @param position The position in which it should be placed. 0 is at the
	 * start of the list. 
	 */
	public void add(String text, int position) {
		TreeIter iter = store.insertRow(position);
		store.setValue(iter, dataBlock, text);
		size++;
	}

	/**
	 * Returns the string at the given position
	 * @param position The position in the list to investigate
	 * @return The value of the cell at that position
	 */
	public String getValue(int position) {
		return store.getValue(store.getIter("" + position), dataBlock);
	}

	/**
	 * Returns true if the element at <code>index</code> is selected
	 */
	public boolean getSelected(int index) {
		return view.getSelection().getSelected(store.getIter("" + index));
	}

	/**
	 * Selects the element at <code>index</code>
	 */
	public void select(int index) {
		view.getSelection().select(store.getIter("" + index));
	}
	/**
	 * Sets how many of the elements can be selected.
	 */
	public void setSelectable(SelectionMode setting) {
		view.getSelection().setMode(setting);
	}

	/**
	 * Removes the element at the given position
	 */
	public void remove(int position) {
		store.removeRow(store.getIter("" + position));
		size--;
	}

	/**
	 * Sets whether the user may edit the cell contents directly.
	 */
	public void setEditable(boolean setting) {
		renderer.setEditable(setting);
	}

	/**
	 * Returns the number of items in the list
	 */
	public int getSize() {
		return size;
	}

}
