/* wptKeysignDlg.cpp - Key signing dialog
 *	Copyright (C) 2001-2005 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <windows.h>
#include <commctrl.h>

#include "../resource.h"
#include "wptGPG.h"
#include "wptNLS.h"
#include "wptW32API.h"
#include "wptVersion.h"
#include "wptTypes.h"
#include "wptErrors.h"
#include "wptCommonCtl.h"
#include "wptContext.h"
#include "wptDlgs.h"
#include "wptUTF8.h"
#include "wptRegistry.h"

static int sig_class_choice = 0;

static const char *
get_printable_fpr (const char * fpr)
{
    static char pfpr[64];    
    int pos = 0;
    size_t i;

    for( i = 0; i < strlen (fpr); i += 4 ) {
	pfpr[pos++] = fpr[i];
	pfpr[pos++] = fpr[i+1];
	pfpr[pos++] = fpr[i+2];
	pfpr[pos++] = fpr[i+3];
	pfpr[pos++] = ' ';
    }
    return pfpr;
} /* get_printable_fpr */


static const char *
get_keyinfo (gpgme_key_t key)
{
    static char buf[64+16];
   
    _snprintf (buf, DIM (buf)-1-16, "%d-bit %s key, ID %s", 
	gpgme_key_get_ulong_attr (key, GPGME_ATTR_LEN, NULL, 0),
	gpgme_key_get_string_attr (key, GPGME_ATTR_ALGO, NULL, 0),
	gpgme_key_get_string_attr (key, GPGME_ATTR_KEYID, NULL, 0) + 8 );
    if (gpgme_key_get_ulong_attr (key, GPGME_ATTR_DIVERT_CARD, NULL, 0))
	strcat (buf, " (Card)");
    return buf;
} /* get_keyinfo */


static int
do_fill_seckeylist (HWND dlg, const char * keyid)
{
    gpgme_keycache_t sec;
    gpgme_key_t pk;
    const char * s;
    char * uid, * p;    
    int i = 0, n=0;

    sec = keycache_get_ctx (0);
    if (!sec)
	BUG (0);
    gpgme_keycache_rewind (sec);
    while (!gpgme_keycache_next_key (sec, 1, &pk)) 
    {
	if (!pk)
	    continue;
	s = gpgme_key_get_string_attr (pk, GPGME_ATTR_KEYID, NULL, 0);
	if (!strcmp (s, keyid))
	    continue;
	/* skip all ElGamal sign+encrypt keys */
	if( gpgme_key_get_ulong_attr( pk, GPGME_ATTR_ALGO, NULL, 0 )
	    == GPGME_PK_ELG_ES )
	    continue;
	/* make sure the public key is okay not: revoked, expired or disabled. */
	if( gpgme_key_get_ulong_attr (pk, GPGME_ATTR_EXPIRE, NULL, 0) 
	    || gpgme_key_get_ulong_attr (pk, GPGME_ATTR_KEY_REVOKED, NULL, 0)
	    || gpgme_key_get_ulong_attr (pk, GPGME_ATTR_KEY_DISABLED, NULL, 0))
	    continue;
	s = gpgme_key_get_string_attr (pk, GPGME_ATTR_NAME, NULL, 0);
	if( !s )
	    continue;
	uid = utf8_to_wincp (s, strlen (s));
	p = new char[strlen( uid ) + 64];
	if( !p )
	    BUG( NULL );	
	_snprintf (p, strlen (uid) + 63, "%s (%s)", uid, get_keyinfo (pk));
	SendDlgItemMessage (dlg, IDC_KEYSIGN_KEYLIST, CB_ADDSTRING, i, (LPARAM)(char *)p);
	SendDlgItemMessage (dlg, IDC_KEYSIGN_KEYLIST, CB_SETITEMDATA, i++, (LPARAM)(DWORD)pk);
	free_if_alloc (p);
	free (uid);
	n++;
    }
    SendDlgItemMessage (dlg, IDC_KEYSIGN_KEYLIST, CB_SETCURSEL, 0, 0);
    if (!n)
	return -1;
    return 0;
} /* do_fill_seckeylist */


static void
do_add_local_user (gpgme_ctx_t ctx, HWND dlg)
{
    int idx;
    const char * s;
    gpgme_key_t key;

    idx = SendDlgItemMessage( dlg, IDC_KEYSIGN_KEYLIST, CB_GETCURSEL, 0, 0 );
    key = (gpgme_key_t)SendDlgItemMessage( dlg, IDC_KEYSIGN_KEYLIST, CB_GETITEMDATA, (WPARAM)idx, 0 );
    if( key && (s = gpgme_key_get_string_attr( key, GPGME_ATTR_KEYID, NULL, 0 )) )
	gpgme_set_local_user( ctx, s );    
} /* do_add_local_user */


static void
do_check_protection (HWND dlg)
{
    int idx, protec;
    gpgme_key_t key;

    idx = SendDlgItemMessage( dlg, IDC_KEYSIGN_KEYLIST, CB_GETCURSEL, 0, 0 );
    key = (gpgme_key_t)SendDlgItemMessage( dlg, IDC_KEYSIGN_KEYLIST, CB_GETITEMDATA, (WPARAM)idx, 0 );
    if (key)
    {
	protec = gpgme_key_get_ulong_attr (key, GPGME_ATTR_IS_PROTECTED, NULL, 0);
	if (!protec)
	    protec = gpgme_key_get_ulong_attr (key, GPGME_ATTR_DIVERT_CARD, NULL, 0);
	EnableWindow (GetDlgItem (dlg, IDC_KEYSIGN_PASSPHRASE), protec? TRUE : FALSE);
    }
} /* do_check_protection */


BOOL CALLBACK
sig_class_dlg_proc (HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam)
{
    switch (msg) 
    {
    case WM_INITDIALOG:
	SetWindowText (dlg, _("Choose Signature Class"));
	SetDlgItemText (dlg, IDC_SIGCLASS_TITLEINF, _("How carefully have you verified the key you are about to sign actually belongs to the person? If you don't know what to anwser, use \"0\"."));
	SetDlgItemText (dlg, IDC_SIGCLASS_CLASS0, _("(0) I will not answer (default)"));
	SetDlgItemText (dlg, IDC_SIGCLASS_CLASS1, _("(1) I have not checked at all."));
	SetDlgItemText (dlg, IDC_SIGCLASS_CLASS2, _("(2) I have done causal checking."));
	SetDlgItemText (dlg, IDC_SIGCLASS_CLASS3, _("(3) I have done very careful checkings."));
	CheckDlgButton (dlg, IDC_SIGCLASS_CLASS0, BST_CHECKED);
	SetForegroundWindow (dlg);
	center_window (dlg);
	return TRUE;

    case WM_COMMAND:
	switch( LOWORD( wparam ) ) {
	case IDOK:
	    if( IsDlgButtonChecked( dlg, IDC_SIGCLASS_CLASS0 ) )
		sig_class_choice = 0;
	    else if( IsDlgButtonChecked( dlg, IDC_SIGCLASS_CLASS1 ) )
		sig_class_choice = 1;
	    else if( IsDlgButtonChecked( dlg, IDC_SIGCLASS_CLASS2 ) )
		sig_class_choice = 2;
	    else if( IsDlgButtonChecked( dlg, IDC_SIGCLASS_CLASS3 ) )
		sig_class_choice = 3;
	    else
		sig_class_choice = 0;
	    EndDialog( dlg, TRUE );
	    break;
	}
	break;
    }

    return FALSE;
} /* sig_class_dlg_proc */


BOOL CALLBACK
keysign_dlg_proc (HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam)
{
    static gpgme_key_t key;
    static int enable=0;
    gpgme_editkey_t ke;
    gpgme_ctx_t ctx;
    gpgme_error_t err;
    SYSTEMTIME st;
    HWND h;
    char keymsg[4096], pwd[256], *uid = NULL;
    const char *keyid, *s;
    u32 created;
    int type, expires=0;

    
    switch ( msg )  {
    case WM_INITDIALOG:
	enable = 0;
        if( lparam == NULL )
            dlg_fatal_error( dlg, "Could not get dialog param." );
	#ifndef LANG_DE
        SetWindowText( dlg, _("Key Signing") );
	#endif
        key = (gpgme_key_t) lparam;
        created = gpgme_key_get_ulong_attr(key, GPGME_ATTR_CREATED, NULL, 0);
        s = gpgme_key_get_string_attr( key, GPGME_ATTR_USERID, NULL, 0 );
        if( s )
	    uid = utf8_to_wincp (s, strlen (s));
        _snprintf( keymsg, sizeof keymsg -1,
                   _("pub %d/%s created: %s\n\n"
                     "Primary key fingerprint: %s\n\n"
                     "\t%s\n\n"
                     "\nAre you really sure that you want to sign this key with YOUR key?\n"),
                   gpgme_key_get_ulong_attr( key, GPGME_ATTR_LEN, NULL, 0 ),
                   gpgme_key_get_string_attr( key, GPGME_ATTR_KEYID, NULL, 0 )+8,
                   gpgme_key_expand_attr( GPGME_ATTR_CREATED, created ),
                   get_printable_fpr( gpgme_key_get_string_attr( key, GPGME_ATTR_FPR, NULL, 0 ) ),
                   uid );
        free (uid);
	s = gpgme_key_get_string_attr (key, GPGME_ATTR_KEYID, NULL, 0);
	if (do_fill_seckeylist (dlg, s)) {
	    msg_box( dlg, _("No valid secret key found."), _("Key Signing"), MB_ERR );
	    EndDialog( dlg, FALSE );
	}
        SetDlgItemText( dlg, IDC_KEYSIGN_INFOS, keymsg );
	#ifndef LANG_DE
        SetDlgItemText( dlg, IDC_KEYSIGN_LOCAL, _("Sign local only (non exportable signature)") );
	SetDlgItemText (dlg, IDC_KEYSIGN_EXPSIG, _("Signature expires on"));
	SetDlgItemText (dlg, IDC_KEYSIGN_NREV, _("Sign non-revocably"));
	#endif
        CheckDlgButton( dlg, IDC_KEYSIGN_LOCAL, BST_CHECKED );
	CheckDlgButton (dlg, IDC_KEYSIGN_EXPSIG, BST_UNCHECKED);
	EnableWindow (GetDlgItem (dlg, IDC_KEYSIGN_EXPIRES), FALSE);
	if (reg_prefs.expert == 0)
	    ShowWindow (GetDlgItem (dlg, IDC_KEYSIGN_NREV), SW_HIDE);
        SetForegroundWindow( dlg );
	h = GetDlgItem( dlg, IDC_KEYSIGN_PASSPHRASE );
	SetFocus( h );
        return FALSE;
        
    case WM_SYSCOMMAND:
        if( LOWORD( wparam ) == SC_CLOSE ) {
            SetDlgItemText( dlg, IDC_KEYSIGN_PASSPHRASE, "" );
            EndDialog( dlg, TRUE );
        }
        return FALSE;
        
    case WM_COMMAND:
	if( HIWORD( wparam ) == CBN_SELCHANGE ) {
	    do_check_protection( dlg );
	    break;
	}
	if (HIWORD (wparam) == BN_CLICKED && LOWORD (wparam) == IDC_KEYSIGN_EXPSIG) {
	    enable ^= 1;
	    EnableWindow (GetDlgItem (dlg, IDC_KEYSIGN_EXPIRES), enable? TRUE : FALSE);
	}

        switch( LOWORD( wparam ) ) {
        case IDOK:
	    dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_SIGCLASS, dlg, 
				sig_class_dlg_proc, NULL,
				_("Choose Signature Class"),  IDS_WINPT_SIGCLASS );
            type = IsDlgButtonChecked (dlg, IDC_KEYSIGN_LOCAL);
	    if (type)
		type = GPGME_EDITKEY_LSIGN;
	    else
		type = GPGME_EDITKEY_SIGN;

	    if (reg_prefs.expert && IsDlgButtonChecked (dlg, IDC_KEYSIGN_NREV))
	    {
		type = GPGME_EDITKEY_NRSIGN;
		if (type == GPGME_EDITKEY_LSIGN)
		    type = GPGME_EDITKEY_NRLSIGN;
	    }
	    if (IsDlgButtonChecked (dlg, IDC_KEYSIGN_EXPSIG)) {
		expires = 1;
		DateTime_GetSystemtime (GetDlgItem (dlg, IDC_KEYSIGN_EXPIRES), &st);
		sprintf (keymsg, "%04d-%02d-%02d", st.wYear, st.wMonth, st.wDay);
	    }

            GetDlgItemText( dlg, IDC_KEYSIGN_PASSPHRASE, pwd, sizeof pwd - 1 );
            keyid = gpgme_key_get_string_attr( key, GPGME_ATTR_KEYID, NULL, 0 );
	    if( !keyid ) {
		msg_box( dlg, _("Could not get Key ID from key."), _("Key Signing"), MB_ERR );
		return FALSE;
	    }
            err = gpgme_new( &ctx );
	    if (err)
		BUG (0);
	    do_add_local_user (ctx, dlg);

            err = gpgme_editkey_new (&ke);
	    if (err)
		BUG (0);
	    err = gpgme_editkey_sign_set (ke, pwd, sig_class_choice, type, 
					  expires? keymsg : NULL);
	    if (err) {
		msg_box (dlg, gpgme_strerror (err), _("Key Signing"), MB_ERR);
		gpgme_release (ctx);
		gpgme_editkey_release (ke);
		return FALSE;
	    }
		
	    gpgme_set_edit_ctx (ctx, ke, type);
            
            err = gpgme_op_editkey( ctx, keyid );
            memset( &pwd, 0, sizeof pwd );
	    if (err == GPGME_Conflict)
		Sleep (500); /* it seems the process is not really finished */
	    gpgme_release( ctx );
            gpgme_editkey_release( ke );
            if( err ) {
                if( err == GPGME_Conflict )
                    msg_box( dlg, _("Key is already signed by your key."), _("Key Signing"), MB_INFO );
		else if( err == GPGME_Invalid_Mode )
		    msg_box( dlg, _("Unusable secret key."), _("Key Signing"), MB_ERR );
                else
                    msg_box( dlg, gpgme_strerror( err ), _("Key Signing"), MB_ERR );
            }
	    else {
		status_box (dlg, _("Key successfully signed."), PGM_NAME);
		keycache_set_reload (1);
		EndDialog (dlg, TRUE);
	    }
            return TRUE;
            
        case IDCANCEL:
            EndDialog( dlg, FALSE );
            return FALSE;
        }
        break;
    }
    
    return FALSE;
} /* keysign_dlg_proc */
