/* wptClipVerifyDlg.cpp - WinPT verify dialog
 *	Copyright (C) 2001-2005 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <windows.h>

#include "../resource.h"
#include "wptGPG.h"
#include "wptTypes.h"
#include "wptCommonCtl.h"
#include "wptKeylist.h"
#include "wptW32API.h"
#include "wptKeyserver.h"
#include "wptNLS.h"
#include "wptContext.h" /* for passwd_s */
#include "wptDlgs.h"
#include "wptErrors.h"
#include "wptVersion.h"

BOOL CALLBACK
clip_verify_dlg_proc (HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam)
{
    static listview_ctrl_t lv = NULL;
    static text_input_s *ctx = NULL;
    const char *ndat, *nurl;
    gpgme_error_t err;
    gpgme_sig_t sig = NULL;
    gpgme_keycache_t kc = NULL;
    char keyid[16+1];
    u16 port = HKP_PORT;
    int rc = 0, tt, algo;
    size_t i;
    
    switch( msg ) {
    case WM_INITDIALOG:
#ifndef LANG_DE
        SetWindowText( dlg, _("Verify") );
#endif
        kc = keycache_get_ctx( KEYCACHE_PUB );
        if( !kc )
	    BUG( NULL );
        ctx = (text_input_s *)lparam;
        if( ctx && ctx->length )
	    err = gpgme_op_clip_verify_detached( kc, &sig, ctx->data, ctx->length );
        else
	    err = gpgme_op_clip_verify (kc, &sig);
        if (err) {
	    if (err == GPGME_Internal_GPG_Problem)
		gnupg_display_error ();
	    else
		msg_box( dlg, gpgme_strerror( err ), _("Verify"), MB_ERR );
            gpgme_sig_release( sig );
            EndDialog( dlg, FALSE );
            return FALSE;
        }        

        tt = gpgme_sig_get_ulong_attr( sig, 0, GPGME_ATTR_VALIDITY );
        if( tt == GPGME_SIG_STAT_NOKEY ) {
	    const char * kserv;
	    u32 kid = gpgme_sig_get_ulong_attr( sig, 0, GPGME_ATTR_KEYID );
	    if( kid == 0 )
		kid = 0xDEADBEEF;
	    tt = gpgme_sig_get_ulong_attr( sig, 0, GPGME_ATTR_CREATED );
	    algo = gpgme_sig_get_ulong_attr( sig, 0, GPGME_ATTR_ALGO );
            _snprintf( keyid, sizeof keyid - 1, "%X", kid );
	    rc = log_box( _("Verify"), MB_INFO|MB_YESNO,
			  _("Signature made %s using %s key ID 0x%s\n"
			  "Cannot check signature: public key not found\n\n"
			  "Do you want to try to retrieve the key from the keyserver?"),
			  strtimestamp( tt ), 
			  gpgme_key_expand_attr( GPGME_ATTR_ALGO, algo ), 
			  keyid );
            if( rc == IDNO ) {
                msg_box( dlg, gpg_sigstat[GPGME_SIG_STAT_NOKEY], _("Verify"), MB_ERR );
                gpgme_sig_release( sig );
                EndDialog( dlg, FALSE );
                return FALSE;
            }
	    kserv = gpgme_sig_get_string_attr (sig, GPGME_ATTR_KEYSERVER);
	    if (kserv && strncmp (kserv, "hkp://", 6)) {
		rc = log_box (_("Verify"), MB_INFO|MB_YESNO,
			      _("The users preferred keyserver is '%s'.\n"
			        "Do you want to use it to fetch the key?"), kserv);
		if (rc == IDNO) {
		    kserv = default_keyserver;
		    port = default_keyserver_port;
		}
	    }
	    else {
		kserv = default_keyserver;
		port = default_keyserver_port;
	    }
            if (!hkp_recv_key (dlg, kserv, port, keyid, 0, 0)) {
		keycache_reload (dlg);
		kc = keycache_get_ctx (KEYCACHE_PUB);
		if (!kc)
		    BUG (dlg);
		gpgme_sig_set_key_ctx (sig, kc);
	    }
        }
        else if( tt == GPGME_SIG_STAT_BAD &&
                 !gpgme_sig_get_ulong_attr( sig, 0, GPGME_ATTR_CREATED ) )
		 ;
        else if( !gpgme_sig_get_ulong_attr( sig, 0, GPGME_ATTR_TYPE )
                 || !gpgme_sig_get_ulong_attr( sig, 0, GPGME_ATTR_CREATED )
                 || !gpgme_sig_get_ulong_attr( sig, 0, GPGME_ATTR_VALIDITY ) ) {
            msg_box( dlg, _("Invalid signature state."), _("Verify"), MB_ERR );
            gpgme_sig_release( sig );
            EndDialog( dlg, FALSE );
            return FALSE;
        }
        verlist_build( &lv, GetDlgItem( dlg, IDC_VERIFY_SIGLIST ), 0 );

	for( i = 0; i < gpgme_sig_get_ulong_attr( sig, 0, GPGME_ATTR_LEVEL ); i++ ) {
	    gpgme_sig_t _sig;
	    _sig = (gpgme_sig_t)gpgme_sig_get_ulong_attr( sig, i, GPGME_ATTR_OPAQUE );
	    rc = verlist_add_sig( lv, _sig );
	    if( rc )
		msg_box( dlg, _("Could not extract key or signature information."), 
			_("Verify"), MB_ERR );
	}
	if( gpgme_sig_get_ulong_attr( sig, 0, GPGME_ATTR_EXPIRE ) )
	    SetDlgItemText( dlg, IDC_VERIFY_INFO, _("The signature is expired!") );
	ndat = gpgme_sig_get_string_attr( sig, GPGME_ATTR_NOTATION );
	nurl = gpgme_sig_get_string_attr( sig, GPGME_ATTR_POLICY_URL );
	if( ndat || nurl ) {
	    size_t n;
	    char * p;
	    n = ndat? strlen( ndat ) + 1 : 0;
	    n += nurl? strlen( nurl ) + 1 : 0;
	    p = new char[n+40];
	    if( !p )
		BUG( NULL );
	    sprintf( p, "%s%s%s%s%s",
		ndat? _("Notation Data: ") : "", ndat? ndat : "", ndat? "\n" : "",
		nurl? _("Policy URL: ") : "", nurl? nurl : "" );
	    msg_box( dlg, p, _("Signature Information"), MB_INFO );
	    free_if_alloc( p );
	}
        gpgme_sig_release( sig );
        SetForegroundWindow( dlg );
        set_active_window( dlg );
        return TRUE;
        
    case WM_DESTROY:
        reset_active_window();
        if( lv ) {
            listview_release( lv );
            lv = NULL;
        }
        return FALSE;
        
    case WM_SYSCOMMAND:
        if( LOWORD( wparam ) == SC_CLOSE )
            EndDialog( dlg, TRUE );
        return FALSE;
        
    case WM_COMMAND:
        switch( LOWORD( wparam ) ) {
        case IDOK:
            EndDialog( dlg, TRUE );
            return FALSE;
            
        case IDC_VERIFY_SAVE:
            dialog_box_param( glob_hinst, (LPCTSTR)IDD_WINPT_FILE_SAVE, dlg, 
                              file_save_dlg_proc, NULL, _("Save Plaintext"), 
			      IDS_WINPT_FILE_SAVE );
            break;
        }
        break;
    }
    
    return FALSE;
} /* clip_verify_dlg_proc */
