/* import.c -  import functions
 *      Copyright (C) 2000 Werner Koch (dd9jn)
 *      Copyright (C) 2001-2005 Timo Schulz
 *
 * This file is part of MyGPGME.
 *
 * MyGPGME is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MyGPGME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <ctype.h>

#include "util.h"
#include "context.h"
#include "ops.h"
#include "key.h"


struct import_result_s {
    gpgme_recipients_t keys;
    gpgme_recipients_t fprlist;
    int import_res[14];
    int imported;
};


void
_gpgme_release_import_result (_import_result_t res)
{
    if (res) {
        gpgme_recipients_release (res->keys);
        gpgme_recipients_release (res->fprlist);
        res->keys = NULL;
        safe_free (res);
    }
} /* _gpgme_release_import_result */


gpgme_error_t
gpgme_get_import_status (gpgme_ctx_t ctx, int *import_res,
                         gpgme_recipients_t *r_keys)
{
    gpgme_recipients_t k;
    struct user_id_s *s;
    int i;

    if (!ctx)
        return mk_error (Invalid_Value);

    assert (ctx->result_type == RESULT_TYPE_IMPORT);
    for (i = 0; i < 14; i++)
        import_res[i] = ctx->result.import->import_res[i];
    if (r_keys) {
        gpgme_recipients_t src;

        if (ctx->result.import->keys)
            src = ctx->result.import->keys;
        else if (ctx->result.import->fprlist)
            src = ctx->result.import->fprlist;
        else
            return 0;

        if (gpgme_recipients_new (&k))
            return mk_error (Out_Of_Core);
        for (s=src->list; s; s=s->next)
            gpgme_recipients_add_name (k, s->name);
        *r_keys = k;
    }

    return 0;
} /* gpgme_get_import_status */



static const char *
import_command_handler (void *opaque, gpg_status_code_t code, const char * key)
{
    gpgme_ctx_t ctx = opaque;
    struct user_id_s * r;

    if (code != STATUS_GET_BOOL)
        return NULL;

    /* We can use the interactive mode to get a summary of the imported
       keys. We always say YES in this case because the IMPORT_CHECK
       contains the user-id and this might be useful */
    if (!ctx->enc_to)
        return "Y";

    for (r = ctx->enc_to->list; r; r = r->next) {
        if (!strncmp (r->name, ctx->tmp_keyid+8, 8)) {
            DEBUG1 ("interactive import `%s: ok\n", ctx->tmp_keyid+8);
            return "Y";
        }
    }
    return "N";
} /* import_command_handler */


static void
import_status_handler (gpgme_ctx_t ctx, gpg_status_code_t code, char * args)
{
    const char * s;

    if (ctx->out_of_core)
        return;

    if( ctx->result_type == RESULT_TYPE_NONE ) {
        assert ( !ctx->result.import );
        ctx->result.import = calloc( 1, sizeof *ctx->result.import );
        if( !ctx->result.import ) {
            ctx->out_of_core = 1;
            return;
        }
        ctx->result.import->imported = 0;
        ctx->result_type = RESULT_TYPE_IMPORT;
    }

    switch (code) {
    case STATUS_NODATA:
        ctx->result.import->imported = -1;
        break;

    case STATUS_IMPORT_CHECK:
        if (!ctx->result.import->keys) {
            if (gpgme_recipients_new (&ctx->result.import->keys))
                ctx->out_of_core = 1;
        }
        s = args;
        while (s && *s != ' ')
            s++;
        s++;
        while (s && *s != ' ')
            s++;
        s++;
        gpgme_recipients_add_name (ctx->result.import->keys, s);
        strncpy (ctx->tmp_keyid, args, 16);
        break;

    case STATUS_IMPORTED:
        ctx->result.import->imported++;
        break;

    case STATUS_IMPORT_OK:
        if (!ctx->result.import->fprlist) {
            if (gpgme_recipients_new (&ctx->result.import->fprlist))
                ctx->out_of_core = 1;
        }
        s = args;
        while (s && *s != ' ')
            s++;
        s++;
        gpgme_recipients_add_name (ctx->result.import->fprlist, s);
        break;

    case STATUS_IMPORT_RES:
        /* IMPORT_RES <count> <no_user_id> <imported> <imported_rsa>
                      <unchanged> <n_uids> <n_subk> <n_sigs> <n_revoc>
                      <sec_read> <sec_imported> <sec_dups> <not_imported> */
        ctx->result.import->imported++;
        sscanf (args, "%d %d %d %d %d %d %d %d %d %d %d %d %d %d",
                &ctx->result.import->import_res[ 0],
                &ctx->result.import->import_res[ 1],
                &ctx->result.import->import_res[ 2],
                &ctx->result.import->import_res[ 3],
                &ctx->result.import->import_res[ 4],
                &ctx->result.import->import_res[ 5],
                &ctx->result.import->import_res[ 6],
                &ctx->result.import->import_res[ 7],
                &ctx->result.import->import_res[ 8],
                &ctx->result.import->import_res[ 9],
                &ctx->result.import->import_res[10],
                &ctx->result.import->import_res[11],
                &ctx->result.import->import_res[12],
                &ctx->result.import->import_res[13]);
        /* we add the RSA keys to have one sum. we don't need to differ
           between RSA and other keys after the patent has been expired. */
        ctx->result.import->import_res[2] += ctx->result.import->import_res[3];
        break;
    }
} /* import_status_handler */


static gpgme_error_t
import_start( gpgme_ctx_t ctx, gpgme_data_t keydata )
{
    gpgme_error_t rc = 0;

    fail_on_pending_request( ctx );
    ctx->pending = 1;

    /* create a process object */
    _gpgme_gpg_release( &ctx->gpg );
    rc = _gpgme_gpg_new( &ctx->gpg );
    if( rc )
        goto leave;

    if (ctx->use_interactive)
        _gpgme_gpg_set_command_handler( ctx->gpg, import_command_handler, ctx );
    if( ctx->use_logging )
    _gpgme_gpg_set_logging_handler( ctx->gpg, ctx );
    _gpgme_gpg_set_status_handler( ctx->gpg, import_status_handler, ctx );

    /* build the commandline */
    if (ctx->force_opt) /* XXX: remove this */
        _gpgme_gpg_add_arg (ctx->gpg, "--allow-non-selfsigned-uid");
    _gpgme_gpg_add_arg (ctx->gpg, "--allow-secret-key-import");
    if (ctx->use_interactive)
        _gpgme_gpg_add_arg (ctx->gpg, "--interactive");
    _gpgme_gpg_add_arg (ctx->gpg, "--import");

    /* Check the supplied data */
    if( gpgme_data_get_type( keydata ) == GPGME_DATA_TYPE_NONE ) {
        rc = mk_error( No_Data );
        goto leave;
    }
    _gpgme_data_set_mode( keydata, GPGME_DATA_MODE_OUT );
    _gpgme_gpg_add_data( ctx->gpg, keydata, 0 );
    rc = _gpgme_gpg_spawn( ctx->gpg, ctx );

leave:
    if( rc ) {
        ctx->pending = 0;
        _gpgme_gpg_release ( &ctx->gpg );
    }
    return rc;
} /* import_start */


static gpgme_error_t
file_import_start( gpgme_ctx_t ctx, const char *input )
{
    gpgme_error_t rc;

    if( !ctx )
        return mk_error( Invalid_Value );

    fail_on_pending_request( ctx );
    ctx->pending = 1;

    _gpgme_gpg_release( &ctx->gpg );
    rc = _gpgme_gpg_new( &ctx->gpg );
    if( rc )
        return rc;

    if( ctx->use_interactive )
        _gpgme_gpg_set_command_handler( ctx->gpg, import_command_handler, ctx );
    _gpgme_gpg_set_status_handler( ctx->gpg, import_status_handler, ctx );
    if( ctx->use_interactive )
        _gpgme_gpg_add_arg( ctx->gpg, "--interactive" );
    _gpgme_gpg_add_arg( ctx->gpg, "--yes" );
    _gpgme_gpg_add_arg( ctx->gpg, "--allow-non-selfsigned-uid" );
    _gpgme_gpg_add_arg( ctx->gpg, "--allow-secret-key-import" );
    _gpgme_gpg_add_arg( ctx->gpg, "--import" );
    _gpgme_gpg_add_arg( ctx->gpg, input );

    rc = _gpgme_gpg_spawn( ctx->gpg, ctx );
    if( rc ) {
        ctx->pending = 0;
        _gpgme_gpg_release( &ctx->gpg );
    }

    return rc;
} /* file_import_start */


static gpgme_error_t
get_import_result( gpgme_ctx_t ctx )
{
    gpgme_error_t err = 0;

    if( ctx->result_type != RESULT_TYPE_IMPORT )
        err = mk_error( General_Error );
    else if( ctx->out_of_core )
        err = mk_error( Out_Of_Core );
    else {
        assert( ctx->result.import );
        if( ctx->result.import->imported == -1 )
            err = mk_error( No_Data );
        else if( !ctx->result.import->imported )
            err = mk_error( General_Error );
        else if( gpgme_get_process_rc( ctx ) == 2
                 && ctx->result.import->import_res[GPGME_IMPSTAT_NSKEYS] ) {
            /* GPG isssues a warning when the secret key is already
               in the secret keyring. Due to the fact GPG can't merge
               secret keys, the return code is incremented but this
               is in fact no real error. */
            err = 0;
        }
        else if( gpgme_get_process_rc( ctx ) )
            err = mk_error( Internal_GPG_Problem );
    }

    return err;
} /* get_import_result */


gpgme_error_t
gpgme_op_import( gpgme_ctx_t ctx, gpgme_recipients_t selkeys, gpgme_data_t keydata )
{
    gpgme_error_t err;

    ctx->enc_to = selkeys;
    err = import_start( ctx, keydata );
    if( !err ) {
        gpgme_wait( ctx, 1 );
        ctx->pending = 0;
        err = get_import_result( ctx );
    }

    return err;
} /* gpgme_op_import */


gpgme_error_t
gpgme_op_file_import( gpgme_ctx_t ctx, gpgme_recipients_t selkeys, const char *input )
{
    gpgme_error_t err;

    ctx->enc_to = selkeys;
    err = file_import_start( ctx, input );
    if( !err ) {
        gpgme_wait( ctx, 1 );
        ctx->pending = 0;
        err = get_import_result( ctx );
    }

    return err;
} /* gpgme_op_file_import */


gpgme_error_t
gpgme_op_clip_import (gpgme_recipients_t selkeys, int * import_res)
{
    gpgme_error_t err = 0;
    gpgme_ctx_t ctx = NULL;
    gpgme_data_t keydata = NULL;

    err = gpgme_new (&ctx);
    if (err)
        return err;
    err = gpgme_data_new_from_clipboard (&keydata);
    if (!err)
        err = gpgme_op_import (ctx, selkeys, keydata);
    if (!err)
        err = gpgme_get_import_status (ctx, import_res, NULL);

    gpgme_data_release (keydata);
    gpgme_release (ctx);

    return err;
} /* gpgme_op_clip_import */


static gpgme_error_t
import_list_start( gpgme_ctx_t ctx, gpgme_data_t in, gpgme_data_t out )
{
    gpgme_error_t rc;

    fail_on_pending_request( ctx );
    ctx->pending = 1;

    _gpgme_gpg_release( &ctx->gpg );
    rc = _gpgme_gpg_new( &ctx->gpg );
    if( rc )
        goto leave;

    /*_gpgme_gpg_add_arg( ctx->gpg, "--fast-list-mode" );*/
    _gpgme_gpg_add_arg( ctx->gpg, "--fixed-list-mode" );
    _gpgme_gpg_add_arg( ctx->gpg, "--with-colons" );

    if( gpgme_data_get_type( in ) == GPGME_DATA_TYPE_NONE ) {
        rc = mk_error( No_Data );
        goto leave;
    }

    _gpgme_data_set_mode( in, GPGME_DATA_MODE_OUT );
    _gpgme_data_set_mode( out, GPGME_DATA_MODE_IN );

    _gpgme_gpg_add_arg( ctx->gpg, "--output" );
    _gpgme_gpg_add_arg( ctx->gpg, "-" );
    _gpgme_gpg_add_data( ctx->gpg, out, 1 );
    _gpgme_gpg_add_arg( ctx->gpg, "--" );
    _gpgme_gpg_add_data( ctx->gpg, in, 0 );

    rc = _gpgme_gpg_spawn( ctx->gpg, ctx );

leave:
    if( rc ) {
        ctx->pending = 0;
        _gpgme_gpg_release( &ctx->gpg );
    }

    return rc;
} /* import_list_start */


gpgme_error_t
gpgme_op_import_list( gpgme_ctx_t ctx, gpgme_data_t in, gpgme_data_t out )
{
    gpgme_error_t err;

    err = import_list_start( ctx, in, out );
    if( !err ) {
        gpgme_wait( ctx, 1 );
        ctx->pending = 0;
        if (gpgme_data_get_type (out) == GPGME_DATA_TYPE_NONE)
            err = mk_error (General_Error);
        else if (gpgme_get_process_rc (ctx))
            err = mk_error (Internal_GPG_Problem);
    }

    return err;
} /* gpgme_op_import_list */


static void
decode_userid( char * pend, gpgme_key_t c )
{
    char * tmp;

    c->uids = calloc( 1, sizeof *c->uids + strlen( pend ) + 1 );
    if( !c->uids )
        return;
    tmp = c->uids->name;
    _gpgme_decode_c_string( pend, &tmp, strlen( pend ) + 1 );
    c->uids->name[strlen( c->uids->name ) -3] = '\0';
} /* decode_userid */


static void
parse_colon_key( char *buf, gpgme_key_t c )
{
    const char * s;
    char *p, *pend;
    int field = 0, rectype = 0;
    struct subkey_s * subk;
    enum {
        KEY_none      = 0,
        KEY_primary   = 1,
        KEY_secondary = 2,
    };

    if( !buf )
        return; /* EOF */

    if( !strncmp( buf, "pub", 3 ) || !strncmp( buf, "sec", 3 ) ) {
        rectype = KEY_primary;
        if( *buf == 's' )
            c->secret = 1;
    }
    else if( !strncmp( buf, "sub", 3 ) || !strncmp( buf, "ssb", 3 ) ) {
        rectype = KEY_secondary;
        if( !c->keys.next )
            c->keys.next = subk = calloc( 1, sizeof *subk );
        else
            subk = c->keys.next;
    }
    else if( !strncmp( buf, "uid", 3 ) ) {
        if( !c->uids ) {
            const char * s = buf;
            s += 3;
            while( s && *s == ':' )
                s++;
            decode_userid( (char *)s, c );

        }
        return;
    }
    else
        return;

    for( p = buf; p; p = pend ) {
        field++;
        pend = strchr( p, ':' );
        if( pend )
            *pend++ = 0;

        switch( field ) {
        case 1:
            if( rectype != KEY_primary )
                break;
            for( s = pend; *s && !isdigit( *s ); s++ ) {
                switch( *s ) {
                case 'd': c->keys.flags.disabled = 1; break;
                case 'e': c->keys.flags.expired = 1; break;
                case 'r': c->keys.flags.revoked = 1; break;
                }
            }
            break;

        case 2:
            if( rectype == KEY_primary )
                c->keys.key_len = atoi( pend );
            else
                subk->key_len = atoi( pend );
            break;

        case 3:
            if( rectype == KEY_primary )
                c->keys.key_algo = atoi( pend );
            else
                subk->key_algo = atoi( pend );
            break;

        case 4:
            if( rectype == KEY_primary ) {
                memcpy( c->keys.keyid, pend, 16 );
                c->keys.keyid[16] = '\0';
            }
            else {
                memcpy( subk->keyid, pend, 16 );
                subk->keyid[16]= '\0';
            }
            break;

        case 5:
            if( rectype == KEY_primary )
                c->keys.timestamp = strtoul( pend, NULL, 10 );
            else
                subk->timestamp = strtoul( pend, NULL, 10 );
            break;

        case 9:
            if( rectype == KEY_primary && !c->uids && strlen( pend ) > 2 ) {
                if( !strchr( pend, '[' ) && !strchr( pend, ']' ) )
                    decode_userid( pend, c );
            }
            break;
        }
    }
} /* parse_colon_key */


gpgme_error_t
gpgme_op_import_list_next( gpgme_data_t out, char **pending_line, gpgme_key_t *r_key )
{
    gpgme_key_t key = NULL;
    int in_cert, got_block = 0;
    char buf[384];

    if( !r_key )
        return mk_error( Invalid_Value );

    if( _gpgme_key_new( &key ) )
        return mk_error( Out_Of_Core );

    if( *pending_line ) {
        parse_colon_key( *pending_line, key );
        safe_free( *pending_line );
        *pending_line = NULL;
        in_cert = 1;
    }
    else
        in_cert = 0;

    while( gpgme_data_readline( out, buf, sizeof buf-1 ) ) {
        if( !strncmp( buf, "pub", 3 )
            || !strncmp( buf, "sec", 3 ) ) {
            if( in_cert ) {
                *pending_line = strdup( buf );
                goto ready;
            }
            in_cert = 1, got_block = 1;
            parse_colon_key( buf, key );
        }
        else if( in_cert ) {
            parse_colon_key( buf, key );
            got_block = 1;
        }
    }
ready:
    *r_key = key;
    if( got_block && _gpgme_data_eof( out ) )
        return 0;
    return _gpgme_data_eof( out ) ? mk_error( EOF ) : 0;
} /* gpgme_op_import_list_next */