/* data.c
 *      Copyright (C) 2000-2003 Werner Koch (dd9jn), g10 Code GmbH
 *      Copyright (C) 2001-2004 Timo Schulz
 *
 * This file is part of MyGPGME.
 *
 * MyGPGME is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MyGPGME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <process.h>
#include <io.h>
#include <windows.h>

#include "util.h"
#include "context.h"
#include "ops.h"

#define ALLOC_CHUNK 1024
#define MAX_FILESIZE 16777216

static char pgm_string[128] = "OpenPrivacy";


/**
 * gpgme_data_new:
 * @r_dh: returns the new data object
 *
 * Create a new data object without any content.
 *
 * Return value: An error value or 0 on success
 **/
gpgme_error_t
gpgme_data_new (gpgme_data_t * r_dh)
{
    gpgme_data_t dh;

    if (!r_dh)
        return mk_error (Invalid_Value);
    *r_dh = NULL;
    dh = calloc (1, sizeof *dh);
    if (!dh)
        return mk_error (Out_Of_Core);
    dh->mode = GPGME_DATA_MODE_INOUT;
    *r_dh = dh;
    return 0;
} /* gpgme_data_new */


/**
 * gpgme_data_new_from_mem:
 * @r_dh:   Returns a new data object.
 * @buffer: Initialize with this.
 * @size: Size of the buffer
 * @copy: Flag wether a copy of the buffer should be used.
 *
 * Create a new data object and initialize with data
 * from the memory.  A @copy with value %TRUE creates a copy of the
 * memory, a value of %FALSE uses the original memory of @buffer and the
 * caller has to make sure that this buffer is valid until gpgme_release_data()
 * is called.
 *
 * Return value:
 **/
gpgme_error_t
gpgme_data_new_from_mem (gpgme_data_t * r_dh,
                         const char * buffer, size_t size, int copy)
{
    gpgme_data_t dh;
    gpgme_error_t err;
    char * p;

    if (!r_dh || !buffer)
        return mk_error (Invalid_Value);
    *r_dh = NULL;
    err = gpgme_data_new (&dh);
    if (err)
        return err;
    dh->len = size;
    if (copy) {
        p = dh->private_buffer = malloc (size? size : 1);
        if (!p) {
            gpgme_data_release (dh);
            return mk_error (Out_Of_Core);
        }
        dh->private_len = size;
        memcpy (dh->private_buffer, buffer, size);
        dh->data = dh->private_buffer;
        dh->writepos = size;
    }
    else
        dh->data = buffer;
    dh->type = GPGME_DATA_TYPE_MEM;
    *r_dh = dh;
    return 0;
}


gpgme_error_t
gpgme_data_new_with_read_cb (gpgme_data_t *r_dh,
                              int (*read_cb)(void*,char *,size_t,size_t*),
                              void *read_cb_value)
{
    gpgme_data_t dh;
    gpgme_error_t err;

    if (!r_dh || !read_cb)
        return mk_error (Invalid_Value);
    *r_dh = NULL;
    err = gpgme_data_new (&dh);
    if (err)
        return err;
    dh->type = GPGME_DATA_TYPE_CB;
    dh->mode = GPGME_DATA_MODE_OUT;
    dh->read_cb = read_cb;
    dh->read_cb_value = read_cb_value;

    *r_dh = dh;
    return 0;
}

/**
 * gpgme_data_new_from_file:
 * @r_dh: returns the new data object
 * @fname: filename
 *
 * Create a new data object and initialize it with the content of
 * the file @file.  If @copy is %True the file is immediately read in
 * adn closed.  @copy of %False is not yet supportted.
 *
 * Return value: An error code or 0 on success. If the error code is
 * %GPGME_File_Error, the OS error code is held in %errno.
 **/
gpgme_error_t
gpgme_data_new_from_file (gpgme_data_t * r_dh, const char * fname)
{
    gpgme_data_t dh;
    gpgme_error_t err;
    struct stat st;
    FILE *fp;

    if (!r_dh)
        return mk_error (Invalid_Value);
    *r_dh = NULL;
    if (!fname)
        return mk_error (Invalid_Value);

    err = gpgme_data_new (&dh);
    if (err)
        return err;

    fp = my_fopen (fname, "rb");
    if (!fp) {
        int save_errno = errno;
        gpgme_data_release (dh);
        errno = save_errno;
        return mk_error (File_Error);
    }

    if (fstat (fileno (fp), &st)) {
        int save_errno = errno;
        fclose (fp);
        gpgme_data_release (dh);
        errno = save_errno;
        return mk_error (File_Error);
    }

    /* Check to disallow to load large files */
    if (st.st_size > MAX_FILESIZE) {
        fclose (fp);
        gpgme_data_release (dh);
        return mk_error (General_Error);
    }
    dh->private_buffer = malloc (st.st_size? st.st_size : 1);
    if (!dh->private_buffer) {
        fclose (fp);
        gpgme_data_release (dh);
        return mk_error (Out_Of_Core);
    }
    dh->private_len = st.st_size;

    if (fread (dh->private_buffer, dh->private_len, 1, fp) != 1) {
        int save_errno = errno;
        fclose (fp);
        gpgme_data_release (dh);
        errno = save_errno;
        return mk_error (File_Error);
    }
    fclose (fp);

    dh->len = dh->private_len;
    dh->data = dh->private_buffer;
    dh->writepos = dh->len;
    dh->type = GPGME_DATA_TYPE_MEM;

    *r_dh = dh;
    return 0;
}


gpgme_error_t
gpgme_data_new_from_filepart (gpgme_data_t *r_dh, const char *fname, FILE *fp,
                               long offset, ulong length)
{
    gpgme_data_t dh;
    gpgme_error_t err;

    if (!r_dh)
        return mk_error (Invalid_Value);
    *r_dh = NULL;
    if ( fname && fp ) /* these are mutual exclusive */
        return mk_error (Invalid_Value);
    if (!fname && !fp)
        return mk_error (Invalid_Value);
    if (!length)
        return mk_error (Invalid_Value);

    err = gpgme_data_new ( &dh );
    if (err)
        return err;

    if (!fp) {
        fp = my_fopen (fname, "rb");
        if (!fp) {
            int save_errno = errno;
            gpgme_data_release (dh);
            errno = save_errno;
            return mk_error (File_Error);
        }
    }

    if ( fseek ( fp, offset, SEEK_SET) ) {
        int save_errno = errno;
        if (fname)
            fclose (fp);
        gpgme_data_release (dh);
        errno = save_errno;
        return mk_error (File_Error);
    }


    dh->private_buffer = malloc ( length? length : 1 );
    if( !dh->private_buffer ) {
        if( fname )
            fclose( fp );
        gpgme_data_release (dh);
        return mk_error (Out_Of_Core);
    }
    dh->private_len = length;

    if ( fread ( dh->private_buffer, dh->private_len, 1, fp ) != 1 ) {
        int save_errno = errno;
        if (fname)
            fclose (fp);
        gpgme_data_release (dh);
        errno = save_errno;
        return mk_error (File_Error);
    }

    if (fname)
        fclose (fp);

    dh->len = dh->private_len;
    dh->data = dh->private_buffer;
    dh->writepos = dh->len;
    dh->type = GPGME_DATA_TYPE_MEM;

    *r_dh = dh;
    return 0;
}


/**
 * gpgme_data_release:
 * @dh: Data object
 *
 * Release the data object @dh.  @dh may be NULL in which case nothing
 * happens.
 **/
void
gpgme_data_release (gpgme_data_t dh)
{
    if (dh) {
        safe_free (dh->private_buffer);
        safe_free (dh);
    }
} /* gpgme_data_release */


char *
_gpgme_data_release_and_return_string (gpgme_data_t dh)
{
    char *val = NULL;

    if (!dh)
        return val;

    if (!_gpgme_data_append (dh, "", 1)) { /* append EOS */
        val = dh->private_buffer;
        if (!val && dh->data) {
            val = malloc (dh->len);
            if (val)
                memcpy (val, dh->data, dh->len);
        }
    }
    safe_free (dh);
    return val;
}


/**
 * gpgme_data_release_and_get_mem:
 * @dh: the data object
 * @r_len: returns the length of the memory
 *
 * Release the data object @dh and return its content and the length
 * of that content.  The caller has to free this data.  @dh maybe NULL
 * in which case NULL is returned.  If there is not enough memory for
 * allocating the return value, NULL is returned and the object is
 * released.
 *
 * Return value: a pointer to an allocated buffer of length @r_len.
 **/
char *
gpgme_data_release_and_get_mem (gpgme_data_t dh, size_t *r_len)
{
    char *val = NULL;

    if (r_len)
        *r_len = 0;
    if (dh) {
        size_t len = dh->len;
        val = dh->private_buffer;
        if ( !val && dh->data ) {
            val = malloc (len);
            if ( val )
                memcpy (val, dh->data, len);
        }
        safe_free (dh);
        if (val && r_len)
            *r_len = len;
    }
    return val;
}


/**
 * gpgme_data_get_type:
 * @dh: the data object
 *
 * Get the type of the data object.
 * Data types are prefixed with %GPGME_DATA_TYPE_
 *
 * Return value: the data type
 **/
gpgme_data_type_t
gpgme_data_get_type ( gpgme_data_t dh )
{
    if ( !dh || (!dh->data && !dh->read_cb))
        return GPGME_DATA_TYPE_NONE;

    return dh->type;
}

void
_gpgme_data_set_mode( gpgme_data_t dh, gpgme_data_mode_t mode )
{
    assert( dh );
    dh->mode = mode;
}


gpgme_data_mode_t
_gpgme_data_get_mode( gpgme_data_t dh )
{
    assert( dh );
    return dh->mode;
}

/**
 * gpgme_data_rewind:
 * @dh: the data object
 *
 * Prepare the data object in a way, that a gpgme_data_read() does start
 * at the beginning of the data.  This has to be done for all types
 * of data objects.
 *
 * Return value: An error code or 0 on success
 **/
gpgme_error_t
gpgme_data_rewind ( gpgme_data_t dh )
{
    if ( !dh )
        return mk_error (Invalid_Value);

    if ( dh->type == GPGME_DATA_TYPE_NONE
         || dh->type == GPGME_DATA_TYPE_MEM ) {
        dh->readpos = 0;
    }
    else if (dh->type == GPGME_DATA_TYPE_CB) {
        dh->len = dh->readpos = 0;
        dh->read_cb_eof = 0;
        /* FIXME: do a special call to the read function to trigger a rewind
           there */
    }
    else
        return mk_error (General_Error);
    return 0;
}

/**
 * gpgme_data_read:
 * @dh: the data object
 * @buffer: A buffer
 * @length: The length of that bufer
 * @nread: Returns the number of bytes actually read.
 *
 * Copy data from the current read position (which may be set by
 * gpgme_data_rewind()) to the supplied @buffer, max. @length bytes
 * are copied and the actual number of bytes are returned in @nread.
 * If there are no more bytes available %GPGME_EOF is returned and @nread
 * is set to 0.
 *
 * Return value: An errorcode or 0 on success, EOF is indcated by the
 * error code GPGME_EOF.
 **/
gpgme_error_t
gpgme_data_read ( gpgme_data_t dh, char *buffer, size_t length, size_t *nread )
{
    size_t nbytes;

    if ( !dh )
        return mk_error (Invalid_Value);
    if (dh->type == GPGME_DATA_TYPE_MEM ) {
        nbytes = dh->len - dh->readpos;
        if ( !nbytes ) {
            *nread = 0;
            return mk_error(EOF);
        }
        if (nbytes > length)
            nbytes = length;
        memcpy ( buffer, dh->data + dh->readpos, nbytes );
        *nread = nbytes;
        dh->readpos += nbytes;
    }
    else if (dh->type == GPGME_DATA_TYPE_CB) {
        nbytes = dh->len - dh->readpos;
        if ( nbytes ) {
            /* we have unread data - return this */
            if (nbytes > length)
                nbytes = length;
            memcpy ( buffer, dh->data + dh->readpos, nbytes );
            *nread = nbytes;
            dh->readpos += nbytes;
        }
        else { /* get the data from the callback */
            if (!dh->read_cb || dh->read_cb_eof) {
                *nread = 0;
                return mk_error (EOF);
            }
            if (dh->read_cb (dh->read_cb_value, buffer, length, nread )) {
                *nread = 0;
                dh->read_cb_eof = 1;
                return mk_error (EOF);
            }
        }
    }
    else
        return mk_error (General_Error);
    return 0;
}


gpgme_error_t
_gpgme_data_eof( gpgme_data_t dh )
{
    if( !dh )
        return mk_error( Invalid_Value );
    if( dh->type == GPGME_DATA_TYPE_MEM ) {
        size_t nbytes = dh->len - dh->readpos;
        if( !nbytes )
            return mk_error( EOF );
    }
    return 0;
}


gpgme_error_t
_gpgme_data_unread (gpgme_data_t dh, const char *buffer, size_t length )
{
   if( !dh )
        return mk_error (Invalid_Value);
   if( dh->type == GPGME_DATA_TYPE_MEM ) {
       /* check that we don't unread more than we have yet read */
       if( dh->readpos < length )
           return mk_error( Invalid_Value );
       /* No need to use the buffer for this data type */
       dh->readpos -= length;
   }
   else {
       return mk_error( General_Error );
   }
   return 0;
}


/*
 * This function does make sense when we know that it contains no nil chars.
 */
char *
_gpgme_data_get_as_string ( gpgme_data_t dh )
{
    char *val = NULL;

    if (dh) {
        val = malloc ( dh->len+1 );
        if ( val ) {
            memcpy ( val, dh->data, dh->len );
            val[dh->len] = 0;
        }
    }
    return val;
}


/**
 * gpgme_data_write:
 * @dh: the context
 * @buffer: data to be written to the data object
 * @length: length o this data
 *
 * Write the content of @buffer to the data object @dh at the current write
 * position.
 *
 * Return value: 0 on succress or an errorcode
 **/
gpgme_error_t
gpgme_data_write (gpgme_data_t dh, const char *buffer, size_t length)
{
    if (!dh || !buffer)
        return mk_error (Invalid_Value);

    return _gpgme_data_append (dh, buffer, length );
}


gpgme_error_t
_gpgme_data_append ( gpgme_data_t dh, const char *buffer, size_t length )
{
    assert (dh);

    if ( dh->type == GPGME_DATA_TYPE_NONE ) {
        /* convert it to a mem data type */
        assert (!dh->private_buffer);
        dh->type = GPGME_DATA_TYPE_MEM;
        dh->private_len = length < ALLOC_CHUNK? ALLOC_CHUNK : length;
        dh->private_buffer = malloc ( dh->private_len );
        if( !dh->private_buffer ) {
            dh->private_len = 0;
            return mk_error (Out_Of_Core);
        }
        dh->writepos = 0;
        dh->data = dh->private_buffer;
    }
    else if ( dh->type != GPGME_DATA_TYPE_MEM )
        return mk_error (Invalid_Type);

    if ( dh->mode != GPGME_DATA_MODE_INOUT
         && dh->mode != GPGME_DATA_MODE_IN  )
        return mk_error (Invalid_Mode);

    if ( !dh->private_buffer ) {
        /* we have to copy it now */
        assert (dh->data);
        dh->private_len = dh->len+length;
        if (dh->private_len < ALLOC_CHUNK)
            dh->private_len = ALLOC_CHUNK;
        dh->private_buffer = malloc ( dh->private_len );
        if (!dh->private_buffer) {
            dh->private_len = 0;
            return mk_error (Out_Of_Core);
        }
        memcpy ( dh->private_buffer, dh->data, dh->len );
        dh->writepos = dh->len;
        dh->data = dh->private_buffer;
    }

    /* allocate more memory if needed */
    if ( dh->writepos + length > dh->private_len ) {
        char *p;
        size_t newlen = dh->private_len
                        + (dh->len < ALLOC_CHUNK? ALLOC_CHUNK : length);
        p = realloc ( dh->private_buffer, newlen );
        if ( !p )
            return mk_error (Out_Of_Core);
        dh->private_buffer = p;
        dh->private_len = newlen;
        dh->data = dh->private_buffer;
        assert ( !(dh->writepos + length > dh->private_len) );
    }

    memcpy ( dh->private_buffer + dh->writepos, buffer, length );
    dh->writepos += length;
    dh->len += length;

    return 0;
}

gpgme_error_t
_gpgme_data_append_string ( gpgme_data_t dh, const char *s )
{
    return _gpgme_data_append ( dh, s, s? strlen(s):0 );
}


gpgme_error_t
_gpgme_data_read_from_tmpfile( gpgme_data_t dh )
{
    FILE *fp;
    size_t n;
    unsigned char buf[512];

    gpgme_data_rewind( dh );
    fp = fopen( _gpgme_get_tmpfile(0), "rb" );
    if( fp == NULL )
        return mk_error( File_Error );
    while( !feof(fp) ) {
        memset(buf, 0, sizeof buf );
        n = fread(buf, 1, sizeof buf - 1, fp );
        if( n > 0 )
            gpgme_data_write( dh, buf, n );
    }
    fclose( fp );
    return 0;
} /* _gpgme_data_read_from_tmpfile */


gpgme_error_t
_gpgme_data_write_to_tmpfile( gpgme_data_t dh )
{
    FILE *fp;
    char *p;

    gpgme_data_rewind( dh );
    fp = fopen( _gpgme_get_tmpfile(1), "wb" );
    if( fp == NULL )
        return mk_error( File_Error );
    p = _gpgme_data_get_as_string( dh );
    if( p ) {
        fwrite( p, 1, strlen( p ), fp );
        safe_free( p );
    }
    fclose( fp );
    return 0;
} /* _gpgme_data_write_to_tmpfile */

char*
gpgme_data_get_as_string( gpgme_data_t dh )
{
    return _gpgme_data_get_as_string( dh );
} /* gpgme_data_get_as_string */

char*
gpgme_data_release_and_return_string( gpgme_data_t dh )
{
    return _gpgme_data_release_and_return_string( dh );
} /* gpgme_data_release_and_return_string */

gpgme_error_t
gpgme_data_release_and_set_file (gpgme_data_t dh, const char *fname)
{
    char *p = NULL;
    FILE *fp;

    fp = my_fopen (fname, "wb");
    if (fp == NULL)
        return mk_error (File_Error);

    p = _gpgme_data_release_and_return_string (dh);
    if (p) {
        fwrite (p, 1, strlen (p), fp);
        fflush (fp);
        memset (p, 0xFF, strlen (p));
        safe_free (p);
    }
    fclose (fp);
    return 0;
} /* gpgme_data_release_and_set_file */


size_t
gpgme_data_readline( gpgme_data_t dh, char * line, size_t nbytes )
{
    char ch = 0;
    size_t nread = 0, pos = 0;

    if( !dh )
        return 0;

    memset( line, 0, nbytes );
    while( !gpgme_data_read( dh, &ch, 1, &nread ) ) {
        if( !nread )
            break;
        if( ch == '\n' ) {
            line[pos++] = ch;
            line[pos++] = '\0';
            break;
        }
        line[pos++] = ch;
        if( pos > nbytes ) {
            line[pos++] = '\0';
            break;
        }
    }

    return pos;
} /* gpgme_data_readline */


static char*
wrap_lines_soft( char *buf, int wraplen )
{
    char *curpos, *lastspc, *lastbrk;

    for( curpos = lastspc = lastbrk = buf; *curpos; curpos++ ) {
        if( *curpos == ' ' )
            lastspc = curpos;
        else {
            if( *curpos == '\n' ) {
                lastbrk = curpos;
                lastspc = lastbrk;
            }
        }
        if( (curpos - lastbrk) >= wraplen ) {
            *lastspc = '\n';
            lastbrk = lastspc;
        }
    }

    return buf;
} /* wrap_lines_soft */

static char*
wrap_lines( char *buf, size_t bufsize, size_t wraplen )
{
    char *soft, *hard, *p;
    size_t pos = 0, size = 0;

    soft = wrap_lines_soft( buf, wraplen );
    if( !soft )
        return NULL;
    hard = malloc (bufsize + (bufsize/10));
    if( !hard )
        return NULL;
    strcpy (hard, soft);
    while( soft ) {
        p = strchr( soft, '\n' );
        if( p == NULL )
            break;
        pos = p - soft + 1;
        if( soft[pos-2] != '\r' ) {
            memcpy( hard + size, soft, pos-1 );
            size += (pos-1);
            memcpy( hard + size, "\r\n", 2 );
            size += 2;
        }
        else {
            memcpy( hard+size, soft, pos );
            size += pos;
        }
        soft += pos;
    }

    memcpy( hard + size, "\0", 1 );
    return hard;
} /* wrap_lines */


int
gpgme_data_wrap_lines( gpgme_data_t *r_dh, size_t wraplen )
{
    gpgme_error_t err = 0;
    gpgme_data_t mdh;
    char *raw, *p;
    size_t nlen;

    err = gpgme_data_new( &mdh );
    if( err )
        return err;

    raw = _gpgme_data_get_as_string( *r_dh );
    if( !raw ) {
        gpgme_data_release( mdh );
        return mk_error( General_Error );
    }
    nlen = strlen( raw );
    if( !strchr( raw, '\n' ) && nlen < wraplen ) {
        gpgme_data_release( mdh );
        safe_free( raw );
        return 0;
    }

    p = wrap_lines( raw, nlen, wraplen );
    if( p )
        _gpgme_data_append_string( mdh, p );
    gpgme_data_release( *r_dh );
    *r_dh = mdh;
    safe_free( raw );
    safe_free( p );
    return 0;
} /* gpgme_data_wrap_lines */


gpgme_error_t
gpgme_data_mail_quote (gpgme_data_t *r_dh)
{
    gpgme_data_t dh;
    char buf[128];

    if (!*r_dh)
        return mk_error (Invalid_Value);
    gpgme_data_new (&dh);
    while (gpgme_data_readline (*r_dh, buf, 127)) {
        gpgme_data_write (dh, "> ", 2);
        gpgme_data_write (dh, buf, strlen (buf));
    }
    gpgme_data_release (*r_dh);
    *r_dh = dh;
    return 0;
}


gpgme_error_t
gpgme_data_extract_plaintext( gpgme_data_t sig, gpgme_data_t *r_plain )
{
    gpgme_data_t plain;
    gpgme_error_t err;
    char line[128+32];
    int pos = 0;
    int sig_begin = 0;

    err = gpgme_data_new( &plain );
    if( err ) {
        if( r_plain )
            *r_plain = NULL;
        return err;
    }

    while( gpgme_data_readline( sig, line, 128 ) ) {
        if( !strncmp( line, "-----BEGIN PGP SIGNED MESSAGE", 29 )
             || !strncmp( line, "Version:", 8 )
             || !strncmp( line, "Comment:", 8 )
             || !strncmp( line, "Charset:", 8 )
             || !strncmp( line, "Hash:", 5 )
             || !strncmp (line, "MessageID", 9))
            continue;
        if( strlen( line ) <= 2 )
            break; /* parsed all headers, now we reached the body */
    }
    /* fixme: handle multi dash escaped sequences! */
    while( gpgme_data_readline( sig, line, 128 ) ) {
        if( !strncmp( line, "-----BEGIN PGP SIGNATURE", 24 ) )
            break; /* end of plaintext */
        if( !strncmp( line, "- -", 3 ) )
            pos = 2;
        _gpgme_data_append_string( plain, line + pos );
        pos = 0;
    }
    if( r_plain )
        *r_plain = plain;
    else
        gpgme_data_release( plain );
    return err;
} /* gpgme_data_extract_plaintext */


gpgme_error_t
gpgme_data_change_version( gpgme_data_t *r_dh )
{
    gpgme_error_t err = 0;
    gpgme_data_t mdh;
    char line[128+32];

    err = gpgme_data_new( &mdh );
    if( err ) {
        if( r_dh )
            *r_dh = NULL;
        return err;
    }
    while( gpgme_data_readline( *r_dh, line, 128 ) ) {
        if( strlen( line ) > 14
             && !strncmp( line, "Version: GnuPG", 14 )
             && !strstr( line, pgm_string ) ) {
            line[strlen( line ) - 2] = '\0';
            strcat( line, " - " );
            strcat( line, pgm_string );
            strcat( line, "\r\n" );
        }
        _gpgme_data_append_string( mdh, line );
    }
    gpgme_data_release( *r_dh );
    if( r_dh )
        *r_dh = mdh;
    else
        gpgme_data_release( mdh );
    return err;
} /* gpgme_data_change_version */


/* Convert two hexadecimal digits from STR to the value they
   represent.  Returns -1 if one of the characters is not a
   hexadecimal digit.  */
static int
_gpgme_hextobyte (const unsigned char *str)
{
    int val = 0;
    int i;

#define NROFHEXDIGITS 2
    for (i = 0; i < NROFHEXDIGITS; i++) {
        if (*str >= '0' && *str <= '9')
            val += *str - '0';
        else if (*str >= 'A' && *str <= 'F')
            val += 10 + *str - 'A';
        else if (*str >= 'a' && *str <= 'f')
            val += 10 + *str - 'a';
        else
            return -1;
        if (i < NROFHEXDIGITS - 1)
            val *= 16;
        str++;
    }
    return val;
}


/* Decode the C formatted string SRC and store the result in the
   buffer *DESTP which is LEN bytes long.  If LEN is zero, then a
   large enough buffer is allocated with malloc and *DESTP is set to
   the result.  Currently, LEN is only used to specify if allocation
   is desired or not, the caller is expected to make sure that *DESTP
   is large enough if LEN is not zero.  */
gpgme_error_t
_gpgme_decode_c_string (const char *src, char **destp, size_t len)
{
  char *dest;

  /* Set up the destination buffer.  */
  if (len)
    {
      if (len < strlen (src) + 1)
        return GPGME_General_Error;

      dest = *destp;
    }
  else
    {
      /* The converted string will never be larger than the original
         string.  */
      dest = malloc (strlen (src) + 1);
      if (!dest)
        return GPGME_Out_Of_Core;
      *destp = dest;
    }

  /* Convert the string.  */
  while (*src)
    {
      if (*src != '\\')
        {
          *(dest++) = *(src++);
          continue;
        }

      switch (src[1])
        {
#define DECODE_ONE(match,result)        \
        case match:                     \
          src += 2;                     \
          *(dest++) = result;           \
          break;

          DECODE_ONE ('\'', '\'');
          DECODE_ONE ('\"', '\"');
          DECODE_ONE ('\?', '\?');
          DECODE_ONE ('\\', '\\');
          DECODE_ONE ('a', '\a');
          DECODE_ONE ('b', '\b');
          DECODE_ONE ('f', '\f');
          DECODE_ONE ('n', '\n');
          DECODE_ONE ('r', '\r');
          DECODE_ONE ('t', '\t');
          DECODE_ONE ('v', '\v');

        case 'x':
          {
            int val = _gpgme_hextobyte (&src[2]);

            if (val == -1)
              {
                /* Should not happen.  */
                *(dest++) = *(src++);
                *(dest++) = *(src++);
                if (*src)
                  *(dest++) = *(src++);
                if (*src)
                  *(dest++) = *(src++);
              }
            else
              {
                if (!val)
                  {
                    /* A binary zero is not representable in a C
                       string.  */
                    *(dest++) = '\\';
                    *(dest++) = '0';
                  }
                else
                  *((unsigned char *) dest++) = val;
                src += 4;
              }
          }

        default:
          {
            /* Should not happen.  */
            *(dest++) = *(src++);
            *(dest++) = *(src++);
          }
        }
    }
  *(dest++) = 0;

  return 0;
}


void
gpgme_set_pgm_string (const char * name)
{
    if (!name || !*name)
        return;
    memset (pgm_string, 0, sizeof pgm_string);
    strncpy (pgm_string, name, sizeof pgm_string-1);
} /* gpgme_set_pgm_string */
