/* sign-encrypt.c -  Sign & Encrypt functions
 *	Copyright (C) 2001-2004 Timo Schulz
 *
 * This file is part of MyGPGME.
 *
 * MyGPGME is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MyGPGME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <string.h>
#include <stdio.h>
#include <assert.h>

#include <windows.h>

#include "util.h"
#include "context.h"
#include "ops.h"

struct sign_encrypt_result_s {
    void * last_pw_handle;
    char * userid_hint;
    char * passphrase_info;
    char cardno[32+1];
    int no_passphrase;
    int okay;	
    int bad_passphrase;
    int idea_cipher;
};


void
_gpgme_release_sign_encrypt_result( _sign_encrypt_result_t res )
{
    if( res ) {
	safe_free( res->userid_hint );
	safe_free( res->passphrase_info );
	safe_free( res );
    }
}


static void
sign_encrypt_status_handler( gpgme_ctx_t ctx, gpg_status_code_t code, char *args )
{
    char * p;
    int i=0;

    if( ctx->out_of_core )
        return;
    
    if( ctx->result_type == RESULT_TYPE_NONE ) {
        assert( !ctx->result.sign_enc );
        ctx->result.sign_enc = calloc ( 1, sizeof * ctx->result.sign_enc );
        if( !ctx->result.sign_enc ) {
            ctx->out_of_core = 1;
            return;
        }
        ctx->result_type = RESULT_TYPE_SIGN_ENCRYPT;
    }
    assert( ctx->result_type == RESULT_TYPE_SIGN_ENCRYPT );
    
    switch( code ) {
    case STATUS_EOF:
        break;
        
    case STATUS_USERID_HINT:
        safe_free( ctx->result.sign_enc->userid_hint );
	ctx->result.sign_enc->userid_hint = strdup( args );
        if( !ctx->result.sign_enc->userid_hint ) {
            ctx->out_of_core = 1;
	    return;
	}
        break;
        
    case STATUS_BAD_PASSPHRASE:
        DEBUG0( "Bad passphrase - once again please\n" );
        ctx->result.sign_enc->bad_passphrase++;		 
        break;
        
    case STATUS_GOOD_PASSPHRASE:
        ctx->result.sign_enc->bad_passphrase = 0;
        break;

    case STATUS_RSA_OR_IDEA:
	ctx->result.sign_enc->idea_cipher = 1;
	break;
        
    case STATUS_NEED_PASSPHRASE:
    case STATUS_NEED_PASSPHRASE_SYM:
        safe_free( ctx->result.sign_enc->passphrase_info );
	ctx->result.sign_enc->passphrase_info = strdup( args );
        if( !ctx->result.sign_enc->passphrase_info ) {
            ctx->out_of_core = 1;
	    return;
	}
        break;        
        
    case STATUS_MISSING_PASSPHRASE:
        DEBUG0( "missing passphrase - stop\n" );
        ctx->result.sign_enc->no_passphrase = 1;
        break;
        
    case STATUS_SIG_CREATED:
        ctx->result.sign_enc->okay = 1;
        break;

    case STATUS_PROGRESS:
	if( ctx->cb.progress )
	    _gpgme_progress_handler( ctx, args );
	break;

    case STATUS_CARDCTRL:
	if( args[i++] != '3' )
	    break;
	i++;
	p = ctx->result.sign_enc->cardno;
	for( ; i-1 < DIM(ctx->result.sign_enc->cardno) && args[i]; i++ )
	    *p++ = args[i];
	*p = 0;
	break;
        
    default:
        break;
    }
}


static const char*
command_handler( void *opqaue, gpg_status_code_t code, const char *key )
{
    gpgme_ctx_t c = opqaue;
    
    if( c->result_type == RESULT_TYPE_NONE ) {
        assert ( !c->result.sign_enc );
        c->result.sign_enc = calloc ( 1, sizeof *c->result.sign_enc );
        if( !c->result.sign_enc ) {
            c->out_of_core = 1;
            return NULL;
        }
        c->result_type = RESULT_TYPE_SIGN_ENCRYPT;
    }
    
    if( !key || !c->cb.passphrase )
        return NULL;
    
    if( code == STATUS_GET_HIDDEN 
	&& (!strcmp( key, "passphrase.enter" )
	||  !strcmp( key, "passphrase.pin.ask" )) ) {
        const char * userid_hint = c->result.sign_enc->userid_hint;
        const char * passphrase_info = c->result.sign_enc->passphrase_info;
	const char * cardno = c->result.sign_enc->cardno;
        int bad_passphrase = c->result.sign_enc->bad_passphrase;
	int is_card=0;
        char * buf;
        const char * s;
        
        c->result.sign_enc->bad_passphrase = 0;
	is_card = !strcmp( key, "passphrase.pin.ask" );
        if( !userid_hint )
            userid_hint = "[User ID hint missing]";
        if( !passphrase_info )
            passphrase_info = "[passphrase info missing]";
        buf = malloc( 20 + strlen( userid_hint )
                         + strlen( passphrase_info ) + 3 );
        if( !buf ) {
            c->out_of_core = 1;
            return NULL;
        }
        sprintf( buf, "%s\n%s\n%s",
                 bad_passphrase? "TRY_AGAIN":"ENTER_PASSPHRASE",
                 userid_hint, passphrase_info );
        
        s = c->cb.passphrase( c->cb.passphrase_value,
			      is_card? cardno : buf,
			      &c->result.sign_enc->last_pw_handle );
        safe_free( buf );
        return s;
    }    
    
    return NULL;
}


gpgme_error_t
gpgme_op_sign_encrypt_start( gpgme_ctx_t ctx, gpgme_recipients_t recp, 
                             gpgme_data_t plain, gpgme_data_t ciph )
{
    gpgme_key_t key;
    int rc = 0;
    int i;
    
    fail_on_pending_request( ctx );
    ctx->pending = 1;
    
    if( !gpgme_recipients_count( recp ) ) {
        rc = mk_error( No_Recipients );
        goto leave;
    }
    
    _gpgme_gpg_release( &ctx->gpg );
    rc = _gpgme_gpg_new( &ctx->gpg );
    if( rc )
        goto leave;
    
    if( ctx->use_logging )
    _gpgme_gpg_set_logging_handler( ctx->gpg, ctx );
    _gpgme_gpg_set_status_handler( ctx->gpg, sign_encrypt_status_handler, ctx );
    if ( ctx->cb.passphrase ) {
        rc = _gpgme_gpg_set_command_handler( ctx->gpg, command_handler, ctx );
        if( rc )
            goto leave;
    }
    else if( ctx->passphrase_value ) {
        rc = _gpgme_add_passphrase( ctx );
        if( rc )
            goto leave;
    }
    
    _gpgme_gpg_add_arg( ctx->gpg, "--encrypt" );
    _gpgme_gpg_add_arg( ctx->gpg, "--sign" );
    if( ctx->use_armor )
        _gpgme_gpg_add_arg( ctx->gpg, "--armor" );
    if( ctx->cb.progress )
	_gpgme_gpg_add_arg (ctx->gpg, "--enable-progress-filter");
    if( ctx->force_trust || _gpgme_recipients_all_valid( recp ) )
        _gpgme_gpg_add_arg (ctx->gpg, "--always-trust");
    
    for( i = 0; (key = gpgme_signers_enum (ctx, i)) != NULL; i++ ) {
        const char * s;
	s = gpgme_key_get_string_attr( key, GPGME_ATTR_KEYID, NULL, 0 );
        if (s) {
            _gpgme_gpg_add_arg (ctx->gpg, "-u");
            _gpgme_gpg_add_arg_concat (ctx->gpg, s, "!");
	    _gpgme_gpg_add_arg (ctx->gpg, "--encrypt-to");
	    _gpgme_gpg_add_arg (ctx->gpg, s);
        }
        gpgme_key_unref (key);
    }
    
    _gpgme_append_gpg_args_from_recipients( recp, ctx->gpg );    
    if( gpgme_data_get_type( plain ) == GPGME_DATA_TYPE_NONE ) {
        rc = mk_error( No_Data );
        goto leave;
    }	
    
    _gpgme_data_set_mode( plain, GPGME_DATA_MODE_OUT );
    
    if( !ciph || gpgme_data_get_type(ciph) != GPGME_DATA_TYPE_NONE ) {
        rc = mk_error( No_Data );
        goto leave;
    }
    
    _gpgme_data_set_mode( ciph, GPGME_DATA_MODE_IN );
    
    if( ctx->use_tmpfiles ) {
        _gpgme_gpg_add_arg ( ctx->gpg, "--output" );
        _gpgme_gpg_add_arg ( ctx->gpg, _gpgme_get_tmpfile(0) );
        _gpgme_data_write_to_tmpfile( plain );
        _gpgme_gpg_add_arg ( ctx->gpg, _gpgme_get_tmpfile(1) );
    }
    else {
        _gpgme_gpg_add_arg ( ctx->gpg, "--output" );
        _gpgme_gpg_add_arg ( ctx->gpg, "-" );
        _gpgme_gpg_add_data ( ctx->gpg, ciph, 1 );
        _gpgme_gpg_add_arg ( ctx->gpg, "--" );
        _gpgme_gpg_add_data ( ctx->gpg, plain, 0 );
    }
    
    rc = _gpgme_gpg_spawn( ctx->gpg, ctx );
    
leave:
    if ( rc ) {
        ctx->pending = 0;
        _gpgme_gpg_release( &ctx->gpg );	
    }
    return rc;
    
} /* gpgme_op_sign_encrypt_start */


gpgme_error_t
gpgme_op_sign_encrypt( gpgme_ctx_t ctx, gpgme_recipients_t rset,
		       gpgme_data_t in, gpgme_data_t out )
{
    gpgme_error_t rc;

    rc = gpgme_op_sign_encrypt_start( ctx, rset, in, out );    
    if( !rc ) {
        gpgme_wait( ctx, 1 );
        ctx->pending = 0;
        if( ctx->use_tmpfiles ) {
            _gpgme_data_read_from_tmpfile( out );
            _gpgme_del_tmpfiles( ctx->wipe_fnc );
        }
        if( ctx->result_type != RESULT_TYPE_SIGN_ENCRYPT )
            rc = mk_error( General_Error );
        else if ( ctx->out_of_core )
            rc = mk_error( Out_Of_Core );
        else {
            assert( ctx->result.sign_enc );
            if( ctx->result.sign_enc->no_passphrase )
                rc = mk_error( No_Passphrase );
            else if( ctx->result.sign_enc->bad_passphrase )
                rc = mk_error( Bad_Passphrase );
	    else if( ctx->result.sign_enc->idea_cipher )
		rc = mk_error( Cipher_IDEA );
	    else if( gpgme_get_process_rc( ctx ) )
		rc =  mk_error( Internal_GPG_Problem );
            else if( !ctx->result.sign_enc->okay )
                rc = mk_error( Signing_Failed );
        }
    }
    return rc;
} /* gpgme_op_sign_encrypt */


gpgme_error_t
gpgme_op_clip_sign_encrypt( gpgme_ctx_t ctx, gpgme_recipients_t rset,
			    const char * keyid, int opts )
{
    gpgme_error_t err;
    gpgme_data_t plain = NULL;
    gpgme_data_t ciph = NULL;
    gpgme_key_t key = NULL;
    
    if( !keyid )
        return mk_error( General_Error );
    
    if( opts & GPGME_CTRL_FORCETRUST )
        gpgme_control( ctx, GPGME_CTRL_FORCETRUST, 1 );
    if( opts & GPGME_CTRL_TMPFILES )
        gpgme_control( ctx, GPGME_CTRL_TMPFILES, 1 );
    gpgme_control( ctx, GPGME_CTRL_ARMOR, 1 );
    
    key = _gpgme_key_new_fromkeyid( keyid );
    if( !key )
	return mk_error( General_Error );
    
    err = gpgme_data_new_from_clipboard (&plain);
    if( !err )        
	err = gpgme_data_new( &ciph );
    if( !err )
	err = gpgme_signers_add( ctx, key );
    if( !err )	
	err = gpgme_op_sign_encrypt( ctx, rset, plain, ciph );
    if( !err ) {
	gpgme_data_change_version( &ciph );
	gpgme_data_release_and_set_clipboard( ciph );
    }
    gpgme_data_release( plain );
    gpgme_key_release( key );

    return err;
} /* gpgme_op_clip_sign_encrypt */