/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.gdk;
import org.gnu.glib.GObject;
/**
 * All drawing operations in GDK take a GC argument.  A graphics context 
 * encapsulates information about the way things are drawn, such as the
 * foreground color or line width.  By using graphics context, the number
 * of arguments to each drawing call is reduced, and communication overhead
 * is minimized.
 */
public class GC extends GObject 
{

	public GC(int handle){
		super(handle);
	}
	
	public GC(Window window) {
		super(gdk_gc_new(window.getHandle()));
	}


	public static GC getDefaultGC(org.gnu.gdk.Drawable d){
		return new GC(GC.gdk_gc_new(d.getHandle()));
	}
	
	/**
	 * Getst he x origin of the clip mask.
	 */
	public int getClipXOrigin() {
		return getClipXOrigin(getHandle());
	}
	
	/**
	 * Gets the y origin of the clip mask.
	 */
	public int getClipYOrigin() {
		return getClipYOrigin(getHandle());
	}
	
	/**
	 * Gets the x origin of the tile or stipple.
	 */
	public int getTsXOrigin() {
		return getTsXOrigin(getHandle());
	}
	
	/**
	 * Gets the y origin of the tile or stipple.
	 */
	public int getTsYOrigin() {
		return getTsYOrigin(getHandle());
	}
	
	/**
	 * Sets the foreground color for the graphics context.
	 */
	public void setForeground(Color color) {
		gdk_gc_set_foreground(getHandle(), color.getHandle());
	}
	
	/**
	 * Sets the background color for the graphics context.
	 */
	public void setBackground(Color color) {
		gdk_gc_set_background(getHandle(), color.getHandle());
	}
	
	/**
	 * Sets the foreground color using an unallocated color.  The
	 * pixel value for the color will be determined using GdkRGB.
	 * If the colormap has not previously been initialized for
	 * GdkRGB, then for pseudo-color colormaps (colormaps with a
	 * small modifiable number of colors), a colorcube will be
	 * allocated in the colormap.
	 * <p>
	 * Calling this function for a GC without a colormap is an
	 * error.
	 */
	public void setRGBForeground(Color color) {
		gdk_gc_set_rgb_fg_color(getHandle(), color.getHandle());
	}
	
	/**
	 * Sets the background color using an unallocated color.  The
	 * pixel value for the color will be determined using GdkRGB.
	 * If the colormap has not previously been initialized for
	 * GdkRGB, then for pseudo-color colormaps (colormaps with a
	 * small modifiable number of colors), a colorcube will be
	 * allocated in the colormap.
	 * <p>
	 * Calling this function for a GC without a colormap is an
	 * error.
	 */
	public void setRGBBackground(Color color) {
		gdk_gc_set_rgb_bg_color(getHandle(), color.getHandle());
	}
	
	/**
	 * Determines how the current pixel values and the pixel values being
	 * drawn are combined to produce the final pixel values.
	 */
	public void setFunction(Function func) {
		gdk_gc_set_function(getHandle(), func.getValue());
	}
	
	/**
	 * Sets the fill mode for the graphic context.
	 */
	public void setFillMode(Fill mode) {
		gdk_gc_set_fill(getHandle(), mode.getValue());
	}
	
	/**
	 * Sets the tile pixmap for the graphics context.  This will only be
	 * used if the fill mode is TILED.
	 */
	public void setTile(Pixmap tile) {
		gdk_gc_set_tile(getHandle(), tile.getHandle());
	}
	
	/**
	 * Sets the stipple bitmap for a graphics context.  The stipple will only be
	 * used if the fill mode is STIPPLED or OPAQUE_STIPPLED.
	 */
	public void setStipple(Pixmap stipple) {
		gdk_gc_set_stipple(getHandle(), stipple.getHandle());
	}
	
	/**
	 * Sets the origin when using tiles or stipples with the GC.  The tile
	 * or stipple will be aligned such that the upper left cornor of the tile
	 * or stipple will coincide with this point.
	 */
	public void setOrigin(int x, int y) {
		gdk_gc_set_ts_origin(getHandle(), x, y);
	}
	
	/**
	 * Sets the origin of the clip mask.  The coordinates are interpreted
	 * relative to the upper-left cornor of the destination drawable of
	 * the current operation.
	 */
	public void setClipOrigin(int x, int y) {
		gdk_gc_set_clip_origin(getHandle(), x, y);
	}
	
	/**
	 * Sets the clip mask for a graphics context from a rectangle.
	 */
	public void setClipRectangle(Rectangle rect) {
		gdk_gc_set_clip_rectangle(getHandle(), rect.getHandle());
	}
	
	/**
	 * Sets the clip mask for a graphics context from a region.
	 */
	public void setClipRegion(Region reg) {
		gdk_gc_set_clip_region(getHandle(), reg.getHandle());
	}

	/**
	 * Sets the clip mask for a graphics context from a bitmap.
	 */
	public void setClipMask(Bitmap mask) {
		gdk_gc_set_clip_mask(getHandle(), mask.getHandle());
	}

	/**
	 * Sets how drawing with this GC on a window will affect
	 * child windows.
	 */
	public void setSubwindow(SubwindowMode mode) {
		gdk_gc_set_subwindow(getHandle(), mode.getValue());
	}	
	
	/**
	 * Sets whether copying non-visible portions of a drawable
	 * using this graphics context generate exposure events for
	 * the corresponding regions of the destination drawable.
	 * 
	 * @see Drawable#drawDrawable
	 */
	
	public void setExposures(boolean exposures) {
		gdk_gc_set_exposures(getHandle(), exposures);
	}
	
	/**
	 * Sets various attributes of how lines are drawn. <!-- See
	 * the corresponding members of GdkGCValues for full
	 * explanations of the arguments. --
	 * 
	 * @param lineWidth the width of lines.
	 * @param lineStyle the dash-style for lines.
	 * @param capStyle the manner in which the ends of lines are drawn.
	 * @param joinStyle the manner in which lines are joined together.
	 */
	public void setLineAttributes(int lineWidth, LineStyle lineStyle, CapStyle capStyle, JoinStyle joinStyle) {
		gdk_gc_set_line_attributes(getHandle(), lineWidth, lineStyle.getValue(), capStyle.getValue(), joinStyle.getValue());
	}
	
	/**
	 * Sets the way dashed-lines are drawn.  Lines will be drawn
	 * with alternating on and off segments of the lengths
	 * specified in dash_list.  The manner in which the on and off
	 * segments are drawn is determined by the line style.
	 * 
	 * @param dashOffset the phase of the pattern for the dashed
	 * line-style you want to see
	 * @param dashList the dash-list for the dashed line-style you
	 * want to set.
	 * @see setLineAttributes 
	 */
	public void setDashes(int dashOffset, int[] dashList) {
		gdk_gc_set_dashes(getHandle(), dashOffset, dashList, dashList.length);
	}
	
	/**
	 * Offset attributes such as the clip and tile-stipple origins
	 * of the GC so that drawing at X - x_offset, y - y_offset
	 * with the offset GC has the same effect as drawing at x, y
	 * with the original GC.
	 * 
	 * @param xOffset amount by which to offset the GC in the X direction
	 * @param yOffset amount by which to offset the GC in the Y direction
	 */
	public void offset(int xOffset, int yOffset) {
		gdk_gc_offset(getHandle(), xOffset, yOffset);
	}
	
	/**
	 * Copy the set of values (settings) from another graphics context.
	 * 
	 * @param source the source graphics context.
	 */
	public void copy(GC source) {
		gdk_gc_copy(getHandle(), source.getHandle());
	}
	
	/**
	 * Sets the colormap to the given colormap.  The depth of the
	 * colormap's visual must match the depth of the drawable for
	 * which the GC was created.
	 */
	public void setColormap(Colormap colormap) {
		gdk_gc_set_colormap(getHandle(), colormap.getHandle());
	}
	
	/**
	 * Gets the colormap, if it exists.  A GC will have a colormap
	 * if the drawable for which it was created has a colormap, or
	 * if a colormap was set explicitely with setColormap().
	 */
	public Colormap getColormap() {
		int colormapHandle = gdk_gc_get_colormap(getHandle());
		if (colormapHandle == 0)
			return null;
		GObject obj = retrieveGObject(colormapHandle);
		if (null != obj)
			return (Colormap)obj;
		return new Colormap(colormapHandle);
	}

    /****************************************
     * BEGINNING OF GENERATED CODE
     ****************************************/
    native static final protected int getClipXOrigin (int obj);
    native static final protected int getClipYOrigin (int obj);
    native static final protected int getTsXOrigin (int obj);
    native static final protected int getTsYOrigin (int obj);
    native static final protected int gdk_gc_get_type ();
    native static final protected int gdk_gc_new (int drawable);
    native static final protected void gdk_gc_set_foreground (int gc, int color);
    native static final protected void gdk_gc_set_background (int gc, int color);
    native static final protected void gdk_gc_set_function (int gc, int function);
    native static final protected void gdk_gc_set_fill (int gc, int fill);
    native static final protected void gdk_gc_set_tile (int gc, int tile);
    native static final protected void gdk_gc_set_stipple (int gc, int stipple);
    native static final protected void gdk_gc_set_ts_origin (int gc, int x, int y);
    native static final protected void gdk_gc_set_clip_origin (int gc, int x, int y);
    native static final protected void gdk_gc_set_clip_mask (int gc, int mask);
    native static final protected void gdk_gc_set_clip_rectangle (int gc, int rectangle);
    native static final protected void gdk_gc_set_clip_region (int gc, int region);
    native static final protected void gdk_gc_set_subwindow (int gc, int mode);
    native static final protected void gdk_gc_set_exposures (int gc, boolean exposures);
    native static final protected void gdk_gc_set_line_attributes (int gc, int lineWidth, int 
        lineStyle, int capStyle, int joinStyle);
    native static final protected void gdk_gc_set_dashes (int gc, int dashOffset, int [] dashList, 
        int n);
    native static final protected void gdk_gc_offset (int gc, int xOffset, int yOffset);
    native static final protected void gdk_gc_copy (int dstGC, int srcGC);
    native static final protected void gdk_gc_set_colormap (int gc, int colormap);
    native static final protected int gdk_gc_get_colormap (int gc);
    native static final protected void gdk_gc_set_rgb_fg_color (int gc, int color);
    native static final protected void gdk_gc_set_rgb_bg_color (int gc, int color);
    /****************************************
     * END OF GENERATED CODE
     ****************************************/
}

