/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.melange.jvmmodel;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import fr.inria.diverse.melange.ast.ModelingElementExtensions;
import fr.inria.diverse.melange.ast.NamingHelper;
import fr.inria.diverse.melange.lib.EcoreExtensions;
import fr.inria.diverse.melange.metamodel.melange.ModelingElement;
import fr.inria.diverse.melange.utils.TypeReferencesHelper;
import java.util.List;
import org.eclipse.emf.codegen.ecore.genmodel.GenClass;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtend2.lib.StringConcatenationClient;
import org.eclipse.xtext.common.types.JvmAnnotationReference;
import org.eclipse.xtext.common.types.JvmFormalParameter;
import org.eclipse.xtext.common.types.JvmOperation;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.xbase.jvmmodel.JvmAnnotationReferenceBuilder;
import org.eclipse.xtext.xbase.jvmmodel.JvmTypeReferenceBuilder;
import org.eclipse.xtext.xbase.jvmmodel.JvmTypesBuilder;
import org.eclipse.xtext.xbase.lib.ExclusiveRange;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * A collection of utilities around JVM model generation
 * to ease some common tasks.
 */
@Singleton
@SuppressWarnings("all")
public class JvmModelInferrerHelper {
  @Inject
  private JvmTypeReferenceBuilder.Factory typeBuilderFactory;

  @Inject
  private JvmAnnotationReferenceBuilder.Factory annotationBuilderFactory;

  @Inject
  @Extension
  private JvmTypeReferenceBuilder typeBuilder;

  @Inject
  @Extension
  private JvmAnnotationReferenceBuilder annotationBuilder;

  @Inject
  @Extension
  private JvmTypesBuilder _jvmTypesBuilder;

  @Inject
  @Extension
  private TypeReferencesHelper _typeReferencesHelper;

  @Inject
  @Extension
  private NamingHelper _namingHelper;

  @Inject
  @Extension
  private ModelingElementExtensions _modelingElementExtensions;

  @Inject
  @Extension
  private EcoreExtensions _ecoreExtensions;

  /**
   * Should be invoked first to inject the required dependencies as the
   * framework doesn't know anything about this class and won't inject
   * it automatically.
   */
  public void setContext(final ResourceSet rs) {
    this.typeBuilder = this.typeBuilderFactory.create(rs);
    this.annotationBuilder = this.annotationBuilderFactory.create(rs);
  }

  /**
   * Returns the {@link JvmOperation} defining the getter signature
   * of the {@link EStructuralFeature} {@code f} whose type is {@code type}
   * in the context of {@code o}.
   */
  public JvmOperation toGetterSignature(final EObject o, final EStructuralFeature f, final JvmTypeReference type) {
    final Procedure1<JvmOperation> _function = (JvmOperation it) -> {
      it.setAbstract(true);
    };
    return this._jvmTypesBuilder.toMethod(o, this._namingHelper.getGetterName(f), type, _function);
  }

  /**
   * Returns the {@link JvmOperation} defining the setter signature
   * of the {@link EStructuralFeature} {@code f} whose type is {@code type}
   * in the context of {@code o}.
   */
  public JvmOperation toSetterSignature(final EObject o, final EStructuralFeature f, final JvmTypeReference type) {
    final Procedure1<JvmOperation> _function = (JvmOperation it) -> {
      it.setAbstract(true);
      EList<JvmFormalParameter> _parameters = it.getParameters();
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("new");
      String _firstUpper = StringExtensions.toFirstUpper(f.getName());
      _builder.append(_firstUpper);
      JvmFormalParameter _parameter = this._jvmTypesBuilder.toParameter(o, _builder.toString(), type);
      this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, _parameter);
    };
    return this._jvmTypesBuilder.toMethod(o, this._namingHelper.getSetterName(f), this.typeBuilder.typeRef(Void.TYPE), _function);
  }

  /**
   * Returns the {@link JvmOperation} defining the unsetter ({@code eUnset()})
   * of the {@link EStructuralFeature} {@code f} in the context of {@code o}.
   */
  public JvmOperation toUnsetter(final EObject o, final EStructuralFeature f, final ModelingElement m) {
    final GenClass genCls = this._modelingElementExtensions.getGenClsFor(m, f.getEContainingClass());
    final Procedure1<JvmOperation> _function = (JvmOperation it) -> {
      boolean _needsUnsetterInterface = this._ecoreExtensions.needsUnsetterInterface(f);
      if (_needsUnsetterInterface) {
        EList<JvmAnnotationReference> _annotations = it.getAnnotations();
        JvmAnnotationReference _annotationRef = this.annotationBuilder.annotationRef(Override.class);
        this._jvmTypesBuilder.<JvmAnnotationReference>operator_add(_annotations, _annotationRef);
      }
      StringConcatenationClient _client = new StringConcatenationClient() {
        @Override
        protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
          {
            boolean _needsUnsetterInterface = JvmModelInferrerHelper.this._ecoreExtensions.needsUnsetterInterface(f);
            if (_needsUnsetterInterface) {
              _builder.append("adaptee.");
              String _unsetterName = JvmModelInferrerHelper.this._namingHelper.getUnsetterName(f);
              _builder.append(_unsetterName);
              _builder.append("() ;");
              _builder.newLineIfNotEmpty();
            } else {
              _builder.append("((");
              String _qualifiedClassName = genCls.getQualifiedClassName();
              _builder.append(_qualifiedClassName);
              _builder.append(") adaptee).");
              String _unsetterName_1 = JvmModelInferrerHelper.this._namingHelper.getUnsetterName(f);
              _builder.append(_unsetterName_1);
              _builder.append("() ;");
              _builder.newLineIfNotEmpty();
            }
          }
        }
      };
      this._jvmTypesBuilder.setBody(it, _client);
    };
    return this._jvmTypesBuilder.toMethod(o, this._namingHelper.getUnsetterName(f), this.typeBuilder.typeRef(Void.TYPE), _function);
  }

  /**
   * Returns the {@link JvmOperation} defining the unsetter check
   * ({@code eIsSet()}) of the {@link EStructuralFeature} {@code f}
   * in the context of {@code o}.
   */
  public JvmOperation toUnsetterCheck(final EObject o, final EStructuralFeature f, final ModelingElement m) {
    final GenClass genCls = this._modelingElementExtensions.getGenClsFor(m, f.getEContainingClass());
    final Procedure1<JvmOperation> _function = (JvmOperation it) -> {
      boolean _needsUnsetterInterface = this._ecoreExtensions.needsUnsetterInterface(f);
      if (_needsUnsetterInterface) {
        EList<JvmAnnotationReference> _annotations = it.getAnnotations();
        JvmAnnotationReference _annotationRef = this.annotationBuilder.annotationRef(Override.class);
        this._jvmTypesBuilder.<JvmAnnotationReference>operator_add(_annotations, _annotationRef);
      }
      StringConcatenationClient _client = new StringConcatenationClient() {
        @Override
        protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
          {
            boolean _needsUnsetterInterface = JvmModelInferrerHelper.this._ecoreExtensions.needsUnsetterInterface(f);
            if (_needsUnsetterInterface) {
              _builder.append("return adaptee.");
              String _unsetterCheckName = JvmModelInferrerHelper.this._namingHelper.getUnsetterCheckName(f);
              _builder.append(_unsetterCheckName);
              _builder.append("() ;");
              _builder.newLineIfNotEmpty();
            } else {
              _builder.append("return ((");
              String _qualifiedClassName = genCls.getQualifiedClassName();
              _builder.append(_qualifiedClassName);
              _builder.append(") adaptee).");
              String _unsetterCheckName_1 = JvmModelInferrerHelper.this._namingHelper.getUnsetterCheckName(f);
              _builder.append(_unsetterCheckName_1);
              _builder.append("() ;");
              _builder.newLineIfNotEmpty();
            }
          }
        }
      };
      this._jvmTypesBuilder.setBody(it, _client);
    };
    return this._jvmTypesBuilder.toMethod(o, this._namingHelper.getUnsetterCheckName(f), this.typeBuilder.typeRef(Boolean.TYPE), _function);
  }

  /**
   * Returns the {@link JvmOperation} defining the unsetter signature
   * ({@code eUnset()}) of the {@link EStructuralFeature} {@code f}
   * in the context of {@code o}.
   */
  public JvmOperation toUnsetterSignature(final EObject o, final EStructuralFeature f) {
    final Procedure1<JvmOperation> _function = (JvmOperation it) -> {
      it.setAbstract(true);
    };
    return this._jvmTypesBuilder.toMethod(o, this._namingHelper.getUnsetterName(f), this.typeBuilder.typeRef(Void.TYPE), _function);
  }

  /**
   * Returns the {@link JvmOperation} defining the unsetter check signature
   * ({@code eIsSet()}) of the {@link EStructuralFeature} {@code f}
   * in the context of {@code o}.
   */
  public JvmOperation toUnsetterCheckSignature(final EObject o, final EStructuralFeature f) {
    final Procedure1<JvmOperation> _function = (JvmOperation it) -> {
      it.setAbstract(true);
    };
    return this._jvmTypesBuilder.toMethod(o, this._namingHelper.getUnsetterCheckName(f), this.typeBuilder.typeRef(Boolean.TYPE), _function);
  }

  /**
   * Checks whether the {@link JvmOperation} {@code op1} overrides (in the
   * Java sense) the {@link JvmOperation} {@code op2}
   */
  public boolean overrides(final JvmOperation o1, final JvmOperation o2) {
    return ((Objects.equal(o1.getSimpleName(), o2.getSimpleName()) && (((o1.getReturnType() == null) && (o2.getReturnType() == null)) || this._typeReferencesHelper.isSubtypeOf(o2.getReturnType(), o1.getReturnType()))) && this.parameterEquals(o1.getParameters(), o2.getParameters()));
  }

  private boolean parameterEquals(final List<JvmFormalParameter> p1, final List<JvmFormalParameter> p2) {
    int _size = p1.size();
    int _size_1 = p2.size();
    boolean _notEquals = (_size != _size_1);
    if (_notEquals) {
      return false;
    }
    int _size_2 = p1.size();
    ExclusiveRange _doubleDotLessThan = new ExclusiveRange(0, _size_2, true);
    for (final Integer i : _doubleDotLessThan) {
      String _qualifiedName = p1.get((i).intValue()).getParameterType().getQualifiedName();
      String _qualifiedName_1 = p2.get((i).intValue()).getParameterType().getQualifiedName();
      boolean _notEquals_1 = (!Objects.equal(_qualifiedName, _qualifiedName_1));
      if (_notEquals_1) {
        return false;
      }
    }
    return true;
  }
}
