/* MakeTeXFMT.c  -- Make a TeX format file
   Time-stamp: "97/08/12 20:46:45 mik"

   Copyright (C) 1995, 96, 97
	Christian Schenk  <cschenk@berlin.snafu.de>

   This file is part of MiKTeX.

   MiKTeX is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
   
   MiKTeX is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with MiKTeX; if not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

#include <stdlib.h>
#include <stdio.h>
#include <getopt.h>
#include <io.h>
#include <direct.h>
#include <process.h>

#include "miktex.h"
#include "MakeTeXFMT.rc"

#if defined (_WIN32)
#  define WIN32_LEAN_AND_MEAN
#  include <windows.h>
#endif /* _WIN32 */

#define PATH_SEP_CHAR ';'
#define isslash(c) ((c) == '\\' || (c) == '/')

static int		verbose_p;
static int		debug_p;
static int		quiet_p;
static int		print_only_p;

static const char *	destdir;

static char *		name;

/* _________________________________________________________________________

   USAGE / HELP
   _________________________________________________________________________ */


static void
show_version (void)

{
  printf ("This is MakeTeXFMT version %s (%s %s)\n",
	  VER_FILEVERSION_STR, VER_PRODUCTNAME_STR,
	  VER_PRODUCTVERSION_STR);
}

static void
usage (void)

{
  puts ("Usage: MakeTeXFMT [OPTION]... name");
  puts ("");
  puts ("This program makes a TeX format file.");
  puts ("");
  puts ("NAME is the name of the format, such as `latex'.");
  puts ("");
  puts ("Options:");
  puts ("--debug, -d                         Print debugging information.");
  puts ("--dest-dir DIRECTORY, -D DIRECTORY  Specify the destination fmt directory.");
  puts ("--help, -h                          Print this help screen and exit.");
  puts ("--print-only, -n                    Print what commands would be executed.");
  puts ("--verbose, -v                       Print information on what is being done.");
  puts ("--version, -V                       Print the version number and exit.");
  puts ("");
  puts ("Email problem reports to cschenk@berlin.snafu.de");
}

/* _________________________________________________________________________

   GETTING COMMAND LINE OPTIONS.
   _________________________________________________________________________ */

static struct option	long_options[] =

{
  "debug",		no_argument,		0,	'd',
  "help",		no_argument,		0,	'h',
  "print-only",		no_argument,		0,	'n',
  "dest-dir",		required_argument,	0,	'D',
  "quiet",		no_argument,		0,	'q',
  "verbose",		no_argument,		0,	'v',
  "version",		no_argument,		0,	'V',
  0,			no_argument,		0,	0,
};

static void
get_options (int	argc,
	     char **	argv)

{
  int option_index;
  int c;

  destdir = 0;
  print_only_p = 0;
  quiet_p = 0;
#if defined (DEBUG)
  verbose_p = 1;
  debug_p = 1;
#else
  verbose_p = 0;
  debug_p = 0;
#endif

  while ((c=getopt_long (argc, argv, "dD:hnqvV", long_options, &option_index))
	 != EOF)
    {
      switch (c)
	{
	case 'h':
	  usage ();
	  exit (0);
	  break;
	case 'n':
	  print_only_p = 1;
	  break;
	case 'D':
	  destdir = strdup (optarg);
	  break;
	case 'd':
	  debug_p = 1;
	  break;
	case 'v':
	  verbose_p = 1;
	  break;
	case 'q':
	  quiet_p = 1;
	  break;
	case 'V':
	  show_version ();
	  exit (0);
	  break;
	default:
	  usage ();
	  exit (1);
	  break;
	}
    }
  if (argc - optind < 1)
    {
      usage ();
      exit (1);
    }

  name = strdup (argv[optind++]);

  if (optind != argc)
    {
      usage ();
      exit (1);
    }
}

/* _________________________________________________________________________

   CREATING THE DESTINATION DIRECTORY.
   _________________________________________________________________________ */


static char *
create_dest_directory ()

{
  char dest_directories[_MAX_PATH];
  char *destdir;
  char *next_destdir;
  char *result;

  get_cfg_value ("TeX", "Format Temp Dir",
		 dest_directories, sizeof (dest_directories),
		 "%R\\miktex\\fmt");
  
  /* Initialize search. */
  destdir = dest_directories;
  next_destdir = strchr (destdir, PATH_SEP_CHAR);
  if (next_destdir)
    *next_destdir++ = 0; 

  result = 0;
  while (destdir && result == 0)
    {
      if (destdir[0] == '%' && destdir[1] == 'R' && isslash (destdir[2]))
	{
	  char alt_destdir[_MAX_PATH];
	  size_t i;

	  for (i = 0; i < get_number_of_texmf_roots () && result == 0; i++)
	    {
	      xassert (get_root_directory (i) != 0);
	      _makepath (alt_destdir, 0, get_root_directory (i),
			 &destdir[3], 0);
	      if (print_only_p)
		{
		  printf ("mkdir %s\n", alt_destdir);
		  result = strdup (alt_destdir);
		}
	      else if (make_path_ex (alt_destdir, 0))
		result = strdup (alt_destdir);
	    }
	}
      else if (print_only_p)
	{
	  printf ("mkdir %s\n", destdir);
	  result = strdup (destdir);
	}
      else if (make_path_ex (destdir, 0))
	result = strdup (destdir);
      
      if (next_destdir && *next_destdir)
	{
	  destdir = next_destdir;
	  next_destdir = strchr (next_destdir, PATH_SEP_CHAR);
	  if (next_destdir)
	    *next_destdir++ = 0;
	}
      else
	destdir = 0;
    }

  if (result == 0)
    {
      fprintf (stderr, "Don't know where to install the format file!\n");
      exit (1);
    }

  return (result);
}

/* _________________________________________________________________________

   MAKING FILE NAMES.
   _________________________________________________________________________ */


static char *
make_fmt_filename (const char *	name)

{
  char result[_MAX_PATH];
  sprintf (result, "%s.fmt", name);
  return (strdup (result));
}

static char *
make_dest_path (const char *	destdir,
		const char *	fmtname)

{
  char result[_MAX_PATH];
  sprintf (result, "%s\\%s", destdir, fmtname);
  return (strdup (result));
}

/* _________________________________________________________________________

   INVOKING INITEX.
   _________________________________________________________________________ */


static int
invoke_initex (const char *	fmtname,
	       const char *	filename)

{
  char command_line[2 * _MAX_PATH + 100];
  int pargc;
  const char *pargv[20];
  int rc;

  if (! find_executable ("initex.exe", command_line))
    {
      fprintf (stderr, "initex.exe not found!\n");
      if (debug_p)
	fprintf (stderr, "Intermediate files were *not* removed!\n");
      else
	leave_temporary_directory ();
      exit (1);
    }

  pargc = 0;
  pargv[pargc++] = command_line;
  pargv[pargc++] = filename;
  pargv[pargc++] = "\\dump";
  pargv[pargc++] = 0;

  if (! quiet_p || print_only_p)
    printf ("%s %s \\dump\n", command_line, filename);

  if (print_only_p)
    rc = 0;
  else
    {
      _flushall ();
      rc = _spawnv (_P_WAIT, command_line, pargv);
    }

  if (rc < 0)
    {
      perror (command_line);
      leave_temporary_directory ();
      exit (1);
    }

  if (rc > 0)
    {
      fprintf (stderr, "IniTeX failed on %s! Return code: %d\n", name, rc);
      if (debug_p)
	fprintf (stderr, "Intermediate files were *not* removed!\n");
      else
	leave_temporary_directory ();
      exit (1);
    }

  return (rc);
}

/* _________________________________________________________________________

   INSTALLING FORMAT FILES.
   _________________________________________________________________________ */


static void
install_fmt_file (const char *	fmtname,
		  const char *	destdir,
		  const char *	destname)

{
  if (! print_only_p && _access (fmtname, 0) < 0)
    {
      fprintf (stderr, "Cannot access FMT file %s\n", fmtname);
      if (debug_p)
	fprintf (stderr, "Intermediate files were *not* removed!\n");
      else
	leave_temporary_directory ();
      exit (1);
    }
  if (print_only_p || verbose_p)
    printf ("move %s %s\n", fmtname, destname);
  if (! print_only_p)
    {
      if (make_path (destdir))
	{
	  if (_access (destname, 0) == 0)
	    {
	      if (_unlink (destname) != 0)
		{
		  perror (destname);
		  exit (1);
		}
	    }
	  if (rename (fmtname, destname) != 0)
	    {
	      perror (fmtname);
	      leave_temporary_directory ();
	      exit (1);
	    }
	}
    }
}

static char *filename_extensions [] =

{
  ".tex", ".ltx", 0
};

static char *
get_input_filename (const char *format_name)

{
  char search_path[_MAX_PATH * 3];
  char filename[_MAX_FNAME];
  char full_filename[_MAX_PATH];
  char *cp;
  size_t i;

  strcpy (filename, format_name);
  cp = filename + strlen (filename);

  get_search_path ("TeX", "Input Dirs", search_path, sizeof (search_path),
		   ".;%R\tex//");
  
  for (i = 0; filename_extensions[i] != 0; i++)
    {
      strcpy (cp, filename_extensions[i]);
      if (find_file (filename, search_path, full_filename))
	return (strdup (full_filename));
    }

  return (0);
}

/* _________________________________________________________________________

   MAIN.
   _________________________________________________________________________ */


int
main (int	argc,
      char **	argv)

{
  char *fmtname;
  char *destname;
  char *inputname;

  /* Get command line options. */
  c4pargv[0] = argv[0];
  get_options (argc, argv);

  /* Figure out the extension of the source file. */
  inputname = get_input_filename (name);

  /* Make destination directory name if none was specified. */
  if (destdir == 0)
    destdir = create_dest_directory ();

  /* Make FMT file name. */
  fmtname = make_fmt_filename (name);

  /* Make full destination path name. */
  destname = make_dest_path (destdir, fmtname);

  if (verbose_p && ! print_only_p)
    printf ("Creating %s...\n", fmtname);

  /* Create temporary directory; make it current. */
  enter_temporary_directory ("fmt", print_only_p);

  /* Invoke IniTeX to make a format file. */
  invoke_initex (name, inputname);

  /* Install format file in destination directory. */
  install_fmt_file (fmtname, destdir, destname);

  /* Remove temporary directory. */
  leave_temporary_directory ();

  return (0);
}

/* MakeTeXFMT.c ends here */
