/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.maven.plugin.compiler;

import javax.lang.model.SourceVersion;
import javax.tools.DiagnosticListener;
import javax.tools.JavaCompiler;
import javax.tools.JavaFileObject;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.io.Writer;
import java.lang.module.ModuleDescriptor;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.TreeMap;
import java.util.stream.Stream;

import static org.apache.maven.plugin.compiler.AbstractCompilerMojo.SUPPORT_LEGACY;
import static org.apache.maven.plugin.compiler.DirectoryHierarchy.META_INF;
import static org.apache.maven.plugin.compiler.SourceDirectory.CLASS_FILE_SUFFIX;
import static org.apache.maven.plugin.compiler.SourceDirectory.MODULE_INFO;

/**
 * A task which configures and executes the Java compiler for the test classes.
 * This executor contains additional configurations compared to the base class.
 *
 * @author Martin Desruisseaux
 */
class ToolExecutorForTest extends ToolExecutor {
    /**
     * The output directory of the main classes.
     * This directory will be added to the class-path or module-path.
     *
     * @see TestCompilerMojo#mainOutputDirectory
     */
    private final Path mainOutputDirectory;

    /**
     * The main output directory of each module. This is usually {@code mainOutputDirectory/<module>},
     * except if some modules are defined only for some Java versions higher than the base version.
     */
    private final Map<String, Path> mainOutputDirectoryForModules;

    /**
     * Whether to place the main classes on the module path when {@code module-info} is present.
     * The default and recommended value is {@code true}. The user may force to {@code false},
     * in which case the main classes are placed on the class path, but this is deprecated.
     * This flag may be removed in a future version if we remove support of this practice.
     *
     * @deprecated Use {@code "claspath-jar"} dependency type instead, and avoid {@code module-info.java} in tests.
     *
     * @see TestCompilerMojo#useModulePath
     */
    @Deprecated(since = "4.0.0")
    private final boolean useModulePath;

    /**
     * Whether a {@code module-info.java} file is defined in the test sources.
     * In such case, it has precedence over the {@code module-info.java} in main sources.
     * This is defined for compatibility with Maven 3, but not recommended.
     *
     * @deprecated Avoid {@code module-info.java} in tests.
     */
    @Deprecated(since = "4.0.0")
    private final boolean hasTestModuleInfo;

    /**
     * Name of the module when using package hierarchy, or {@code null} if not applicable.
     * This is used for setting {@code --patch-module} option during compilation of tests.
     * This field is null in a class-path project or in a multi-module project.
     *
     * <p>This field exists mostly for compatibility with the Maven 3 way to build a modular project.
     * It is recommended to use the {@code <sources>} element instead. We may remove this field in a
     * future version if we abandon compatibility with the Maven 3 way to build modular projects.</p>
     *
     * @deprecated Declare modules in {@code <source>} elements instead.
     */
    @Deprecated(since = "4.0.0")
    private String moduleNameFromPackageHierarchy;

    /**
     * Whether {@link #addModuleOptions(Options)} has already been invoked.
     * The options shall be completed only once, otherwise conflicts may occur.
     */
    private boolean addedModuleOptions;

    /**
     * If non-null, the {@code module} part to remove in {@code target/test-classes/module/package}.
     * This {@code module} directory is generated by {@code javac} for some compiler options.
     * We keep that directory when the project is configured with the new {@code <source>} element,
     * but have to remove it for compatibility reason if the project is compiled in the old way.
     *
     * @deprecated Exists only for compatibility with the Maven 3 way to do a modular project.
     * Is likely to cause confusion, for example with incremental builds.
     * New projects should use the {@code <source>} elements instead.
     */
    @Deprecated(since = "4.0.0")
    private String directoryLevelToRemove;

    /**
     * Creates a new task by taking a snapshot of the current configuration of the given <abbr>MOJO</abbr>.
     * This constructor creates the {@linkplain #outputDirectory output directory} if it does not already exist.
     *
     * @param mojo the <abbr>MOJO</abbr> from which to take a snapshot
     * @param listener where to send compilation warnings, or {@code null} for the Maven logger
     * @param mainModulePath path to the {@code module-info.class} file of the main code, or {@code null} if none
     * @throws MojoException if this constructor identifies an invalid parameter in the <abbr>MOJO</abbr>
     * @throws IOException if an error occurred while creating the output directory or scanning the source directories
     */
    @SuppressWarnings("deprecation")
    ToolExecutorForTest(
            final TestCompilerMojo mojo,
            final DiagnosticListener<? super JavaFileObject> listener,
            final Path mainModulePath)
            throws IOException {
        super(mojo, listener);
        /*
         * Notable work done by the parent constructor (examples with default paths):
         *
         *  - Set `outputDirectory` to a single "target/test-classes".
         *  - Set `sourceDirectories` to many "src/<module>/test/java".
         *  - Set `sourceFiles` to the content of `sourceDirectories`.
         *  - Set `dependencies` with class-path and module-path, but not including main output directory.
         *
         * We will need to add the main output directory to the class-path or module-path, but not here.
         * It will be done by `ToolExecutor.compile(…)` if `getOutputDirectoryOfPreviousPhase()` returns
         * a non-null value.
         */
        useModulePath = mojo.useModulePath;
        hasTestModuleInfo = mojo.hasTestModuleInfo;
        mainOutputDirectory = mojo.mainOutputDirectory;
        mainOutputDirectoryForModules = new HashMap<>();
        if (Files.notExists(mainOutputDirectory)) {
            return;
        }
        if (mainModulePath != null) {
            try (InputStream in = Files.newInputStream(mainModulePath)) {
                moduleNameFromPackageHierarchy = ModuleDescriptor.read(in).name();
            }
        }
        // Following is non-null only for modular project using package hierarchy.
        final String testModuleName = mojo.moduleNameFromPackageHierarchy(sourceDirectories);
        if (testModuleName != null) {
            moduleNameFromPackageHierarchy = testModuleName;
        }
        /*
         * If compiling the test classes of a modular project, we will need `--patch-modules` options.
         * In this case, the option values are directories of main class files of the patched module.
         * This block only prepares an empty map for each module. Maps are filled in the next block.
         */
        final var patchedModules = new LinkedHashMap<String, NavigableMap<SourceVersion, Path>>();
        for (SourceDirectory dir : sourceDirectories) {
            String moduleToPatch = dir.moduleName;
            if (moduleToPatch == null) {
                moduleToPatch = moduleNameFromPackageHierarchy;
                if (moduleToPatch == null) {
                    continue; // No module-info found.
                }
                /*
                 * Modular project using package hierarchy (Maven 3 way).
                 * We will need to move directories after compilation for reproducing the Maven 3 output.
                 */
                directoryLevelToRemove = moduleToPatch;
            }
            if (testModuleName != null && !moduleToPatch.equals(testModuleName)) {
                // Mix of package hierarchy and module source hierarchy.
                throw new CompilationFailureException(
                        "The \"" + testModuleName + "\" module must be declared in a <module> element of <sources>.");
            }
            patchedModules.put(moduleToPatch, new TreeMap<>()); // Signal that this module exists in the test.
        }
        // Shortcut for class-path projects.
        if (patchedModules.isEmpty()) {
            return;
        }
        /*
         * The values of `patchedModules` are empty maps. Now, add the real paths to the
         * main classes for each module that exists in both the main code and the tests.
         * Note that a module may exist only in the `META-INF/versions-modular/` directory.
         */
        addDirectoryIfModule(
                mainOutputDirectory, moduleNameFromPackageHierarchy, SourceVersion.RELEASE_0, patchedModules);
        addModuleDirectories(mainOutputDirectory, SourceVersion.RELEASE_0, patchedModules);
        Path versionsDirectory = DirectoryHierarchy.MODULE_SOURCE.outputDirectoryForReleases(mainOutputDirectory);
        if (Files.exists(versionsDirectory)) {
            List<Path> asList;
            try (Stream<Path> paths = Files.list(versionsDirectory)) {
                asList = paths.toList();
            }
            for (Path path : asList) {
                SourceVersion version;
                try {
                    version = SourceDirectory.parse(path.getFileName().toString());
                } catch (UnsupportedVersionException e) {
                    logger.debug(e);
                    continue;
                }
                addModuleDirectories(path, version, patchedModules);
            }
        }
        /*
         * At this point, we finished to scan the main output directory for modules.
         * Remembers the directories of each module. They are usually sub-directories
         * of the main directory, but could also be in `META-INF/versions-modular/`.
         */
        patchedModules.forEach((moduleToPatch, directories) -> {
            Map.Entry<SourceVersion, Path> base = directories.firstEntry();
            if (base != null) {
                mainOutputDirectoryForModules.putIfAbsent(moduleToPatch, base.getValue());
            }
        });
    }

    /**
     * Performs a shallow scan of the given directory for modules.
     * This method searches for {@code module-info.class} files.
     *
     * <p>The keys of the {@code addTo} map are module names. Values are paths for all versions where
     * {@code module-info.class} has been found. Note that this is not an exhaustive list of paths for
     * all versions, because most {@code versions} directories do not have a {@code module-info.class} file.
     * Therefore, the {@code SortedMap} will usually contain only the base directory. But we check versions
     * anyway because sometime, a module does not exist in the base directory and is first defined only for
     * a higher version.</p>
     *
     * <p>This method adds paths to existing entries only, and ignores modules that are not already in the map.
     * This is done that way for collecting modules that are both in the main code and in the tests.</p>
     *
     * @param directory the directory to scan
     * @param version target Java version of the directory to add
     * @param addTo where to add the module paths
     * @throws IOException if an error occurred while scanning the directories
     */
    private void addModuleDirectories(
            Path directory, SourceVersion version, Map<String, NavigableMap<SourceVersion, Path>> addTo)
            throws IOException {

        try (Stream<Path> paths = Files.list(directory)) {
            paths.forEach(
                    (path) -> addDirectoryIfModule(path, path.getFileName().toString(), version, addTo));
        } catch (UncheckedIOException e) {
            throw e.getCause();
        }
    }

    /**
     * Adds the given directory in {@code addTo} if the directory contains a {@code module-info.class} file.
     *
     * @param directory the directory to scan
     * @param moduleName name of the module to add
     * @param version target Java version of the directory to add
     * @param addTo where to add the module paths
     */
    private static void addDirectoryIfModule(
            Path directory,
            String moduleName,
            SourceVersion version,
            Map<String, NavigableMap<SourceVersion, Path>> addTo) {

        NavigableMap<SourceVersion, Path> versions = addTo.get(moduleName);
        if (versions != null && Files.isRegularFile(directory.resolve(MODULE_INFO + CLASS_FILE_SUFFIX))) {
            versions.putIfAbsent(version, directory);
        }
    }

    /**
     * Completes the given configuration with module options the first time that this method is invoked.
     * If at least one {@value ModuleInfoPatch#FILENAME} file is found in a root directory of test sources,
     * then these files are parsed and the options that they declare are added to the given configuration.
     * Otherwise, if {@link #hasModuleDeclaration} is {@code true}, then this method generates the
     * {@code --add-modules} and {@code --add-reads} options for dependencies that are not in the main compilation.
     * If this method is invoked more than once, all invocations after the first one have no effect.
     *
     * @param configuration where to add the options
     * @throws IOException if the module information of a dependency or the module-info patch cannot be read
     */
    @SuppressWarnings({"checkstyle:MissingSwitchDefault", "fallthrough"})
    private void addModuleOptions(final Options configuration) throws IOException {
        if (addedModuleOptions) {
            return;
        }
        addedModuleOptions = true;
        ModuleInfoPatch info = null;
        ModuleInfoPatch defaultInfo = null;
        final var patches = new LinkedHashMap<String, ModuleInfoPatch>();
        for (SourceDirectory source : sourceDirectories) {
            Path file = source.root.resolve(ModuleInfoPatch.FILENAME);
            String moduleName;
            if (Files.notExists(file)) {
                if (SUPPORT_LEGACY && useModulePath && hasTestModuleInfo && hasModuleDeclaration) {
                    /*
                     * Do not add any `--add-reads` parameters. The developers should put
                     * everything needed in the `module-info`, including test dependencies.
                     */
                    continue;
                }
                /*
                 * No `patch-module-info` file. Generate a default module patch instance for the
                 * `--add-modules TEST-MODULE-PATH` and `--add-reads TEST-MODULE-PATH` options.
                 * We generate that patch only for the first module. If there is more modules
                 * without `patch-module-info`, we will copy the `defaultInfo` instance.
                 */
                moduleName = source.moduleName;
                if (moduleName == null) {
                    moduleName = moduleNameFromPackageHierarchy;
                    if (moduleName == null) {
                        continue;
                    }
                }
                if (defaultInfo != null) {
                    patches.putIfAbsent(moduleName, null); // Remember that we will need to compute a value later.
                    continue;
                }
                defaultInfo = new ModuleInfoPatch(moduleName, info);
                defaultInfo.setToDefaults();
                info = defaultInfo;
            } else {
                info = new ModuleInfoPatch(moduleNameFromPackageHierarchy, info);
                try (BufferedReader reader = Files.newBufferedReader(file)) {
                    info.load(reader);
                }
                moduleName = info.getModuleName();
            }
            if (patches.put(moduleName, info) != null) {
                throw new ModuleInfoPatchException(
                        "\"module-info-patch " + moduleName + "\" is defined more than once.");
            }
        }
        /*
         * Replace all occurrences of `TEST-MODULE-PATH` by the actual dependency paths.
         * Add `--add-modules` and `--add-reads` options with default values equivalent to
         * `TEST-MODULE-PATH` for every module that do not have a `module-info-patch` file.
         */
        for (Map.Entry<String, ModuleInfoPatch> entry : patches.entrySet()) {
            info = entry.getValue();
            if (info != null) {
                info.replaceProjectModules(sourceDirectories);
                info.replaceTestModulePath(dependencyResolution);
            } else {
                // `defaultInfo` cannot be null if this `info` value is null.
                entry.setValue(defaultInfo.patchWithSameReads(entry.getKey()));
            }
        }
        /*
         * Write the runtime dependencies in the `META-INF/maven/module-info-patch.args` file.
         * Note that we unconditionally write in the root output directory, not in the module directory,
         * because a single option file applies to all modules.
         */
        if (!patches.isEmpty()) {
            Path directory = // TODO: replace by Path.resolve(String, String...) with JDK22.
                    Files.createDirectories(outputDirectory.resolve(META_INF).resolve("maven"));
            try (BufferedWriter out = Files.newBufferedWriter(directory.resolve("module-info-patch.args"))) {
                for (ModuleInfoPatch m : patches.values()) {
                    m.writeTo(configuration, out);
                }
            }
        }
    }

    /**
     * @hidden
     */
    @Override
    public boolean applyIncrementalBuild(AbstractCompilerMojo mojo, Options configuration) throws IOException {
        addModuleOptions(configuration); // Effective only once.
        return super.applyIncrementalBuild(mojo, configuration);
    }

    /**
     * @hidden
     */
    @Override
    public boolean compile(JavaCompiler compiler, Options configuration, Writer otherOutput) throws IOException {
        addModuleOptions(configuration); // Effective only once.
        try (var r = ModuleDirectoryRemover.create(outputDirectory, directoryLevelToRemove)) {
            return super.compile(compiler, configuration, otherOutput);
        }
    }

    /**
     * Returns the output directory of the main classes. This is the directory to prepend to
     * the class-path or module-path before to compile the classes managed by this executor.
     *
     * @return the directory to prepend to the class-path or module-path
     */
    @Override
    Path getOutputDirectoryOfPreviousPhase() {
        return mainOutputDirectory;
    }

    /**
     * Returns the directory of the classes compiled for the specified module.
     * If the project is multi-release, this method returns the directory for the base version.
     *
     * @param outputDirectory the output directory which is the root of modules
     * @param moduleName the name of the module for which the class directory is desired
     * @return directories of classes for the given module
     */
    @Override
    Path resolveModuleOutputDirectory(Path outputDirectory, String moduleName) {
        if (outputDirectory.equals(mainOutputDirectory)) {
            Path path = mainOutputDirectoryForModules.get(moduleName);
            if (path != null) {
                return path;
            }
        }
        return super.resolveModuleOutputDirectory(outputDirectory, moduleName);
    }

    /**
     * Name of the module when using package hierarchy, or {@code null} if not applicable.
     * This is null in a class-path project or in a multi-module project.
     *
     * @deprecated This information exists only for compatibility with the Maven 3 way to build a modular project.
     */
    @Override
    @Deprecated(since = "4.0.0")
    final String moduleNameFromPackageHierarchy() {
        return moduleNameFromPackageHierarchy;
    }
}
