/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.io.input;

import com.google.common.base.Stopwatch;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.time.Duration;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Stream;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.input.QueueInputStream;
import org.apache.commons.io.output.QueueOutputStream;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Assumptions;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.DynamicTest;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestFactory;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;

public class QueueInputStreamTest {
    public static Stream<Arguments> inputData() {
        return Stream.of(Arguments.of((Object[])new Object[]{""}), Arguments.of((Object[])new Object[]{"1"}), Arguments.of((Object[])new Object[]{"12"}), Arguments.of((Object[])new Object[]{"1234"}), Arguments.of((Object[])new Object[]{"12345678"}), Arguments.of((Object[])new Object[]{StringUtils.repeat((String)"A", (int)4095)}), Arguments.of((Object[])new Object[]{StringUtils.repeat((String)"A", (int)4096)}), Arguments.of((Object[])new Object[]{StringUtils.repeat((String)"A", (int)4097)}), Arguments.of((Object[])new Object[]{StringUtils.repeat((String)"A", (int)8191)}), Arguments.of((Object[])new Object[]{StringUtils.repeat((String)"A", (int)8192)}), Arguments.of((Object[])new Object[]{StringUtils.repeat((String)"A", (int)8193)}), Arguments.of((Object[])new Object[]{StringUtils.repeat((String)"A", (int)32768)}));
    }

    @TestFactory
    public DynamicTest[] bulkReadErrorHandlingTests() {
        QueueInputStream queueInputStream = new QueueInputStream();
        return new DynamicTest[]{DynamicTest.dynamicTest((String)"Offset too big", () -> Assertions.assertThrows(IndexOutOfBoundsException.class, () -> queueInputStream.read(ArrayUtils.EMPTY_BYTE_ARRAY, 1, 0))), DynamicTest.dynamicTest((String)"Offset negative", () -> Assertions.assertThrows(IndexOutOfBoundsException.class, () -> queueInputStream.read(ArrayUtils.EMPTY_BYTE_ARRAY, -1, 0))), DynamicTest.dynamicTest((String)"Length too big", () -> Assertions.assertThrows(IndexOutOfBoundsException.class, () -> queueInputStream.read(ArrayUtils.EMPTY_BYTE_ARRAY, 0, 1))), DynamicTest.dynamicTest((String)"Length negative", () -> Assertions.assertThrows(IndexOutOfBoundsException.class, () -> queueInputStream.read(ArrayUtils.EMPTY_BYTE_ARRAY, 0, -1)))};
    }

    private int defaultBufferSize() {
        return 8192;
    }

    private void doTestReadLineByLine(String inputData, InputStream inputStream, OutputStream outputStream) throws IOException {
        String[] lines = inputData.split("\n");
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8));){
            for (String line : lines) {
                outputStream.write(line.getBytes(StandardCharsets.UTF_8));
                outputStream.write(10);
                String actualLine = reader.readLine();
                Assertions.assertEquals((Object)line, (Object)actualLine);
            }
        }
    }

    private String readUnbuffered(InputStream inputStream) throws IOException {
        return this.readUnbuffered(inputStream, Integer.MAX_VALUE);
    }

    private String readUnbuffered(InputStream inputStream, int maxBytes) throws IOException {
        if (maxBytes == 0) {
            return "";
        }
        ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        int n = -1;
        while ((n = inputStream.read()) != -1) {
            byteArrayOutputStream.write(n);
            if (byteArrayOutputStream.size() < maxBytes) continue;
            break;
        }
        return byteArrayOutputStream.toString(StandardCharsets.UTF_8.name());
    }

    @ParameterizedTest(name="inputData={0}")
    @MethodSource(value={"inputData"})
    void testAvailableAfterClose(String inputData) throws IOException {
        QueueInputStream shadow;
        LinkedBlockingQueue queue = new LinkedBlockingQueue();
        try (QueueInputStream inputStream = new QueueInputStream(queue);){
            shadow = inputStream;
        }
        Assertions.assertEquals((int)0, (int)shadow.available());
    }

    @ParameterizedTest(name="inputData={0}")
    @MethodSource(value={"inputData"})
    void testAvailableAfterOpen(String inputData) throws IOException {
        LinkedBlockingQueue queue = new LinkedBlockingQueue();
        try (QueueInputStream inputStream = new QueueInputStream(queue);){
            Assertions.assertEquals((int)0, (int)inputStream.available());
            IOUtils.toString((InputStream)inputStream, (Charset)StandardCharsets.UTF_8);
            Assertions.assertEquals((int)0, (int)inputStream.available());
        }
    }

    @ParameterizedTest(name="inputData={0}")
    @MethodSource(value={"inputData"})
    void testBufferedReads(String inputData) throws IOException {
        LinkedBlockingQueue queue = new LinkedBlockingQueue();
        try (BufferedInputStream inputStream = new BufferedInputStream((InputStream)new QueueInputStream(queue));
             QueueOutputStream outputStream = new QueueOutputStream(queue);){
            outputStream.write(inputData.getBytes(StandardCharsets.UTF_8));
            String actualData = IOUtils.toString((InputStream)inputStream, (Charset)StandardCharsets.UTF_8);
            Assertions.assertEquals((Object)inputData, (Object)actualData);
        }
    }

    @ParameterizedTest(name="inputData={0}")
    @MethodSource(value={"inputData"})
    void testBufferedReadWrite(String inputData) throws IOException {
        LinkedBlockingQueue queue = new LinkedBlockingQueue();
        try (BufferedInputStream inputStream = new BufferedInputStream((InputStream)new QueueInputStream(queue));
             BufferedOutputStream outputStream = new BufferedOutputStream((OutputStream)new QueueOutputStream(queue), this.defaultBufferSize());){
            outputStream.write(inputData.getBytes(StandardCharsets.UTF_8));
            outputStream.flush();
            String dataCopy = IOUtils.toString((InputStream)inputStream, (Charset)StandardCharsets.UTF_8);
            Assertions.assertEquals((Object)inputData, (Object)dataCopy);
        }
    }

    @ParameterizedTest(name="inputData={0}")
    @MethodSource(value={"inputData"})
    void testBufferedWrites(String inputData) throws IOException {
        LinkedBlockingQueue queue = new LinkedBlockingQueue();
        try (QueueInputStream inputStream = new QueueInputStream(queue);
             BufferedOutputStream outputStream = new BufferedOutputStream((OutputStream)new QueueOutputStream(queue), this.defaultBufferSize());){
            outputStream.write(inputData.getBytes(StandardCharsets.UTF_8));
            outputStream.flush();
            String actualData = this.readUnbuffered((InputStream)inputStream);
            Assertions.assertEquals((Object)inputData, (Object)actualData);
        }
    }

    @ParameterizedTest(name="inputData={0}")
    @MethodSource(value={"inputData"})
    void testBulkReadWaiting(String inputData) throws IOException {
        Assumptions.assumeFalse((boolean)inputData.isEmpty());
        final CountDownLatch onPollLatch = new CountDownLatch(1);
        final CountDownLatch afterWriteLatch = new CountDownLatch(1);
        LinkedBlockingQueue<Integer> queue = new LinkedBlockingQueue<Integer>(){

            @Override
            public Integer poll(long timeout, TimeUnit unit) throws InterruptedException {
                onPollLatch.countDown();
                afterWriteLatch.await(1L, TimeUnit.HOURS);
                return (Integer)super.poll(timeout, unit);
            }
        };
        try (QueueInputStream queueInputStream = QueueInputStream.builder().setBlockingQueue((BlockingQueue)queue).setTimeout(Duration.ofHours(1L)).get();){
            QueueOutputStream queueOutputStream = queueInputStream.newQueueOutputStream();
            CompletableFuture<Void> future = CompletableFuture.runAsync(() -> {
                try {
                    onPollLatch.await(1L, TimeUnit.HOURS);
                    queueOutputStream.write(inputData.getBytes(StandardCharsets.UTF_8));
                    afterWriteLatch.countDown();
                }
                catch (Exception e) {
                    throw new RuntimeException(e);
                }
            });
            byte[] data = new byte[inputData.length()];
            int read = queueInputStream.read(data, 0, data.length);
            Assertions.assertEquals((int)inputData.length(), (int)read);
            String outputData = new String(data, 0, read, StandardCharsets.UTF_8);
            Assertions.assertEquals((Object)inputData, (Object)outputData);
            Assertions.assertDoesNotThrow(() -> (Void)future.get());
        }
    }

    @Test
    void testBulkReadZeroLength() {
        QueueInputStream queueInputStream = new QueueInputStream();
        int read = queueInputStream.read(ArrayUtils.EMPTY_BYTE_ARRAY, 0, 0);
        Assertions.assertEquals((int)0, (int)read);
    }

    @Test
    void testInvalidArguments() {
        Assertions.assertThrows(IllegalArgumentException.class, () -> QueueInputStream.builder().setTimeout(Duration.ofMillis(-1L)).get(), (String)"waitTime must not be negative");
    }

    @ParameterizedTest(name="inputData={0}")
    @MethodSource(value={"inputData"})
    void testReadAfterClose(String inputData) throws IOException {
        QueueInputStream shadow;
        LinkedBlockingQueue queue = new LinkedBlockingQueue();
        try (QueueInputStream inputStream = new QueueInputStream(queue);){
            shadow = inputStream;
        }
        Assertions.assertEquals((int)-1, (int)shadow.read());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @ParameterizedTest(name="inputData={0}")
    @MethodSource(value={"inputData"})
    void testReadLineByLineFile(String inputData) throws IOException {
        Path tempFile = Files.createTempFile(this.getClass().getSimpleName(), ".txt", new FileAttribute[0]);
        try (InputStream inputStream = Files.newInputStream(tempFile, new OpenOption[0]);
             OutputStream outputStream = Files.newOutputStream(tempFile, new OpenOption[0]);){
            this.doTestReadLineByLine(inputData, inputStream, outputStream);
        }
        finally {
            Files.delete(tempFile);
        }
    }

    @ParameterizedTest(name="inputData={0}")
    @MethodSource(value={"inputData"})
    void testReadLineByLineQueue(String inputData) throws IOException {
        String[] lines = inputData.split("\n");
        LinkedBlockingQueue queue = new LinkedBlockingQueue();
        try (QueueInputStream inputStream = QueueInputStream.builder().setBlockingQueue(queue).setTimeout(Duration.ofHours(1L)).get();
             QueueOutputStream outputStream = inputStream.newQueueOutputStream();){
            this.doTestReadLineByLine(inputData, (InputStream)inputStream, (OutputStream)outputStream);
        }
    }

    @Test
    void testResetArguments() throws IOException {
        try (QueueInputStream queueInputStream = QueueInputStream.builder().setTimeout(null).get();){
            Assertions.assertEquals((Object)Duration.ZERO, (Object)queueInputStream.getTimeout());
            Assertions.assertEquals((int)0, (int)queueInputStream.getBlockingQueue().size());
        }
        queueInputStream = QueueInputStream.builder().setBlockingQueue(null).get();
        try {
            Assertions.assertEquals((Object)Duration.ZERO, (Object)queueInputStream.getTimeout());
            Assertions.assertEquals((int)0, (int)queueInputStream.getBlockingQueue().size());
        }
        finally {
            if (queueInputStream != null) {
                queueInputStream.close();
            }
        }
    }

    @Test
    @DisplayName(value="If read is interrupted while waiting, then exception is thrown")
    void testTimeoutInterrupted() throws Exception {
        try (QueueInputStream inputStream = QueueInputStream.builder().setTimeout(Duration.ofMinutes(2L)).get();
             QueueOutputStream outputStream = inputStream.newQueueOutputStream();){
            AtomicBoolean result = new AtomicBoolean();
            CountDownLatch latch = new CountDownLatch(1);
            Thread thread = new Thread(() -> {
                Assertions.assertThrows(IllegalStateException.class, () -> this.readUnbuffered((InputStream)inputStream, 3));
                Assertions.assertTrue((boolean)Thread.currentThread().isInterrupted());
                result.set(true);
                latch.countDown();
            });
            thread.setDaemon(true);
            thread.start();
            thread.interrupt();
            latch.await(500L, TimeUnit.MILLISECONDS);
            Assertions.assertTrue((boolean)result.get());
        }
    }

    @Test
    @DisplayName(value="If data is not available in queue, then read will wait until wait time elapses")
    void testTimeoutUnavailableData() throws IOException {
        try (QueueInputStream inputStream = QueueInputStream.builder().setTimeout(Duration.ofMillis(500L)).get();
             QueueOutputStream outputStream = inputStream.newQueueOutputStream();){
            Stopwatch stopwatch = Stopwatch.createStarted();
            String actualData = (String)Assertions.assertTimeout((Duration)Duration.ofSeconds(1L), () -> this.readUnbuffered((InputStream)inputStream, 3));
            stopwatch.stop();
            Assertions.assertEquals((Object)"", (Object)actualData);
            Assertions.assertTrue((stopwatch.elapsed(TimeUnit.MILLISECONDS) >= 500L ? 1 : 0) != 0, () -> stopwatch.toString());
        }
    }

    @ParameterizedTest(name="inputData={0}")
    @MethodSource(value={"inputData"})
    void testUnbufferedReadWrite(String inputData) throws IOException {
        try (QueueInputStream inputStream = new QueueInputStream();
             QueueOutputStream outputStream = inputStream.newQueueOutputStream();){
            this.writeUnbuffered(outputStream, inputData);
            String actualData = this.readUnbuffered((InputStream)inputStream);
            Assertions.assertEquals((Object)inputData, (Object)actualData);
        }
    }

    @ParameterizedTest(name="inputData={0}")
    @MethodSource(value={"inputData"})
    void testUnbufferedReadWriteWithTimeout(String inputData) throws IOException {
        Duration timeout = Duration.ofMinutes(2L);
        try (QueueInputStream inputStream = QueueInputStream.builder().setTimeout(timeout).get();
             QueueOutputStream outputStream = inputStream.newQueueOutputStream();){
            Assertions.assertEquals((Object)timeout, (Object)inputStream.getTimeout());
            this.writeUnbuffered(outputStream, inputData);
            String actualData = (String)Assertions.assertTimeout((Duration)Duration.ofSeconds(1L), () -> this.readUnbuffered((InputStream)inputStream, inputData.length()));
            Assertions.assertEquals((Object)inputData, (Object)actualData);
        }
    }

    private void writeUnbuffered(QueueOutputStream outputStream, String inputData) throws IOException {
        byte[] bytes = inputData.getBytes(StandardCharsets.UTF_8);
        outputStream.write(bytes, 0, bytes.length);
    }
}

