/******************************************************************************
 * (c) Copyright 2002,2003, 1060 Research Ltd
 *
 * This Software is licensed to You, the licensee, for use under the terms of
 * the 1060 Public License v1.0. Please read and agree to the 1060 Public
 * License v1.0 [www.1060research.com/license] before using or redistributing
 * this software.
 *
 * In summary the 1060 Public license has the following conditions.
 * A. You may use the Software free of charge provided you agree to the terms
 * laid out in the 1060 Public License v1.0
 * B. You are only permitted to use the Software with components or applications
 * that provide you with OSI Certified Open Source Code [www.opensource.org], or
 * for which licensing has been approved by 1060 Research Limited.
 * You may write your own software for execution by this Software provided any
 * distribution of your software with this Software complies with terms set out
 * in section 2 of the 1060 Public License v1.0
 * C. You may redistribute the Software provided you comply with the terms of
 * the 1060 Public License v1.0 and that no warranty is implied or given.
 * D. If you find you are unable to comply with this license you may seek to
 * obtain an alternative license from 1060 Research Limited by contacting
 * license@1060research.com or by visiting www.1060research.com
 *
 * NO WARRANTY:  THIS SOFTWARE IS NOT COVERED BY ANY WARRANTY. SEE 1060 PUBLIC
 * LICENSE V1.0 FOR DETAILS
 *
 * THIS COPYRIGHT NOTICE IS *NOT* THE 1060 PUBLIC LICENSE v1.0. PLEASE READ
 * THE DISTRIBUTED 1060_Public_License.txt OR www.1060research.com/license
 *
 * File:          $RCSfile: DOMXDA.java,v $
 * Version:       $Name:  $ $Revision: 1.6 $
 * Last Modified: $Date: 2004/10/06 13:47:02 $
 *****************************************************************************/
package org.ten60.netkernel.xml.xda;

import org.ten60.netkernel.xml.util.*;
import java.util.*;
import org.w3c.dom.*;
import org.apache.xpath.*;
import org.apache.xpath.objects.*;
import org.apache.xml.utils.*;
import org.xml.sax.*;
import javax.xml.transform.TransformerException;
import java.io.*;

/**
 * Implementation of IXDA, IXDAIterator, IXDAReadOnly and IXDAReadOnlyIterator using DOM
 * @author  pjr
 */
public class DOMXDA implements IXDA,IXDAIterator, IXDAReadOnly, IXDAReadOnlyIterator
{
    private Document mDocument;
    
    public static final String XMLNS_DECLARATION_URI="http://www.w3.org/2000/xmlns/";
    
    /** Internal storage of context node   */
    private Node mRoot;
    /** If we are an IReadableIterator then this is the NodeList to iterate over.
     */
    private NodeList mNodes;
    private ArrayList mNewNodes;
    
    /** If we are an IReadbleIterator then this is the nodelist to iterate over.
     */
    private int mNodeIndex;
    /** Reference to an owning XMLBean if any.
     */
    //private IXDA mParent;
    /** if this is an iterator then this is the writable of the writable it
     * came from */
    private DOMXDA mParentXDA;
    
    /** XPath api cached for performance */
    private java.lang.ref.SoftReference mXPathAPI;
    
    private DOMXPathImplementation getXPathAPI()
    {	DOMXPathImplementation result=null;
        if (mXPathAPI!=null)
        {	result = (DOMXPathImplementation)mXPathAPI.get();
        }
        if (result==null)
        {	result = new DOMXPathImplementation();
            mXPathAPI = new java.lang.ref.SoftReference(result);
        }
        return result;
    }
    
    private PrefixResolver getPrefixResolver()
    {	DOMXPathImplementation xpath = getXPathAPI();
        PrefixResolver result = xpath.getPrefixResolver();
        if (result==null)
        {	result = new DOMPrefixResolver(mDocument);
        }
        return result;
    }
    
    
    public DOMXDA(Document aDocument, boolean clone)
    {	construct(aDocument, null, clone);
    }
    
    /** Creates a new instance of DOMXDA */
    public DOMXDA(Document aDocument)
    {	this(aDocument,true);
    }
    
    public DOMXDA(Node aRoot, DOMXDA aParentXDA)
    {	construct(aRoot, aParentXDA, true);
    }
    
    public DOMXDA(Node aRoot, DOMXDA aParentXDA, boolean clone)
    {	construct(aRoot, aParentXDA, clone);
    }
    
    private void construct(Node aRoot, DOMXDA aParentXDA, boolean clone)
    {	if (aRoot instanceof Document)
        {   mDocument=(Document)aRoot;
            if(clone)
            {	mDocument=(Document)safeDeepClone((Node)mDocument);
            }
            mRoot = mDocument.getDocumentElement();
        }
        else
        {   mDocument=aRoot.getOwnerDocument();
            mRoot=aRoot;
        }
        
        mParentXDA=aParentXDA;
    }
    
    /** Creates a new DOMXDA with a node list to iterate over.
     * @param aNodes The NodeList.
     * @param aParent The parent bean which we belong to.
     */
    DOMXDA(NodeList aNodes, IXDA aParentXDA)
    {	initIterator(aNodes, aParentXDA);
    }
    
    private void initIterator(NodeList aNodes, IXDA aParentXDA)
    {	//	mNodes=aNodes;
        mNewNodes=new ArrayList();
        for(int i=0;i<aNodes.getLength();i++)
        {
            mNewNodes.add(aNodes.item(i));
        }
        //mParent=aParent;
        mParentXDA=(DOMXDA)aParentXDA;
        mNodeIndex=-1;
        if (aNodes.getLength()>0)
        {   //mRoot=mNodes.item(0);
            mRoot=(Node)mNewNodes.get(0);
            if(mRoot.getNodeType()==Node.DOCUMENT_NODE)
            {
                mDocument=(Document)mRoot;
                mRoot=mDocument.getDocumentElement();
            }
            else mDocument = mRoot.getOwnerDocument();
        }
    }
    
    public Node getRoot()
    {	return mRoot;
    }
    
    public Document getDocument()
    {	return mDocument;
    }
    
    
    public void setDirty()
    {
        if (mParentXDA!=null)
        {	mParentXDA.setDirty();
        }
        getXPathAPI().reset();
    }
    
    public void setDirtyNS()
    {	setDirty();
        getXPathAPI().resetPrefixResolver();
    }
    
    
    /** Evaluates the given XPath expression on the current document.
     * @param aXPath The XPath to evaluate
     * @throws MalformedTargetException Thrown if the XPath cannot be parsed.
     * @return Returns a NodeList with zero or more matching nodes.
     */
    private synchronized NodeList evaluateXPath(String aXPath)throws XPathLocationException
    {   try
        {	if (FastXPath.isSuitable(aXPath))
            {	Node context = mRoot;
                if (mRoot==null)
                {	context=mDocument;
                }
                return FastXPath.eval(context,aXPath);
            }
            else
            {	DOMXPathResult xo = evaluateComplexXPath(aXPath);
				NodeList result = xo.getNodeList();
				if (result==null)
				{	result = new NodeSet();
				}
				return result;
            }
        } catch (TransformerException e)
        {   throw new XPathLocationException("result isn't nodelist", XPathLocationException.MALFORMEDTARGET);
        }
    }
    
    
    /** Xalan XPath call
     */
    private DOMXPathResult evaluateComplexXPath(String aXPath)
    throws XPathLocationException
    {
        
        try
        {   Node context = mRoot;
            if (mRoot==null)
            {	context=mDocument;
            }
            DOMXPathImplementation xpath = getXPathAPI();
            DOMXPathResult result = (DOMXPathResult)xpath.eval(context,aXPath);
            return result;
        } catch (XPathImplementationException e)
        {   throw new XPathLocationException("cannot evaluate "+aXPath+": "+e.getMessage(), XPathLocationException.MALFORMEDTARGET);
        }
    }
    
    
    /** Returns the one and only node from an Xpath query.
     * @param aXpath The Xpath which is expected to have only one node.
     * @throws NullTargetException Thrown if the Xpath has zero nodes.
     * @throws AmbiguousTargetException Thrown if the Xpath has more than one node.
     * @throws MalformedTargetException Thrown if the Xpath is not valid.
     * @throws AccessViolationException Thrown when lock is not held on bean.
     * @return The node.
     */
    public Node getSingleNode(String aXpath) throws XPathLocationException
    {	Node n=null;
        try
        {	n=getSingleNode(evaluateXPath(aXpath));
        }
        catch(XPathLocationException e)
        {	if(e.getType()==e.NULLTARGET) throw new XPathLocationException("Target "+aXpath+" does not exist",XPathLocationException.NULLTARGET);
            if(e.getType()==e.AMBIGUOUSTARGET) throw new XPathLocationException("More than one target for "+aXpath,XPathLocationException.AMBIGUOUSTARGET);
            else throw e;
        }
        return n;
    }
    
    /** Returns a the one and only node from a NodeList.
     * @param aNodeList The NodeList which is expected to have only one node.
     * @throws NullTargetException Thrown if the NodeList has zero nodes.
     * @throws AmbiguousTargetException Thrown if the node list has more than one node.
     * @return The node.
     */
    public Node getSingleNode(NodeList aNodeList)
    throws XPathLocationException
    {   int count=aNodeList.getLength();
        if (count==0) throw new XPathLocationException(null,XPathLocationException.NULLTARGET);
        if (count>1) throw new XPathLocationException(null,XPathLocationException.AMBIGUOUSTARGET);
        return aNodeList.item(0);
    }
    
    /** Evaluates the given XPath expression on the current document
     *  and returns a NodeList.
     * @param aXPath The XPath to evaluate
     * @throws MalformedTargetException Thrown if the XPath cannot be parsed.
     * @return Returns a NodeList with zero or more matching nodes.
     */
    public synchronized NodeList getNodeList(String aXPath)
    throws XPathLocationException
    {   return evaluateXPath(aXPath);
    }
    public synchronized List getNodesFor(String aXPath)
    throws XPathLocationException
    {  NodeList nl= evaluateXPath(aXPath);
	   int length=nl.getLength();
	   List result = new ArrayList(length);
		for (int i=0; i<length; i++)
		{	result.add(nl.item(i));
		}
	   return result;
    }
    
    /** Returns the document fragment located by an unambiguous XPath */
    public Document getFragment(String aXPath) throws XPathLocationException
    {	try
        {
            Node node = getSingleNode(aXPath);
            if (node==null)
            {   throw new XPathLocationException("XPath must evaluate to element or attribute :"+aXPath, XPathLocationException.MALFORMEDTARGET);
            }
            return getFragment(node);
        }
        catch (DOMException e)
        {   e.printStackTrace();
            throw new XPathLocationException(e.getMessage(), XPathLocationException.MALFORMEDTARGET);
        }
    }
    
    /** Returns the document fragment located by a Node */
    public Document getFragment(Node node) throws DOMException
    {	if(node.getNodeType()==node.DOCUMENT_NODE)
        {
            Document d=(Document)node;
            node=d.getDocumentElement();
        }
        Document doc=XMLUtils.getInstance().newDocument();
        if (node!=null)
        {	synchronized(node.getOwnerDocument())
            {	Element imp=(Element) doc.importNode(node,true);
                doc.appendChild(imp);
            }
        }
        
        //Deal with Namespace declarations
        Element root=doc.getDocumentElement();
        DOMPrefixResolver dpr=(DOMPrefixResolver)getPrefixResolver();
        Map map=dpr.getPrefixMap();
        map.keySet();
        Iterator i=map.keySet().iterator();
        while(i.hasNext())
        {	String prefix=(String)i.next();
            if(!prefix.equals("xml"))
            {
                if(prefix.equals(""))
                {	String ns=dpr.getNamespaceForPrefix(prefix);
                    if(!ns.equals("")) root.setAttributeNS(XMLNS_DECLARATION_URI, "xmlns", ns);
                }
                else root.setAttributeNS(XMLNS_DECLARATION_URI, "xmlns:"+prefix, dpr.getNamespaceForPrefix(prefix));
                
            }
        }
        
    /*
    String prefix=root.getPrefix();
    if(prefix!=null)
    {	boolean test=false;
        NamedNodeMap nnm=root.getAttributes();
        if(nnm!=null)
        {	for(int i=0;i<nnm.getLength();i++)
            {	Node n=nnm.item(i);
                String name=n.getNodeName();
                if (name.equals("xmlns:"+prefix))
                {	test=true;
                }
            }
        }
        if(!test)
        {	root.setAttribute("xmlns:"+prefix, getPrefixResolver().getNamespaceForPrefix(prefix));
        }
    }
     */
        return doc;
    }
    
    /** Builds a document fragment from the given "Simple XPath" expression. This
     * expression consists of a path of zero or more elements optionally terminated
     * by an attribute. I.e. a/b/c/@d
     * @return Returns the topmost node in the fragment.
     * @param aSimpleXPath The "Simple XPath" expression
     * @param aValue The value to put into the created leaf node (either text into an element or
     * value into an attribute) This value may be null.
     * @throws MalformedTargetException Thrown if the Simple XPath cannot be parsed.
     */
    private Node buildFragment(String aSimpleXPath, String aValue)
    throws XPathLocationException
    {   Node result=null;
        Node top=null;
        StringTokenizer st=new StringTokenizer(aSimpleXPath,"/");
        while(st.hasMoreTokens())
        {   String token = st.nextToken();
            if (token.startsWith("@"))
            {   // create an attribute
                if (st.hasMoreTokens())
                {   throw new XPathLocationException("Build Fragment - Attributes can only be placed on leaf nodes :"+aSimpleXPath, XPathLocationException.MALFORMEDTARGET);
                }
                int idx=token.indexOf(":");
                Attr attr=null;
                if(idx>0)
                {   /*Namespace case*/
                    String prefix=token.substring(1,idx);
                    String namespace=getPrefixResolver().getNamespaceForPrefix(prefix);
                    attr = mDocument.createAttributeNS(namespace, token.substring(1));
                }
                else attr = mDocument.createAttribute(token.substring(1));
                if (aValue!=null)
                {   attr.setNodeValue(aValue);
                }
                if (result==null)
                {   result=attr;
                }
                else
                {   ((Element)result).setAttributeNode(attr);
                }
            }
            else
            {   //create an element
                int idx=token.indexOf(":");
                Element element=null;
                if(idx>0)
                {   /*Namespace case*/
                    String prefix=token.substring(0,idx);
                    String namespace=getPrefixResolver().getNamespaceForPrefix(prefix);
                    element = mDocument.createElementNS(namespace, token);
                }
                else element = mDocument.createElement(token);
                if (aValue!=null && !st.hasMoreTokens())
                {   Text textNode = mDocument.createTextNode(aValue);
                    element.appendChild(textNode);
                }
                if (result==null)
                {   result=element;
                }
                else
                {   result.appendChild(element);
                    if (top==null)
                    {   top=result;
                    }
                    result=element;
                }
            }
        }
        if (top!=null)
        {   result=top;
        }
        return result;
        
    }
    
    /** Adds the given fragment to the given parent.
     * @param aParent The parent element to add the fragment too.
     * @param aFragment The fragment must be either an element or attribute.
     */
    private void addFragment(Node aParent, Node aFragment)
    {
        switch(aFragment.getNodeType())
        {   case Node.ELEMENT_NODE:
            case Node.PROCESSING_INSTRUCTION_NODE:
                aParent.appendChild(aFragment);
                if (aParent==mDocument && mRoot==null)
                {	mRoot=aFragment;
                }
                break;
            case Node.ATTRIBUTE_NODE:
				if (aParent instanceof Element)
				{	((Element)aParent).setAttributeNode((Attr)aFragment);
				}
                break;
        }
    }
    
    public static Node safeDeepClone(Node aNode)
    {	Document d = (aNode instanceof Document)?(Document)aNode:aNode.getOwnerDocument();
        synchronized(d)
        {	return aNode.cloneNode(true);
        }
    }
    
    private void checkType(IXDAReadOnly aXDA) throws XDOIncompatibilityException
    {	if(!( aXDA instanceof DOMXDA )) throw new XDOIncompatibilityException("Only DOM supported");
    }
    
    /*Slow search for all namespace declarations.  Can be improved?*/
    private void testNamespaceDeclarationExists(Node node, String prefix, String auri) throws XPathLocationException
    {   String uri=getPrefixResolver().getNamespaceForPrefix(prefix);
        if(uri!=null && !uri.equalsIgnoreCase(auri))
        {	throw new XPathLocationException("Namespace prefix has already been used with different uri", XPathLocationException.MALFORMEDTARGET);
        }
    }
    
    private void testExistingNamespace(Node node) throws XPathLocationException
    {   if(node.getNamespaceURI()!=null)
        {   throw new XPathLocationException("Cannot overwrite descendents namespace", XPathLocationException.MALFORMEDTARGET);
        }
        for (Node n =node.getFirstChild(); n!=null; n=n.getNextSibling())
        {   if (n instanceof Element)
            {   testExistingNamespace(n);
            }
        }
    }
    
    private void testNamespaceIsEmpty(Node node, String auri) throws XPathLocationException
    {	String uri=node.getNamespaceURI();
        if(uri!=null && uri.equalsIgnoreCase(auri)) throw new XPathLocationException("Elements still exist in the namespace "+auri, XPathLocationException.MALFORMEDTARGET);
        if(node.hasChildNodes())
        {
            NodeList children=node.getChildNodes();
            for(int i=0;i<children.getLength();i++)
            {
                testNamespaceIsEmpty(children.item(i), auri);
            }
        }
    }
    
    private void removeNamespacePrefix(Node node)
    {
        if(node.getNamespaceURI()!=null)
        {
            node.setPrefix("");
            if(node.hasChildNodes())
            {
                NodeList children=node.getChildNodes();
                for(int i=0;i<children.getLength();i++)
                {
                    removeNamespacePrefix(children.item(i));
                }
            }
        }
    }
    
    /*Safe replacement of one node with another - takes care of root element corner cases*/
    private void replaceNode(Node src, Node dest)
    {	Node parent = dest.getParentNode();
        Node oldRoot=mRoot;
        if(parent == null)
        {	parent=dest;
        }
        if(parent.getNodeType()==Node.DOCUMENT_NODE)
        {	Document d=mDocument;//(Document)parent;
            Element oldDocEl = d.getDocumentElement();
            d.removeChild(oldDocEl);
            d.appendChild(src);
            if (mRoot==oldDocEl)
            {	mRoot=src;
            }
        }
        else
        {
            parent.insertBefore(src,dest);
            parent.removeChild(dest);
        }
        //Take care of mRoot in interator
        if(mNewNodes!=null)
        {
            if (dest==oldRoot)
            {	mNewNodes.set(mNodeIndex, src);
                mRoot=src;
            }
        }
    }
    
    /** Removes the given node from the document.
     * @param aNode The node may be either an element or attribute.
     */
    private void deleteNode(Node aNode)
    {   switch(aNode.getNodeType())
        {   case Node.TEXT_NODE:
            case Node.ELEMENT_NODE:
                aNode.getParentNode().removeChild(aNode);
                break;
            case Node.ATTRIBUTE_NODE:
                ((Attr)aNode).getOwnerElement().removeAttributeNode((Attr)aNode);
                break;
        }
    }
    
    /** Returns the value within a node.
     * @param aNode The node may be either and element or attribute.
     * @param aTrim If true the whitespace before and after characters in the String are removed.
     * @return The string value of the node.
     */
    private String getNodeValue(Node aNode, boolean aTrim)
    {   String result=null;
        switch(aNode.getNodeType())
        {   case Node.ELEMENT_NODE:
                result = getText((Element)aNode);
                break;
            case Node.DOCUMENT_NODE:
            {	Element docEl = ((Document)aNode).getDocumentElement();
                if (docEl!=null)
                {	result = getText(docEl);
                }
                break;
            }
            case Node.TEXT_NODE:
            case Node.ATTRIBUTE_NODE:
                result = aNode.getNodeValue();
                break;
        }
        if (aTrim && result!=null)
        {   result = result.trim();
        }
        return result;
    }
    
    /** Gets a concatenation of all text nodes directly below and element.
     * @param aElement The element which holds the text.
     * @return The concatenation of all the text.
     */
    private synchronized String getText(Element aElement)
    {   return XMLUtils.getText(aElement);
    }
    
    /** Sets the value of a node.
     * @param aNode The node may be either and element or attribute.
     * @param aValue The value to set.
     */
    private void setValue(Node aNode, String aValue)
    {   switch (aNode.getNodeType())
        {   case Node.ELEMENT_NODE:
                setText((Element)aNode,aValue);
                break;
            case Node.ATTRIBUTE_NODE:
                aNode.setNodeValue(aValue);
                break;
        }
    }
    
    /** Replaces any text nodes directly below and element with one text node containing
     * the given value.
     * @param aElement The element to replace the text of.
     * @param aValue The text.
     */
    private void setText(Element aElement, String aValue)
    {   // remove any existing text
        XMLUtils.setText(aElement, aValue);
    }
    
    /** Returns
     * @return Returns a Map of node name to value for all nodes found by the XPath expression.
     * @param aXPath An XPath expression which should evaluate to zero or more elements or attributes
     * within the document.
     * @param aTrim If true the resulting string is trimed at beginning and end for whitespace.
     * @throws AccessViolationException Thrown when lock is not held on bean.
     * @throws MalformedTargetException Thrown if aXPath cannot be parsed.
     */
    public Map getMap(String aTargetXPath, boolean aTrim) throws XPathLocationException
    {   try
        {   NodeList nl = evaluateXPath(aTargetXPath);
            HashMap result = new HashMap();
            for (int i=nl.getLength()-1; i>=0; i--)
            {   Node node = nl.item(i);
                String value=getNodeValue(node,aTrim);
                if (result==null)
                {   throw new XPathLocationException("XPath must evaluate to elements or attributes :"+aTargetXPath, XPathLocationException.MALFORMEDTARGET);
                }
                result.put(XMLUtils.getPathFor(node),value);
            }
            return result;
        } catch (DOMException e)
        {   throw new XPathLocationException(e.toString(), XPathLocationException.MALFORMEDTARGET);
        }
    }
    
    
    public void append(IXDAReadOnly aSource, String aSourceXPath, String aTargetXPath) throws XPathLocationException, XDOIncompatibilityException
    {	checkType(aSource);
        DOMXDA source=(DOMXDA)aSource;
        
        Node s=source.getSingleNode(aSourceXPath);
        if(s.getNodeType()==Node.ATTRIBUTE_NODE)
        {	// get destination node
            Node dstNode = getSingleNode(aTargetXPath);
            if(dstNode.getNodeType()==Node.DOCUMENT_NODE)
            { Document d=(Document)dstNode;
              dstNode=d.getDocumentElement();
            }
            if(dstNode.getNodeType()!=Node.ELEMENT_NODE)
            {	throw new XPathLocationException("destination xpath location not an element", XPathLocationException.MALFORMEDTARGET);
            }
            Element dstElement = (Element)dstNode;
            
            // get source node
            String attrString = aSource.eval("concat(name("+aSourceXPath+"), '=', "+aSourceXPath+" )").getStringValue();
            int i=attrString.indexOf('=');
            String attrName = attrString.substring(0,i);
            String attrValue= attrString.substring(i+1);
            i = attrName.indexOf(':');
            if (i>=0)
            {   String attrNS = attrName.substring(0,i);
                //attrName = attrName.substring(i+1);
                String uri = this.getPrefixResolver().getNamespaceForPrefix(attrNS);
                if (dstElement.getAttributeNS(uri,attrName).length()>0) throw new XPathLocationException("attribute "+attrNS+":"+attrName+" already exists", XPathLocationException.MALFORMEDTARGET);
                dstElement.setAttributeNS(uri,attrName,attrValue);
            }
            else
            {   if (dstElement.getAttribute(attrName).length()>0) throw new XPathLocationException("attribute "+attrName+" already exists", XPathLocationException.MALFORMEDTARGET);
                dstElement.setAttribute(attrName,attrValue);
            }
        }
        else
        {	// get source node
            Document fragment = source.getFragment(aSourceXPath);
            Node srcClone = mDocument.importNode(fragment.getDocumentElement(),true);
            
            // get destination node
            Node dstNode = getSingleNode(aTargetXPath);
            
            // remove destination node and replace with source node
            Node dstParent = dstNode.getParentNode();
            if (dstParent==null)
            {   //must be document - handle this case
                Document d=(Document)dstNode;
                dstNode=d.getDocumentElement().appendChild(srcClone);
            }
            else
            {   // the general case - preserve order
                dstNode.appendChild(srcClone);
            }
        }
        setDirtyNS();
    }
    
    public void appendPath(String aTargetXPath, String aNewRelativeXPath, String aOptionalValue) throws XPathLocationException
    {	try
        {   NodeList nl = evaluateXPath(aTargetXPath);
            Node fragment = buildFragment(aNewRelativeXPath,aOptionalValue);
            for (int i=nl.getLength()-1; i>=0; i--)
            {   Node node = nl.item(i);
                Node fragmentToAppend = (i==0)? fragment : safeDeepClone(fragment);
                addFragment(node,fragmentToAppend);
                setDirty();
            }
        } catch (DOMException e)
        {   throw new XPathLocationException(e.toString(), XPathLocationException.MALFORMEDTARGET);
        }
    }
    
    public void applyNS(String aTargetXPath, String prefix, String uri) throws XPathLocationException
    {	try
        {   Node node = getSingleNode(aTargetXPath);
            testNamespaceDeclarationExists(mDocument, prefix, uri);
            testExistingNamespace(node);
            
            if ((node.getNodeType()==node.ELEMENT_NODE || node.getNodeType()==node.DOCUMENT_NODE))
            {
                Document dfrag=null;
                try
                {
                    DOM2SAX b2s=new DOM2SAX(node);
                    ApplyNamespaceHandler handler=new ApplyNamespaceHandler(uri, prefix);
                    b2s.setContentHandler(handler);
                    b2s.parse();
                    Element outnode=handler.getDocument().getDocumentElement();
                    outnode.setAttributeNS(XMLNS_DECLARATION_URI, "xmlns:"+prefix, uri);
                    Node innode=mDocument.importNode(outnode, true);
                    replaceNode(innode,node);
                    setDirtyNS();
                }
                catch(SAXException e)
                {
                    throw new XPathLocationException("Apply Namespace - Problems parsing namespaced document :"+aTargetXPath, XPathLocationException.MALFORMEDTARGET);
                }
            }
            else if(node.getNodeType()==node.ATTRIBUTE_NODE)
            {
                throw new XPathLocationException("Apply Namespace to attribute not supported "+aTargetXPath, XPathLocationException.MALFORMEDTARGET);
            }
            else   throw new XPathLocationException("Apply Namespace - XPath must evaluate to element or attribute :"+aTargetXPath, XPathLocationException.MALFORMEDTARGET);
        } catch (DOMException e)
        {   throw new XPathLocationException(e.toString(), XPathLocationException.UNKNOWN);
        }
    }
    
    /*
     * Create namespace declaration attribute at a given xpath location.  Use applyNS if you want tosimultaneously move the target Xpath into
     * the namespace. If prefix is null the default xmlns is declared.
     */
    public void declareNS(String aTargetXPath, String prefix, String uri) throws XPathLocationException
    {	try
        {   Node node = getSingleNode(aTargetXPath);
            if(node.getNodeType()==node.ELEMENT_NODE)
            {	Element e=(Element)node;
                e.setAttributeNS(XMLNS_DECLARATION_URI, prefix==null ? "xmlns" : "xmlns:"+prefix, uri);
                setDirtyNS();
            }
            else throw new XPathLocationException("Xpath must point to an element to declare a namespace", XPathLocationException.MALFORMEDTARGET);
        }
        catch (DOMException e)
        {   throw new XPathLocationException(e.toString(), XPathLocationException.UNKNOWN);
        }
    }
    
    public void delete(String aTargetXPath) throws XPathLocationException
    {	try
        {   NodeList nl = evaluateXPath(aTargetXPath);
            NodeSet ns = new NodeSet(nl);
            for (int i=ns.getLength()-1; i>=0; i--)
            {   Node node = ns.item(i);
                deleteNode(node);
                setDirtyNS();
            }
        } catch (DOMException e)
        {   throw new XPathLocationException(e.toString(), XPathLocationException.MALFORMEDTARGET);
        }
    }
    
    public synchronized IXPathResult eval(String aTargetXPath) throws XPathLocationException
    {
        return evaluateComplexXPath(aTargetXPath);
    }
    
    public synchronized String getText(String aTargetXPath, boolean aTrim) throws XPathLocationException
    {	try
        {   Node node = getSingleNode(aTargetXPath);
            String result=getNodeValue(node,aTrim);
            if (result==null)
            {   throw new XPathLocationException("XPath must evaluate to element or attribute :"+aTargetXPath, XPathLocationException.MALFORMEDTARGET);
            }
            return result;
        } catch (DOMException e)
        {   throw new XPathLocationException(e.toString(), XPathLocationException.MALFORMEDTARGET);
        }
    }
    
    public boolean hasNext()
    {
        return mNodeIndex<(mNewNodes.size()-1);
    }
    
    public void insertAfter(IXDAReadOnly aSource, String aSourceXPath, String aTargetXPath) throws XPathLocationException, XDOIncompatibilityException
    {	checkType(aSource);
        DOMXDA source=(DOMXDA)aSource;
        // get source node
        Document fragment = source.getFragment(aSourceXPath);
        Node srcClone = mDocument.importNode(fragment.getDocumentElement(),true);
        
        // get destination node
        Node dstNode = getSingleNode(aTargetXPath);
        
        Node dstParent = dstNode.getParentNode();
        if (dstParent==null)
        {   //must be document - handle this case
            throw new XPathLocationException("Cannot have more than one document root", XPathLocationException.MALFORMEDTARGET);
        }
        else
        {
            Node next=dstNode.getNextSibling();
            dstParent.insertBefore(srcClone,next);
        }
        setDirtyNS();
    }
    
    public void insertBefore(IXDAReadOnly aSource, String aSourceXPath, String aTargetXPath) throws XPathLocationException, XDOIncompatibilityException
    {	checkType(aSource);
        DOMXDA source=(DOMXDA)aSource;
        // get source node
        Document fragment = source.getFragment(aSourceXPath);
        Node srcClone = mDocument.importNode(fragment.getDocumentElement(),true);
        
        // get destination node
        Node dstNode = getSingleNode(aTargetXPath);
        
        Node dstParent = dstNode.getParentNode();
        if (dstParent==null)
        {   //must be document - handle this case
            throw new XPathLocationException("Cannot have more than one document root", XPathLocationException.MALFORMEDTARGET);
        }
        else
        {
            Node next=dstNode.getNextSibling();
            dstParent.insertBefore(srcClone,dstNode);
        }
        setDirtyNS();
    }
    
    public synchronized boolean isTrue(String aTargetXPath) throws XPathLocationException
    {	try
        {
            boolean result;
            if (FastXPath.isSuitable(aTargetXPath))
            {	Node context = mRoot;
                if (mRoot==null)
                {	context=mDocument;
                }
                result = FastXPath.eval(context,aTargetXPath).getLength()>0;
            }
            else
            {	DOMXPathResult r = (DOMXPathResult)evaluateComplexXPath(aTargetXPath);
                result = r.isTrue();
            }
            return result;
        }
        catch(Exception e)
        { XPathLocationException e2 = new XPathLocationException("Not boolean", XPathLocationException.MALFORMEDTARGET);
          e2.initCause(e);
          throw e2;
        }
    }
    
    public IXDAIterator iterator(String aTargetXPath) throws XPathLocationException
    {
        if(mRoot.getNodeType()==mRoot.DOCUMENT_NODE)
        {	mRoot=mDocument.getDocumentElement();
        }
        NodeList results = evaluateXPath(aTargetXPath);
        if(results!=null)
        {	DOMXDA iterator=new DOMXDA(results,this);
            //Take care of root node / document element
            if(results.getLength()==1)
            {	Node n=results.item(0);
                if(n.getNodeType()!=Node.ATTRIBUTE_NODE)
                {	if( n.getNodeType()==Node.DOCUMENT_NODE || n.getParentNode().getNodeType()==Node.DOCUMENT_NODE)
                    {	this.initIterator(results, null);
                        iterator=this;
                    }
                }
            }
            return iterator;
        }
        else
		{	return null;
		}
    }
    
    public IXDAReadOnlyIterator readOnlyIterator(String aTargetXPath) throws XPathLocationException
    {	return (IXDAReadOnlyIterator)iterator(aTargetXPath);
    }
    
    public void move(String aSourceXPath, String aTargetXPath) throws XPathLocationException
    {	Node s=getSingleNode(aSourceXPath);
        Node t=getSingleNode(aTargetXPath);
        if(s.getNodeType()==Node.DOCUMENT_NODE || s.equals(mDocument.getDocumentElement()))
        {	throw new XPathLocationException("Cannot move document root", XPathLocationException.MALFORMEDTARGET);
        }
        if(s.getNodeType()==Node.ELEMENT_NODE)
        {	Document temp=getFragment(aSourceXPath);
            DOMXDA tempxda=new DOMXDA(temp, null, false);
            try
            {	append(tempxda, "/" , aTargetXPath);
            }
            catch(XDOIncompatibilityException e)
            {  /*Can't get thrown since source is same as destination*/ }
        }
        else if(s.getNodeType()==Node.ATTRIBUTE_NODE)
        {	try
            {	append(this, aSourceXPath, aTargetXPath);
            }
            catch(XDOIncompatibilityException e)
            {  /*Can't get thrown since source is same as destination*/ }
        }
        else if(s.getNodeType()==Node.TEXT_NODE)
        {	String text=getText(aSourceXPath,false);
            setText(aTargetXPath, text);
        }
        delete(aSourceXPath);
    }
    
    public boolean next()
    {	if (hasNext())
        {   mRoot=(Node)mNewNodes.get(++mNodeIndex);
            return true;
        }
        return false;
    }
    
    public void removeNS(String aTargetXPath, String prefix) throws XPathLocationException
    {	try
        {   Node node = getSingleNode(aTargetXPath);
            if(node.getNodeType()==node.DOCUMENT_NODE)
            {
                Document d=(Document)node;
                node=d.getDocumentElement();
            }
            if(node.getNamespaceURI()==null)
            {
                throw new XPathLocationException("Target XPath does not have a namespace"+aTargetXPath, XPathLocationException.MALFORMEDTARGET);
            }
            if ((node.getNodeType()==node.ELEMENT_NODE))
            {
                Element e=(Element)node;
                getPrefixResolver();
                try
                {	e.setAttributeNS(XMLNS_DECLARATION_URI, "xmlns:"+prefix, getPrefixResolver().getNamespaceForPrefix(prefix));
                }
                catch(Exception ex)
                {	//Ignore. NS Declaration already exists
                }
                
                try
                {
                    DOM2SAX b2s=new DOM2SAX(node);
                    
                    RemoveNamespaceHandler handler=new RemoveNamespaceHandler(getPrefixResolver().getNamespaceForPrefix(prefix), prefix);
                    b2s.setContentHandler(handler);
                    b2s.parse();
                    Node outnode=handler.getDocument().getDocumentElement();
                    Node innode=mDocument.importNode(outnode, true);
                    replaceNode(innode,node);
                    setDirtyNS();
                }
                catch(SAXException se)
                {
                    throw new XPathLocationException("problem parsing unnamespaced document :"+aTargetXPath, XPathLocationException.MALFORMEDTARGET);
                }
            }
            else if(node.getNodeType()==node.ATTRIBUTE_NODE)
            {
                throw new XPathLocationException("Remove Namespace from attribute not supported "+aTargetXPath, XPathLocationException.MALFORMEDTARGET);
            }
            else throw new XPathLocationException("Remove Namespace - XPath must evaluate to element or attribute :"+aTargetXPath, XPathLocationException.MALFORMEDTARGET);
        } catch (DOMException e)
        {   throw new XPathLocationException(e.toString(), XPathLocationException.MALFORMEDTARGET);
        }
    }
    
    public void rename(String aTargetXPath, String name) throws XPathLocationException
    {	try
        {   Node node = getSingleNode(aTargetXPath);
            if ((node.getNodeType()==node.ELEMENT_NODE || node.getNodeType()==node.DOCUMENT_NODE))
            {
                if(node.getNodeType()==node.DOCUMENT_NODE)
                {	Document d=(Document)node;
                    node=d.getDocumentElement();
                }
                //Set the new node name
                Element e;
                if(node.getNamespaceURI()!=null)
                {
                    String qname="";
                    if(node.getPrefix()!=null) qname=node.getPrefix()+":"+name;
                    e=node.getOwnerDocument().createElementNS(node.getNamespaceURI(), qname);
                }
                else e=node.getOwnerDocument().createElement(name);
                NodeList nodechildren=node.getChildNodes();
                for(int i=0; i<nodechildren.getLength();i++)
                {
                    Node n=nodechildren.item(i).cloneNode(true);  //Had to add this to get deep copy of child nodes
                    e.appendChild(n);
                }
                
                //Surely there's a better way to do this!!!
                NamedNodeMap nnm=node.getAttributes();
                for(int j=0;j<nnm.getLength();j++)
                {
                    Attr att=(Attr)nnm.item(j);
                    Attr attClone = (Attr)att.cloneNode(false);
                    e.setAttributeNode(attClone);
                }
                replaceNode(e, node);
                setDirty();
            }
            else if(node.getNodeType()==node.ATTRIBUTE_NODE)
            {
                Attr att=(Attr)node;
                //Set the new node name
                Attr a;
                if(node.getNamespaceURI()!=null)
                {
                    String qname="";
                    if(att.getPrefix()!=null) qname=att.getPrefix()+":"+name;
                    a=att.getOwnerDocument().createAttributeNS(att.getNamespaceURI(), qname);
                }
                else a=att.getOwnerDocument().createAttribute(name);
                a.setValue(att.getValue());
                Element parent=(Element)att.getOwnerElement();
                if(a.getNamespaceURI()!=null) parent.setAttributeNodeNS(a);
                else parent.setAttributeNode(a);
                parent.removeAttributeNode(att);
                setDirty();
            }
            else   throw new XPathLocationException("XPath must evaluate to element or attribute :"+aTargetXPath, XPathLocationException.MALFORMEDTARGET);
        } catch (DOMException e)
        {   throw new XPathLocationException(e.toString(), XPathLocationException.MALFORMEDTARGET);
        }
    }
    
    public void replace(IXDAReadOnly aSource, String aSourceXPath, String aTargetXPath) throws XPathLocationException, XDOIncompatibilityException
    {	checkType(aSource);
        DOMXDA source=(DOMXDA)aSource;
        DOMXPathResult xpr = (DOMXPathResult)source.evaluateComplexXPath(aSourceXPath);
        switch(xpr.getType())
        {	case IXPathResult.XPR_SINGLE_ELEMENT:
            {	Document d = source.getFragment(aSourceXPath);
                Node n = mDocument.importNode(d.getDocumentElement(), true);
                Node dstNode = getSingleNode(aTargetXPath);
                replaceNode(n,dstNode);
                break;
            }
            case IXPathResult.XPR_NODESET:
            {	//Only case for a nodeset is if the Xpath points to multiple text nodes in an element.
                //System.err.println("DEBUG using new XPR_NODESET code - remove when tested");
                if (xpr.getNodeList().getLength()!=0)
                {	Node item=xpr.getNodeList().item(0);
                    String s=null;
                    if(item.getNodeType()==Node.ATTRIBUTE_NODE)
                    {	Attr attr=(Attr)item;
                        s=attr.getValue();
                    }
                    else if (item.getNodeType()==Node.TEXT_NODE)
                    {	Node parent=xpr.getNodeList().item(0).getParentNode();
                        s=XMLUtils.getInstance().getText(parent);
                    }
                    replaceByText(aTargetXPath, s);
                }
                else
                {	delete(aTargetXPath);
                }
                break;
            }
            case IXPathResult.XPR_NUMBER:
            case IXPathResult.XPR_STRING:
            {	replaceByText(aTargetXPath, xpr.getStringValue());
                break;
            }
        }
        setDirtyNS();
    }
    
    public void replaceByText(String aTargetXPath, String aText) throws XPathLocationException
    {	Node node = getSingleNode(aTargetXPath);
        if(node.getNodeType()==node.DOCUMENT_NODE || node.getParentNode().getNodeType()==node.DOCUMENT_NODE)
        {	throw new XPathLocationException("Cannot replace document element with text!", XPathLocationException.MALFORMEDTARGET);
        }
        Document d=node.getOwnerDocument();
        Node t=d.createTextNode(aText);
        replaceNode(t, node);
        setDirty();
    }
    
    public void setText(String aTargetXPath, String aText) throws XPathLocationException
    {	try
        {   Node node = getSingleNode(aTargetXPath);
            setValue(node,aText);
            setDirty();
        } catch (DOMException e)
        {   throw new XPathLocationException(e.toString(), XPathLocationException.MALFORMEDTARGET);
        }
    }
    
    public void serialize(Writer aWriter, String aTargetXPath, boolean indent) throws XPathLocationException, IOException
    {   Node node=getSingleNode(aTargetXPath);
        innerSerialize(aWriter, node, indent);
    }
    public void serialize(Writer aWriter, boolean indent) throws IOException
    {   innerSerialize(aWriter, mRoot, indent);
    }
    
    /** Returns a debug representation of the contained document.
     * @return Returns a debug representation of the contained document.
     * @param aIndent Set to true to cause tidy (human-readable) output of the document.
     * @throws AccessViolationException Thrown when the client thread doesn't have a lock.
     */
    public String toString(boolean aIndent)
    {   return innerToString(mRoot,aIndent);
    }
    
    public String toString()
    {   return innerToString(mRoot,true);
    }
    
    private String innerToString(Node aNode, boolean aIndent)
    { 	return XMLUtils.getInstance().toXML(aNode, aIndent, true);
    }
    
    private void innerSerialize(Writer aWriter, Node aNode, boolean aIndent) throws IOException
    {	XMLUtils.getInstance().toXML(aWriter, aNode, aIndent, true);
    }
    
    /*WARNING UNSAFE for expert use only - Gets the underlying document*/
    public Document toDOMQuick()
    {	return mDocument;
    }
    
    /*Get a safe copy of the underlying document*/
    public Document toDOM()
    {	return (Document)safeDeepClone(mDocument);
    }
    
    /*Return the XPath of the root node of the current iteration*/
    public String getCurrentXPath()
    {	return XMLUtils.getPathFor(mRoot);
    }
    
    /*Given a namespace URI find the first occurrence below the current context node of the prefix assigned to the namespace and return the prefix.
     Returns empty string for the default namespace.*/
    public String getPrefix(String aNamespace)
    {   return testAttributesForNamespaceDecl((Element)mRoot, aNamespace);
    }
    /*Traverse Elements looking for an attribute namespace declaration matching the namespace URI.*/
    private String testAttributesForNamespaceDecl(Element e, String namespace)
    {   NamedNodeMap nnm=e.getAttributes();
        for(int i=0; i<nnm.getLength(); i++)
        {	Node n=nnm.item(i);
            if(n.getNodeType()==Node.ATTRIBUTE_NODE)
            {   String namespaceURI=n.getNamespaceURI();
                if(namespaceURI!=null && namespaceURI.equals(XMLNS_DECLARATION_URI))
                {	if(n.getNodeValue().equals(namespace))
                    {   String nname=n.getNodeName();
                        int idx=nname.indexOf(":");
                        if(idx>0)
                        {	return nname.substring(idx+1);
                        }
                        else return "";
                    }
                }
            }
        }
        for (Node n=e.getFirstChild(); n!=null; n=n.getNextSibling())
        {	if(n.getNodeType()==Node.ELEMENT_NODE) return testAttributesForNamespaceDecl((Element)n,namespace);
        }
        return null;
    }
}
