package com.streamsicle.songinfo;

import java.util.*;
// Import log4j classes.
import org.ten60.orchextra.*;

/**
 *  <P>
 *
 *  SongInfo that is read in from an ID3v2 tag.</P> <P>
 *
 *  More info can be found at <A href="http://www.id3.org/">http://www.id3.org/
 *  </A>.</P>
 *
 *@author     mhall
 *@created    September 17, 2002
 */
public class SongInfoID3v2 extends SongInfo {
    /**
     *  The delimiter with which a valid tag starts.
     */
    public final static String DELIMITER = "ID3";


    /**
     *  Description of the Method
     *
     *@param  str  Description of the Parameter
     *@return      Description of the Return Value
     */
    private final static String clean(String str) {
        String tmp = str.trim();
        if (tmp.length() == 0) {
            return null;
        } else {
            return tmp;
        }
    }


    private String m_title, m_artist, m_album, m_year, m_comment, m_trackNumber,
            m_genre, m_composer, m_originalArtist, m_url, m_length;
    private boolean m_valid = false;
    // not valid until TIT2 frame found

    private static byte[] PADDING_FRAME_ID = {0x00, 0x00, 0x00, 0x00};


    /**
     *  Create a SongInfo object from an ID3v2 tag.
     *
     *@param  header  Description of the Parameter
     *@param  tag     Description of the Parameter
     */
    public SongInfoID3v2(byte[] header, byte[] tag) {
        byte major = header[3];
        byte revision = header[4];

        if (major <= 2) {
            // ID3v2.2.0 support coming soon...
            return;
        }

        int pos = 0;
        while (pos < tag.length) {
            byte[] frameID = new byte[4];

            if (pos + 4 > tag.length) {
                // no more frames
                return;
            }
            System.arraycopy(tag, pos, frameID, 0, 4);
            if (frameID[0] == PADDING_FRAME_ID[0] &&
                    frameID[1] == PADDING_FRAME_ID[1] &&
                    frameID[2] == PADDING_FRAME_ID[2] &&
                    frameID[3] == PADDING_FRAME_ID[3]) {
                // if( Arrays.equals( frameID, PADDING_FRAME_ID ) ) {
                // end of tag, all done
                return;
            } else {
                String frameIDStr = new String(frameID);

                int frameSize;
                if (major >= 4) {
                    frameSize = syncSafeValue(tag, pos + 4);
                } else {
                    frameSize = nonSyncSafeValue(tag, pos + 4);
                }

                String frameContent = new String();
                try {
                    frameContent = new String(tag, pos + 10, frameSize);
                } catch (java.lang.StringIndexOutOfBoundsException e) {
                    // this seems to occur when an ID3v2 tag exists but doesn't
                    // have any information associated with it, resulting in an
                    // index out of bounds error.  it seems to be safe to ignore it
                    // and just continue
                    OrchextraAccessor.log(OrchextraAccessor.WARNING, this, "Tag might be empty?  Exception:");
                    //log.warn(e);
                }

                setValue(frameIDStr, frameContent);

                pos += 10 + frameSize;
            }
        }
    }


    /**
     *  Sets the value attribute of the SongInfoID3v2 object
     *
     *@param  frameID       The new value value
     *@param  frameContent  The new value value
     */
    private void setValue(String frameID, String frameContent) {
        // DSH Note: if needed, this method could be made faster by
        // representing the frameID as an int and using a case statement, since
        // there are sooo many string comparisons...
        if (frameID.equals("TIT2")) {
            m_title = frameContent.trim();
            m_valid = true;
        } else if (frameID.equals("TALB")) {
            m_album = frameContent.trim();
        } else if (frameID.equals("TRCK")) {
            m_trackNumber = frameContent.trim();
        } else if (frameID.equals("TPE1")) {
            m_artist = frameContent.trim();
        } else if (frameID.equals("TOPE")) {
            m_originalArtist = frameContent.trim();
        } else if (frameID.equals("TCOM")) {
            m_composer = frameContent.trim();
        } else if (frameID.equals("TCON")) {
            m_genre = frameContent.trim();
        } else if (frameID.equals("WXXX")) {
            m_url = frameContent.trim();
        } else if (frameID.equals("TYER")) {
            m_year = frameContent.trim();
        } else if (frameID.equals("COMM")) {
            m_comment = frameContent.trim();
        } else if (frameID.equals("TLEN")) {
            m_length = frameContent.trim();
        } else {
            OrchextraAccessor.log(OrchextraAccessor.WARNING, this, "Couldn't find frame ID: " + frameID);
        }
    }


    /**
     *  Gets the valid attribute of the SongInfoID3v2 object
     *
     *@return    The valid value
     */
    public final boolean isValid() {
        return m_valid;
    }


    /**
     *  Description of the Method
     *
     *@return    Description of the Return Value
     */
    public boolean hasTitle() {
        return (m_title != null);
    }


    /**
     *  Gets the title attribute of the SongInfoID3v2 object
     *
     *@return    The title value
     */
    public String getTitle() {
        return m_title;
    }


    /**
     *  Description of the Method
     *
     *@return    Description of the Return Value
     */
    public boolean hasArtist() {
        return (m_artist != null);
    }


    /**
     *  Gets the artist attribute of the SongInfoID3v2 object
     *
     *@return    The artist value
     */
    public String getArtist() {
        return m_artist;
    }


    /**
     *  Description of the Method
     *
     *@return    Description of the Return Value
     */
    public boolean hasAlbum() {
        return (m_album != null);
    }


    /**
     *  Gets the album attribute of the SongInfoID3v2 object
     *
     *@return    The album value
     */
    public String getAlbum() {
        return m_album;
    }


    /**
     *  Description of the Method
     *
     *@return    Description of the Return Value
     */
    public boolean hasTrackNumber() {
        return (m_trackNumber != null);
    }


    /**
     *  Gets the trackNumber attribute of the SongInfoID3v2 object
     *
     *@return    The trackNumber value
     */
    public String getTrackNumber() {
        return m_trackNumber;
    }


    /**
     *  Description of the Method
     *
     *@return    Description of the Return Value
     */
    public boolean hasYear() {
        return (m_year != null);
    }


    /**
     *  Gets the year attribute of the SongInfoID3v2 object
     *
     *@return    The year value
     */
    public String getYear() {
        return m_year;
    }


    /**
     *  Description of the Method
     *
     *@return    Description of the Return Value
     */
    public boolean hasGenre() {
        return (m_genre != null);
    }


    /**
     *  Gets the genre attribute of the SongInfoID3v2 object
     *
     *@return    The genre value
     */
    public String getGenre() {
        return m_genre;
    }


    /**
     *  Description of the Method
     *
     *@return    Description of the Return Value
     */
    public boolean hasComment() {
        return (m_comment != null);
    }


    /**
     *  Gets the comment attribute of the SongInfoID3v2 object
     *
     *@return    The comment value
     */
    public String getComment() {
        return m_comment;
    }


    /**
     *  Description of the Method
     *
     *@return    Description of the Return Value
     */
    public boolean hasComposer() {
        return (m_composer != null);
    }


    /**
     *  Gets the composer attribute of the SongInfoID3v2 object
     *
     *@return    The composer value
     */
    public String getComposer() {
        return m_composer;
    }


    /**
     *  Description of the Method
     *
     *@return    Description of the Return Value
     */
    public boolean hasOriginalArtist() {
        return (m_originalArtist != null);
    }


    /**
     *  Gets the originalArtist attribute of the SongInfoID3v2 object
     *
     *@return    The originalArtist value
     */
    public String getOriginalArtist() {
        return m_originalArtist;
    }


    /**
     *  Description of the Method
     *
     *@return    Description of the Return Value
     */
    public boolean hasURL() {
        return (m_url != null);
    }


    /**
     *  Gets the uRL attribute of the SongInfoID3v2 object
     *
     *@return    The uRL value
     */
    public String getURL() {
        return m_url;
    }


    /**
     *  Description of the Method
     *
     *@return    Description of the Return Value
     */
    public boolean hasTimeLength() {
        return true;
    }


    /**
     *  Gets the timeLength attribute of the SongInfoID3v2 object
     *
     *@return    The timeLength value
     */
    public String getTimeLength() {
        // in milliseconds
        String song_length;
        long size = 0;
        try {
            if (bitRate > 0) {
                size = fileSize * 8L / (long) bitRate;
            }
        } catch (Exception e) {
            //log.debug("There was an exception in the getTimeLength(), probably a bitrate=0 problem.");
        }
        song_length = Long.toString(size);
        return song_length;
    }


    // converts a syncsafe value as described in the ID3v2 spec into real value
    /**
     *  Description of the Method
     *
     *@param  bytes   Description of the Parameter
     *@param  offset  Description of the Parameter
     *@return         Description of the Return Value
     */
    public static int syncSafeValue(byte[] bytes, int offset) {
        int value = (bytes[offset] & 0x7F) << 21;
        value += (bytes[offset + 1] & 0x7F) << 14;
        value += (bytes[offset + 2] & 0x7F) << 7;
        value += bytes[offset + 3] & 0x7F;
        return value;
    }


    // converts a  non syncsafe value as described in the ID3v2 spec into
    // real value
    /**
     *  Description of the Method
     *
     *@param  bytes   Description of the Parameter
     *@param  offset  Description of the Parameter
     *@return         Description of the Return Value
     */
    public static int nonSyncSafeValue(byte[] bytes, int offset) {
        int value = (bytes[offset] & 0xFF) << 24;
        value += (bytes[offset + 1] & 0xFF) << 16;
        value += (bytes[offset + 2] & 0xFF) << 8;
        value += bytes[offset + 3] & 0xFF;
        return value;
    }
}
