package com.streamsicle.fluid;

import java.io.*;
import java.util.Properties;
// Import log4j classes.
import org.ten60.orchextra.*;

/**
 * The superclass of all streaming classes.
 * <P>
 * A MediaInputStream is a streamer, ie a class from
 * which packets can be read and later fed to an
 * output stream. As there are many different kinds
 * of streamed media there will be a need for many
 * different kinds of MediaInputStream classes.
 * Each new format should be handled by subclassing
 * this class and providing the necessary methods
 * in that class.
 * <P>
 * A MediaTransceiver is used as a tie between a
 * MediaInputStream and an output stream. The media
 * transceiver will use the getPacket() method to
 * read a packet from the stream and then send
 * it to the associated output stream.
 * <P>
 * It is up to the subclasses to implement
 * how long the MediaInputStream should wait between
 * making packets available. The MediaInputStream
 * will compensate this delay with any additional
 * time spent on computations.
 *
 * @author Lars Samuelsson
 */
public abstract class MediaInputStream implements Runnable {
    private InputStream mediaInput;
    private Thread engine;
    private ThreadQueue queue;
    private byte[] packet;
    private int retries = 0;
    private double streamDelayTime;
    private Delay lastDelay;
    private Properties props;
    /**
     * Creates a MediaInputStream without specifying an underlying
     * input stream.
     */
    public MediaInputStream() {
    	this(null);
    }
    /**
     * Creates a MediaInputStream with a specified underlying
     * input stream.
     *
     * @param mediaInput An input stream
     */
    public MediaInputStream(InputStream mediaInput) {
    	queue = new ThreadQueue();
    	this.mediaInput = mediaInput;
      streamDelayTime = System.currentTimeMillis();
      this.lastDelay = null;
    }
    /**
     * Fetches the underlying input stream associated
     * with this media stream.
     *
     * @return The associated input stream
     */
    public InputStream getInputStream() {
    	return mediaInput;
    }
    /**
     * Sets the underlying input stream with which this
     * media stream will be associated.
     *
     * @param mediaInput An input stream
     */
    public void setInputStream(InputStream mediaInput) {
	    this.mediaInput = mediaInput;
    }

    /**
     * Sets properties for this stream
     */
    public void setProperties (Properties value) {
      props = value;
    }

    /**
     * Gets properties for this stream
     */
    public Properties getProperties () {
      return props;
    }

    /**
     * Tells a configurable to configure itself with
     * the specified configuration.
     *
     * This method is used by the notifyConfigurables()
     * method in the Configuration class.
     *
     * @param conf The associated configuration
     */
    public abstract void configure () throws StreamingEngineException;
    /**
     * Gets the number of available bytes that can be
     * read from the underlying input stream without
     * blocking.
     *
     * @return The number of available bytes or -1
     *         if there is no associated input stream
     *         or the MediaInputStream is unconfigured
     */
    public int available() throws IOException {
        if(!engine.isAlive() || mediaInput == null)
            return -1;
        return mediaInput.available();
    }
    /**
     * Sets the number of retries that will be done
     * when no data can be read from the underlying
     * input stream.
     *
     * @param retries The number of retries before
     *                stopping (0 means infinite
     *                retries, default)
     */
    public void setRetries(int retries) {
	    this.retries = retries;
    }
    /**
     * Starts parallel execution.
     *
     * A new thread is created and started on this
     * object.
     */
    public void start() {
        engine = new Thread(this, "MediaInputStream Engine");
        engine.start();
    }
    /**
     * Closes the underlying input stream.
     */
    public void close() throws IOException {
    	mediaInput.close();
    }
    /**
     * Ensures that the thread will sleep for a
     * given amount of time since the last time
     * the method was entered. Will compensate for
     * any time spent away from sleeping.
     *
     * @param delay The amount of time the thread
     *              will be put to sleep
     */
    public void sleep(Delay delay) {
        if(delay == null)
            return;        // no sleeping for null delays
        streamDelayTime += delay.getDelay();
        lastDelay = delay;
        try {
            engine.sleep((long)(streamDelayTime - System.currentTimeMillis()));
        }
        catch(InterruptedException e) {
            // should not happen
            OrchextraAccessor.log(OrchextraAccessor.INFO, this, "My sleep got interrupted, shouldn't have happened.");
        }
        catch(IllegalArgumentException e) {
            // negative time nothing to bother about
        }
    }
    /**
     * Returns a packet from the media stream when
     * such is available.
     *
     * Will block the calling thread until a new
     * packet becomes available.
     *
     * @return A packet read from the media stream
     */
    public byte[] getPacket() {
        queue.enqueue();
        return packet;
    }
    /**
     * Sets a packet and releases the blocked threads.
     *
     * @param packet A packet containing media data
     */
    public void setPacket(byte[] packet) {
        this.packet = packet;
        queue.dequeueAll();
    }
    /**
     * Parallel execution starts here.
     *
     * This thread will read a packet from the underlying
     * stream using the deferred read() method and
     * directly afterwards set the packet using the
     * setPacket() method, hence releasing any waiting
     * threads.
     * <P>
     * Between each read the sleep() method is called
     * with the delay time returned from the getDelay()
     * method. The thread compensates internally for
     * the computational time spent on reading and
     * writing data.
     */
    public void run() {
        int failures = 0;
        byte[] input;
        loop:
            while(engine.isAlive()) {
            try {
                while((input = read()) != null) {
                    setPacket(input);
                    failures = 0;
                    sleep(getDelay());
                }
                mediaInput.close(); // Close when done
            }
            catch(IOException e) {
                    failures++;
                    if(failures == retries)
                break loop;
            }
            }
    }

    /**
     * A method for reading a packet from the underlying input
     * stream.
     *
     * This method needs to be implemented by the subclass as
     * only them knows how to interpret the media format that
     * they handle.
     *
     * @return A packet read from the underlying stream
     */
    public abstract byte[] read() throws IOException;
    /**
     * For fetching the delay that the MediaInputStream will
     * wait before reading another packet.
     *
     * This is directly proportional to the bitrate
     * and framesize of the media that is being sent.
     *
     * @return Between-packet delay
     */
    public abstract Delay getDelay();
    /**
     * This method is used to simplify dynamic loading
     * or media stream classes.
     *
     * @param type A type we would like to know if this
     *             MediaInputStream subclass handles
     * @return     true if the subclass handles the type
     */
    public abstract boolean handlesMedia(String type);

    /**
     * Set the MetaDataListener for this stream.  A stream may or may not
     * produce meta data.
     */
    public void setMetaDataListener( IMetaDataListener listener ) {
      // do nothing by default
    }
}
