package com.streamsicle.songinfo;

import java.io.*;
import com.streamsicle.fluid.MP3FrameReader;
// Import log4j classes.
import org.ten60.orchextra.*;

/**
 *  <P>
 *
 *  SongInfo is the abstract base class for all classes that can represent song
 *  info (generally, the information that could be stored in an ID3v1 or ID3v2
 *  tag). There are two sets of methods: the getXXX and hasXXX methods. One of
 *  each method exists for each property, and all are abstract. None of these
 *  methods take any arguements, the getXXX methods return strings and the
 *  hasXXX methods return booleans.<P>
 *
 *  <P>
 *
 *  The only method implemented in SongInfo is createSongInfo(), which is a
 *  factory method for creating SongInfo objects. In order to add other methods
 *  of reading song info, you should create a subclass of SongInfo and add it to
 *  createSongInfo().</P>
 *
 *@author     Dan Heller (dan@streamsicle.com)
 *@created    September 17, 2002
 */
public abstract class SongInfo {

    protected int bitRate;
    protected long fileSize;


    /**
     *  Ideally every SongInfo object includes a title, so hasTitle <I>should
     *  </I> always return true.
     *
     *@return    Description of the Return Value
     */
    public abstract boolean hasTitle();


    /**
     *  Gets the title attribute of the SongInfo object
     *
     *@return    The title value
     */
    public abstract String getTitle();


    /**
     *  Description of the Method
     *
     *@return    Description of the Return Value
     */
    public abstract boolean hasArtist();


    /**
     *  Gets the artist attribute of the SongInfo object
     *
     *@return    The artist value
     */
    public abstract String getArtist();


    /**
     *  Description of the Method
     *
     *@return    Description of the Return Value
     */
    public abstract boolean hasAlbum();


    /**
     *  Gets the album attribute of the SongInfo object
     *
     *@return    The album value
     */
    public abstract String getAlbum();


    /**
     *  Description of the Method
     *
     *@return    Description of the Return Value
     */
    public abstract boolean hasTrackNumber();


    /**
     *  Gets the trackNumber attribute of the SongInfo object
     *
     *@return    The trackNumber value
     */
    public abstract String getTrackNumber();


    /**
     *  Description of the Method
     *
     *@return    Description of the Return Value
     */
    public abstract boolean hasYear();


    /**
     *  Gets the year attribute of the SongInfo object
     *
     *@return    The year value
     */
    public abstract String getYear();


    /**
     *  Description of the Method
     *
     *@return    Description of the Return Value
     */
    public abstract boolean hasGenre();


    /**
     *  Gets the genre attribute of the SongInfo object
     *
     *@return    The genre value
     */
    public abstract String getGenre();


    /**
     *  Description of the Method
     *
     *@return    Description of the Return Value
     */
    public abstract boolean hasComment();


    /**
     *  Gets the comment attribute of the SongInfo object
     *
     *@return    The comment value
     */
    public abstract String getComment();


    /**
     *  Description of the Method
     *
     *@return    Description of the Return Value
     */
    public abstract boolean hasComposer();


    /**
     *  Gets the composer attribute of the SongInfo object
     *
     *@return    The composer value
     */
    public abstract String getComposer();


    /**
     *  Description of the Method
     *
     *@return    Description of the Return Value
     */
    public abstract boolean hasOriginalArtist();


    /**
     *  Gets the originalArtist attribute of the SongInfo object
     *
     *@return    The originalArtist value
     */
    public abstract String getOriginalArtist();


    /**
     *  Description of the Method
     *
     *@return    Description of the Return Value
     */
    public abstract boolean hasURL();


    /**
     *  Gets the uRL attribute of the SongInfo object
     *
     *@return    The uRL value
     */
    public abstract String getURL();


    /**
     *  Description of the Method
     *
     *@return    Description of the Return Value
     */
    public abstract boolean hasTimeLength();


    /**
     *  Gets the timeLength attribute of the SongInfo object
     *
     *@return    The timeLength value
     */
    public abstract String getTimeLength();


    

    /**
     *  Description of the Method
     *
     *@return    Description of the Return Value
     */
    public String toString() {
        if (hasArtist() && hasTitle()) {
            return getArtist() + " - " + getTitle();
        } else {
            return getTitle();
        }
    }


    /**
     *  Factory method for creating SongInfo objects. Reads in the file and
     *  creates an appropriate SongInfo object for the file.
     *
     *@param  songFile  Description of the Parameter
     *@return           Description of the Return Value
     */
    public static SongInfo createSongInfo(File songFile) {
        // DSH note: this method knows way too much about the different file
        // formats, but to have the subclasses responsible for figuring out it
        // they were applicable to the file would be too inefficient for this
        // processor intensive task
        FileInputStream fis = null;
        FileInputStream songInfoFis = null;
        try {
            fis = new FileInputStream(songFile);
            byte[] v2header = new byte[10];
            fis.read(v2header);
            if (v2header[0] == (byte) 'I' &&
                    v2header[1] == (byte) 'D' &&
                    v2header[2] == (byte) '3') {
                int tagSize = SongInfoID3v2.syncSafeValue(v2header, 6);
                byte[] v2tag = new byte[tagSize];
                fis.read(v2tag);
                SongInfo info = new SongInfoID3v2(v2header, v2tag);
                try {
                    songInfoFis = new FileInputStream(songFile);
                    info = (SongInfoID3v2) getTimeInfo(songInfoFis, info, songFile);
                } catch (java.io.IOException ioe) {
                    OrchextraAccessor.log(OrchextraAccessor.WARNING, null, "While looking for song length (ID3v2): " + songFile);
                } finally {
                    // Try to close the input stream
                    if (songInfoFis != null) {
                        try {
                            songInfoFis.close();
                        } catch (IOException ignore) {
                            // Nothing we can do
                        }
                    }
                }
                /*
                 *  fis = new FileInputStream( songFile );
                 *  info.setFileSize(songFile.length());
                 *  MP3FrameReader frameReader = new MP3FrameReader((InputStream)fis);
                 *  frameReader.nextFrame();
                 *  info.setBitRate(frameReader.getBitrate());
                 */
                if (((SongInfoID3v2) info).isValid()) {
                    fis.close();
                    return info;
                }
            }

            fis.skip(fis.available() - 128);
            byte[] v1tag = new byte[128];
            fis.read(v1tag);
            if (v1tag[0] == (byte) 'T' &&
                    v1tag[1] == (byte) 'A' &&
                    v1tag[2] == (byte) 'G') {
                SongInfo info = new SongInfoID3v1(v1tag);
                try {
                    songInfoFis = new FileInputStream(songFile);
                    info = (SongInfoID3v1) getTimeInfo(songInfoFis, info, songFile);
                } catch (java.io.IOException ioe) {
                    OrchextraAccessor.log(OrchextraAccessor.WARNING, null, "While looking for song length (ID3v1): " + songFile);
                } finally {
                    // Try to close the input stream
                    if (songInfoFis != null) {
                        try {
                            songInfoFis.close();
                        } catch (IOException ignore) {
                            // Nothing we can do
                        }
                    }
                }
                /*
                 *  fis = new FileInputStream( songFile );
                 *  info.setFileSize(songFile.length());
                 *  MP3FrameReader frameReader = new MP3FrameReader((InputStream)fis);
                 *  frameReader.nextFrame();
                 *  info.setBitRate(frameReader.getBitrate());
                 */
                fis.close();
                return info;
            }
        } catch (Exception e) {
            OrchextraAccessor.log(OrchextraAccessor.WARNING, null, "Problem reading ID3 tag for file: " + songFile);
        } finally {
            if (fis != null) {
               try {
                  fis.close();
               } catch (IOException ignore) {
                  // Nothing we can do
               }
            }
        }

        // if there was no ID3 info, or an IOException, just use file name
        SongInfo info = new SongInfoFile(songFile);
        try {
            songInfoFis = new FileInputStream(songFile);
            info = getTimeInfo(songInfoFis, info, songFile);
            /*
             *  info.setFileSize(songFile.length());
             *  MP3FrameReader frameReader = new MP3FrameReader((InputStream)fis);
             *  frameReader.nextFrame();
             *  info.setBitRate(frameReader.getBitrate());
             */
        } catch (Exception e) {
            OrchextraAccessor.log(OrchextraAccessor.WARNING, null, "Problem getting info for file: " + songFile);
        } finally {
            // Try to close the input stream
            if (songInfoFis != null) {
                try {
                    songInfoFis.close();
                } catch (IOException ignore) {
                    // Nothing we can do
                }
            }
        }
        // We cannot return null info
        if (info == null) {
            OrchextraAccessor.log(OrchextraAccessor.WARNING, null, "Had to use degenerate song info for: " + songFile + ".");
            info = new SongInfoDegenerate(songFile);
        }
        return info;
    }


    /**
     *  Gets the timeInfo attribute of the SongInfo class
     *
     *@param  fis       Description of the Parameter
     *@param  info      Description of the Parameter
     *@param  songFile  Description of the Parameter
     *@return           The timeInfo value
     */
    private static SongInfo getTimeInfo(FileInputStream fis, SongInfo info, File songFile) {
        info.setFileSize(songFile.length());
        MP3FrameReader frameReader = new MP3FrameReader((InputStream) fis);
        try {
            frameReader.nextFrame();
        } catch (java.io.IOException e) {
            e.printStackTrace();
        }
        info.setBitRate(frameReader.getBitrate());
        return info;
    }


    /**
     *  Sets the fileSize attribute of the SongInfo object
     *
     *@param  mp3FileSize  The new fileSize value
     */
    public void setFileSize(long mp3FileSize) {
        fileSize = mp3FileSize;
    }


    /**
     *  Sets the bitRate attribute of the SongInfo object
     *
     *@param  fileBitRate  The new bitRate value
     */
    public void setBitRate(int fileBitRate) {
        bitRate = fileBitRate;
    }


    /**
     *  main for testing.
     *
     *@param  args  The command line arguments
     */
    public static void main(String[] args) {
        for (int i = 0; i < args.length; i++) {
            File file = new File(args[i]);
            SongInfo songInfo = createSongInfo(file);

            if (songInfo != null) {

                if (songInfo.hasTitle()) {
                    //log.info("Title: " + songInfo.getTitle());
                }

                if (songInfo.hasArtist()) {
                    //log.info("Artist: " + songInfo.getArtist());
                }

                if (songInfo.hasAlbum()) {
                    //log.info("Album: " + songInfo.getAlbum());
                }

                if (songInfo.hasTrackNumber()) {
                    //log.info("TrackNumber: " + songInfo.getTrackNumber());
                }

                if (songInfo.hasYear()) {
                    //log.info("Year: " + songInfo.getYear());
                }

                if (songInfo.hasGenre()) {
                    //log.info("Genre: " + songInfo.getGenre());
                }

                if (songInfo.hasComment()) {
                    //log.info("Comment: " + songInfo.getComment());
                }

                if (songInfo.hasComposer()) {
                    //log.info("Composer: " + songInfo.getComposer());
                }

                if (songInfo.hasOriginalArtist()) {
                    //log.info("OriginalArtist: " + songInfo.getOriginalArtist());
                }

                if (songInfo.hasURL()) {
                    //log.info("URL: " + songInfo.getURL());
                }

                if (songInfo.hasTimeLength()) {
                    //log.info("Time: " + songInfo.getTimeLength());
                }
            }
        }
    }
}
