package com.streamsicle.fluid;

import java.util.*;
import org.ten60.orchextra.*;


/**
 * Keeps track of the requested shut-off times of the clients and shuts them
 * down as needed.
 * <p>
 * This class wouldn't be needed, but we want to maintain 1.1 compatibility.
 *
 * @author John Watkinson
 */
public class PlayTimer implements Runnable
{

   ///////////////////////////////////////////////////////////////////////////
   // INNER CLASSES
   ///////////////////////////////////////////////////////////////////////////

   private static class TimerEvent
   {
      Client client;
      long time;

      public TimerEvent (Client c, long t) {
         client = c;
         time = t;
      }
   }


   ///////////////////////////////////////////////////////////////////////////
   // FIELDS
   ///////////////////////////////////////////////////////////////////////////

   /**
    * List of events.
    */
   private Vector events;

   /**
    * Running status of the timer.
    */
   private boolean alive;

   ///////////////////////////////////////////////////////////////////////////
   // CONSTRUCTORS
   ///////////////////////////////////////////////////////////////////////////

   /**
    * Creates a new play timer.
    */
   public PlayTimer() {
      events = new Vector();
   }

   ///////////////////////////////////////////////////////////////////////////
   // METHODS
   ///////////////////////////////////////////////////////////////////////////

   /**
    * Called to add an event to the queue.
    */
   public synchronized void addEvent (Client c, long time) {
      // Use a linear insert
      int i;
      for (i = 0; i < events.size(); i++) {
         TimerEvent event = (TimerEvent)events.elementAt(i);
         if (time < event.time) {
            break;
         }
      }
      TimerEvent newEvent = new TimerEvent(c, time);
      events.insertElementAt(newEvent, i);
      OrchextraAccessor.log(OrchextraAccessor.INFO, this, "Client " + c + " sets a timer for " + new Date(time) + ".");
      // Make the timer re-determine its next event
      notify();
   }

   /**
    * The main loop of the PlayTimer.
    */
   public void run () {
      while (alive) {
         synchronized (this) {
            if (events.size() == 0) {
               // Wait for an event to show up
               try {
                  wait();
               } catch (InterruptedException e) {}
            } else {
               long now = System.currentTimeMillis();
               TimerEvent event = (TimerEvent)events.elementAt(0);
               while ((event != null) && (event.time <= now)) {
                  OrchextraAccessor.log(OrchextraAccessor.INFO, this, "Client " + event.client + "'s timer has gone off.");
                  // Shut this client down
                  event.client.shutdown();
                  // Remove it from the queue
                  events.removeElementAt(0);
                  if (events.size() > 0) {
                     event = (TimerEvent)events.elementAt(0);
                  } else {
                     event = null;
                  }
               }
               if (event != null) {
                  // Compute how long before we need to wake up and shut the next
                  // client down.
                  long waitTime = event.time - now;
                  try {
                     wait(waitTime);
                  } catch (InterruptedException e) {}
               }
            }
         }
      }
   }

   /**
    * Starts the timer.
    */
   public void start () {
      alive = true;
      new Thread(this, "Play Timer").start();
   }

   /**
    * Shuts down the timer.
    */
   public synchronized void shutdown () {
      alive = false;
      // Get it to wake up right now
      notify();
   }
}