package com.streamsicle.fluid;

import java.io.*;
import java.util.*;
import org.ten60.orchextra.*;

/**
 * A class for extracting the ID3 tag from an MP3 file.
 *
 * An ID3 tag can be found at the very end of many
 * MP3 files and contains information about the
 * content.
 *
 * @author Lars Samuelsson
 */
public class ID3Tag {
    /**
     * The size of an ID3 tag in bytes.
     *
     * This field is valid for ID3 v1.0 and
     * maybe also for later versions. The
     * size of the tag might have changed to
     * be able to hold more information
     * though.
     */
    public static final int TAGSIZE = 128;
    /**
     * The delimiter with which a valid tag
     * starts.
     */
    public static final String delimiter = "TAG";
    /**
     * Contains the genres as specified in the
     * ID3 v1.0 specification.
     */

    // This is the log4j main logging class

    public static final String[] genres = {
        "Blues",
        "Classic Rock",
        "Country",
        "Dance",
        "Disco",
        "Funk",
        "Grunge",
        "Hip-Hop",
        "Jazz",
        "Metal",
        "New Age",
        "Oldies",
        "Other",
        "Pop",
        "R&B",
        "Rap",
        "Reggae",
        "Rock",
        "Techno",
        "Industrial",
        "Alternative",
        "Ska",
        "Death Metal",
        "Pranks",
        "Soundtrack",
        "Euro-Techno",
        "Ambient",
        "Trip-Hop",
        "Vocal",
        "Jazz+Funk",
        "Fusion",
        "Trance",
        "Classical",
        "Instrumental",
        "Acid",
        "House",
        "Game",
        "Sound Clip",
        "Gospel",
        "Noise",
        "AlternRock",
        "Bass",
        "Soul",
        "Punk",
        "Space",
        "Meditative",
        "Instrumental Pop",
        "Instrumental Rock",
        "Ethnic",
        "Gothic",
        "Darkwave",
        "Techno-Industrial",
        "Electronic",
        "Pop-Folk",
        "Eurodance",
        "Dream",
        "Southern Rock",
        "Comedy",
        "Cult",
        "Gangsta",
        "Top 40",
        "Christian Rap",
        "Pop/Funk",
        "Jungle",
        "Native American",
        "Cabaret",
        "New Wave",
        "Psychadelic",
        "Rave",
        "Showtunes",
        "Trailer",
        "Lo-Fi",
        "Tribal",
        "Acid Punk",
        "Acid Jazz",
        "Polka",
        "Retro",
        "Musical",
        "Rock & Roll",
        "Hard Rock",
        "Folk",
        "Folk-Rock",
        "National Folk",
        "Swing",
        "Fast Fusion",
        "Bebob",
        "Latin",
        "Revival",
        "Celtic",
        "Bluegrass",
        "Avantgarde",
        "Gothic Rock",
        "Progressive Rock",
        "Psychedelic Rock",
        "Symphonic Rock",
        "Slow Rock",
        "Big Band",
        "Chorus",
        "Easy Listening",
        "Acoustic",
        "Humour",
        "Speech",
        "Chanson",
        "Opera",
        "Chamber Music",
        "Sonata",
        "Symphony",
        "Booty Bass",
        "Primus",
        "Porn Groove",
        "Satire",
        "Slow Jam",
        "Club",
        "Tango",
        "Samba",
        "Folklore",
        "Ballad",
        "Power Ballad",
        "Rhythmic Soul",
        "Freestyle",
        "Duet",
        "Punk Rock",
        "Drum Solo",
        "A capella",
        "Euro-House",
        "Dance Hall"
     };
    private String title;
    private String artist;
    private String album;
    private String year;
    private String comment;
    private String genre;
    private String complete;
    /**
     * Extracts an ID3 tag from the given filename or
     * uses the filename itself to deduce information.
     *
     * The filename is used if there's no tag in the
     * file. If the filename is on the form
     * artist - track both fields will be available
     * for access after initialization.
     *
     * @param filename The name of the file containing
     *                 an ID3 tag
     */
    public ID3Tag(String filename) throws IOException {
	String nada = "";
	byte whitespace = 32;
        byte[] mp3tag = new byte[TAGSIZE];
        for(int i=0; i<TAGSIZE; i++) {
            mp3tag[i] = whitespace;
        }
        FileInputStream mp3input = new FileInputStream(filename);
        mp3input.skip(mp3input.available() - TAGSIZE);
        mp3input.read(mp3tag);
	mp3input.close();
        if(new String(mp3tag, 0, 3).equals(delimiter)) {
            title = new String(mp3tag, 3, 30).trim();
            artist = new String(mp3tag, 33, 30).trim();
            album = new String(mp3tag, 63, 30).trim();
            year = new String(mp3tag, 93, 4).trim();
            comment = new String(mp3tag, 97, 30).trim();
            if(mp3tag[TAGSIZE - 1] < 126 && mp3tag[TAGSIZE - 1] >= 0) {
                genre = genres[mp3tag[TAGSIZE - 1]];
            }
            else {
                genre = nada;
            }
            complete = artist + " - " + title;
        }
        else {
            String str = filename;
            StringTokenizer path = new StringTokenizer(filename, File.separator);
            while(path.hasMoreTokens()) {
                str = path.nextToken();
            }
            StringTokenizer suffix = new StringTokenizer(str, ".");
            if(suffix.hasMoreTokens()) {
                str = suffix.nextToken();
            }
            complete = str;
            StringTokenizer st = new StringTokenizer(str, "-");
            if(st.countTokens() >= 2) {
                artist = st.nextToken().trim();
		title = nada;
		while(st.hasMoreTokens()) {
		    title += st.nextToken();
		    if(st.hasMoreTokens())
			title += "-";
		}
		title = title.trim();
            }
            else if(st.hasMoreTokens()) {
                artist = nada;
                title = st.nextToken().trim();
            }
	    else {
		artist = nada;
		title = str;
	    }
            album = nada;
            year = nada;
            comment = nada;
            genre = nada;
        }
    }
    /**
     * Fetches the title of this track.
     *
     * @return The title of the track
     */
    public String getTitle() {
        return title;
    }
    /**
     * Fetches the artist who made the track.
     *
     * @return The artist
     */
    public String getArtist() {
        return artist;
    }
    /**
     * Fetches the name of the album from which
     * this track is taken.
     *
     * @return An album name
     */
    public String getAlbum() {
        return album;
    }
    /**
     * Fetches the year this track was recorded.
     *
     * @return A year (String format)
     */
    public String getYear() {
        return year;
    }
    /**
     * Fetches the comment contained in the tag.
     *
     * @return A comment for this track
     */
    public String getComment() {
        return comment;
    }
    /**
     * Fetches the genre for this track.
     *
     * @return A genre from the list of genres
     */
    public String getGenre() {
        return genre;
    }
    /**
     * Fetches the full name of this track, ie
     * artist - track.
     *
     * @return The full name of this track
     */
    public String getCompleteName() {
        return complete;
    }
}
