package com.streamsicle.fluid;

import java.io.*;
import java.util.Properties;

/**
 * This class handles a response from a given client query.
 *
 * Maybe not as general as it ought to be, and right now it
 * is aware of only a few clients and what response these expect
 * to receive.
 *
 * @author Lars Samuelsson
 * @author forked by John Watkinson
 */
public class ClientResponse {
    private String EOL = "\r\n"; // end of line
    private String EOR = "\r\n"; // end of response
    private ClientQuery query;
    private byte[] response;
    private Properties props;

    /**
     * Creates a client response object from the given
     * client query.
     *
     * @param query A client query object
     */
    public ClientResponse( ClientQuery query, Properties properties ) {
      props = properties;
   	this.query = query;
   	response = createResponse();
    }

    /**
     * Writes a response to the given output stream.
     *
     * @param out   The output stream that the response
     *              will be written to
     */
    public void respond(OutputStream out) throws IOException {
	if(response != null) {
	    out.write(response);
	    out.flush();
	}
    }

    /**
     * Used to check if this client is one that
     * needs extra information sent back in the response.
     *
     * @param client The name of the client program
     * @return       true if the client needs extra information
     */
    public boolean isExtended(String clientName) {
	if(clientName == null)
	    return false;
	if(clientName.indexOf("MSIE") >= 0)
	    return true;
	if(clientName.indexOf("RMA") >= 0)
	    return true;
	if(clientName.indexOf("NSPlayer") >= 0)
	    return true;
	return false;
    }

    /**
     * This is the method creating the response.
     *
     * It will first add the general response strings
     * and after that the extended response strings if
     * this client is one that needs them.
     *
     * @return the full response as an array of bytes
     */
    protected byte[] createResponse() {
       // these waste memory if they are instance variables...
       // doh, scratch that, but too late now... DSH
       String[] generalResponse, extendedResponse;

       String response = new String();

       generalResponse = getGeneralResponses();
       for(int i = 0; i < generalResponse.length; i++) {
          response += generalResponse[i] + EOL;
       }

       String clientName = query.getProperty("User-Agent");
       if(isExtended(clientName)) {
          extendedResponse = getExtendedResponses();
          for(int i = 0; i < extendedResponse.length; i++) {
             response += extendedResponse[i] + EOL;
          }
       }

       response += EOR;
       return response.getBytes();
    }

    private String[] getGeneralResponses() {
       String serverName = props.getProperty(
             "streamsicle.description",
             "- - - S T R E A M S I C L E - - - take a lick" );
       String mimeType = props.getProperty( "streamsicle.mime",
             "audio/mpeg" );

       // DSH TODO: add icy-url to web interface
       // DSH TODO: add icy-genre
       // DSH TODO: add icy-notice to streamsicle.sf.net?
       return new String[] {
          "HTTP/1.0 200 OK",
          "Server: Fluid Streaming Server",
          "Content-type: " + mimeType,
          "icy-name: " + serverName,
          "icy-metaint: " + IMetaDataListener.META_DATA_INTERVAL
             // should be ignored if unsupported by client
       };
    }

    private String[] getExtendedResponses() {
       return new String[] {
          "Cache-control: no-cache",
          "Pragma: no-cache",
          "Connection: close",
          "Content-length: -1"
       };
    }

    /**
     * Returns the response in string format
     *
     * @return the response in string format
     */
    public String toString() {
	if(response == null)
	    return "";
	return new String(response);
    }
}
