/*
 *
 *    Artistic License
 *
 *    Preamble
 *
 *    The intent of this document is to state the conditions under which a
 *    Package may be copied, such that the Copyright Holder maintains some
 *    semblance of artistic control over the development of the package,
 *    while giving the users of the package the right to use and distribute
 *    the Package in a more-or-less customary fashion, plus the right to make
 *    reasonable modifications.
 *
 *    Definitions:
 *
 *    "Package" refers to the collection of files distributed by the
 *    Copyright Holder, and derivatives of that collection of files created
 *    through textual modification.
 *
 *    "Standard Version" refers to such a Package if it has not been
 *    modified, or has been modified in accordance with the wishes of the
 *    Copyright Holder.
 *
 *    "Copyright Holder" is whoever is named in the copyright or copyrights
 *    for the package.
 *
 *    "You" is you, if you're thinking about copying or distributing this Package.
 *
 *    "Reasonable copying fee" is whatever you can justify on the basis of
 *    media cost, duplication charges, time of people involved, and so
 *    on. (You will not be required to justify it to the Copyright Holder,
 *    but only to the computing community at large as a market that must bear
 *    the fee.)
 *
 *    "Freely Available" means that no fee is charged for the item itself,
 *    though there may be fees involved in handling the item. It also means
 *    that recipients of the item may redistribute it under the same
 *    conditions they received it.
 *
 *    1. You may make and give away verbatim copies of the source form of the
 *    Standard Version of this Package without restriction, provided that you
 *    duplicate all of the original copyright notices and associated
 *    disclaimers.
 *
 *    2. You may apply bug fixes, portability fixes and other modifications
 *    derived from the Public Domain or from the Copyright Holder. A Package
 *    modified in such a way shall still be considered the Standard Version.
 *
 *    3. You may otherwise modify your copy of this Package in any way,
 *    provided that you insert a prominent notice in each changed file
 *    stating how and when you changed that file, and provided that you do at
 *    least ONE of the following:
 *
 *        a) place your modifications in the Public Domain or otherwise make
 *        them Freely Available, such as by posting said modifications to
 *        Usenet or an equivalent medium, or placing the modifications on a
 *        major archive site such as ftp.uu.net, or by allowing the Copyright
 *        Holder to include your modifications in the Standard Version of the
 *        Package.
 *
 *        b) use the modified Package only within your corporation or
 *        organization.
 *
 *        c) rename any non-standard executables so the names do not conflict
 *        with standard executables, which must also be provided, and provide
 *        a separate manual page for each non-standard executable that
 *        clearly documents how it differs from the Standard Version.
 *
 *        d) make other distribution arrangements with the Copyright Holder.
 *
 *    4. You may distribute the programs of this Package in object code or
 *    executable form, provided that you do at least ONE of the following:
 *
 *        a) distribute a Standard Version of the executables and library
 *        files, together with instructions (in the manual page or
 *        equivalent) on where to get the Standard Version.
 *
 *        b) accompany the distribution with the machine-readable source of
 *        the Package with your modifications.
 *
 *        c) accompany any non-standard executables with their corresponding
 *        Standard Version executables, giving the non-standard executables
 *        non-standard names, and clearly documenting the differences in
 *        manual pages (or equivalent), together with instructions on where
 *        to get the Standard Version.
 *
 *        d) make other distribution arrangements with the Copyright Holder.
 *
 *    5. You may charge a reasonable copying fee for any distribution of this
 *    Package. You may charge any fee you choose for support of this
 *    Package. You may not charge a fee for this Package itself.  However,
 *    you may distribute this Package in aggregate with other (possibly
 *    commercial) programs as part of a larger (possibly commercial) software
 *    distribution provided that you do not advertise this Package as a
 *    product of your own.
 *
 *    6. The scripts and library files supplied as input to or produced as
 *    output from the programs of this Package do not automatically fall
 *    under the copyright of this Package, but belong to whomever generated
 *    them, and may be sold commercially, and may be aggregated with this
 *    Package.
 *
 *    7. C or perl subroutines supplied by you and linked into this Package
 *    shall not be considered part of this Package.
 *
 *    8. The name of the Copyright Holder may not be used to endorse or
 *    promote products derived from this software without specific prior
 *    written permission.
 *
 *    9. THIS PACKAGE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED
 *    WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 *    MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 */
package org.chiba.xml.xforms.ui;

import org.apache.log4j.Category;
import org.chiba.xml.xforms.Binding;
import org.chiba.xml.xforms.Initializer;
import org.chiba.xml.xforms.Model;
import org.chiba.xml.xforms.NamespaceCtx;
import org.chiba.xml.xforms.exception.XFormsException;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * Helper class to wrap a single entry of a repeat. In the internal DOM of the processor xforms:group
 * Elements are created for every repeat entry. This element has an additional chiba:transient attribute
 * which signals a stylesheet writer, that this group was not part of the original form and can be ignored
 * for rendering. Adding these transient groups helps to wrap mixed markup in the input document and also
 * simplifies writing UI transformations for repeated data.
 *
 * @author <a href="mailto:unl@users.sourceforge.net">Ulrich Nicolas Liss&eacute;</a>
 * @version $Id: RepeatEntry.java,v 1.19 2004/01/27 09:59:56 joernt Exp $
 */
public class RepeatEntry extends Group {
	private static final Category LOGGER = Category.getInstance(RepeatEntry.class);
	private Repeat repeat = null;
	private int position = 0;

	/**
	 * Creates a new RepeatEntry object.
	 *
	 * @param element the DOM Element
	 * @param model the Model this repeatentry belongs to
	 */
	public RepeatEntry(Element element, Model model) {
		super(element, model);
	}

    // implementation of 'org.chiba.xml.xforms.Binding'

	/**
	 * Returns the binding expression.
	 *
	 * @return the binding expression.
	 */
	public String getBindingExpression() {
		// filter the enclosing repeat
		return getRepeat().getBindingExpression() + "[" + this.position + "]";
	}

	/**
	 * Returns the id of the binding element.
	 *
	 * @return the id of the binding element.
	 */
	public String getBindingId() {
		// filter the enclosing repeat
		return getRepeat().getBindingId();
	}

	/**
	 * Returns the enclosing element.
	 *
	 * @return the enclosing element.
	 */
	public Binding getEnclosingBinding() {
		// filter the enclosing repeat
		return getRepeat().getEnclosingBinding();
	}

	/**
	 * Returns the location path.
	 *
	 * @return the location path.
	 */
	public String getLocationPath() {
		// filter the enclosing repeat
		return getRepeat().getLocationPath() + "[" + this.position + "]";
	}

	/**
	 * Returns the model id of the binding element.
	 *
	 * @return the model id of the binding element.
	 */
	public String getModelId() {
		// filter the enclosing repeat
		return getRepeat().getModelId();
	}

    // repeat entry specific methods

	/**
	 * Returns the position of this <code>repeat entry</code>.
	 *
	 * @return the position of this <code>repeat entry</code>.
	 */
	public int getPosition() {
		return this.position;
	}

	/**
	 * Returns the parent <code>repeat</code>.
	 *
	 * @return the parent <code>repeat</code>.
	 */
	public Repeat getRepeat() {
		if (this.repeat == null) {
			this.repeat = (Repeat)getParentObject();
		}

		return this.repeat;
	}

    // lifecycle methods

	/**
	 * Performs element disposal.
	 *
	 * @throws XFormsException if any error occurred during disposal.
	 */
	public void dispose() throws XFormsException {
		if (getLogger().isDebugEnabled()) {
			getLogger().debug(this + " dispose");
		}

		disposeChildren();
		disposeRepeatEntry();
		disposeSelf();
	}

	/**
	 * Performs element init.
	 *
	 * @throws XFormsException if any error occurred during init.
	 */
	public void init() throws XFormsException {
		if (getLogger().isDebugEnabled()) {
			getLogger().debug(this + " init");
		}

		initializeRepeatEntry();
		Initializer.initializeUIElements(this.model, this.element, this.id);
	}

	/**
	 * Performs element update.
	 *
	 * @throws XFormsException if any error occurred during update.
	 */
	public void update() throws XFormsException {
		if (getLogger().isDebugEnabled()) {
			getLogger().debug(this + " update");
		}

		updateChildren();
	}

    // standard methods

    /**
     * Returns a string representation of this object.
     *
     * @return a string representation of this object.
     */
    public String toString() {
        return "[" + this.element.getNodeName() + "/repeatentry id='" + getId() + "']";
    }

	/**
	 * Returns the logger object.
	 *
	 * @return the logger object.
	 */
	protected Category getLogger() {
		return LOGGER;
	}

    // lifecycle template methods

	/**
	 * Disposes this repeat entry by releasing its reference to the
     * enclosing repeat.
	 */
	protected final void disposeRepeatEntry() {
		this.repeat = null;
	}

    /**
     * Initializes this repeat entry by cloning the repeat prototype.
     */
	protected final void initializeRepeatEntry() {
		this.position = Integer.valueOf(this.element.getAttributeNS(NamespaceCtx.CHIBA_NS, "position"))
		                       .intValue();

		Element prototype = getRepeat().getPrototype();
		NodeList children = prototype.getChildNodes();

		for (int i = 0; i < children.getLength(); i++) {
			this.element.appendChild(children.item(i).cloneNode(true));
		}
	}

}
//end of class
