/*
 *
 *    Artistic License
 *
 *    Preamble
 *
 *    The intent of this document is to state the conditions under which a Package may be copied, such that
 *    the Copyright Holder maintains some semblance of artistic control over the development of the
 *    package, while giving the users of the package the right to use and distribute the Package in a
 *    more-or-less customary fashion, plus the right to make reasonable modifications.
 *
 *    Definitions:
 *
 *    "Package" refers to the collection of files distributed by the Copyright Holder, and derivatives
 *    of that collection of files created through textual modification.
 *
 *    "Standard Version" refers to such a Package if it has not been modified, or has been modified
 *    in accordance with the wishes of the Copyright Holder.
 *
 *    "Copyright Holder" is whoever is named in the copyright or copyrights for the package.
 *
 *    "You" is you, if you're thinking about copying or distributing this Package.
 *
 *    "Reasonable copying fee" is whatever you can justify on the basis of media cost, duplication
 *    charges, time of people involved, and so on. (You will not be required to justify it to the
 *    Copyright Holder, but only to the computing community at large as a market that must bear the
 *    fee.)
 *
 *    "Freely Available" means that no fee is charged for the item itself, though there may be fees
 *    involved in handling the item. It also means that recipients of the item may redistribute it under
 *    the same conditions they received it.
 *
 *    1. You may make and give away verbatim copies of the source form of the Standard Version of this
 *    Package without restriction, provided that you duplicate all of the original copyright notices and
 *    associated disclaimers.
 *
 *    2. You may apply bug fixes, portability fixes and other modifications derived from the Public Domain
 *    or from the Copyright Holder. A Package modified in such a way shall still be considered the
 *    Standard Version.
 *
 *    3. You may otherwise modify your copy of this Package in any way, provided that you insert a
 *    prominent notice in each changed file stating how and when you changed that file, and provided that
 *    you do at least ONE of the following:
 *
 *        a) place your modifications in the Public Domain or otherwise make them Freely
 *        Available, such as by posting said modifications to Usenet or an equivalent medium, or
 *        placing the modifications on a major archive site such as ftp.uu.net, or by allowing the
 *        Copyright Holder to include your modifications in the Standard Version of the Package.
 *
 *        b) use the modified Package only within your corporation or organization.
 *
 *        c) rename any non-standard executables so the names do not conflict with standard
 *        executables, which must also be provided, and provide a separate manual page for each
 *        non-standard executable that clearly documents how it differs from the Standard
 *        Version.
 *
 *        d) make other distribution arrangements with the Copyright Holder.
 *
 *    4. You may distribute the programs of this Package in object code or executable form, provided that
 *    you do at least ONE of the following:
 *
 *        a) distribute a Standard Version of the executables and library files, together with
 *        instructions (in the manual page or equivalent) on where to get the Standard Version.
 *
 *        b) accompany the distribution with the machine-readable source of the Package with
 *        your modifications.
 *
 *        c) accompany any non-standard executables with their corresponding Standard Version
 *        executables, giving the non-standard executables non-standard names, and clearly
 *        documenting the differences in manual pages (or equivalent), together with instructions
 *        on where to get the Standard Version.
 *
 *        d) make other distribution arrangements with the Copyright Holder.
 *
 *    5. You may charge a reasonable copying fee for any distribution of this Package. You may charge
 *    any fee you choose for support of this Package. You may not charge a fee for this Package itself.
 *    However, you may distribute this Package in aggregate with other (possibly commercial) programs as
 *    part of a larger (possibly commercial) software distribution provided that you do not advertise this
 *    Package as a product of your own.
 *
 *    6. The scripts and library files supplied as input to or produced as output from the programs of this
 *    Package do not automatically fall under the copyright of this Package, but belong to whomever
 *    generated them, and may be sold commercially, and may be aggregated with this Package.
 *
 *    7. C or perl subroutines supplied by you and linked into this Package shall not be considered part of
 *    this Package.
 *
 *    8. The name of the Copyright Holder may not be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 *    9. THIS PACKAGE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED
 *    WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 *    MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 */
package org.chiba.xml.xforms.ui;

import org.apache.log4j.Category;
import org.apache.xerces.dom.ElementImpl;
import org.chiba.xml.util.DOMUtil;
import org.chiba.xml.xforms.Bind;
import org.chiba.xml.xforms.Instance;
import org.chiba.xml.xforms.Model;
import org.chiba.xml.xforms.NamespaceCtx;
import org.chiba.xml.xforms.exception.XFormsException;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

import java.util.List;

/**
 * Implementation of XForms Repeat element.
 *
 * @author <a href="mailto:unl@users.sourceforge.net">Ulrich Nicolas Liss&eacute;</a>
 * @version $Id: Repeat.java,v 1.28 2004/02/06 15:15:11 joernt Exp $
 */
public class Repeat extends ContainerElement {
	private static final Category LOGGER = Category.getInstance(Repeat.class);
	private Element prototype = null;
    private int contextSize=0;

    /**
	 * Creates a new Repeat object.
	 *
	 * @param element the DOM Element
	 * @param model the Model this repeat belongs to
	 */
	public Repeat(Element element, Model model) {
		super(element, model);
	}

	// implementation of 'org.chiba.xml.xforms.BindingElement'

	/**
	 * Returns the binding expression.
	 *
	 * @return the binding expression.
	 */
	public String getBindingExpression() {
		if (hasNodesetAttribute()) {
			return this.element.getAttributeNS(NamespaceCtx.XFORMS_NS, NODESET_ATTRIBUTE);
		}

		if (hasRepeatNodesetAttribute()) {
			return this.element.getAttributeNS(NamespaceCtx.XFORMS_NS, REPEAT_NODESET_ATTRIBUTE);
		}

		return null;
	}

	/**
	 * Sets the index of this <code>repeat</code>.
	 *
	 * @param index the index of this <code>repeat</code>.
	 */
	public void setIndex(int index) {
		setIndexAttribute(index);

		if (isBound()) {
			this.model.getInstance(getInstanceId()).setNodesetPosition(getLocationPath(), index);
		}
	}

	/**
	 * Returns the index of this <code>repeat</code>.
	 *
	 * @return the index of this <code>repeat</code>.
	 */
	public int getIndex() {
//		if (isBound()) {
//			return this.model.getInstance(getInstanceId()).getNodesetPosition(getLocationPath());
//		}

		return Integer.valueOf(this.element.getAttributeNS(NamespaceCtx.CHIBA_NS, "index")).intValue();
	}

    /**
     * returns the size of the context (nodeset) of this repeat
     * @return the size of the context (nodeset) of this repeat
     */
    public int getContextSize() {
        return contextSize;
    }

	/**
	 * Returns the model binding of this element.
	 *
	 * @return the model binding of this element.
	 */
	public Bind getModelBinding() {
		if (hasRepeatBindAttribute()) {
			String bindId = this.element.getAttributeNS(NamespaceCtx.XFORMS_NS, REPEAT_BIND_ATTRIBUTE);

			return (Bind)this.container.lookup(bindId);
		}

		return super.getModelBinding();
	}

	// repeat specific methods

	/**
	 * Returns the prototype of this <code>repeat</code>.
	 *
	 * @return the prototype of this <code>repeat</code>.
	 */
	public Element getPrototype() {
		return this.prototype;
	}

	/**
	 * Performs element disposal.
	 *
	 * @throws XFormsException if any error occurred during disposal.
	 */
	public void dispose() throws XFormsException {
		if (getLogger().isDebugEnabled()) {
			getLogger().debug(this + " dispose");
		}

		disposeChildren();
		disposeDataElement();
		disposeRepeat();
		disposeSelf();
	}

	// binding related methods

	/**
	 * Checks wether this element has a model binding.
	 * <p>
	 * This element has a model binding if it has a <code>bind</code> or a
	 * <code>repeat-bind</code> attribute.
	 *
	 * @return <code>true</code> if this element has a model binding, otherwise
	 * <code>false</code>.
	 */
	public boolean hasModelBinding() {
		return super.hasModelBinding() || hasRepeatBindAttribute();
	}

	/**
	 * Checks wether this element has an ui binding.
	 * <p>
	 * This element has an ui binding if it has a <code>nodeset</code> or a
	 * <code>repeat-nodeset</code> attribute.
	 *
	 * @return <code>true</code> if this element has an ui binding, otherwise
	 * <code>false</code>.
	 */
	public boolean hasUIBinding() {
		return hasNodesetAttribute() || hasRepeatNodesetAttribute();
	}

	// lifecycle methods

	/**
	 * Performs element init.
	 *
	 * @throws XFormsException if any error occurred during init.
	 */
	public void init() throws XFormsException {
		if (getLogger().isDebugEnabled()) {
			getLogger().debug(this + " init");
		}

		initializeInstanceNode();
		initializeRepeat();
		initializeDataElement();
	}

	/**
	 * Performs element update.
	 *
	 * @throws XFormsException if any error occurred during update.
	 */
	public void update() throws XFormsException {
		if (getLogger().isDebugEnabled()) {
			getLogger().debug(this + " update");
		}

		updateRepeat();
		updateDataElement();
		updateChildren();
	}

	/**
	 * Returns the logger object.
	 *
	 * @return the logger object.
	 */
	protected Category getLogger() {
		return LOGGER;
	}

	/**
	 * free all resources bound to this repeat
	 */
	protected final void disposeRepeat() {
		this.prototype = null;
	}

	// lifecycle template methods

	/**
	 * Initializes the <code>repeat</code> element.
	 * <ol>
	 * <li>The repeat prototype is stored.</li>
	 * <li>The repeat index is set to <code>1</code>.</li>
	 * <li>For each member of the nodeset this repeat is bound to, the
	 * user interface is generated by cloning the repeat prototype.</li>
	 * <li>The generated user interface is initialized.</li>
	 * </ol>
	 */
	protected final void initializeRepeat() throws XFormsException {
		// store prototype
		storeRepeatPrototype();

		// count nodeset
		int count;

		if (isBound()) {
			String path = getLocationPath();
			Instance instance = this.model.getInstance(getInstanceId());
			count = instance.countNodeset(path);

			if (count == 0) {
				// todo: fatal ?
				getLogger().warn(this + " init: nodeset " + path + " does not exist");

				// set count to 1 to enforce instance node creation
				count = 1;
			}

			if (getLogger().isDebugEnabled()) {
				getLogger().debug(this + " init: creating " + count + " repeat " +
				                  ((count == 1)
				                   ? "entry"
				                   : "entries"));
			}

			// init new repeat entry for each collection member
			addRepeatEntries(0, count);

			// set index attribute
			setIndexAttribute(1);
		} else {
			// todo: fatal ?
			getLogger().warn(this + " init: repeat is not bound");

			// set index attribute
			setIndexAttribute(0);
		}
	}

	/**
	 * update the repeat
	 *
	 * @throws XFormsException __UNDOCUMENTED__
	 */
	protected final void updateRepeat() throws XFormsException {
		if (isBound()) {
			// get number of children (minus 1 for data element)
			int size = DOMUtil.getChildElements(this.element).size() - 1;

			// get instance releated data
			String path = getLocationPath();
			Instance instance = this.model.getInstance(getInstanceId());
			int count = instance.countNodeset(path);

			if (size < count) {
				if (getLogger().isDebugEnabled()) {
					int entries = count - size;
					getLogger().debug(this + " update: adding " + entries + " repeat " +
					                  ((entries == 1)
					                   ? "entry"
					                   : "entries"));
				}

				// init and initialize missing repeat entries
				addRepeatEntries(size, count);
			}

			if (size > count) {
				int entries = size - count;

				if (getLogger().isDebugEnabled()) {
					getLogger().debug(this + " update: removing " + entries + " repeat " +
					                  ((entries == 1)
					                   ? "entry"
					                   : "entries"));
				}

				// remove obsolete repeat entries
				removeRepeatEntries(entries);
			}

			// set index attribute
			setIndexAttribute(instance.getNodesetPosition(path));
		}
	}

	private void setIndexAttribute(int index) {
		if (getLogger().isDebugEnabled()) {
			getLogger().debug(this + " set index: " + index);
		}

		this.element.setAttributeNS(NamespaceCtx.CHIBA_NS, NamespaceCtx.CHIBA_PREFIX + ":index",
		                            String.valueOf(index));
	}

	// helper methods
	private void addRepeatEntries(int start, int max) throws XFormsException {
		Node data = DOMUtil.findFirstChildNS(this.element, NamespaceCtx.CHIBA_NS, "data");

        this.contextSize = 0;
		for (int position = start + 1; position <= max; position++) {
			// create group for repeat entry
			Element group = this.element.getOwnerDocument().createElementNS(NamespaceCtx.XFORMS_NS,
			                                                                this.xformsPrefix + ":" + GROUP);
			group.setAttributeNS(NamespaceCtx.CHIBA_NS, NamespaceCtx.CHIBA_PREFIX + ":position",
			                     String.valueOf(position));
			group.setAttributeNS(NamespaceCtx.CHIBA_NS, NamespaceCtx.CHIBA_PREFIX + ":transient",
			                     String.valueOf(true));
			this.element.insertBefore(group, data);

			// build and init repeat entry
			RepeatEntry entry = (RepeatEntry)getContainerObject().getElementFactory().createXFormsElement(group,
			                                                                                              getModel());
			entry.init();
            this.contextSize++;
		}
	}

	/**
	 * Checks wether this element has a <code>nodeset</code> attribute.
	 *
	 * @return <code>true</code> if this element has a <code>nodeset</code>
	 * attribute, otherwise <code>false</code>.
	 */
	private boolean hasNodesetAttribute() {
		return this.element.hasAttributeNS(NamespaceCtx.XFORMS_NS, NODESET_ATTRIBUTE);
	}

	/**
	 * Checks wether this element has a <code>repeat-bind</code> attribute.
	 *
	 * @return <code>true</code> if this element has a <code>repeat-bind</code>
	 * attribute, otherwise <code>false</code>.
	 */
	private boolean hasRepeatBindAttribute() {
		return this.element.hasAttributeNS(NamespaceCtx.XFORMS_NS, REPEAT_BIND_ATTRIBUTE);
	}

	/**
	 * Checks wether this element has a <code>repeat-nodeset</code> attribute.
	 *
	 * @return <code>true</code> if this element has a <code>repeat-nodeset</code>
	 * attribute, otherwise <code>false</code>.
	 */
	private boolean hasRepeatNodesetAttribute() {
		return this.element.hasAttributeNS(NamespaceCtx.XFORMS_NS, REPEAT_NODESET_ATTRIBUTE);
	}

	private void removeRepeatEntries(int count) throws XFormsException {
		// get entries without data element
		List entries = DOMUtil.getChildElements(this.element);
		entries.remove(entries.size() - 1);

		int index = entries.size() - 1;
		int removed = 0;

		while (removed < count) {
			// dispose repeat entry
			ElementImpl elementImpl = (ElementImpl)entries.get(index);
			RepeatEntry repeatEntry = (RepeatEntry)elementImpl.getUserData();
			repeatEntry.dispose();

			// remove repeat entry
			this.element.removeChild(elementImpl);
			removed++;
			index--;
		}
	}

	private void storeRepeatPrototype() {
		// store repeat prototype
		this.prototype = (Element)getElement().cloneNode(true);

		// remove original children
		Node child;

		while ((child = getElement().getFirstChild()) != null) {
			getElement().removeChild(child);
		}
	}
}
//end of class



