/*
 *
 *    Artistic License
 *
 *    Preamble
 *
 *    The intent of this document is to state the conditions under which a Package may be copied, such that
 *    the Copyright Holder maintains some semblance of artistic control over the development of the
 *    package, while giving the users of the package the right to use and distribute the Package in a
 *    more-or-less customary fashion, plus the right to make reasonable modifications.
 *
 *    Definitions:
 *
 *    "Package" refers to the collection of files distributed by the Copyright Holder, and derivatives
 *    of that collection of files created through textual modification.
 *
 *    "Standard Version" refers to such a Package if it has not been modified, or has been modified
 *    in accordance with the wishes of the Copyright Holder.
 *
 *    "Copyright Holder" is whoever is named in the copyright or copyrights for the package.
 *
 *    "You" is you, if you're thinking about copying or distributing this Package.
 *
 *    "Reasonable copying fee" is whatever you can justify on the basis of media cost, duplication
 *    charges, time of people involved, and so on. (You will not be required to justify it to the
 *    Copyright Holder, but only to the computing community at large as a market that must bear the
 *    fee.)
 *
 *    "Freely Available" means that no fee is charged for the item itself, though there may be fees
 *    involved in handling the item. It also means that recipients of the item may redistribute it under
 *    the same conditions they received it.
 *
 *    1. You may make and give away verbatim copies of the source form of the Standard Version of this
 *    Package without restriction, provided that you duplicate all of the original copyright notices and
 *    associated disclaimers.
 *
 *    2. You may apply bug fixes, portability fixes and other modifications derived from the Public Domain
 *    or from the Copyright Holder. A Package modified in such a way shall still be considered the
 *    Standard Version.
 *
 *    3. You may otherwise modify your copy of this Package in any way, provided that you insert a
 *    prominent notice in each changed file stating how and when you changed that file, and provided that
 *    you do at least ONE of the following:
 *
 *        a) place your modifications in the Public Domain or otherwise make them Freely
 *        Available, such as by posting said modifications to Usenet or an equivalent medium, or
 *        placing the modifications on a major archive site such as ftp.uu.net, or by allowing the
 *        Copyright Holder to include your modifications in the Standard Version of the Package.
 *
 *        b) use the modified Package only within your corporation or organization.
 *
 *        c) rename any non-standard executables so the names do not conflict with standard
 *        executables, which must also be provided, and provide a separate manual page for each
 *        non-standard executable that clearly documents how it differs from the Standard
 *        Version.
 *
 *        d) make other distribution arrangements with the Copyright Holder.
 *
 *    4. You may distribute the programs of this Package in object code or executable form, provided that
 *    you do at least ONE of the following:
 *
 *        a) distribute a Standard Version of the executables and library files, together with
 *        instructions (in the manual page or equivalent) on where to get the Standard Version.
 *
 *        b) accompany the distribution with the machine-readable source of the Package with
 *        your modifications.
 *
 *        c) accompany any non-standard executables with their corresponding Standard Version
 *        executables, giving the non-standard executables non-standard names, and clearly
 *        documenting the differences in manual pages (or equivalent), together with instructions
 *        on where to get the Standard Version.
 *
 *        d) make other distribution arrangements with the Copyright Holder.
 *
 *    5. You may charge a reasonable copying fee for any distribution of this Package. You may charge
 *    any fee you choose for support of this Package. You may not charge a fee for this Package itself.
 *    However, you may distribute this Package in aggregate with other (possibly commercial) programs as
 *    part of a larger (possibly commercial) software distribution provided that you do not advertise this
 *    Package as a product of your own.
 *
 *    6. The scripts and library files supplied as input to or produced as output from the programs of this
 *    Package do not automatically fall under the copyright of this Package, but belong to whomever
 *    generated them, and may be sold commercially, and may be aggregated with this Package.
 *
 *    7. C or perl subroutines supplied by you and linked into this Package shall not be considered part of
 *    this Package.
 *
 *    8. The name of the Copyright Holder may not be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 *    9. THIS PACKAGE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED
 *    WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 *    MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 */
package org.chiba.xml.xforms;

import org.apache.log4j.Category;
import org.chiba.xml.xforms.config.Config;
import org.chiba.xml.xforms.config.XFormsConfigException;
import org.chiba.xml.xforms.exception.XFormsException;
import org.chiba.xml.xforms.ui.AbstractFormControl;
import org.chiba.xml.xforms.ui.Upload;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.xml.sax.InputSource;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import java.io.*;
import org.chiba.xml.util.DOMSerializer;

/**
 * Chiba Facade Class.
 * <p>
 * This class provides the interface to process W3C XForms 1.0 conformant
 * documents.
 *
 * @author Joern Turner
 * @author Ulrich Nicolas Liss&eacute;
 * @version $Id: ChibaBean.java,v 1.30 2004/03/05 17:48:49 joernt Exp $
 */
public class ChibaBean implements Serializable {
	private static Category LOGGER = Category.getInstance(ChibaBean.class);
	private static String APP_INFO = null;

    /**
     * The document container object model.
     */
	private Container container = null;


	/**
     * The base URI for resolution of relative URIs.
     */
	private String baseURI = null;

    /**
     * signals if shutdown() has been called.
     */
    private boolean shutdown=false;
    private ChibaContext context=null;

    /**
	 * Creates a new ChibaBean object.
	 */
	public ChibaBean() {
		LOGGER.info(getAppInfo());
	}

	/**
	 * Returns Chiba version string.
	 *
	 * @return Chiba version string.
	 */
	public static String getAppInfo() {
		synchronized (ChibaBean.class) {
			if (APP_INFO == null) {
				try {
					BufferedInputStream stream = new BufferedInputStream(ChibaBean.class.getResourceAsStream("version.info"));
					StringBuffer buffer = new StringBuffer("Chiba/");
					int c;

					while ((c = stream.read()) > -1) {
						buffer.append((char)c);
					}

					stream.close();

					APP_INFO = buffer.toString();
				} catch (IOException e) {
					APP_INFO = "Chiba";
				}
			}

			return APP_INFO;
		}
	}

    /**
     * Sets the config path.
     * <p>
     * Checks existence of the config path and creates a config instance.
     *
     * @param path the absolute path to the config file.
     */
    public void setConfig(String path) throws XFormsException {
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("set config: " + path);
        }

        if ((path == null) || (new File(path).exists() == false)) {
            throw new XFormsConfigException("path not found: " + path);
        }

//        Config.resetInstance();
        Config.getInstance(path);
    }

	/**
	 * Sets the base URI.
     * <p>
     * The base URI is used for resolution of relative URIs occurring
     * in the document, e.g. instance sources or submission actions.
	 *
	 * @param uri the base URI.
	 */
	public void setBaseURI(String uri) {
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("set base uri: " + uri);
        }

		this.baseURI = uri;
	}

	/**
	 * Returns the base URI.
	 *
	 * @return the base URI.
     * @see #setBaseURI(String)
	 */
	public String getBaseURI() {
		return this.baseURI;
	}

    /**
     * Allows to set a ChibaContext object for storing application-specific parameters.
     *
     * @param context the ChibaContext to use
     */
    public void setContext(ChibaContext context){
        this.context = context;
    }

    /**
     * returns the ChibaContext object which stores application-specific params.
     * @return the ChibaContext object which stores application-specific params.
     */
    public ChibaContext getContext(){
        return this.context;
    }

    /**
     * Returns the document container associated with this processor.
     *
     * @return the document container associated with this processor.
     */
    public Container getContainer() throws XFormsException {
        if(shutdown){
//            throw new XFormsException("The form session has been terminated.");
            throw new XFormsException(Config.getInstance().getErrorMessage("session-invalid"));
        }
        return this.container;
    }

	/**
	 * Sets the containing document.
     * <p>
     * A new document container is created.
	 *
	 * @param document the containing document as DOM.
     * @throws XFormsException if the document container could not be created.
	 */
    public void setXMLContainer (Document document) throws XFormsException {
        try {
            Document xformsDocument = new XFormsDocument();
            Node node = xformsDocument.importNode(document.getDocumentElement(), true);
            xformsDocument.appendChild(node);
            createContainer().setDocument(xformsDocument);
        }
        catch (Exception e) {
            throw new XFormsException("could not create document container", e);
        }
    }

    /**
     * Sets the containing document.
     * <p>
     * A new document container is created.
     * <p>
     * Todo: This should be changed to take an URI and use URI resolution
     * instead of an absolute file name.
     *
     * @param fileName the containing document absolute file name.
     * @throws XFormsException if the document container could not be created.
	 */
	public void setXMLContainer(String fileName) throws XFormsException {
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("using form: " + fileName);
        }

		try {
			Document input = getDocumentBuilder().parse(new File(fileName));
            createContainer().setDocument(input);
        }
        catch (Exception e) {
            throw new XFormsException("could not create document container", e);
        }
	}

	/**
     * Sets the containing document.
     * <p>
     * A new document container is created.
     *
     * @param stream the containing document as input stream.
     * @throws XFormsException if the document container could not be created.
	 */
	public void setXMLContainer(InputStream stream) throws XFormsException {
		try {
			Document input = getDocumentBuilder().parse(stream);
            createContainer().setDocument(input);
        }
        catch (Exception e) {
            throw new XFormsException("could not create document container", e);
        }
	}

	/**
     * Sets the containing document.
     * <p>
     * A new document container is created.
     *
     * @param source the containing document as input source.
     * @throws XFormsException if the document container could not be created.
	 */
	public void setXMLContainer(InputSource source) throws XFormsException {
		try {
			Document input = getDocumentBuilder().parse(source);
            createContainer().setDocument(input);
        }
        catch (Exception e) {
            throw new XFormsException("could not create document container", e);
        }
	}

	/**
	 * Returns the containing document as DOM.
	 * <p>
	 * This returns the live DOM processed by Chiba internally. Changes will affect
     * internal state and may cause malfunction. Should we better be more restrictive
	 * and return a clone to prevent this ?
	 *
	 * @return the containing document.
     * @throws XFormsException if no document container is present.
	 */
	public Document getXMLContainer() throws XFormsException {
        if (this.container == null) {
            throw new XFormsException("no document container present");
        }

		return this.container.getDocument();
	}

    /**
     * Convenience method to import instance data prior to initializing the processor (calling
     * its init-method). This inlines the passed instance data into the containing document and
     * makes them available for editing. They'll be handled exactly like inline preset data.
     *
     * @param id the id of the target instance element.
     * @param element the imported element.
     * @throws XFormsException if no document container is present
     * or an error occurred during instance import.
     */
    public void importInstance(String id, Element element) throws XFormsException {
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("import instance: instance: " + id + ", element: " + element);
        }

        if (this.container == null) {
            throw new XFormsException("no document container present");
        }

        this.container.importInstance(id,element);
    }

    /**
     * Allows to set or overwrite a instance's src URI.
     * <p>
     * This method can be used to provide a parametrized URI to the
     * URI Resolver which handles the instance's src URI.
     * <p>
     * If no id or an empty id is specified, the default instance is selected.
     * The default instance is the first instance in document order.
     *
     * @param id the id of the instance.
     * @param srcURI the source URI.
     * @throws XFormsException if no document container is present
     * or the specified instance does not exist.
     * @see org.chiba.xml.xforms.connector.URIResolver
     */
    public void setInstanceURI(String id, String srcURI) throws XFormsException {
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("set instance uri: instance: " + id + ", uri: " + srcURI);
        }

        if (this.container == null) {
            throw new XFormsException("no document container present");
        }

        this.container.setInstanceURI(id, srcURI);
    }

    /**
     * Allows to set or overwrite a submission's action URI.
     * <p>
     * This method can be used to provide a parametrized URI to the
     * Submission Driver which handles the submission's action URI.
     * <p>
     * If no id or an empty id is specified, the default submission is selected.
     * The default submission is the first submission in document order.
     *
     * @param id the id of the submission.
     * @param actionURI the action URI.
     * @throws XFormsException if no document container is present
     * or the specified submission does not exist.
     * @see org.chiba.xml.xforms.connector.SubmissionHandler
     */
    public void setSubmissionURI(String id, String actionURI) throws XFormsException {
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("set submission uri: submission: " + id + ", uri: " + actionURI);
        }

        if (this.container == null) {
            throw new XFormsException("no document container present");
        }

        this.container.setSubmissionURI(id, actionURI);
    }

    /**
     * Bootstraps processor initialization.
     * <p>
     * Use this method after setXMLContainer()
     * and (optionally) setInstanceData() have been called to actually start the
     * processing.
     *
     * @throws XFormsException if no document container is present
     * or an error occurred during init.
     */
    public void init() throws XFormsException {
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("init");
        }

        if (this.container == null) {
            throw new XFormsException("no document container present");
        }

        this.container.init();
    }

	/**
	 * Dispatches an event of the specified type to the given event target.
	 *
	 * @param target the id of the event target.
	 * @param event the event type.
     * @throws XFormsException if no document container is present
     * or an error occurred during dispatch.
	 */
	public boolean dispatch(String target, String event) throws XFormsException {
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("dispatch: target: " + target + ", event: " + event);
        }

        if (this.container == null) {
            throw new XFormsException("no document container present");
        }

		return this.container.dispatch(target, event);
	}

    /**
     * This method emulates the setting of a new value through an UI control. The value will only be changed if
     * there was a change to the data. This method mainly exists to allow the separation of the actual UI handling.
     * Applications have to call this method to propagate their UI value changes to the Chiba processor.
     *
     * @param id the id of the control
     * @param newValue the new value for the control
     * @throws XFormsException
     */
    public void updateControlValue(String id, String newValue) throws XFormsException {
        // check old control value
        AbstractFormControl control = (AbstractFormControl) getContainer().lookup(id);
        String oldValue = control.getValue();

        if (newValue.equals(oldValue)) {
            // NOP
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("data '" + id + "' has no changes");
            }
        } else {
            // update control value
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("data '" + id + "' changes to '" + newValue + "'");
            }
            control.setValue(newValue);
        }
    }


    /**
     *
     * @param id the id of the control
     * @param contentType the content-type for the uploaded resource
     * @param filename the filename the uploaded data should be saved under
     * @param data the uploaded data as byte array
     * @throws XFormsException
     */
    public void updateControlValue(String id, String contentType, String filename, byte[] data)
    throws XFormsException {
        AbstractFormControl control = (AbstractFormControl) getContainer().lookup(id);
        if (!(control instanceof Upload)) {
            throw new XFormsException("Only Update control can be updated with file data");
        }
        Upload uploadControl = (Upload) control;
        uploadControl.setValue(contentType, filename, data);
    }




    /**
     * Finishes processor operation.
     *
     * @throws XFormsException if no document container is present
     * or an error occurred during shutdown.
     */
    public void shutdown() throws XFormsException {
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("shutdown");
        }

        if (this.container == null) {
            throw new XFormsException("no document container present");
        }
        this.shutdown=true;
        this.container.shutdown();
        this.container = null;
    }

    private Container createContainer(){
        this.container = new Container(this);
        return this.container;
    }

	private DocumentBuilder getDocumentBuilder() throws XFormsException {
		try {
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            factory.setNamespaceAware(true);
            factory.setValidating(false);
            factory.setAttribute("http://apache.org/xml/properties/dom/document-class-name",
                                 "org.chiba.xml.xforms.XFormsDocument");
			return factory.newDocumentBuilder();
		}
        catch (Exception e) {
			throw new XFormsException(e);
		}
	}

    //=============== Added for NetKernel ==================

	/**
	 * returns a serialized Document Container
	 *
	 *	this is necessary to avoid DOM Linkage Errors crossing from Chiba to URA
	 *
	 * @return - returns the internal DOM-tree in serialized form
	 */
	public String getXMLContainerAsString() throws XFormsException
	{	return serializer(container.getDocument().getDocumentElement());
	}
	
	public static String serializer(org.w3c.dom.Element e) throws XFormsException
	{	try
		{
		ByteArrayOutputStream stream = new ByteArrayOutputStream();
        DOMSerializer serializer = new DOMSerializer();
		serializer.setDeclarationOmitting(true);
        serializer.setEncoding("UTF-8");
        serializer.setIndentation("\t");
        serializer.setLineBreaking(true);
        serializer.setOutputStream(stream);
        serializer.setWhitespaceIgnoring(true);
        serializer.serialize(e);

        return stream.toString();
		}
		catch(Exception ex)
		{	throw new XFormsException(ex.getMessage());
		}
	}

}
