/*
 *
 *    Artistic License
 *
 *    Preamble
 *
 *    The intent of this document is to state the conditions under which a Package may be copied, such that
 *    the Copyright Holder maintains some semblance of artistic control over the development of the
 *    package, while giving the users of the package the right to use and distribute the Package in a
 *    more-or-less customary fashion, plus the right to make reasonable modifications.
 *
 *    Definitions:
 *
 *    "Package" refers to the collection of files distributed by the Copyright Holder, and derivatives
 *    of that collection of files created through textual modification.
 *
 *    "Standard Version" refers to such a Package if it has not been modified, or has been modified
 *    in accordance with the wishes of the Copyright Holder.
 *
 *    "Copyright Holder" is whoever is named in the copyright or copyrights for the package.
 *
 *    "You" is you, if you're thinking about copying or distributing this Package.
 *
 *    "Reasonable copying fee" is whatever you can justify on the basis of media cost, duplication
 *    charges, time of people involved, and so on. (You will not be required to justify it to the
 *    Copyright Holder, but only to the computing community at large as a market that must bear the
 *    fee.)
 *
 *    "Freely Available" means that no fee is charged for the item itself, though there may be fees
 *    involved in handling the item. It also means that recipients of the item may redistribute it under
 *    the same conditions they received it.
 *
 *    1. You may make and give away verbatim copies of the source form of the Standard Version of this
 *    Package without restriction, provided that you duplicate all of the original copyright notices and
 *    associated disclaimers.
 *
 *    2. You may apply bug fixes, portability fixes and other modifications derived from the Public Domain
 *    or from the Copyright Holder. A Package modified in such a way shall still be considered the
 *    Standard Version.
 *
 *    3. You may otherwise modify your copy of this Package in any way, provided that you insert a
 *    prominent notice in each changed file stating how and when you changed that file, and provided that
 *    you do at least ONE of the following:
 *
 *        a) place your modifications in the Public Domain or otherwise make them Freely
 *        Available, such as by posting said modifications to Usenet or an equivalent medium, or
 *        placing the modifications on a major archive site such as ftp.uu.net, or by allowing the
 *        Copyright Holder to include your modifications in the Standard Version of the Package.
 *
 *        b) use the modified Package only within your corporation or organization.
 *
 *        c) rename any non-standard executables so the names do not conflict with standard
 *        executables, which must also be provided, and provide a separate manual page for each
 *        non-standard executable that clearly documents how it differs from the Standard
 *        Version.
 *
 *        d) make other distribution arrangements with the Copyright Holder.
 *
 *    4. You may distribute the programs of this Package in object code or executable form, provided that
 *    you do at least ONE of the following:
 *
 *        a) distribute a Standard Version of the executables and library files, together with
 *        instructions (in the manual page or equivalent) on where to get the Standard Version.
 *
 *        b) accompany the distribution with the machine-readable source of the Package with
 *        your modifications.
 *
 *        c) accompany any non-standard executables with their corresponding Standard Version
 *        executables, giving the non-standard executables non-standard names, and clearly
 *        documenting the differences in manual pages (or equivalent), together with instructions
 *        on where to get the Standard Version.
 *
 *        d) make other distribution arrangements with the Copyright Holder.
 *
 *    5. You may charge a reasonable copying fee for any distribution of this Package. You may charge
 *    any fee you choose for support of this Package. You may not charge a fee for this Package itself.
 *    However, you may distribute this Package in aggregate with other (possibly commercial) programs as
 *    part of a larger (possibly commercial) software distribution provided that you do not advertise this
 *    Package as a product of your own.
 *
 *    6. The scripts and library files supplied as input to or produced as output from the programs of this
 *    Package do not automatically fall under the copyright of this Package, but belong to whomever
 *    generated them, and may be sold commercially, and may be aggregated with this Package.
 *
 *    7. C or perl subroutines supplied by you and linked into this Package shall not be considered part of
 *    this Package.
 *
 *    8. The name of the Copyright Holder may not be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 *    9. THIS PACKAGE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED
 *    WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 *    MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 */
package org.chiba.adapter.web;

import org.apache.commons.fileupload.DiskFileUpload;
import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileUpload;
import org.apache.commons.fileupload.FileUploadException;
import org.apache.log4j.Category;
import org.chiba.tools.xslt.StylesheetLoader;
import org.chiba.tools.xslt.UIGenerator;
import org.chiba.tools.xslt.XSLTGenerator;
import org.chiba.xml.xforms.ChibaBean;
import org.chiba.xml.xforms.ChibaContext;
import org.chiba.xml.xforms.config.Config;
import org.chiba.xml.xforms.events.EventFactory;
import org.chiba.xml.xforms.events.XFormsEvent;
import org.chiba.xml.xforms.exception.XFormsException;
import org.chiba.xml.xforms.ui.Repeat;
import org.w3c.dom.events.Event;
import org.w3c.dom.events.EventTarget;

import javax.servlet.http.HttpServletRequest;
import java.io.File;
import java.io.Writer;
import java.util.*;

/**
 * integrates XForms Processor into Web-applications and handles request processing.
 *
 * @author joern turner
 * @version $Id: ServletAdapter.java,v 1.19 2004/03/05 17:48:49 joernt Exp $
 */
public class ServletAdapter implements org.w3c.dom.events.EventListener {

    private static final Category LOGGER = Category.getInstance(ServletAdapter.class);
    private static final String DATA_PREFIX_PROPERTY = "chiba.web.dataPrefix";
    private static final String TRIGGER_PREFIX_PROPERTY = "chiba.web.triggerPrefix";
    private static final String SELECTOR_PREFIX_PROPERTY = "chiba.web.selectorPrefix";
    private static final String DATA_PREFIX_DEFAULT = "d_";
    private static final String TRIGGER_PREFIX_DEFAULT = "t_";
    private static final String SELECTOR_PREFIX_DEFAULT = "s_";
    private ChibaBean chibaBean = null;
    private Map parameterNames = null;
    private String dataPrefix = null;
    private String selectorPrefix = null;
    private String triggerPrefix = null;
    private String uploadDir = null;

    private String configPath = null;
    private String formPath = null;
    private String actionUrl = null;
    private String CSSFile = null;
    private String stylesheet = null;
    private String contextRoot = null;
    private Map instanceURIMap = null;
    private Map submissionURIMap = null;
    private UIGenerator generator = null;
    private String stylesheetPath = null;

    //processing-phase handling - signals to UIGenerator
    private static final String INIT_PHASE = "init";
    private static final String SESSION_PHASE = "session";
    private static final String SUBMIT_PHASE = "submit";
    private String phase;

    private ChibaContext webContext = null;

    /**
     * Creates a new ServletAdapter object.
     */
    public ServletAdapter() {
        this.parameterNames = new HashMap();
        this.webContext = new WebContext(this);
        this.webContext.setProperty(WebContext.REQUEST_PARAMETERS,this.parameterNames);
    }

    /**
     * creates an instance of ChibaBean, configures it and creates a generator instance
     *
     * @throws XFormsException If an error occurs
     */
    public void init() throws XFormsException {
        File f = new File(locateFile(formPath));
        // create bean
        this.chibaBean = new ChibaBean(); //using default configuration
        chibaBean.setBaseURI(f.getParentFile().toURI().toString());
        //use custom config-file if this param is existent
        if ((configPath != null) && !(configPath.equals(""))) {
            chibaBean.setConfig(configPath);
        }
        chibaBean.setContext(this.webContext);
        chibaBean.setXMLContainer(f.getAbsolutePath());

        // import instances if any
        if (null != instanceURIMap && instanceURIMap.size() > 0) {
            Iterator instanceIds = instanceURIMap.keySet().iterator();
            // get each instance id/uri and set it in the bean
            while (instanceIds.hasNext()) {
                String instanceId = (String) instanceIds.next();
                chibaBean.setInstanceURI(instanceId, (String) instanceURIMap.get(instanceId));

                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("InstanceURI: id: " + instanceId + " value: " + (String) instanceURIMap.get(instanceId));
                }
            }
        }

        // import submission uri's if any
        if (null != submissionURIMap && submissionURIMap.size() > 0) {
            Iterator submissionIds = submissionURIMap.keySet().iterator();
            // get each submission id/uri and set it in the bean
            while (submissionIds.hasNext()) {
                String submissionId = (String) submissionIds.next();
                chibaBean.setSubmissionURI(submissionId, (String) submissionURIMap.get(submissionId));

                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("SubmissionURI: id: " + submissionId + " value: " + (String) submissionURIMap.get(submissionId));
                }
            }
        }

        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Formpath: " + formPath);
            LOGGER.debug("CSS-File: " + CSSFile);
            LOGGER.debug("XSLT stylesheet: " + stylesheet);
            LOGGER.debug("action URL: " + actionUrl);
        }

        //create and configure StylesheetLoader
        StylesheetLoader stylesLoader = new StylesheetLoader(stylesheetPath);

        //if there's a stylesheet specified in the request
        if (stylesheet != null) {
            stylesLoader.setStylesheetFile(stylesheet);
        }

        //create and configure User Interface generator and pass loader to it
        generator = createTransformer(stylesLoader, chibaBean, this.actionUrl);

        //attach listener that sets submit processing phase
        addPhaseListener(chibaBean);
        chibaBean.init();
    }

    /**
     * Instructs the application environment to forward the given response.
     *
     * @param response a map containing at least a response stream and optional
     *                 header information.
     */
    public void forward(Map response) {
        this.chibaBean.getContext().setProperty(ChibaContext.SUBMISSION_RESPONSE, response);
    }

    /**
     * returns a Map object containing a forward uri. this is used by the 'load' action
     *
     * @return a Map object containing a forward uri
     */
    public Map getForwardMap() {
        return (Map) chibaBean.getContext().getProperty(ChibaContext.SUBMISSION_RESPONSE);
    }

    /**
     * checks whether we have multipart or urlencoded request and processes it accordingly. After updating
     * the data, a reacalculate, revalidate refresh sequence is fired and the found trigger is executed.
     *
     * @param request Servlet request
     * @throws XFormsException todo: implement action block behaviour
     */
    public void handleRequest(HttpServletRequest request)
            throws XFormsException {
        String trigger = null;

        // Check that we have a file upload request
        boolean isMultipart = FileUpload.isMultipartContent(request);
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("request isMultipart: " + isMultipart);
            LOGGER.debug("Upload dir: " + uploadDir);
            LOGGER.debug("base URI: " + chibaBean.getBaseURI());
            LOGGER.debug("context root: " + contextRoot);
        }

        if (isMultipart) {
            trigger =
                    processMultiPartRequest(request,
                            trigger,
                            this.uploadDir);
        } else {
            trigger = processUrlencodedRequest(request, trigger);
        }

        // finally activate trigger if any
        if (trigger != null) {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("trigger '" + trigger + "'");
            }

            this.chibaBean.dispatch(trigger, EventFactory.DOM_ACTIVATE);
        }
    }

    /**
     * generates the user interface.
     *
     * @param responseWriter the Writer to use for the result stream
     * @throws XFormsException
     */
    public void buildUI(Writer responseWriter) throws XFormsException {

        generator.setParameter("phase", this.phase);
        generator.setInputNode(this.chibaBean.getXMLContainer());
        generator.generate(responseWriter);

        if (this.phase.equals(INIT_PHASE)) {
            this.phase = SESSION_PHASE;
        }
    }


    /**
     * This method is called whenever an event occurs of the type for which
     * the <code> EventListener</code> interface was registered.
     *
     * @param evt The <code>Event</code> contains contextual information
     *            about the event. It also contains the <code>stopPropagation</code>
     *            and <code>preventDefault</code> methods which are used in
     *            determining the event's flow and default action.
     */
    public void handleEvent(Event evt) {
        if (evt instanceof XFormsEvent) {

            if (evt.getType().equals(EventFactory.READY)) {
                this.phase = INIT_PHASE;
            }

            if (evt.getType().equals(EventFactory.SUBMIT)) {
                this.phase = SUBMIT_PHASE;
            }

        }

    }

    /**
     * Instructs the application environment to setRedirect to the given URI.
     *
     * @param uri an absolute URI.
     */
    public void setRedirect(String uri) {
        chibaBean.getContext().setProperty(ChibaContext.REDIRECT_URI, uri);
    }

    /**
     * returns the redirect Uri
     *
     * @return the redirect Uri
     */
    public String getRedirectUri() {
        return (String) chibaBean.getContext().getProperty(ChibaContext.REDIRECT_URI);
    }

    // ************************* ACCESSORS ********************************************


    /**
     * returns the configured prefix which identifies 'selector' parameters. These are used to transport
     * the state of repeat indices via http.
     *
     * @return the prefix for selector parameters from the configuration
     */
    public String getSelectorPrefix() {
        if (this.selectorPrefix == null) {
            try {
                this.selectorPrefix =
                        Config.getInstance().getProperty(SELECTOR_PREFIX_PROPERTY,
                                SELECTOR_PREFIX_DEFAULT);
            } catch (Exception e) {
                this.selectorPrefix = SELECTOR_PREFIX_DEFAULT;
            }
        }

        return this.selectorPrefix;
    }

    /**
     * sets the Url for the action target
     *
     * @param actionUrl the Url for the action target
     */
    public void setActionUrl(String actionUrl) {
        this.actionUrl = actionUrl;
    }

    /**
     * sets the context root for the webapp. This is used to build the correct pathes of relative path-statements
     *
     * @param contextRoot the root of the webapp
     */
    public void setContextRoot(String contextRoot) {
        this.contextRoot = contextRoot;
    }

    /**
     * set the path to the config file
     *
     * @param configPath the path to the config file
     */
    public void setConfigPath(String configPath) {
        this.configPath = configPath;
    }

    /**
     * sets the path where to find XForms documents
     *
     * @param formPath the path where to find XForms documents
     */
    public void setFormPath(String formPath) {
        this.formPath = formPath;
    }

    /**
     * returns the ChibaBean instance used with this servletAdapter
     *
     * @return the ChibaBean instance used with this servletAdapter
     */
    public ChibaBean getChibaBean() {
        return chibaBean;
    }

    /**
     * sets the ChibaBean instance to use with this servletAdapter
     *
     * @param chibaBean the ChibaBean instance to use with this servletAdapter
     */
    public void setChibaBean(ChibaBean chibaBean) {
        this.chibaBean = chibaBean;
    }

    /**
     * sets the reference to the using Servlet.
     *
     * @param chibaServlet the Servlet using this Adapter
     */
    //    public void setChibaServlet(ChibaServlet chibaServlet) {
    //        this.chibaServlet = chibaServlet;
    //    }

    /**
     * sets the directory where uploaded files are stored.
     *
     * @param uploadDir the directory where uploaded files are stored
     */
    public void setUploadDir(String uploadDir) {
        this.uploadDir = uploadDir;
    }

    /**
     * sets the path where to find the xslt stylesheets
     *
     * @param stylesPath the path where to find the xslt stylesheets
     */
    public void setStylesheetPath(String stylesPath) {
        this.stylesheetPath = stylesPath;
    }

    /**
     * sets the instance URI path map
     *
     * @param aInstanceURIMap the instance URI path map
     */
    public void setInstanceURIMap(Map aInstanceURIMap) {
        this.instanceURIMap = aInstanceURIMap;
    }

    /**
     * sets the submission URI map
     *
     * @param aSubmissionURIMap the submission URI map
     */
    public void setSubmissionURIMap(Map aSubmissionURIMap) {
        this.submissionURIMap = aSubmissionURIMap;
    }

    /**
     * set the CSS file to use for styling the user interface
     *
     * @param css the CSS file to use for styling the user interface
     */
    public void setCSS(String css) {
        this.CSSFile = css;
    }

    /**
     * sets the name of the xslt stylesheet to use for building the UI
     *
     * @param stylesheetFile the name of the xslt stylesheet to use for building the UI
     */
    public void setStylesheet(String stylesheetFile) {
        this.stylesheet = stylesheetFile;
    }

    /**
     * @param request Servlet request
     * @param trigger Trigger control
     * @param path    Base path for uploading files
     * @return the calculated trigger
     * @throws XFormsException If an error occurs
     */
    private String processMultiPartRequest(HttpServletRequest request,
                                           String trigger, String path) throws XFormsException {
        DiskFileUpload upload = new DiskFileUpload();
        String rootDir = contextRoot + File.separator + path;
        upload.setRepositoryPath(rootDir);

        if(LOGGER.isDebugEnabled()){
            LOGGER.debug("root dir for uploads: " + rootDir );
        }

        List items = null;
        try {
            items = upload.parseRequest(request);
        } catch (FileUploadException fue) {
            throw new XFormsException(fue);
        }

        Iterator iter = items.iterator();
        while (iter.hasNext()) {
            FileItem item = (FileItem) iter.next();
            String itemName = item.getName();
            String fieldName = item.getFieldName();
            String id = (String) this.parameterNames.get(fieldName);

            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Multipart item name is: " + itemName
                        + " and fieldname is: " + fieldName
                        + " and id is: " + id);
                LOGGER.debug("Is formfield: " + item.isFormField());
                LOGGER.debug("Content: " + item.getString());
            }

            if (item.isFormField()) {
                // It's a field name, it means that we got a non-file
                // form field. Upload is not required. We must treat it as we
                // do in processUrlencodedRequest()
                String[] values = new String[1];
                values[0] = item.getString();   //for compatibility

                // [1] handle data
                handleData(fieldName, values);

                // [3] handle selector
                handleSelector(fieldName, values[0]);

                // [2] handle trigger
                trigger = handleTrigger(trigger, fieldName);

            } else {
                String uniqueFilename = "/" + uploadDir + "/"
                        + getUniqueParameterName("file") + "/" + itemName;

                contextRoot = request.getSession().getServletContext().getRealPath("");
                if (contextRoot == null)
                    contextRoot = request.getSession().getServletContext().getRealPath(".");

                File savedFile = new File(contextRoot + uniqueFilename);

                byte[] tmpData=item.get();

                //TODO: Save the file when anyURI datatype is used
                if (item.getSize() > 0) {
                    /*
                    AbstractFormControl control =
                    (AbstractFormControl) chibaBean.getContainer().lookup(id);

                    if (!(control instanceof Upload)) {
                        throw new XFormsException("Invalid control " +control
                        +"for id: " +id);
                    }
                     */


                    //TODO: How to figure out which datatype is used?
                    // control.getModelBinding() has protected access!!
                    String datatype = "anyURI";


                    if ("anyURI".equals(datatype)) {

                        try {
                            savedFile.getParentFile().mkdir();
                            item.write(savedFile);
                        } catch (Exception e) {
                            throw new RuntimeException(e);
                        }
                    }
                }
//                chibaBean.updateControlValue(id, item.getContentType(),
//                        uniqueFilename, item.get());
                chibaBean.updateControlValue(id, item.getContentType(),
                        uniqueFilename, tmpData);
            }
        }

        return trigger;
    }

    /**
     * @param request
     * @param trigger
     * @return
     * @throws XFormsException
     */
    private String processUrlencodedRequest(HttpServletRequest request,
                                            String trigger)
            throws XFormsException {
        // iterate request parameters
        Enumeration names = request.getParameterNames();
        while (names.hasMoreElements()) {
            String name = names.nextElement().toString();
            String[] values = request.getParameterValues(name);

            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug(this + " parameter-name: " + name);
                for (int i = 0; i < values.length; i++) {
                    LOGGER.debug(this + " value: " + values[i]);
                }
            }

            // [1] handle data
            handleData(name, values);

            // [3] handle selector
            handleSelector(name, values[0]);

            // [2] handle trigger
            trigger = handleTrigger(trigger, name);
        }
        return trigger;
    }

    /**
     * @param name
     * @throws XFormsException
     */
    private void handleSelector(String name, String value)
            throws XFormsException {
        if (name.startsWith(getSelectorPrefix())) {
            String id = name.substring(getSelectorPrefix().length());
            int index = Integer.valueOf(value).intValue();
            Repeat repeat = (Repeat) this.chibaBean.getContainer().lookup(id);

            // set repeat index
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("selecting '" + id + "' repeat entry " + index);
            }

            repeat.setIndex(index);
        }
    }

    /**
     * @param trigger
     * @param name
     * @return
     */
    private String handleTrigger(String trigger, String name) {
        if ((trigger == null) && name.startsWith(getTriggerPrefix())) {
            String parameter = name;
            int x = parameter.lastIndexOf(".x");
            int y = parameter.lastIndexOf(".y");

            if (x > -1) {
                parameter = parameter.substring(0, x);
            }

            if (y > -1) {
                parameter = parameter.substring(0, y);
            }

            // keep trigger id
            trigger = (String) this.parameterNames.get(parameter);
        }
        return trigger;
    }


    /**
     * @param name
     * @throws XFormsException
     */
    private void handleData(String name, String[] values)
            throws XFormsException {
        if (name.startsWith(getDataPrefix())) {
            String id = (String) this.parameterNames.get(name);

            // assemble new control value
            String newValue;

            if (values.length > 1) {
                StringBuffer buffer = new StringBuffer(values[0]);

                for (int i = 1; i < values.length; i++) {
                    buffer.append(" ").append(values[i]);
                }

                newValue = buffer.toString().trim();
            } else {
                newValue = values[0];
            }

            this.chibaBean.updateControlValue(id, newValue);
        }
    }

    // helper methods
    /**
     * returns the prefix which is used to identify trigger parameters.
     *
     * @return the prefix which is used to identify trigger parameters
     */
    public String getTriggerPrefix() {
        if (this.triggerPrefix == null) {
            try {
                this.triggerPrefix =
                        Config.getInstance().getProperty(TRIGGER_PREFIX_PROPERTY, TRIGGER_PREFIX_DEFAULT);
            } catch (Exception e) {
                this.triggerPrefix = TRIGGER_PREFIX_DEFAULT;
            }
        }

        return this.triggerPrefix;
    }

    /**
     * returns the prefix which is used to identify data parameters.
     *
     * @return the prefix which is used to identify data parameters.
     */
    public String getDataPrefix() {
        if (this.dataPrefix == null) {
            try {
                this.dataPrefix = Config.getInstance().getProperty(DATA_PREFIX_PROPERTY, DATA_PREFIX_DEFAULT);
            } catch (Exception e) {
                this.dataPrefix = DATA_PREFIX_DEFAULT;
            }
        }

        return this.dataPrefix;
    }

    /**
     * @param prefix
     * @return
     */
    public String getUniqueParameterName(String prefix) {
        String id = prefix + Integer.toHexString((int) (Math.random() * 10000));

        while (this.parameterNames.get(id) != null) {
            id = prefix + Integer.toHexString((int) (Math.random() * 10000));
        }

        return id;
    }


    /**
     * build the absolute path to the requested file and test its
     * existence. <br><br>
     *
     * @param uri - the relative uri of the file
     * @return returns the absolute path to the file
     */
    private String locateFile(String uri) throws XFormsException {
        if (uri == null) {
            throw new XFormsException("No form file specified");
        }

        //construct absolute path to file and check existence
        String filePath = contextRoot + uri;

        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("requested file: " + filePath);
        }

        if (!(new File(filePath).exists())) {
            throw new XFormsException("File does not exist: " + filePath);
        }

        return filePath;
    }

    //this is a quick hack to provide phase information e.g. signal that you're currently in submit-phase.
    //this is convenient to steer the rendering of validation error messages
    /**
     * @param chibaBean
     * @throws XFormsException
     */
    private void addPhaseListener(ChibaBean chibaBean)
            throws XFormsException {
        EventTarget eventTarget = (EventTarget) chibaBean.getXMLContainer().getDocumentElement();
        //for setting init phase
        eventTarget.addEventListener(EventFactory.READY, this, true);
        //for setting submit phase
        eventTarget.addEventListener(EventFactory.SUBMIT, this, true);
    }

    private UIGenerator createTransformer(StylesheetLoader stylesLoader,
                                          ChibaBean cp,
                                          String actionURL)
            throws XFormsException {
        if (this.generator == null) {
            this.generator = new XSLTGenerator(stylesLoader);
        }
        this.generator.setParameter("action-url", actionURL);
        this.generator.setParameter("debug-enabled", String.valueOf(LOGGER.isDebugEnabled()));
        this.generator.setParameter("selector-prefix", getSelectorPrefix());
        if (CSSFile != null) {
            LOGGER.debug("CSS: " + CSSFile);
            this.generator.setParameter("css-file", CSSFile);
        }
        return this.generator;
    }

    public Object getParameter(String key) {
        return webContext.getProperty((String) key);
    }

    public Object setParameter(String key, Object val) {
        webContext.setProperty((String) key, val);
        return val;
    }

}

//end of class
