/*
 *
 *    Artistic License
 *
 *    Preamble
 *
 *    The intent of this document is to state the conditions under which a Package may be copied, such that
 *    the Copyright Holder maintains some semblance of artistic control over the development of the
 *    package, while giving the users of the package the right to use and distribute the Package in a
 *    more-or-less customary fashion, plus the right to make reasonable modifications.
 *
 *    Definitions:
 *
 *    "Package" refers to the collection of files distributed by the Copyright Holder, and derivatives
 *    of that collection of files created through textual modification.
 *
 *    "Standard Version" refers to such a Package if it has not been modified, or has been modified
 *    in accordance with the wishes of the Copyright Holder.
 *
 *    "Copyright Holder" is whoever is named in the copyright or copyrights for the package.
 *
 *    "You" is you, if you're thinking about copying or distributing this Package.
 *
 *    "Reasonable copying fee" is whatever you can justify on the basis of media cost, duplication
 *    charges, time of people involved, and so on. (You will not be required to justify it to the
 *    Copyright Holder, but only to the computing community at large as a market that must bear the
 *    fee.)
 *
 *    "Freely Available" means that no fee is charged for the item itself, though there may be fees
 *    involved in handling the item. It also means that recipients of the item may redistribute it under
 *    the same conditions they received it.
 *
 *    1. You may make and give away verbatim copies of the source form of the Standard Version of this
 *    Package without restriction, provided that you duplicate all of the original copyright notices and
 *    associated disclaimers.
 *
 *    2. You may apply bug fixes, portability fixes and other modifications derived from the Public Domain
 *    or from the Copyright Holder. A Package modified in such a way shall still be considered the
 *    Standard Version.
 *
 *    3. You may otherwise modify your copy of this Package in any way, provided that you insert a
 *    prominent notice in each changed file stating how and when you changed that file, and provided that
 *    you do at least ONE of the following:
 *
 *        a) place your modifications in the Public Domain or otherwise make them Freely
 *        Available, such as by posting said modifications to Usenet or an equivalent medium, or
 *        placing the modifications on a major archive site such as ftp.uu.net, or by allowing the
 *        Copyright Holder to include your modifications in the Standard Version of the Package.
 *
 *        b) use the modified Package only within your corporation or organization.
 *
 *        c) rename any non-standard executables so the names do not conflict with standard
 *        executables, which must also be provided, and provide a separate manual page for each
 *        non-standard executable that clearly documents how it differs from the Standard
 *        Version.
 *
 *        d) make other distribution arrangements with the Copyright Holder.
 *
 *    4. You may distribute the programs of this Package in object code or executable form, provided that
 *    you do at least ONE of the following:
 *
 *        a) distribute a Standard Version of the executables and library files, together with
 *        instructions (in the manual page or equivalent) on where to get the Standard Version.
 *
 *        b) accompany the distribution with the machine-readable source of the Package with
 *        your modifications.
 *
 *        c) accompany any non-standard executables with their corresponding Standard Version
 *        executables, giving the non-standard executables non-standard names, and clearly
 *        documenting the differences in manual pages (or equivalent), together with instructions
 *        on where to get the Standard Version.
 *
 *        d) make other distribution arrangements with the Copyright Holder.
 *
 *    5. You may charge a reasonable copying fee for any distribution of this Package. You may charge
 *    any fee you choose for support of this Package. You may not charge a fee for this Package itself.
 *    However, you may distribute this Package in aggregate with other (possibly commercial) programs as
 *    part of a larger (possibly commercial) software distribution provided that you do not advertise this
 *    Package as a product of your own.
 *
 *    6. The scripts and library files supplied as input to or produced as output from the programs of this
 *    Package do not automatically fall under the copyright of this Package, but belong to whomever
 *    generated them, and may be sold commercially, and may be aggregated with this Package.
 *
 *    7. C or perl subroutines supplied by you and linked into this Package shall not be considered part of
 *    this Package.
 *
 *    8. The name of the Copyright Holder may not be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 *    9. THIS PACKAGE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED
 *    WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 *    MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 */
package org.chiba.adapter.web;

import org.apache.log4j.Category;
import org.chiba.xml.xforms.ChibaContext;
import org.chiba.xml.xforms.config.Config;
import org.chiba.xml.xforms.exception.XFormsException;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * The ChibaServlet handles all interactions between client and
 * form-processor (ChibaBean) for the whole lifetime of a form-filling session.
 * <br>
 * The Processor will be started through a Get-request from the client
 * pointing to the desired form-container. The Processor instance will
 * be stored in a Session-object.<br>
 * <br>
 * All further interaction will be handled through Post-requests.
 * Incoming request params will be mapped to data and action-handlers.
 *
 * @author  joern turner (joernt@chiba.sourceforge.net)
 * @version $Version: $
 */
public class ChibaServlet extends HttpServlet {
	//init-params
	private static Category cat = Category.getInstance(ChibaServlet.class);

	// the absolute path to the Chiba config-file
	private String configPath = null;

	// the rootdir of this app; forms + documents fill be searched under this root
	private String contextRoot = null;

	//path for Form Document (relative to context)
	private String formPath = null;
	private Map instanceURIMap = null;
	private Map submissionURIMap = null;
	private String css = null;
	private String stylesheetFile = null;
	private String stylesPath = null;

	// where uploaded files are stored
	private String uploadDir = null;
	private ServletAdapter servletAdapter = null;

	/**
	 * Returns a short description of the servlet.
	 * @return - Returns a short description of the servlet.
	 */
	public String getServletInfo() {
		return "Servlet Controller for Chiba XForms Processor";
	}

	/** Destroys the servlet.
	 */
	public void destroy() {
	}

	/**
	 * Initializes the servlet.
	 * @param config - the ServletConfig object
	 * @throws javax.servlet.ServletException
	 */
	public void init(ServletConfig config) throws ServletException {
		super.init(config);

		//read some params from web-inf
		contextRoot = getServletConfig().getServletContext().getRealPath("");
		if (contextRoot == null)
			contextRoot = getServletConfig().getServletContext().getRealPath(".");

		//get the relative path to the chiba config-file
		String path = getServletConfig().getInitParameter("chiba.config");

		//get the real path for the config-file
		if (path != null) {
			configPath = getServletConfig().getServletContext().getRealPath(path);
		}

		//get the path for the stylesheets
		path = getServletConfig().getServletContext().getInitParameter("chiba.xforms.stylesPath");

		//get the real path for the stylesheets and configure a new StylesheetLoader with it
		if (path != null) {
			stylesPath = getServletConfig().getServletContext().getRealPath(path);
			cat.info("stylesPath: " + stylesPath);
		}

		//uploadDir = contextRoot	+ "/" + getServletConfig().getServletContext().getInitParameter("chiba.upload");
                uploadDir = getServletConfig().getServletContext().getInitParameter("chiba.upload");
                
                //Security constraint
                if (uploadDir != null) {
                    if (uploadDir.toUpperCase().indexOf("WEB-INF") >= 0) {
                        throw new ServletException("Chiba security constraint: uploadDir '" +uploadDir +"' not allowed");
                    }
                }
	}

	/**
	 * Starts a new form-editing session.<br>
     *
     * The default value of a number of settings can be overridden as follows:
     *
     * 1. The uru of the xform to be displayed can be specified by using a param name of 'form' and a param value
     * of the location of the xform file as follows, which will attempt to load the current xforms file.
     *
     * http://localhost:8080/chiba-0.9.3/XFormsServlet?form=/forms/hello.xhtml
     *
     * 2. The uru of the CSS file used to style the form can be specified using a param name of 'css' as follows:
     *
     * http://localhost:8080/chiba-0.9.3/XFormsServlet?form=/forms/hello.xhtml&css=/chiba/my.css
     *
     * 3. The uri of the XSLT file used to generate the form can be specified using a param name of 'xslt' as follows:
     *
     * http://localhost:8080/chiba-0.9.3/XFormsServlet?form=/forms/hello.xhtml&xslt=/chiba/my.xslt
     *
     * 4. The servlet currently supports setting the source uri of 1 or more instances by passing in parameters.
     * The parameter value should be 'instance' + 'the instance id' and the parameter value
     * should be the uri of the location of the actual instance data for that id. For example,
     * with the current hello.xhtml sample form entering the following URL in the browser will
     * populate the form from the xml file found at c:\test\hello-instance.xml on a Windows box:
     *
     * http://localhost:8080/chiba-0.9.3/XFormsServlet?form=/forms/hello.xhtml&instanceC-2=file://C:/test/hello-instance.xml
     *
     * The uri of the default instance can be set by using a parameter name of 'instance' without a suffix
     * i.e. the following URL is equivalent (because the hello form only has one instance data declaration):
     *
     * http://localhost:8080/chiba-0.9.3/XFormsServlet?form=/forms/hello.xhtml&instance=file://C:/test/hello-instance.xml
     *
     * 5. It is also possible to pass in the uri's of one or more submission elements, enabling dynamic routing of xform submitting.
     * The param name should be 'submit' + 'the id of the  submission element' while the param value
     * should be the uri where the form data should be submitted. Using the hello example the following browser url will result in
     * the form data being sent to the specialDebug jsp when the Debug button is pressed:
     *
     * http://127.0.0.1:8080/chiba-0.9.3/XFormsServlet?form=/forms/hello.xhtml&&submissiondebug=jsp/specialDebug.jsp
     *
     * As before the default submission action attribute can be set simply by using a param name of 'submit'.
     *
     * 6. The action attribute of the html form built by this servlet can be set by passing in the param name of 'action_url'
     * and the param value of the uri where the form should be posted. This is useful when running Chiba in an existing
     * application with its own url-space, such as behind the Apache web server. As before with the hello xform the
     * following url will have the html form posting data to the http://www.example.com/test uri:
     *
     * http://127.0.0.1:8080/chiba-0.9.3/XFormsServlet?form=/forms/hello.xhtml&action_url=http://www.example.com/test
     *
	 * @param request servlet request
	 * @param response servlet response
	 * @throws javax.servlet.ServletException
	 * @throws java.io.IOException
	 */
	protected void doGet(
		HttpServletRequest request,
		HttpServletResponse response)
		throws ServletException, IOException {
		HttpSession session = request.getSession(true);
		//        ChibaBean chibaBean = null;

		try {
			response.setContentType("text/html");
			java.io.PrintWriter out = response.getWriter();

			// determine Form to load
			this.formPath = request.getParameter("form");
			if (formPath == null) {
				throw new IOException("File: " + formPath + " not found");
			}

            storeURIParams(request);

            // determine CSS file to be used if any is specified
			this.css = request.getParameter("css");

            //determine stylesheetfile to be used if any is specified
			this.stylesheetFile = request.getParameter("xslt");

            // build actionURL where forms are submitted to
            String actionURL = getActionURL(request, response);

            this.servletAdapter = setupServletAdapter(actionURL, session);
            this.servletAdapter.buildUI(out);

			session.setAttribute("chiba.adapter", servletAdapter);
			out.close();
		} catch (Exception e) {
			if (this.servletAdapter.getChibaBean() != null) {
				try {
					this.servletAdapter.getChibaBean().shutdown();
				} catch (XFormsException xfe) {
					xfe.printStackTrace();
				}
			}

			session.setAttribute("chiba.exception", e);
			// redirect to error page (after encoding session id if required)
			response.sendRedirect(
				response.encodeRedirectURL(
					request.getContextPath() + "/jsp/error.jsp"));
		}
	}


    /**
	 * handles all interaction with the user during a form-session.
	 *
	 * @param request servlet request
	 * @param response servlet response
	 * @throws javax.servlet.ServletException
	 * @throws java.io.IOException
	 */
	protected void doPost(
		HttpServletRequest request,
		HttpServletResponse response)
		throws ServletException, IOException {
		HttpSession session = request.getSession(true);
		ServletAdapter servletAdapter = null;

		try {
			servletAdapter = (ServletAdapter) session.getAttribute("chiba.adapter");
			if (servletAdapter == null) {
				//				throw new ServletException("Session contains no ServletAdapter instance.");
				throw new ServletException(
					Config.getInstance().getErrorMessage("session-invalid"));
			}
			servletAdapter.handleRequest(request);

			// handle setRedirect <xforms:load show='replace'/>
			// and redirect from submission as well
			// NOTE - this needs to be checked *before* the this.getForwardMap()
			// as a submission handler may force a redirect
			if (servletAdapter.getRedirectUri() != null) {
				String redirectTo = servletAdapter.getRedirectUri();
				// todo: remove from session ?
				// shutdown processor
				servletAdapter.getChibaBean().shutdown();

				// send setRedirect (after encoding session id if required)
				response.sendRedirect(response.encodeRedirectURL(redirectTo));

				// remove setRedirect uri and terminate
				servletAdapter.setRedirect(null);
				return;
			}

			// handle forward <xforms:submission replace='all'/>
            Map forwardMap = servletAdapter.getForwardMap();
            InputStream forwardStream = (InputStream) forwardMap.get(ChibaContext.FORWARD_RESPONSE_STREAM);
			if (forwardStream != null) {
				// todo: remove from session ?
				// shutdown processor
				servletAdapter.getChibaBean().shutdown();

				// fetch response stream
				InputStream responseStream =(InputStream) forwardMap.remove( ChibaContext.FORWARD_RESPONSE_STREAM);


				// copy header information
				Iterator iterator =	servletAdapter.getForwardMap().keySet().iterator();
				while (iterator.hasNext()) {
					String name = iterator.next().toString();
					String value = servletAdapter.getForwardMap().get(name).toString();
					response.setHeader(name, value);
				}

				// copy stream content
				OutputStream outputStream = new BufferedOutputStream(response.getOutputStream());
				for (int b = responseStream.read();
					b > -1;
					b = responseStream.read()) {
					outputStream.write(b);
				}

				// close streams
				responseStream.close();
				outputStream.close();

				// remove forward response  and terminate
				servletAdapter.forward(null);
				return;
			}

			// set content type
			response.setContentType("text/html");

			// render result to output
			this.servletAdapter.buildUI(response.getWriter());
			response.getWriter().close();
		} catch (Exception e) {
			if (servletAdapter != null) {
				try {
					servletAdapter.getChibaBean().shutdown();
				} catch (XFormsException xfe) {
					//nop
				}
			}
			session.setAttribute("chiba.exception", e);
			// redirect to error page (after encoding session id if required)
			response.sendRedirect(response.encodeRedirectURL(request.getContextPath() + "/jsp/error.jsp"));
		}
	}

    /**
     * create, configure and init the ServletAdapter which does the actual request processing.
     * @param actionURL - the URL to submit to
     * @param session - the Servlet session
     * @throws XFormsException
     */
    private ServletAdapter setupServletAdapter(String actionURL, HttpSession session) throws XFormsException {
        //setup and configure the adapter
        ServletAdapter aAdapter = new ServletAdapter();
        aAdapter.setContextRoot(contextRoot);
        if ((configPath != null) && !(configPath.equals(""))) {
            aAdapter.setConfigPath(configPath);
        }
        aAdapter.setFormPath(formPath);
        aAdapter.setStylesheetPath(stylesPath);
        aAdapter.setActionUrl(actionURL);
        aAdapter.setUploadDir(uploadDir);

        //set parameters if they exist
        //todo: integrate with WebContext
        if (instanceURIMap != null) {
            aAdapter.setInstanceURIMap(instanceURIMap);
        }
        if (submissionURIMap != null) {
            aAdapter.setSubmissionURIMap(submissionURIMap);
        }
        if (css != null) {
            aAdapter.setCSS(css);
        }
        if (stylesheetFile != null) {
            aAdapter.setStylesheet(stylesheetFile);
        }

//        Map servletMap = (Map)aAdapter.setParameter("httpservlet.adaptermap", new HashMap());
        Map servletMap = (Map)aAdapter.setParameter(ChibaContext.SUBMISSION_RESPONSE, new HashMap());
//        servletMap.put("tomcat.sessionid", session.getId());
        servletMap.put(ChibaContext.SESSION_ID, session.getId());

        aAdapter.init();
        return aAdapter;
    }


    private String getActionURL(HttpServletRequest request, HttpServletResponse response) {
        String defaultActionURL =
            request.getScheme()
                + "://"
                + request.getServerName()
                + ":"
                + request.getServerPort()
                + request.getContextPath()
                + request.getServletPath();
        String encodedDefaultActionURL = response.encodeURL(defaultActionURL);
        int sessIdx = encodedDefaultActionURL.indexOf(";jsession");
        String sessionId = null;
        if (sessIdx > -1) {
            sessionId = encodedDefaultActionURL.substring(sessIdx);
        }
        String actionURL = request.getParameter("action_url");
        if (null == actionURL) {
            actionURL = encodedDefaultActionURL;
        } else if (null != sessionId) {
            actionURL += sessionId;
        }

        cat.info("actionURL: " + actionURL);
        // encode the URL to allow for session id rewriting
        actionURL = response.encodeURL(actionURL);
        return actionURL;
    }

    /**
     * parses request params for some instance or submission URIs and if any stores them in
     * instanceURIMap or submissionURIMap respectively.
     *
     * @param request the ServleRequest to handle.
     */
    private void storeURIParams(HttpServletRequest request) {
        // determine instances and submission URI's if any are specified
        instanceURIMap = new HashMap();
        submissionURIMap = new HashMap();
        // *TODO* change to get param base names from web.xml
        String instanceParamBaseName = "instance";
        String submissionParamBaseName = "submit";
        Map paramsMap = request.getParameterMap();
        Iterator paramNames = paramsMap.keySet().iterator();
        // check each parameter, if is an instance path or submission uri then save it
        while (paramNames.hasNext()) {
            String paramName = (String) paramNames.next();
            String[] paramValues = (String[]) paramsMap.get(paramName);
            if (paramValues.length > 0) {
                String paramValue = paramValues[0];
                if (paramValue.length() > 0) {
                    if (paramName.startsWith(instanceParamBaseName)) {
                        if (paramValues.length != 1) {
                            cat.warn(
                                "Only 1 instance path per instance id is supported, defaulting to first occurrance");
                        }
                        instanceURIMap.put(
                            paramName.substring(
                                instanceParamBaseName.length()),
                            paramValue);
                    } else if (
                        paramName.startsWith(submissionParamBaseName)) {
                        if (paramValues.length != 1) {
                            cat.warn(
                                "Only 1 submission URI per id is supported, defaulting to first occurrance");
                        }
                        submissionURIMap.put(
                            paramName.substring(
                                submissionParamBaseName.length()),
                            paramValue);
                    }
                } else {
                    cat.warn("Empty value for param: " + paramName);
                }
            } else {
                cat.warn("No values for param: " + paramName);
            }
        }
    }

}
//end of class


