﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/neptune/NeptuneRequest.h>
#include <aws/neptune/Neptune_EXPORTS.h>
#include <aws/neptune/model/Parameter.h>

#include <utility>

namespace Aws {
namespace Neptune {
namespace Model {

/**
 */
class ModifyDBParameterGroupRequest : public NeptuneRequest {
 public:
  AWS_NEPTUNE_API ModifyDBParameterGroupRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ModifyDBParameterGroup"; }

  AWS_NEPTUNE_API Aws::String SerializePayload() const override;

 protected:
  AWS_NEPTUNE_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The name of the DB parameter group.</p> <p>Constraints:</p> <ul> <li> <p>If
   * supplied, must match the name of an existing DBParameterGroup.</p> </li> </ul>
   */
  inline const Aws::String& GetDBParameterGroupName() const { return m_dBParameterGroupName; }
  inline bool DBParameterGroupNameHasBeenSet() const { return m_dBParameterGroupNameHasBeenSet; }
  template <typename DBParameterGroupNameT = Aws::String>
  void SetDBParameterGroupName(DBParameterGroupNameT&& value) {
    m_dBParameterGroupNameHasBeenSet = true;
    m_dBParameterGroupName = std::forward<DBParameterGroupNameT>(value);
  }
  template <typename DBParameterGroupNameT = Aws::String>
  ModifyDBParameterGroupRequest& WithDBParameterGroupName(DBParameterGroupNameT&& value) {
    SetDBParameterGroupName(std::forward<DBParameterGroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An array of parameter names, values, and the apply method for the parameter
   * update. At least one parameter name, value, and apply method must be supplied;
   * subsequent arguments are optional. A maximum of 20 parameters can be modified in
   * a single request.</p> <p>Valid Values (for the application method):
   * <code>immediate | pending-reboot</code> </p>  <p>You can use the immediate
   * value with dynamic parameters only. You can use the pending-reboot value for
   * both dynamic and static parameters, and changes are applied when you reboot the
   * DB instance without failover.</p>
   */
  inline const Aws::Vector<Parameter>& GetParameters() const { return m_parameters; }
  inline bool ParametersHasBeenSet() const { return m_parametersHasBeenSet; }
  template <typename ParametersT = Aws::Vector<Parameter>>
  void SetParameters(ParametersT&& value) {
    m_parametersHasBeenSet = true;
    m_parameters = std::forward<ParametersT>(value);
  }
  template <typename ParametersT = Aws::Vector<Parameter>>
  ModifyDBParameterGroupRequest& WithParameters(ParametersT&& value) {
    SetParameters(std::forward<ParametersT>(value));
    return *this;
  }
  template <typename ParametersT = Parameter>
  ModifyDBParameterGroupRequest& AddParameters(ParametersT&& value) {
    m_parametersHasBeenSet = true;
    m_parameters.emplace_back(std::forward<ParametersT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_dBParameterGroupName;

  Aws::Vector<Parameter> m_parameters;
  bool m_dBParameterGroupNameHasBeenSet = false;
  bool m_parametersHasBeenSet = false;
};

}  // namespace Model
}  // namespace Neptune
}  // namespace Aws
