/* eclean-kernel2
 * (c) 2017 Michał Górny
 * 2-clause BSD license
 */

#ifdef HAVE_CONFIG_H
#	include "config.h"
#endif

#include "ek2/bootloaders/grub2.h"

#include "ek2/util/directorystream.h"
#include "ek2/util/error.h"
#include "ek2/util/relativepath.h"
#include "ek2/util/subprocess.h"

#include <cerrno>
#include <cstring>
#include <iostream>
#include <string>
#include <vector>

extern "C"
{
#	include <fcntl.h>
#	include <unistd.h>
};

static const char grub2_autogen_header[] =
	"#\n"
	"# DO NOT EDIT THIS FILE\n"
	"#\n"
	"# It is automatically generated by grub-mkconfig";

GRUB2::GRUB2(const Options& opts)
	: BootLoader(opts), is_autogenerated_(false), is_multislot_(false)
{
	DirectoryStream topds;

	try
	{
		topds.open(opts.boot_path);
	}
	catch (IOError& e)
	{
		if (e.err() != ENOENT)
			throw;
		return;
	}

	// try /boot/grub/grub.cfg first
	try
	{
		grub_dir_.reset(new DirectoryStream(RelativePath(topds, "grub")));
		grub_cfg_.reset(new RelativePath(grub_dir_, "grub.cfg"));
		process_config();
		return;
	}
	catch (IOError& e)
	{
		if (e.err() != ENOENT)
			throw;
	}

	// fallback to /boot/grub2/grub.cfg
	try
	{
		grub_dir_.reset(new DirectoryStream(RelativePath(topds, "grub2")));
		grub_cfg_.reset(new RelativePath(grub_dir_, "grub.cfg"));
		process_config();
		is_multislot_ = true;
		return;
	}
	catch (IOError& e)
	{
		if (e.err() != ENOENT)
			throw;
	}

	// no success? make sure it's null again
	grub_cfg_.reset(nullptr);
}

bool GRUB2::is_installed() const
{
	return !!grub_cfg_;
}

void GRUB2::process_config()
{
	int fd = grub_cfg_->file_fd(O_RDONLY);

	// check if the config is autogenerated
	char buf[sizeof(grub2_autogen_header)-1];
	ssize_t rd = read(fd, &buf, sizeof(buf));
	if (rd == -1)
		throw IOError("Reading failed on " + grub_cfg_->filename(), errno);
	if (rd == sizeof(buf) && !memcmp(buf, grub2_autogen_header, sizeof(buf)))
		is_autogenerated_ = true;
}

std::unique_ptr<BootLoader> GRUB2::construct(const Options& opts)
{
	std::unique_ptr<GRUB2> b{new GRUB2(opts)};

	if (b->is_installed())
		return std::move(b);

	return nullptr;
}

void GRUB2::postrm()
{
	// if the config is autogenerated, and we removed some kernels
	// we should update it
	if (is_autogenerated_)
	{
		std::vector<std::string> argv{
			is_multislot_ ? "grub2-mkconfig" : "grub-mkconfig",
			"-o", grub_cfg_->path()};

		if (opts_.pretend)
		{
			std::cerr << "\nThe following command would be run: "
				<< argv[0] << " " << argv[1] << " " << argv[2] << std::endl;
			return;
		}

		std::cerr << "\nRunning: " << argv[0] << " " << argv[1]
			<< " " << argv[2] << std::endl;
		run_subprocess(argv);
	}
}
