/**
 * <copyright> 
 * 
 * Copyright (c) 2004-2005 IBM Corporation and others. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License - v 1.0 
 * which accompanies this distribution, and is available at 
 * http://opensource.org/licenses/eclipse-1.0.txt 
 * 
 * Contributors: 
 *   IBM - Initial API and implementation 
 * 
 * </copyright> 
 * 
 * $Id: RDFS2Ecore.java,v 1.1 2007/03/18 08:39:04 lzhang Exp $
 */

package org.eclipse.eodm.rdf.transformer;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EAnnotation;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EEnum;
import org.eclipse.emf.ecore.EEnumLiteral;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EcoreFactory;
import org.eclipse.emf.ecore.EcorePackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.xmi.impl.EcoreResourceFactoryImpl;
import org.eclipse.eodm.rdf.rdfbase.PlainLiteral;
import org.eclipse.eodm.rdf.rdfbase.RDFGraph;
import org.eclipse.eodm.rdf.rdfbase.RDFProperty;
import org.eclipse.eodm.rdf.rdfbase.RDFSLiteral;
import org.eclipse.eodm.rdf.rdfbase.RDFSResource;
import org.eclipse.eodm.rdf.rdfbase.URIReference;
import org.eclipse.eodm.rdf.rdfs.RDFBag;
import org.eclipse.eodm.rdf.rdfs.RDFSClass;
import org.eclipse.eodm.rdf.rdfs.RDFSDatatype;
import org.eclipse.eodm.rdf.rdfweb.Document;
import org.eclipse.eodm.rdf.resource.RDFXMLResource;
import org.eclipse.eodm.rdf.resource.RDFXMLResourceFactoryImpl;
import org.eclipse.eodm.rdf.transformer.EODMRDFSTransformerException;
import org.eclipse.eodm.vocabulary.RDF;
import org.eclipse.eodm.vocabulary.RDFS;


/**
 * The API realizes translation from RDFS model to ECore model.
 *  
 */
public class RDFS2Ecore {

	public static void rdf2ecore(String rdfFile, String ecoreFile, Map options) throws EODMRDFSTransformerException {
		
		EPackage epackage = rdf2ecore(rdfFile,options);
		
        //save ecore model
        ResourceSet resourceSet = new ResourceSetImpl();
        resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap()
                .put(Resource.Factory.Registry.DEFAULT_EXTENSION,
                        new EcoreResourceFactoryImpl());

        Resource resource = resourceSet.createResource(URI
                .createFileURI(new File(ecoreFile).getAbsolutePath()));
        resource.getContents().add(epackage);
        
        try {
            resource.save(Collections.EMPTY_MAP);
        } catch (Exception e) {
        	throw new EODMRDFSTransformerException(
                    "Error during serialize into XMI file: " + e.toString());
        }
	}
	
	public static EPackage rdf2ecore(String rdfFile, Map options) throws EODMRDFSTransformerException {
		RDFGraph graph = null;
		try {
			graph = loadRDF(rdfFile);
		} catch(Exception e) {
			throw new EODMRDFSTransformerException("Can not load rdf files: " + e.getMessage());
		}
		
		return rdf2ecore(graph, options);
	}

	public static EPackage rdf2ecore(RDFGraph graph, Map options)  throws EODMRDFSTransformerException {
		EcoreFactory efactory = EcoreFactory.eINSTANCE;
		EPackage ePackage = efactory.createEPackage();
		
		//set info for ePackage
		URIReference graphUri = graph.getGraphName();
		
		if(graphUri != null) {
			if(graphUri.getNamespace() != null ) {
				//ePackage.setNsPrefix( graphUri.getNamespace().getNamespaceURIRef());
			}
			ePackage.setName(graphUri.getUri().getName());
		}
		
		
		try {
			List clist = graph.getTypeResources(RDFS.C_CLASS_STR);
			
			HashMap rdf2eclassifierMap = new HashMap();
	        
	        // RDFBag -> EEnum
	        List baglist = graph.getTypeResources(RDF.C_BAG_STR);
	        if( baglist != null) {
		        for(Iterator it=baglist.iterator();it.hasNext();){
		        	EEnum eenum = rdfBag2EEnum((RDFBag)it.next(),rdf2eclassifierMap);
		        	ePackage.getEClassifiers().add(eenum);
		        }
	        }
	        
			// rdf class to eclass
			for(Iterator it=clist.iterator();it.hasNext();) {
				RDFSClass rclass = (RDFSClass)it.next();
				EClassifier eclass = rdfs2eclassifier(rclass, rdf2eclassifierMap);
				
				if(eclass!=null) {
					// add super class
					for(Iterator itSuper=rclass.getRDFSsubClassOf().iterator();itSuper.hasNext();){
						RDFSClass superClass = (RDFSClass)itSuper.next();
						EClass efather = (EClass)rdfs2eclassifier(superClass, rdf2eclassifierMap);
						if(efather!=null) {
							((EClass)eclass).getESuperTypes().add(efather);
						}
					}
					
					// RDFProperty -> EReference, EAttribute
					for(Iterator itP = rclass.getPropertyForDomain().iterator();itP.hasNext();){
						RDFProperty p = (RDFProperty) (itP.next());
						List rangeList = p.getRDFSrange();
						if(!rangeList.isEmpty()) {
							RDFSClass range = (RDFSClass)rangeList.get(0);
							EClassifier erange = rdfs2eclassifier(range, rdf2eclassifierMap);
							
							if(erange instanceof EDataType) {
								EAttribute attribute = EcoreFactory.eINSTANCE.createEAttribute();
								attribute.setName( getName(p));
								attribute.getEAnnotations().addAll( comments2EAnnotation(p) );
								attribute.setEType(erange);
								((EClass)eclass).getEStructuralFeatures().add(attribute);
							} else {
								EReference eref = EcoreFactory.eINSTANCE.createEReference();
								eref.setName(getName(p));
								eref.getEAnnotations().addAll( comments2EAnnotation(p) );
								eref.setEType(erange);
								((EClass)eclass).getEStructuralFeatures().add(eref);
							}
							
						}

					}
					
					ePackage.getEClassifiers().add(eclass);
				}

			}

	        
		} catch(Exception e) {
			throw new EODMRDFSTransformerException("There are something wrong during transformation" +
					e.getMessage());
		}
		
		return ePackage;
	}
	
	private static List comments2EAnnotation(RDFSResource rsource) {
		List lst = new ArrayList();
		
		for(Iterator it=rsource.getRDFScomment().iterator();it.hasNext();){
			EAnnotation an = EcoreFactory.eINSTANCE.createEAnnotation();
			an.setSource( ((RDFSLiteral)it.next()).getLexicalForm());
			lst.add(an);
		}
		
		for(Iterator it=rsource.getRDFSisDefinedBy().iterator();it.hasNext();){
			EAnnotation an = EcoreFactory.eINSTANCE.createEAnnotation();
			an.setSource(  getName((RDFSResource)it.next()) );
		}
		
		for(Iterator it=rsource.getRDFSlabel().iterator();it.hasNext();){
			EAnnotation an = EcoreFactory.eINSTANCE.createEAnnotation();
			an.setSource( ((PlainLiteral) it.next()).getLexicalForm());
		}
		
		return lst;
	}
	
	// it could return null when rclass is datatype and dtURI = null
	private static EClassifier rdfs2eclassifier(RDFSClass rclass, Map rdfs2ecoreMap) {
		EClassifier eclass = (EClassifier)rdfs2ecoreMap.get(rclass);
		
		if(eclass==null) {
			if(rclass instanceof RDFSDatatype) {
	            RDFSDatatype dt = (RDFSDatatype) rclass;
	            String dtURI = dt.getURI();
	            if( dtURI != null ) {
	            	if(dtURI.equals(XSD_BOOLEAN) )
	            		eclass = EcorePackage.eINSTANCE.getEBoolean();
	            	else if(dtURI.equals(XSD_FLOAT) )
	            		eclass = EcorePackage.eINSTANCE.getEFloat();
	            	else if(dtURI.equals(XSD_INT))
	            		eclass = EcorePackage.eINSTANCE.getEInt();
	            	else if(dtURI.equals(XSD_BYTE))
	            		eclass = EcorePackage.eINSTANCE.getEByte();
	            	else if(dtURI.equals(XSD_LONG))
	            		eclass = EcorePackage.eINSTANCE.getELong();
	            	else if(dtURI.equals(XSD_DOUBLE))
	            		eclass = EcorePackage.eINSTANCE.getEDouble();
	            	else if(dtURI.equals(XSD_SHORT))
	            		eclass = EcorePackage.eINSTANCE.getEShort();
	            	else if(dtURI.equals(XSD_STRING)|| dtURI.equals(RDFS.C_LITERAL_STR) 
							|| dtURI.equals(RDF.C_XMLLITERAL_STR))
	            		eclass = EcorePackage.eINSTANCE.getEString();
	            	else if(RDFBag.class.isInstance(
		        	  (dt.getRDFSisDefinedBy().size() > 0 ? dt.getRDFSisDefinedBy().get(0) : null) )) {
	            		eclass = rdfBag2EEnum((RDFBag) dt.getRDFSisDefinedBy().get(0), rdfs2ecoreMap);
	            	}
	            	else {
		                // Create a user-defined EDatatType
		                eclass = EcoreFactory.eINSTANCE.createEDataType();
		                eclass.setName( getName(dt) );
		                eclass.getEAnnotations().addAll( comments2EAnnotation(dt) );
	            	}
	            }
			} else {
				// rdfs class
				eclass = EcoreFactory.eINSTANCE.createEClass();
				eclass.setName( getName(rclass) );
				eclass.getEAnnotations().addAll(comments2EAnnotation(rclass));
			}
			
			if(eclass!=null) {
				rdfs2ecoreMap.put(rclass,eclass);
			}

		}
		
		return eclass;
	}
	
    /**
     * Create EEnum from RDFBag
     * 
     * @param bag
     * @return
     */
    private static EEnum rdfBag2EEnum(RDFBag bag, Map rdfs2ecoreMap) {
        EEnum eenum = (EEnum)rdfs2ecoreMap.get(bag);
        
        if(eenum==null) {
	        List memberList = bag.getRDFSmember();
	        // if bag.size > 0, create EEnum
	        if (memberList.size()>0) {
	            eenum = EcoreFactory.eINSTANCE.createEEnum();
	            eenum.setName( getName(bag) );
	            eenum.getEAnnotations().addAll( comments2EAnnotation(bag) );
	
	            for(int i=0; i< memberList.size();i++){
	            	RDFSResource r = (RDFSResource) (memberList.get(i));
	            	
	            	EEnumLiteral eliteral = EcoreFactory.eINSTANCE.createEEnumLiteral();
	            	String name = (r instanceof RDFSLiteral) ? ((RDFSLiteral) r).getLexicalForm() : getName(r);
	            	eliteral.setName(name);
	            	eliteral.setValue(i);
	            	eliteral.getEAnnotations().addAll( comments2EAnnotation(r) );
	            	eenum.getELiterals().add(eliteral);
	            }
	            
	            rdfs2ecoreMap.put(bag,eenum);
	        }
	        
        }
        return eenum;
    }
	
    private static RDFGraph loadRDF(String rdfFile) throws IOException {
        // create a resource set
        ResourceSet resourceSet = new ResourceSetImpl();

        // register the RDF resource factory for all ".rdf" files/URIs;
        // you can also register for ".xml" or even the wild card "*" extension
        // depending
        // on the URI of the RDF/XML content you are going to load;       
        resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap()
                .put(RDFXMLResource.DEFAULT_RDFXML_EXTENSION,new RDFXMLResourceFactoryImpl() );


        // get the URI of the RDFS file.
        // If the content is a file, use the following code
        URI fileURI = URI.createFileURI(new File(rdfFile).getAbsolutePath());

        // create a resource for this URI.
        Resource resource = resourceSet.createResource(fileURI);

        // load the RDF/XML content as a EODM RDFS ontology model
        resource.load(Collections.EMPTY_MAP);

        // obtain the RDFS ontology
        return ((Document) resource.getContents().get(0)).getComplementalGraph();
    }
    
    private static String getName(RDFSResource r) {
    	// create name for eclass
    	String name = null;
    	
    	if( !r.getUriRef().isEmpty() ) {
    		URIReference uriRef = (URIReference)r.getUriRef().get(0);
    		name = (uriRef.getFragmentIdentifier()==null) ? 
    				uriRef.getUri().getName() : uriRef.getFragmentIdentifier().getName();
    	} else {
    		name = r.getURI();
    	}
    	
    	return name;
    }
	    
    private static String XSD_BOOLEAN = "http://www.w3.org/2001/XMLSchema#boolean";
    private static String XSD_FLOAT = "http://www.w3.org/2001/XMLSchema#float";
    private static String XSD_BYTE = "http://www.w3.org/2001/XMLSchema#byte";
    private static String XSD_INT = "http://www.w3.org/2001/XMLSchema#int";
    private static String XSD_LONG = "http://www.w3.org/2001/XMLSchema#long";
    private static String XSD_DOUBLE = "http://www.w3.org/2001/XMLSchema#double";
    private static String XSD_SHORT = "http://www.w3.org/2001/XMLSchema#short";
    private static String XSD_STRING = "http://www.w3.org/2001/XMLSchema#string";

}
