package org.eclipse.eodm.owl.transformer;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.common.util.EMap;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EAnnotation;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EEnum;
import org.eclipse.emf.ecore.EEnumLiteral;
import org.eclipse.emf.ecore.ENamedElement;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EcorePackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.xmi.impl.XMIResourceFactoryImpl;
import org.eclipse.eodm.OWLFactory;
import org.eclipse.eodm.RDFFactory;
import org.eclipse.eodm.exceptions.UnsupportedViewTypeException;
import org.eclipse.eodm.owl.owlbase.CardinalityRestriction;
import org.eclipse.eodm.owl.owlbase.EnumeratedClass;
import org.eclipse.eodm.owl.owlbase.Individual;
import org.eclipse.eodm.owl.owlbase.MaxCardinalityRestriction;
import org.eclipse.eodm.owl.owlbase.MinCardinalityRestriction;
import org.eclipse.eodm.owl.owlbase.OWLClass;
import org.eclipse.eodm.owl.owlbase.OWLDatatypeProperty;
import org.eclipse.eodm.owl.owlbase.OWLGraph;
import org.eclipse.eodm.owl.owlbase.OWLObjectProperty;
import org.eclipse.eodm.owl.resource.OWLXMLSaver;
import org.eclipse.eodm.owl.transformer.EODMOWLTransformerException;
import org.eclipse.eodm.rdf.rdfbase.PlainLiteral;
import org.eclipse.eodm.rdf.rdfbase.RDFSResource;
import org.eclipse.eodm.rdf.rdfbase.TypedLiteral;
import org.eclipse.eodm.rdf.rdfs.RDFSDatatype;
import org.eclipse.eodm.rdf.rdfweb.Document;

public class Ecore2OWL {
	
	public static void ecore2OWL(String ecoreFilePath, String owlFilePath, Map options) throws EODMOWLTransformerException {
		OWLGraph graph = ecore2OWL(ecoreFilePath,options);
		
		saveOWLGraph(graph, owlFilePath, null);
	}
	
    public static OWLGraph ecore2OWL(String ecoreFilePath, Map options)throws EODMOWLTransformerException {
        //register default resource factory
        Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(Resource.Factory.Registry.DEFAULT_EXTENSION, new XMIResourceFactoryImpl());
        
        //create resource set to hold the resource we're loading and its dependent resources
        ResourceSet resourceSet = new ResourceSetImpl();
        
        // load file
        resourceSet.getResource(URI.createFileURI(ecoreFilePath), true);
        File inputFile = new File(ecoreFilePath);
        URI absoluteInputURI = URI.createFileURI(inputFile.getAbsolutePath());
        
        Resource resource = resourceSet.getResource(absoluteInputURI, true);

        return ecore2OWL(resource, options);
    }
	
    public static OWLGraph ecore2OWL(Resource ecoreResource, Map options) throws EODMOWLTransformerException {
        if(options==null) {
            options = Collections.EMPTY_MAP;
        }
        
        EPackage ePackage = (EPackage)ecoreResource.getContents().get(0);

        return ecore2OWL(ePackage,options);
    }
    
	public static OWLGraph ecore2OWL(EPackage ePackage, Map options) throws EODMOWLTransformerException {
		OWLGraph graph = null;
		
		String nsUri = ePackage.getNsURI();
		
		try {
			OWLFactory factory = OWLFactory.eINSTANCE;
			
			// retrieve all eclassifier need to be transformed
			graph = factory.createOWLGraph(nsUri);
			
			// retrieve all eclassifier need to be transformed
			List eclassifierLst = getEclassifier(ePackage);
			
			// EClass -> OWLClass
			for(Iterator it = eclassifierLst.iterator();it.hasNext();) {
				EClassifier eclassifier = (EClassifier) it.next();
				
	            // transform classifier
				RDFSResource rclass = eclass2OWL(eclassifier, graph);				
				
				if(eclassifier instanceof EClass) {
					OWLClass oclass = (OWLClass)rclass;
					
	                // eSuperTypes -> RDFSSubClassOf
	                EClass eclass = (EClass) eclassifier;
	                for(Iterator itSuper=eclass.getESuperTypes().iterator();itSuper.hasNext();) {
	                	EClassifier eSuper = (EClassifier)itSuper.next();
	                	if(eSuper.eIsProxy()) {
	                		continue;
	                	}
	                    OWLClass parents = (OWLClass)eclass2OWL(eSuper,graph);
	                    parents.getSubClass().add(rclass);
	                }
	                    
	                // EAttribute -> OWLDatatypeProperty
	                for(Iterator itAtt=eclass.getEAttributes().iterator();itAtt.hasNext();) {
	                    EAttribute eatt = (EAttribute) itAtt.next();
	                    if(eatt.eIsProxy())
	                    	continue;	                    
	                    OWLDatatypeProperty dp = eatt2DP(eatt, graph);
	                    dp.getRDFSdomain().add(oclass);

	                }

	                // EReference -> OWLObjectProperty
	                for(Iterator itRef=eclass.getEReferences().iterator();itRef.hasNext();) {
	                    EReference eref = (EReference) itRef.next();
	                    OWLObjectProperty op = eref2OP(eref, graph);
	                    op.getRDFSdomain().add(oclass);

	                    /*
	                     * determine multiplicity: 5 situation situation
	                     *      lowBound upperBound 
	                     * 0..*    0          -1 
	                     * m..*    m          -1 
	                     * 0..n    0           n 
	                     * m..n    m           n 
	                     * m..m    m           m
	                     */
	                    int lowerBound = eref.getLowerBound();
	                    int upperBound = eref.getUpperBound();
	                    if(lowerBound > 0 ) {
	                        if(upperBound==-1) {
	                            // minCardinality  m..*  (m, -1)
	                            oclass.getRDFSsubClassOf().add( createMinCardinarlity(graph, op,lowerBound));
	                        } else if(lowerBound==upperBound){
	                            // Cardinality (m,m)-->m..m
	                            oclass.getRDFSsubClassOf().add( createCardinarlity(graph, op,lowerBound));
	                        } else {
	                            // minCardinality & maxCardinality (m,n)-->m..n 
	                            oclass.getRDFSsubClassOf().add( createMinCardinarlity(graph, op, lowerBound));
	                            oclass.getRDFSsubClassOf().add( createMaxCardinarlity(graph, op, upperBound));
	                        }
	                    } else if(upperBound!=-1){
	                        // maxCardinality  (0,n)-->0..n
	                        oclass.getRDFSsubClassOf().add( createMaxCardinarlity(graph,op,upperBound));
	                    }
	                }
				}
			}
			
			
		} catch (URISyntaxException e) {
        	throw new EODMOWLTransformerException("Something wrong when creating OWL Resource:" + e.getMessage());
		}
		
		return graph; 
	}
	
	private static RDFSResource eclass2OWL(EClassifier eclassifier, OWLGraph graph) throws EODMOWLTransformerException {
		OWLFactory factory = OWLFactory.eINSTANCE;
		
		String uri = getURI(eclassifier);
		RDFSResource rsource = graph.getRDFSResource(uri);
		
		if(rsource == null){
			try {
				if(eclassifier instanceof EClass ){
	                // EClass -> OWLClass
	                EClass eclass = (EClass)eclassifier;
	                OWLClass oclass = factory.createOWLClass(graph, uri);
	
	                //use namespace of eContainer
	                oclass.getRDFScomment().addAll( eAnnotation2Literal(eclass) );
	                oclass.getRDFSlabel().add( RDFFactory.eINSTANCE.createPlainLiteral(eclass.getName()) );
	                
	                rsource = oclass;
				} else if (eclassifier instanceof EEnum) {
	                // EEnum -> OWLOneOf
	                EEnum eenum = (EEnum) eclassifier;
	                
	                EnumeratedClass enumclass = factory.createEnumeratedClass(graph, uri);
	                enumclass.getRDFScomment().addAll( eAnnotation2Literal(eenum) );
	                enumclass.getRDFSlabel().add(RDFFactory.eINSTANCE.createPlainLiteral(eenum.getName()));
	
	                // EEnumLiteral -> OWLMember
	                for(Iterator et = eenum.getELiterals().iterator(); et.hasNext();) {
	                    EEnumLiteral enumLiteral = (EEnumLiteral) et.next();
	                    
	                    Individual individual = factory.createIndividual(graph, enumLiteral.getName());
	                    individual.getRDFSlabel().add( RDFFactory.eINSTANCE.createPlainLiteral(enumLiteral.getName()) );
	                    enumclass.getOWLoneOf().add(individual);
	                }
	
	                rsource = enumclass;
	            } else if (eclassifier instanceof EDataType) {
	                // EDataType -> RDFSDatatype
	                EDataType datatype = (EDataType) eclassifier;
	                RDFSDatatype rdatatype = RDFFactory.eINSTANCE.createRDFSDatatype(graph, datatype.getName());
	
	                rdatatype.getRDFSlabel().add( RDFFactory.eINSTANCE.createPlainLiteral(datatype.getName()) );
	                rdatatype.getRDFScomment().addAll( eAnnotation2Literal(datatype) );
	                
	                rsource = rdatatype;
	            }
				
			} catch (URISyntaxException e) {
	        	throw new EODMOWLTransformerException("Something wrong when creating OWL Resource:" + e.getMessage());
			}
		}
		
		return rsource;
	}
	
	private static OWLDatatypeProperty eatt2DP(EAttribute eatt, OWLGraph graph) throws EODMOWLTransformerException {
		OWLFactory factory = OWLFactory.eINSTANCE;
		OWLDatatypeProperty odp = null;
		
		String uri = getURI(eatt);
		
		RDFSResource rsource = graph.getRDFSResource(uri);
		
		try {
			if(rsource == null) {
				odp = factory.createOWLDatatypeProperty(graph,uri);		
			} else if (graph.getRDFSResource(uri).canAsType(OWLDatatypeProperty.class)) {
				odp = (OWLDatatypeProperty)rsource;
			}
	
			EClassifier etype = eatt.getEAttributeType();
			
			if(etype.eIsProxy()) {
				
			} else if (!EEnum.class.isInstance(etype) ) {
	        	String xsdName = null;
	            //check if XSD datatype
	            if (etype == EcorePackage.eINSTANCE.getEBooleanObject())
	            	xsdName = XSD_BOOLEAN;
	            else if (etype == EcorePackage.eINSTANCE.getEFloatObject())
	            	xsdName = XSD_FLOAT;
	            else if (etype == EcorePackage.eINSTANCE.getEByteObject())
	            	xsdName = XSD_BYTE;
	            else if (etype == EcorePackage.eINSTANCE.getEInt())
	            	xsdName = XSD_INT;
	            else if (etype == EcorePackage.eINSTANCE.getELongObject())
	            	xsdName = XSD_LONG;
	            else if (etype == EcorePackage.eINSTANCE.getEDoubleObject())
	            	xsdName = XSD_DOUBLE;
	            else if (etype == EcorePackage.eINSTANCE.getEShortObject())
	            	xsdName = XSD_SHORT;
	            else if (etype == EcorePackage.eINSTANCE.getEIntegerObject())
	            	xsdName = XSD_INTEGER;
	            else if (etype == EcorePackage.eINSTANCE.getEString())
	            	xsdName = XSD_STRING;
	
	            if (xsdName != null) {
	            	RDFSDatatype type = RDFFactory.eINSTANCE.createRDFSDatatype(graph,xsdName);
	            	odp.getRDFSrange().add(type);
	            } else {
	                // user-defined datatype
	            	String euri = getURI(etype);
	            	
	            	RDFSDatatype rtype = (RDFSDatatype)graph.getRDFSResource(euri);
	
	                if (rtype == null) {
	                    rtype = RDFFactory.eINSTANCE.createRDFSDatatype(graph,euri);
	                } 
	                
	                odp.getRDFSrange().add(rtype);
	            }
	        	
	        } else {
	            // a oneof attribute
	            EnumeratedClass enumclass = (EnumeratedClass) graph.getRDFSResource( getURI(etype) );
	
	            if (enumclass != null) {
	                //TODO:RDFSDatatype dummy = RDFFactory.eINSTANCE.createRDFSDatatype(graph,  );
	            }
	        }
	        
		} catch (URISyntaxException e) {
        	throw new EODMOWLTransformerException("Something wrong when creating OWL Resource:" + e.getMessage());
		}
		
		return odp;
	}
	
	private static OWLObjectProperty eref2OP(EReference eref, OWLGraph graph) throws EODMOWLTransformerException {
		String uri = getURI(eref);
		
		RDFSResource rs = graph.getRDFSResource(uri);
		OWLObjectProperty oop = null;
		
		try {
			
			if(rs == null || !rs.canAsType(OWLObjectProperty.class)) {
				// new property
				oop = OWLFactory.eINSTANCE.createOWLObjectProperty(graph, uri);
				
				// range
				EClassifier erange = eref.getEReferenceType();
				if(!erange.eIsProxy()) {
					OWLClass range = (OWLClass) eclass2OWL(erange,graph);
					oop.getRDFSrange().add(range);
				}
			} else {
				oop = (OWLObjectProperty)rs.asType(OWLObjectProperty.class);
			}
			
		} catch (UnsupportedViewTypeException e) {
			throw new EODMOWLTransformerException("Cannot retrieve resource:" + e.getMessage());
		} catch (URISyntaxException e) {
			throw new EODMOWLTransformerException("Something wrong when creating OWL Resource:" + e.getMessage());
		}
		
		return oop;
	}
	
	private static String getURI(ENamedElement elem) {
		
		EObject nsOwner = elem;
		
		String ns = null;
        // use EPackage's Namespace 
        while (nsOwner != null) {
            if (nsOwner instanceof EPackage) {
                ns = ((EPackage) nsOwner).getNsURI();               
                break;
            } else {
                nsOwner = nsOwner.eContainer();
            }
        }
        
		return ns.endsWith("#") ? (ns + elem.getName() ): (ns + "#" + elem.getName());
	}
	
    /**
     * 
     * @param eNameElem
     * @return a list of PlainLiterals from eclassifier
     */
    private static List eAnnotation2Literal(ENamedElement eNameElem) {
    	List alist = new ArrayList();
    	
        for(Iterator itAn = eNameElem.getEAnnotations().iterator();itAn.hasNext();){
        	EAnnotation an = (EAnnotation)itAn.next();
        	if(an!=null){
        		PlainLiteral literal = RDFFactory.eINSTANCE.createPlainLiteral(eAnnotation2String(an));
        		alist.add(literal);
        	}       	
        }
    	
    	return alist;
    }
    
    
    private static String eAnnotation2String(EAnnotation an) {
        String str = an.getSource() + ": ";
        EMap detail = (EMap) an.getDetails();
        for (Iterator iter = detail.keySet().iterator(); iter.hasNext();) {
            Object key = iter.next();
            str += "  " + key + ":" + detail.get(key);
        }
        return replaceKeywords(str);
    }

    private static String replaceKeywords(String str) {
        str = str.replaceAll("&", "&amp;");
        str = str.replaceAll("<", "&lt;");
        str = str.replaceAll(">", "&gt;");
        str = str.replaceAll("'", "&apos;");
        str = str.replaceAll("\"", "&quot;");

        return str;
    }
    
    private static MinCardinalityRestriction createMinCardinarlity(OWLGraph graph, OWLObjectProperty property, int bound) throws EODMOWLTransformerException {
        // cardinality
        MinCardinalityRestriction minCr = OWLFactory.eINSTANCE.createMinCardinalityRestriction(graph);

        try {
	        TypedLiteral literal = RDFFactory.eINSTANCE.createTypedLiteral(Integer.toString(bound),XSD_NON_NEGATIVE_INT);
	
	        minCr.setOWLonProperty(property);
	        minCr.setOWLminCardinality(literal);
        } catch (Exception e) {
        	throw new EODMOWLTransformerException("Something wrong when creating OWL Resource:" + e.getMessage());
        }
        
        return minCr;
    }

    private static MaxCardinalityRestriction createMaxCardinarlity(OWLGraph graph, OWLObjectProperty property, int bound) throws EODMOWLTransformerException  {
        // cardinality
        MaxCardinalityRestriction maxCr = OWLFactory.eINSTANCE.createMaxCardinalityRestriction(graph);

        try {
	        TypedLiteral literal = RDFFactory.eINSTANCE.createTypedLiteral(Integer.toString(bound),XSD_NON_NEGATIVE_INT);
	        maxCr.setOWLonProperty(property);
	        maxCr.setOWLmaxCardinality(literal);
        } catch (Exception e) {
        	throw new EODMOWLTransformerException("Something wrong when creating OWL Resource:" + e.getMessage());
        }

        return maxCr;
    }

    private static CardinalityRestriction createCardinarlity(OWLGraph graph, OWLObjectProperty property, int bound) throws EODMOWLTransformerException {
        // cardinality
        CardinalityRestriction cr = OWLFactory.eINSTANCE.createCardinalityRestriction(graph);

        try {
	        TypedLiteral literal = RDFFactory.eINSTANCE.createTypedLiteral(Integer.toString(bound),XSD_NON_NEGATIVE_INT);
	
	        cr.setOWLonProperty(property);
	        cr.setOWLcardinality(literal);
        } catch (Exception e) {
        	throw new EODMOWLTransformerException("Something wrong when creating OWL Resource:" + e.getMessage());
        }
        return cr;
    }
    
    private static void saveOWLGraph(OWLGraph graph, String filePath,String docURI) {

        try {
        	Document document = RDFFactory.eINSTANCE.createDocument(docURI);
            document.setComplementalGraph(graph);
            OWLXMLSaver.saveToFile(document, filePath, "UTF-8");

        } catch (URISyntaxException e) {
        	e.printStackTrace();
        } catch (UnsupportedViewTypeException e) {
        	e.printStackTrace();
        }
        catch (IOException e1) {
            e1.printStackTrace();
        }
        
    }
    
	static List getEclassifier(EPackage ePackage) {
		List lst = ePackage.getEClassifiers();
		
		for (Iterator iter = ePackage.getESubpackages().iterator(); iter.hasNext();) {
			EPackage sube = (EPackage) iter.next();
			lst.addAll( getEclassifier(sube)  );
		}
		
		return lst;
	}
    
    private static String XSD_BOOLEAN = "http://www.w3.org/2001/XMLSchema#boolean";
    private static String XSD_FLOAT = "http://www.w3.org/2001/XMLSchema#float";
    private static String XSD_BYTE = "http://www.w3.org/2001/XMLSchema#byte";
    private static String XSD_INT = "http://www.w3.org/2001/XMLSchema#int";
    private static String XSD_LONG = "http://www.w3.org/2001/XMLSchema#long";
    private static String XSD_DOUBLE = "http://www.w3.org/2001/XMLSchema#double";
    private static String XSD_SHORT = "http://www.w3.org/2001/XMLSchema#short";
    private static String XSD_INTEGER = "http://www.w3.org/2001/XMLSchema#integer";
    private static String XSD_STRING = "http://www.w3.org/2001/XMLSchema#string";
    private static String XSD_NON_NEGATIVE_INT = "http://www.w3.org/2001/XMLSchema#nonNegativeInteger";
    
    
}
