/*****************************************************************************
 * Copyright (c) 2009, 2015 CEA, Christian W. Damus, and others
 *
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Atos Origin - Initial API and implementation
 *   Christian W. Damus - bug 433206
 *
 *****************************************************************************/
package org.eclipse.papyrus.uml.diagram.sequence.edit.policies;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.eclipse.core.commands.operations.IUndoableOperation;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.draw2d.ConnectionAnchor;
import org.eclipse.draw2d.PositionConstants;
import org.eclipse.draw2d.geometry.Dimension;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.Request;
import org.eclipse.gef.commands.Command;
import org.eclipse.gef.commands.CompoundCommand;
import org.eclipse.gef.commands.UnexecutableCommand;
import org.eclipse.gef.requests.ChangeBoundsRequest;
import org.eclipse.gef.requests.CreateConnectionRequest;
import org.eclipse.gmf.runtime.common.core.command.CompositeCommand;
import org.eclipse.gmf.runtime.common.core.command.ICommand;
import org.eclipse.gmf.runtime.diagram.core.commands.AddCommand;
import org.eclipse.gmf.runtime.diagram.core.edithelpers.CreateElementRequestAdapter;
import org.eclipse.gmf.runtime.diagram.core.util.ViewUtil;
import org.eclipse.gmf.runtime.diagram.ui.commands.CommandProxy;
import org.eclipse.gmf.runtime.diagram.ui.commands.ICommandProxy;
import org.eclipse.gmf.runtime.diagram.ui.commands.SetBoundsCommand;
import org.eclipse.gmf.runtime.diagram.ui.editparts.ConnectionNodeEditPart;
import org.eclipse.gmf.runtime.diagram.ui.editparts.DiagramEditPart;
import org.eclipse.gmf.runtime.diagram.ui.editparts.GraphicalEditPart;
import org.eclipse.gmf.runtime.diagram.ui.editparts.IGraphicalEditPart;
import org.eclipse.gmf.runtime.diagram.ui.editparts.INodeEditPart;
import org.eclipse.gmf.runtime.diagram.ui.requests.CreateConnectionViewRequest;
import org.eclipse.gmf.runtime.diagram.ui.requests.CreateViewRequest;
import org.eclipse.gmf.runtime.diagram.ui.requests.CreateViewRequest.ViewDescriptor;
import org.eclipse.gmf.runtime.diagram.ui.requests.CreateViewRequestFactory;
import org.eclipse.gmf.runtime.diagram.ui.requests.DropObjectsRequest;
import org.eclipse.gmf.runtime.diagram.ui.requests.EditCommandRequestWrapper;
import org.eclipse.gmf.runtime.emf.core.util.EObjectAdapter;
import org.eclipse.gmf.runtime.emf.type.core.IElementType;
import org.eclipse.gmf.runtime.emf.type.core.IHintedType;
import org.eclipse.gmf.runtime.emf.type.core.requests.CreateElementRequest;
import org.eclipse.gmf.runtime.emf.type.core.requests.MoveRequest;
import org.eclipse.gmf.runtime.notation.Node;
import org.eclipse.gmf.runtime.notation.View;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.osgi.util.NLS;
import org.eclipse.papyrus.commands.wrappers.CommandProxyWithResult;
import org.eclipse.papyrus.infra.gmfdiag.common.adapter.SemanticAdapter;
import org.eclipse.papyrus.infra.gmfdiag.common.utils.DiagramEditPartsUtil;
import org.eclipse.papyrus.uml.diagram.common.commands.DeferredCreateCommand;
import org.eclipse.papyrus.uml.diagram.common.editpolicies.CommonDiagramDragDropEditPolicy;
import org.eclipse.papyrus.uml.diagram.common.helper.DurationConstraintHelper;
import org.eclipse.papyrus.uml.diagram.common.helper.DurationObservationHelper;
import org.eclipse.papyrus.uml.diagram.sequence.command.OLDCreateGateViewCommand;
import org.eclipse.papyrus.uml.diagram.sequence.command.CreateLocatedConnectionViewCommand;
import org.eclipse.papyrus.uml.diagram.sequence.command.RestoreDurationConstraintLinkCommand;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.ActionExecutionSpecificationEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.BehaviorExecutionSpecificationEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.CCombinedCompartmentEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.CombinedFragmentEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.CommentAnnotatedElementEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.CommentBodyEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.CommentEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.ConsiderIgnoreFragmentEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.Constraint2EditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.ConstraintConstrainedElementEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.ConstraintEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.ContinuationEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.OLDCustomCombinedFragmentEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.DestructionOccurrenceSpecificationEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.DurationConstraintEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.DurationConstraintInMessageEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.DurationObservationEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.GateEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.OLDGateEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.GeneralOrderingEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.InteractionEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.InteractionInteractionCompartmentEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.InteractionOperandEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.InteractionUseEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.LifelineEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.MessageAsyncEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.MessageReplyEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.MessageCreateEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.MessageDeleteEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.MessageLostEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.MessageFoundEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.MessageSyncEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.SequenceDiagramEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.StateInvariantEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.TimeConstraintEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.edit.parts.TimeObservationEditPart;
import org.eclipse.papyrus.uml.diagram.sequence.part.UMLDiagramEditorPlugin;
import org.eclipse.papyrus.uml.diagram.sequence.part.UMLVisualIDRegistry;
import org.eclipse.papyrus.uml.diagram.sequence.providers.UMLElementTypes;
import org.eclipse.papyrus.uml.diagram.sequence.util.CombinedFragmentMoveHelper;
import org.eclipse.papyrus.uml.diagram.sequence.util.GateHelper;
import org.eclipse.papyrus.uml.diagram.sequence.util.SequenceLinkMappingHelper;
import org.eclipse.papyrus.uml.diagram.sequence.util.SequenceRequestConstant;
import org.eclipse.papyrus.uml.diagram.sequence.util.SequenceUtil;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.uml2.uml.CombinedFragment;
import org.eclipse.uml2.uml.ConnectableElement;
import org.eclipse.uml2.uml.DestructionOccurrenceSpecification;
import org.eclipse.uml2.uml.DurationConstraint;
import org.eclipse.uml2.uml.DurationObservation;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.ExecutionOccurrenceSpecification;
import org.eclipse.uml2.uml.ExecutionSpecification;
import org.eclipse.uml2.uml.Gate;
import org.eclipse.uml2.uml.GeneralOrdering;
import org.eclipse.uml2.uml.InteractionFragment;
import org.eclipse.uml2.uml.InteractionOperand;
import org.eclipse.uml2.uml.IntervalConstraint;
import org.eclipse.uml2.uml.Lifeline;
import org.eclipse.uml2.uml.Message;
import org.eclipse.uml2.uml.MessageEnd;
import org.eclipse.uml2.uml.MessageOccurrenceSpecification;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.OccurrenceSpecification;
import org.eclipse.uml2.uml.PackageableElement;
import org.eclipse.uml2.uml.StateInvariant;
import org.eclipse.uml2.uml.TimeObservation;

/**
 * A policy to support dNd from the Model Explorer in the sequence diagram
 *
 */
public class CustomDiagramDragDropEditPolicy extends CommonDiagramDragDropEditPolicy {

	public static final String LIFELINE_MISSING = "There is no representation of lifeline {0}";

	public static final String DIALOG_TITLE = "Element missing";

	public CustomDiagramDragDropEditPolicy() {
		super(SequenceLinkMappingHelper.getInstance());
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Set<String> getDroppableElementVisualId() {
		Set<String> elementsVisualId = new HashSet<String>();
		elementsVisualId.add(LifelineEditPart.VISUAL_ID);
		elementsVisualId.add(CCombinedCompartmentEditPart.VISUAL_ID);
		elementsVisualId.add(ActionExecutionSpecificationEditPart.VISUAL_ID);
		elementsVisualId.add(BehaviorExecutionSpecificationEditPart.VISUAL_ID);
		elementsVisualId.add(InteractionUseEditPart.VISUAL_ID);
		elementsVisualId.add(InteractionEditPart.VISUAL_ID);
		elementsVisualId.add(InteractionOperandEditPart.VISUAL_ID);
		elementsVisualId.add(CombinedFragmentEditPart.VISUAL_ID);
		elementsVisualId.add(CommentAnnotatedElementEditPart.VISUAL_ID);
		elementsVisualId.add(ConsiderIgnoreFragmentEditPart.VISUAL_ID);
		elementsVisualId.add(ContinuationEditPart.VISUAL_ID);
		elementsVisualId.add(StateInvariantEditPart.VISUAL_ID);
		elementsVisualId.add(CommentEditPart.VISUAL_ID);
		elementsVisualId.add(CommentBodyEditPart.VISUAL_ID);
		elementsVisualId.add(ConstraintEditPart.VISUAL_ID);
		elementsVisualId.add(Constraint2EditPart.VISUAL_ID);
		elementsVisualId.add(ConstraintConstrainedElementEditPart.VISUAL_ID);
		elementsVisualId.add(TimeConstraintEditPart.VISUAL_ID);
		elementsVisualId.add(TimeObservationEditPart.VISUAL_ID);
		elementsVisualId.add(DurationConstraintEditPart.VISUAL_ID);
		elementsVisualId.add(SequenceDiagramEditPart.VISUAL_ID);
		elementsVisualId.add(MessageSyncEditPart.VISUAL_ID);
		elementsVisualId.add(MessageAsyncEditPart.VISUAL_ID);
		elementsVisualId.add(MessageReplyEditPart.VISUAL_ID);
		elementsVisualId.add(MessageCreateEditPart.VISUAL_ID);
		elementsVisualId.add(MessageCreateEditPart.VISUAL_ID);
		elementsVisualId.add(MessageDeleteEditPart.VISUAL_ID);
		elementsVisualId.add(MessageLostEditPart.VISUAL_ID);
		elementsVisualId.add(MessageFoundEditPart.VISUAL_ID);
		elementsVisualId.add(MessageLostEditPart.VISUAL_ID);
		elementsVisualId.add(GeneralOrderingEditPart.VISUAL_ID);
		elementsVisualId.add(DestructionOccurrenceSpecificationEditPart.VISUAL_ID);
		elementsVisualId.add(StateInvariantEditPart.VISUAL_ID);
		elementsVisualId.add(TimeConstraintEditPart.VISUAL_ID);
		elementsVisualId.add(DurationConstraintEditPart.VISUAL_ID);
		elementsVisualId.add(DurationConstraintInMessageEditPart.VISUAL_ID);
		elementsVisualId.add(TimeObservationEditPart.VISUAL_ID);
		elementsVisualId.add(DurationObservationEditPart.VISUAL_ID);
		elementsVisualId.add(LifelineEditPart.VISUAL_ID);
		//elementsVisualId.add(GateEditPart.VISUAL_ID);
		// handle nodes on messages (no visual ID detected for them)
		//elementsVisualId.add(null);
		return elementsVisualId;
	}

	@Override
	public Command getCommand(Request request) {
		Command command = super.getCommand(request);
		if (false == request instanceof ChangeBoundsRequest) {
			return command;
		}

		boolean someCombinedFragment = false;
		boolean someNonCombinedFragment = false;

		List<?> editParts = ((ChangeBoundsRequest) request).getEditParts();

		if (editParts != null) {
			for (Object part : editParts) {
				someCombinedFragment |= (part instanceof OLDCustomCombinedFragmentEditPart);
				someNonCombinedFragment |= !(part instanceof OLDCustomCombinedFragmentEditPart);
			}
		}

		if (someCombinedFragment && someNonCombinedFragment) {
			// Can't Drop CombinedFragment and other nodes at the same time
			return UnexecutableCommand.INSTANCE;
		} else if (someNonCombinedFragment) {
			return command;
		} else {
			return getMoveCombinedFragmentCommand((ChangeBoundsRequest) request);
		}
	}

	/*
	 * "In-place" drag-and-drop command for Combined Fragment
	 */
	protected Command getMoveCombinedFragmentCommand(ChangeBoundsRequest request) {
		CompoundCommand cc = new CompoundCommand("move CombinedFragments to new parent"); //$NON-NLS-1$

		Rectangle rectangleDroppedCombined = CombinedFragmentMoveHelper.calcCombinedRect(request);
		GraphicalEditPart newParentEP = CombinedFragmentMoveHelper.findNewParentEP(request, getHost());

		List<?> editParts = request.getEditParts();

		// Move the request's CFs models and views
		if (editParts != null) {
			for (Object part : editParts) {
				OLDCustomCombinedFragmentEditPart combinedFragmentEP = (OLDCustomCombinedFragmentEditPart) part;
				CombinedFragment combinedFragment = (CombinedFragment) ViewUtil.resolveSemanticElement((View) ((IGraphicalEditPart) combinedFragmentEP).getModel());

				if (combinedFragmentEP.getParent() == newParentEP) {
					continue; // no change of the parent
				}

				View containerNewParent = (View) newParentEP.getModel();
				EObject contextNewParent = ViewUtil.resolveSemanticElement(containerNewParent);
				TransactionalEditingDomain editingDomain = ((IGraphicalEditPart) getHost()).getEditingDomain();

				// Move semantic
				Command moveSemanticCmd = getHost().getCommand(new EditCommandRequestWrapper(
						new MoveRequest(editingDomain, contextNewParent, combinedFragment)));
				if (moveSemanticCmd == null) {
					return UnexecutableCommand.INSTANCE;
				}
				cc.add(moveSemanticCmd);

				// Move view
				View container = (View) newParentEP.getModel();
				View view = (View) combinedFragmentEP.getModel();
				cc.add(new ICommandProxy(new AddCommand(combinedFragmentEP.getEditingDomain(), new EObjectAdapter(container),
						new EObjectAdapter(view))));
			}
		}

		// Calc new parent rect
		Rectangle newParentOldRect = newParentEP.getFigure().getBounds().getCopy();
		newParentEP.getFigure().translateToAbsolute(newParentOldRect);
		Rectangle newParentNewRect = new Rectangle(newParentOldRect.getUnion(rectangleDroppedCombined));

		if (getHost().getParent() instanceof OLDCustomCombinedFragmentEditPart) {
			CombinedFragmentMoveHelper.adjustNewParentOperands(cc, newParentNewRect, newParentOldRect, getHost());
		}
		// TODO: resize parent's parent (and so on)

		// Move & resize parent CF
		Point newParentOffsetSW = new Point(newParentNewRect.x - newParentOldRect.x, newParentNewRect.y - newParentOldRect.y);
		if (newParentEP.getParent().getParent() != null) {
			final ChangeBoundsRequest moveParentRequest = new ChangeBoundsRequest();
			moveParentRequest.setType(REQ_MOVE);
			moveParentRequest.setMoveDelta(newParentOffsetSW);
			moveParentRequest.setEditParts(newParentEP.getParent().getParent());
			moveParentRequest.setSizeDelta(new Dimension(newParentNewRect.width - newParentOldRect.width,
					newParentNewRect.height - newParentOldRect.height));
			moveParentRequest.setResizeDirection(PositionConstants.SOUTH_WEST);
			cc.add(newParentEP.getParent().getParent().getCommand(moveParentRequest));
		}

		if (editParts != null) {
			for (Object part : request.getEditParts()) {
				OLDCustomCombinedFragmentEditPart combinedFragmentEP = (OLDCustomCombinedFragmentEditPart) part;
				CombinedFragmentMoveHelper.moveCombinedFragmentEP(cc, request, combinedFragmentEP, newParentEP, newParentOffsetSW);
			}
		}

		return cc;
	}

	@Override
	protected IUndoableOperation getDropObjectCommand(DropObjectsRequest dropRequest, final EObject droppedObject) {
		IUndoableOperation dropObjectCommand = super.getDropObjectCommand(dropRequest, droppedObject);
		if (dropObjectCommand != null && dropObjectCommand.canExecute()) {
			return dropObjectCommand;
		}
		// fix bug 364696(https://bugs.eclipse.org/bugs/show_bug.cgi?id=364696)
		if (droppedObject instanceof ConnectableElement) {
			return doDropConnectableElement(dropRequest, (ConnectableElement) droppedObject);
		}
		return dropObjectCommand;
	}

	private IUndoableOperation doDropConnectableElement(DropObjectsRequest dropRequest, final ConnectableElement droppedObject) {
		Point location = dropRequest.getLocation();
		CreateViewRequest createShapeRequest = CreateViewRequestFactory.getCreateShapeRequest(UMLElementTypes.Lifeline_Shape, UMLDiagramEditorPlugin.DIAGRAM_PREFERENCES_HINT);
		createShapeRequest.setLocation(location);
		ViewDescriptor viewDescriptor = createShapeRequest.getViewDescriptors().get(0);
		CreateElementRequestAdapter elementAdapter = (CreateElementRequestAdapter) viewDescriptor.getElementAdapter();
		CreateElementRequest createElementRequest = (CreateElementRequest) elementAdapter.getAdapter(CreateElementRequest.class);
		// parameter "ConnectableElement" used in LifelineCreateCommand#doConfigure()
		createElementRequest.setParameter(SequenceRequestConstant.CONNECTABLE_ELEMENT, droppedObject);
		EditPart host = getHost();
		Command theRealCmd = ((IGraphicalEditPart) host).getCommand(createShapeRequest);
		if (theRealCmd != null && theRealCmd.canExecute()) {
			return new CommandProxy(theRealCmd);
		}
		return org.eclipse.gmf.runtime.common.core.command.UnexecutableCommand.INSTANCE;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public IElementType getUMLElementType(String elementID) {
		return UMLElementTypes.getElementType(elementID);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getNodeVisualID(View containerView, EObject domainElement) {
		return UMLVisualIDRegistry.getNodeVisualID(containerView, domainElement);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getLinkWithClassVisualID(EObject domainElement) {
		return UMLVisualIDRegistry.getLinkWithClassVisualID(domainElement);
	}

	@Override
	protected Command getSpecificDropCommand(DropObjectsRequest dropRequest, Element semanticElement, String nodeVISUALID, String linkVISUALID) {
		Point location = dropRequest.getLocation().getCopy();
		// handle gate creation
		if (semanticElement instanceof Gate) {
			return dropGate((Gate) semanticElement, location);
		}
		if (semanticElement instanceof DurationObservation) {
			return new ICommandProxy(getDefaultDropNodeCommand(nodeVISUALID, location, semanticElement, dropRequest));
		}
		// handle specifically the case when node is on a message
		Command cmd = handleNodeOnMessage(semanticElement, nodeVISUALID, linkVISUALID, location);
		if (cmd != null) {
			return cmd;
		}
		if (nodeVISUALID != null) {
			switch (nodeVISUALID) {
			case ActionExecutionSpecificationEditPart.VISUAL_ID:
			case BehaviorExecutionSpecificationEditPart.VISUAL_ID:
			case CCombinedCompartmentEditPart.VISUAL_ID:
				return dropExecutionSpecification((ExecutionSpecification) semanticElement, nodeVISUALID, location);
			case DestructionOccurrenceSpecificationEditPart.VISUAL_ID:
				return dropDestructionOccurrence((DestructionOccurrenceSpecification) semanticElement, nodeVISUALID, location);
			case StateInvariantEditPart.VISUAL_ID:
				return dropStateInvariant((StateInvariant) semanticElement, nodeVISUALID, location);
			case TimeConstraintEditPart.VISUAL_ID:
			case DurationConstraintEditPart.VISUAL_ID:
				return dropIntervalConstraintInLifeline((IntervalConstraint) semanticElement, nodeVISUALID);
			case TimeObservationEditPart.VISUAL_ID:
				return dropTimeObservationInLifeline((TimeObservation) semanticElement, nodeVISUALID, location);
			case CommentEditPart.VISUAL_ID:
			case ConstraintEditPart.VISUAL_ID:
				if (semanticElement instanceof DurationConstraint) {
					return dropDurationConstraint((DurationConstraint) semanticElement, location);
				}
			case Constraint2EditPart.VISUAL_ID:
			case InteractionUseEditPart.VISUAL_ID:
			case LifelineEditPart.VISUAL_ID:
				return dropNodeElement(semanticElement, nodeVISUALID, location);
			case ConsiderIgnoreFragmentEditPart.VISUAL_ID:
			case CombinedFragmentEditPart.VISUAL_ID:
				return dropCombinedFragment((CombinedFragment) semanticElement, nodeVISUALID, location);
			case ContinuationEditPart.VISUAL_ID:
			case InteractionOperandEditPart.VISUAL_ID:
				return dropCompartmentNodeElement(semanticElement, nodeVISUALID, location);
			}
		}
		if (linkVISUALID != null) {
			switch (linkVISUALID) {
			case MessageSyncEditPart.VISUAL_ID:
			case MessageAsyncEditPart.VISUAL_ID:
			case MessageReplyEditPart.VISUAL_ID:
			case MessageCreateEditPart.VISUAL_ID:
			case MessageDeleteEditPart.VISUAL_ID:
			case MessageLostEditPart.VISUAL_ID:
			case MessageFoundEditPart.VISUAL_ID:
				return dropMessage(dropRequest, semanticElement, linkVISUALID);
			case GeneralOrderingEditPart.VISUAL_ID:
				return dropGeneralOrdering(dropRequest, semanticElement, linkVISUALID);
			}
		}
		return UnexecutableCommand.INSTANCE;
	}

	/**
	 * Get the drop command for the Element
	 *
	 * @param element
	 *            the Element
	 * @param nodeVISUALID
	 *            the node visual id
	 * @return the drop command if the Element can be dropped
	 */
	private Command dropNodeElement(Element element, String nodeVISUALID, Point location) {
		if(LifelineEditPart.VISUAL_ID==nodeVISUALID ) {
			location.setY(50);
		}
		Element parent = element.getOwner();
		if (getHostObject().equals(parent)) {
			List<View> existingViews = DiagramEditPartsUtil.findViews(parent, getViewer());
			if (!existingViews.isEmpty()) {
				return new ICommandProxy(getDefaultDropNodeCommand(getHost(), nodeVISUALID, location, element));
			}
		}
		return UnexecutableCommand.INSTANCE;
	}

	/**
	 * Get the drop command for the Element
	 *
	 * @param element
	 *            the Element
	 * @param nodeVISUALID
	 *            the node visual id
	 * @return the drop command if the element can be dropped
	 */
	private Command dropCombinedFragment(CombinedFragment combinedFragment, String nodeVISUALID, Point location) {
		Element parent = combinedFragment.getOwner();
		Element parentContainer = parent.getOwner();
		if (!(parentContainer instanceof CombinedFragment)) {
			parentContainer = parent;
		}
		if (getHostObject().equals(parentContainer)) {
			List<View> existingViews = DiagramEditPartsUtil.findViews(parent, getViewer());
			if (!existingViews.isEmpty()) {
				EditPart parentEditPart = getHost();
				if (parentEditPart instanceof GraphicalEditPart) {
					// check if all lifelines coversby exist in diagram.
					Rectangle bounds = null;
					for (Lifeline lifeline : combinedFragment.getCovereds()) {
						EditPart lifelineEditPart = lookForEditPart(lifeline);
						if (lifelineEditPart == null) {
							Shell shell = Display.getCurrent().getActiveShell();
							MessageDialog.openError(shell, DIALOG_TITLE, NLS.bind(LIFELINE_MISSING, lifeline.getName()));
							return UnexecutableCommand.INSTANCE;
						}
						if (lifelineEditPart instanceof GraphicalEditPart) {
							GraphicalEditPart graphicalEditPart = (GraphicalEditPart) lifelineEditPart;
							Rectangle rectangle = graphicalEditPart.getFigure().getBounds().getCopy();
							graphicalEditPart.getFigure().translateToAbsolute(rectangle);
							if (bounds == null) {
								bounds = rectangle;
							} else {
								bounds = bounds.union(rectangle);
							}
						}
					}
					if (bounds == null) {
						return new ICommandProxy(getDefaultDropNodeCommand(parentEditPart, nodeVISUALID, location, combinedFragment));
					}
					location.x = bounds.x;
					return new ICommandProxy(dropCombinedFragment(getHost(), nodeVISUALID, location, new Dimension(bounds.width, 100), combinedFragment));
				}
			}
		}
		return UnexecutableCommand.INSTANCE;
	}

	/*
	 * To extend the method in superclass with an option Dimension size,
	 *
	 *
	 * @param hostEP
	 *
	 * @param nodeVISUALID
	 *
	 * @param absoluteLocation
	 *
	 * @param size
	 *
	 * @param droppedObject
	 *
	 * @return
	 */
	protected ICommand dropCombinedFragment(EditPart hostEP, String nodeVISUALID, Point absoluteLocation, Dimension size, EObject droppedObject) {
		IHintedType type = ((IHintedType) getUMLElementType(nodeVISUALID));
		String semanticHint = null;
		if (type != null) {
			semanticHint = type.getSemanticHint();
		}
		List<View> existingViews = DiagramEditPartsUtil.findViews(droppedObject, getViewer());
		// only allow one view instance of a single element by diagram
		if (existingViews.isEmpty()) {
			IAdaptable elementAdapter = new EObjectAdapter(droppedObject);
			ViewDescriptor descriptor = new ViewDescriptor(elementAdapter, Node.class, semanticHint, ViewUtil.APPEND, true, getDiagramPreferencesHint());
			CreateViewRequest createViewRequest = new CreateViewRequest(descriptor);
			createViewRequest.setLocation(absoluteLocation);
			createViewRequest.setSize(size);
			// "ask" the host for a command associated with the
			// CreateViewRequest
			Command command = hostEP.getCommand(createViewRequest);
			if (createViewRequest.getNewObject() instanceof List) {
				for (Object object : (List<?>) createViewRequest.getNewObject()) {
					if (object instanceof IAdaptable) {
						DeferredCreateCommand createCommand2 = new DeferredCreateCommand(getEditingDomain(), droppedObject, (IAdaptable) object, getHost().getViewer());
						command.chain(new ICommandProxy(createCommand2));
					}
				}
			}
			// set the viewdescriptor as result
			// it then can be used as an adaptable to retrieve the View
			return new CommandProxyWithResult(command, descriptor);
		}
		return org.eclipse.gmf.runtime.common.core.command.UnexecutableCommand.INSTANCE;
	}

	/**
	 * Get the drop command in case the element can be handled as an element in a CombinedFragment
	 *
	 * @param element
	 * @param nodeVISUALID
	 * @param location
	 * @return
	 */
	private Command dropCompartmentNodeElement(Element element, String nodeVISUALID, Point location) {
		IHintedType type = ((IHintedType) getUMLElementType(nodeVISUALID));
		String semanticHint = null;
		if (type != null) {
			semanticHint = type.getSemanticHint();
		}
		Element parent = element.getOwner();
		Element directParent = parent;
		if (parent instanceof InteractionOperand) {
			parent = parent.getOwner();
		}
		if (getHostObject().equals(parent)) {
			List<View> existingViews = DiagramEditPartsUtil.findViews(directParent, getViewer());
			if (!existingViews.isEmpty()) {
				EditPart parentEditPart = lookForEditPart(directParent);
				if (parentEditPart != null) {
					location.setY(0);
					location.setX(0);
					IAdaptable elementAdapter = new EObjectAdapter(element);
					ViewDescriptor descriptor = new ViewDescriptor(elementAdapter, Node.class, semanticHint, ViewUtil.APPEND, true, getDiagramPreferencesHint());
					CreateViewRequest createViewRequest = new CreateViewRequest(descriptor);
					// find best bounds
					createViewRequest.setLocation(location);
					// "ask" the host for a command associated with the CreateViewRequest
					Command command = getHost().getCommand(createViewRequest);
					// set the viewdescriptor as result
					// it then can be used as an adaptable to retrieve the View
					return new ICommandProxy(new CommandProxyWithResult(command, descriptor));
					//	return new ICommandProxy(getDefaultDropNodeCommand(parentEditPart, nodeVISUALID, location, element));
				}
			}
		}
		return UnexecutableCommand.INSTANCE;
	}

	/**
	 * Get the drop command in case the element can be handled as a node on a message
	 *
	 * @param semanticElement
	 *            the element being dropped from the model
	 * @param nodeVISUALID
	 *            node visual id or -1
	 * @param linkVISUALID
	 *            link visual id or -1
	 * @param location
	 * @return the drop command if the element can be dropped as a message label node, or null otherwise
	 */
	private Command handleNodeOnMessage(Element semanticElement, String nodeVISUALID, String linkVISUALID, Point location) {
		if (nodeVISUALID == null && linkVISUALID == null) {
			// detect duration observation on a message
			if (semanticElement instanceof DurationObservation) {
				List<NamedElement> events = ((DurationObservation) semanticElement).getEvents();
				if (events.size() >= 2) {
					return dropMessageNodeBetweenEvents(semanticElement, events.get(0), events.get(1));
				}
			}
		}
		if (isDurationConstraintHint(nodeVISUALID, linkVISUALID)) {
			// detect duration constraint on a message
			if (semanticElement instanceof DurationConstraint) {
				return dropDurationConstraint((DurationConstraint) semanticElement, location);
			}
		}
		return null;
	}

	/**
	 * Drop DurationConstraint at given location. The constrained elements would also be restored if existed.
	 *
	 * @see dropDurationConstraint(Element durationConstraint, Element event1, Element event2, Point location)
	 * @param durationConstraint
	 * @param location
	 */
	protected Command dropDurationConstraint(DurationConstraint durationConstraint, Point location) {
		List<Element> events = durationConstraint.getConstrainedElements();
		Element event1 = null, event2 = null;
		if (events.size() >= 2) {
			event1 = events.get(0);
			event2 = events.get(1);
		} else if (events.size() == 1) {
			event1 = event2 = events.get(0);
		}
		return dropDurationConstraint(durationConstraint, event1, event2, location);
	}

	/**
	 * Drop DurationConstraint at given location. The constrained elements would also be restored if existed.
	 *
	 * @param durationConstraint
	 * @param event1
	 * @param event2
	 * @param location
	 * @return
	 */
	private Command dropDurationConstraint(Element durationConstraint, Element event1, Element event2, Point location) {
		List<View> existingViews = DiagramEditPartsUtil.findViews(durationConstraint, getViewer());
		// only allow one view instance of a single element by diagram
		if (!existingViews.isEmpty()) {
			return UnexecutableCommand.INSTANCE;
		}
		IAdaptable elementAdapter = new EObjectAdapter(durationConstraint);
		IHintedType elementType = (IHintedType) UMLElementTypes.getElementType(DurationConstraintEditPart.VISUAL_ID);
		ViewDescriptor descriptor = new ViewDescriptor(elementAdapter, Node.class, elementType.getSemanticHint(), ViewUtil.APPEND, true, getDiagramPreferencesHint());
		CreateViewRequest createViewRequest = new CreateViewRequest(descriptor);
		createViewRequest.setLocation(location);
		// "ask" the host for a command associated with the
		// CreateViewRequest
		Command command = getHost().getCommand(createViewRequest);
		if (command == null || !command.canExecute()) {
			return UnexecutableCommand.INSTANCE;
		}
		// restore links.
		CompositeCommand result = new CompositeCommand(command.getLabel());
		result.add(new CommandProxyWithResult(command, descriptor));
		Point[] constraintLocations = getLocationForDurationConstraint((DurationConstraint) durationConstraint, event1, event2, location);
		if (constraintLocations[0] != null && constraintLocations[1] != null) {
			if (constraintLocations[0].y <= constraintLocations[1].y) {
				result.add(new RestoreDurationConstraintLinkCommand(getEditingDomain(), descriptor, getViewer(), true, constraintLocations[0], getDiagramPreferencesHint()));
				result.add(new RestoreDurationConstraintLinkCommand(getEditingDomain(), descriptor, getViewer(), false, constraintLocations[1], getDiagramPreferencesHint()));
			} else if (constraintLocations[0].y > constraintLocations[1].y) {
				result.add(new RestoreDurationConstraintLinkCommand(getEditingDomain(), descriptor, getViewer(), false, constraintLocations[0], getDiagramPreferencesHint()));
				result.add(new RestoreDurationConstraintLinkCommand(getEditingDomain(), descriptor, getViewer(), true, constraintLocations[1], getDiagramPreferencesHint()));
			}
		} else if (constraintLocations[0] != null) {
			result.add(new RestoreDurationConstraintLinkCommand(getEditingDomain(), descriptor, getViewer(), null, constraintLocations[0], getDiagramPreferencesHint()));
		} else if (constraintLocations[1] != null) {
			result.add(new RestoreDurationConstraintLinkCommand(getEditingDomain(), descriptor, getViewer(), null, constraintLocations[1], getDiagramPreferencesHint()));
		}
		// it then can be used as an adaptable to retrieve the View
		return new ICommandProxy(result);
	}

	/**
	 * Collect the locations of constrained elements.
	 *
	 * @param durationConstraint
	 * @param event1
	 * @param event2
	 * @param location
	 * @return
	 */
	private Point[] getLocationForDurationConstraint(DurationConstraint durationConstraint, Element event1, Element event2, Point location) {
		Point[] constraintLocations = new Point[2];
		if (event1 instanceof OccurrenceSpecification) {
			constraintLocations[0] = getLocationForDurationConstraintOnEvent((OccurrenceSpecification) event1);
		}
		if (event2 instanceof OccurrenceSpecification) {
			constraintLocations[1] = getLocationForDurationConstraintOnEvent((OccurrenceSpecification) event2);
		}
		if (constraintLocations[0] == null && event1 != null) {
			EditPart editPart = lookForEditPart(event1);
			if (editPart instanceof INodeEditPart) {
				constraintLocations[0] = getLocationForDurationConstraintOnEditPart((INodeEditPart) editPart, location);
			}
		}
		if (constraintLocations[1] == null && event2 != null) {
			EditPart editPart = lookForEditPart(event2);
			if (editPart instanceof INodeEditPart) {
				constraintLocations[1] = getLocationForDurationConstraintOnEditPart((INodeEditPart) editPart, location.getTranslated(0, 40));
			}
		}
		if (event1 != null && event1 == event2 && constraintLocations[0] != null && constraintLocations[0].equals(constraintLocations[1])) {
			EditPart editPart = lookForEditPart(event1);
			if (editPart instanceof IGraphicalEditPart) {
				Rectangle rect = SequenceUtil.getAbsoluteBounds((IGraphicalEditPart) editPart);
				if (rect.contains(constraintLocations[0].getTranslated(0, 40))) {
					constraintLocations[1] = constraintLocations[0].getTranslated(0, 40);
				} else if (rect.contains(constraintLocations[0].getTranslated(0, -40))) {
					constraintLocations[1] = constraintLocations[0].getTranslated(0, -40);
				}
			}
		}
		return constraintLocations;
	}

	/**
	 * Get location from a INodeEditPart.
	 *
	 * @param nodeEditPart
	 * @param reference
	 * @return
	 */
	private Point getLocationForDurationConstraintOnEditPart(INodeEditPart nodeEditPart, Point reference) {
		Rectangle rect = SequenceUtil.getAbsoluteBounds((IGraphicalEditPart) nodeEditPart);
		Point location = new Point();
		if (reference.y < rect.y) {
			location.y = rect.y + 1;
		} else if (reference.y > rect.bottom()) {
			location.y = rect.bottom() - 1;
		} else {
			location.y = reference.y;
		}
		for (int x = rect.x; x <= rect.right(); x++) {
			CreateConnectionRequest request = new CreateConnectionRequest();
			request.setType(REQ_CONNECTION_END);
			request.setTargetEditPart(nodeEditPart);
			location.x = x;
			request.setLocation(location);
			ConnectionAnchor targetAnchor = nodeEditPart.getTargetConnectionAnchor(request);
			if (targetAnchor != null) {
				location = targetAnchor.getLocation(reference);
				break;
			}
		}
		return location;
	}

	/**
	 * Get location of OccurrenceSpecification.
	 *
	 * @param event
	 * @return
	 */
	private Point getLocationForDurationConstraintOnEvent(OccurrenceSpecification event) {
		Point targetLocation = null;
		if (event instanceof MessageOccurrenceSpecification) {
			Message message = ((MessageOccurrenceSpecification) event).getMessage();
			if (message == null) {
				return null;
			}
			ConnectionNodeEditPart MessageSyncEditPart = null;
			DiagramEditPart diag = DiagramEditPartsUtil.getDiagramEditPart(getHost());
			for (Object conn : diag.getConnections()) {
				if (conn instanceof ConnectionNodeEditPart) {
					EObject connElt = ((ConnectionNodeEditPart) conn).resolveSemanticElement();
					if (message.equals(connElt)) {
						MessageSyncEditPart = (ConnectionNodeEditPart) conn;
						break;
					}
				}
			}
			if (MessageSyncEditPart == null) {
				return null;
			}
			if (event == message.getSendEvent()) {
				targetLocation = SequenceUtil.getAbsoluteEdgeExtremity(MessageSyncEditPart, true);
			} else if (event == message.getReceiveEvent()) {
				targetLocation = SequenceUtil.getAbsoluteEdgeExtremity(MessageSyncEditPart, false);
			}
		} else if (event instanceof ExecutionOccurrenceSpecification) {
			ExecutionSpecification execution = ((ExecutionOccurrenceSpecification) event).getExecution();
			if (execution != null) {
				List<View> existingViews = DiagramEditPartsUtil.findViews(execution, getViewer());
				for (View view : existingViews) {
					if (CCombinedCompartmentEditPart.VISUAL_ID.equals(UMLVisualIDRegistry.getVisualID(view)) || BehaviorExecutionSpecificationEditPart.VISUAL_ID.equals(UMLVisualIDRegistry.getVisualID(view))) {
						Object object = getViewer().getEditPartRegistry().get(view);
						if (object instanceof IGraphicalEditPart) {
							Rectangle bounds = SequenceUtil.getAbsoluteBounds((IGraphicalEditPart) object);
							if (event == execution.getStart()) {
								targetLocation = bounds.getTop();
							} else if (event == execution.getFinish()) {
								targetLocation = bounds.getBottom();
							}
						}
					}
				}
			}
		}
		return targetLocation;
	}

	/**
	 * Get the command to drop an element between two events in order to create a message label
	 *
	 * @param droppedElement
	 *            the dropped element
	 * @param event1
	 *            first event (of type MessageOccurrenceSpecification)
	 * @param event2
	 *            second event (of type MessageOccurrenceSpecification)
	 * @param element
	 * @return the command or false if the elements can not be dropped as message label
	 */
	private Command dropMessageNodeBetweenEvents(Element droppedElement, Element event1, Element event2) {
		if (event1 instanceof MessageOccurrenceSpecification && event2 instanceof MessageOccurrenceSpecification) {
			if (!event1.equals(event2)) {
				boolean endsOfSameMessage = false;
				String visualId = null;
				if (droppedElement instanceof DurationConstraint) {
					visualId = DurationConstraintInMessageEditPart.VISUAL_ID;
					endsOfSameMessage = DurationConstraintHelper.endsOfSameMessage((OccurrenceSpecification) event1, (OccurrenceSpecification) event2);
				} else if (droppedElement instanceof DurationObservation) {
					visualId = DurationObservationEditPart.VISUAL_ID;
					endsOfSameMessage = DurationObservationHelper.endsOfSameMessage((OccurrenceSpecification) event1, (OccurrenceSpecification) event2);
				}
				if (endsOfSameMessage) {
					Message message = ((MessageOccurrenceSpecification) event1).getMessage();
					// search a connection which matches the possessing message
					DiagramEditPart diag = DiagramEditPartsUtil.getDiagramEditPart(getHost());
					for (Object conn : diag.getConnections()) {
						if (conn instanceof ConnectionNodeEditPart) {
							EObject connElt = ((ConnectionNodeEditPart) conn).resolveSemanticElement();
							if (message.equals(connElt)) {
								// check that node isn't already represented, or dropping is impossible
								for (Object child : ((ConnectionNodeEditPart) conn).getChildren()) {
									if (child instanceof GraphicalEditPart) {
										EObject childElt = ((GraphicalEditPart) child).resolveSemanticElement();
										if (droppedElement.equals(childElt)) {
											return null;
										}
									}
								}
								return dropNodeOnMessage((PackageableElement) droppedElement, (ConnectionNodeEditPart) conn, visualId);
							}
						}
					}
				}
			}
		}
		return null;
	}

	/**
	 * Test whether visual ids are compatible with a duration constraint element
	 *
	 * @param nodeVISUALID
	 *            the detected node visual id
	 * @param linkVISUALID
	 *            the detected link visual id
	 * @return true if element may be a duration constraint
	 */
	private boolean isDurationConstraintHint(String nodeVISUALID, String linkVISUALID) {
		if (linkVISUALID != null) {
			return false;
		} else {
			return nodeVISUALID == null || ConstraintEditPart.VISUAL_ID.equals(nodeVISUALID) || DurationConstraintEditPart.VISUAL_ID.equals(nodeVISUALID) || DurationConstraintInMessageEditPart.VISUAL_ID.equals(nodeVISUALID);
		}
	}

	/**
	 * Drop a duration observation or a duration constraint on a message edit part
	 *
	 * @param durationLabelElement
	 *            the duration observation or duration constraint to display as message label
	 * @param MessageSyncEditPart
	 *            the containing message edit part
	 * @param nodeVISUALID
	 *            the label node visual id
	 * @return the command or UnexecutableCommand
	 */
	private Command dropNodeOnMessage(PackageableElement durationLabelElement, ConnectionNodeEditPart MessageSyncEditPart, String nodeVISUALID) {
		IAdaptable elementAdapter = new EObjectAdapter(durationLabelElement);
		ViewDescriptor descriptor = new ViewDescriptor(elementAdapter, Node.class, ((IHintedType) getUMLElementType(nodeVISUALID)).getSemanticHint(), ViewUtil.APPEND, false, getDiagramPreferencesHint());
		return MessageSyncEditPart.getCommand(new CreateViewRequest(descriptor));
	}

	/**
	 * Drop a time observation on a lifeline.
	 *
	 * @param observation
	 *            the time constraint
	 * @param nodeVISUALID
	 *            the node visual id
	 * @param dropLocation
	 * @return the command if the lifeline is the correct one or UnexecutableCommand
	 */
	private Command dropTimeObservationInLifeline(TimeObservation observation, String nodeVISUALID, Point dropLocation) {
		CompoundCommand cc = new CompoundCommand("Drop");
		IAdaptable elementAdapter = new EObjectAdapter(observation);
		ViewDescriptor descriptor = new ViewDescriptor(elementAdapter, Node.class, ((IHintedType) getUMLElementType(nodeVISUALID)).getSemanticHint(), ViewUtil.APPEND, true, getDiagramPreferencesHint());
		cc.add(getHost().getCommand(new CreateViewRequest(descriptor)));
		LifelineEditPart lifelinePart = SequenceUtil.getParentLifelinePart(getHost());
		if (lifelinePart != null) {
			NamedElement occ1 = observation.getEvent();
			if (occ1 instanceof OccurrenceSpecification) {
				Point middlePoint = SequenceUtil.findLocationOfEvent(lifelinePart, (OccurrenceSpecification) occ1);
				if (middlePoint != null) {
					int height = getDefaultDropHeight(nodeVISUALID);
					Point startPoint = middlePoint.getCopy();
					if (height > 0) {
						startPoint.translate(0, -height / 2);
					}
					Rectangle newBounds = new Rectangle(startPoint, new Dimension(-1, height));
					lifelinePart.getFigure().translateToRelative(newBounds);
					Point parentLoc = lifelinePart.getLocation();
					newBounds.translate(parentLoc.getNegated());
					SetBoundsCommand setBoundsCommand = new SetBoundsCommand(getEditingDomain(), "move", descriptor, newBounds);
					cc.add(new ICommandProxy(setBoundsCommand));
					return cc;
				}
			}
		}
		if (getHost() instanceof InteractionInteractionCompartmentEditPart) {
			Rectangle newBounds = new Rectangle(dropLocation, new Dimension(-1, -1));
			((InteractionInteractionCompartmentEditPart) getHost()).getFigure().translateToRelative(newBounds);
			SetBoundsCommand setBoundsCommand = new SetBoundsCommand(getEditingDomain(), "move", descriptor, newBounds);
			cc.add(new ICommandProxy(setBoundsCommand));
			return cc;
		}
		return UnexecutableCommand.INSTANCE;
	}

	/**
	 * Drop an interval constraint (duration or time) on a lifeline.
	 *
	 * @param constraint
	 *            the interval constraint
	 * @param nodeVISUALID
	 *            the node visual id
	 * @return the command if the lifeline is the correct one or UnexecutableCommand
	 */
	private Command dropIntervalConstraintInLifeline(IntervalConstraint constraint, String nodeVISUALID) {
		CompoundCommand cc = new CompoundCommand("Drop");
		IAdaptable elementAdapter = new EObjectAdapter(constraint);
		ViewDescriptor descriptor = new ViewDescriptor(elementAdapter, Node.class, ((IHintedType) getUMLElementType(nodeVISUALID)).getSemanticHint(), ViewUtil.APPEND, true, getDiagramPreferencesHint());
		cc.add(getHost().getCommand(new CreateViewRequest(descriptor)));
		LifelineEditPart lifelinePart = SequenceUtil.getParentLifelinePart(getHost());
		if (lifelinePart != null && constraint.getConstrainedElements().size() >= 2) {
			Element occ1 = constraint.getConstrainedElements().get(0);
			Element occ2 = constraint.getConstrainedElements().get(1);
			if (occ1 instanceof OccurrenceSpecification && occ2 instanceof OccurrenceSpecification) {
				Point startPoint = SequenceUtil.findLocationOfEvent(lifelinePart, (OccurrenceSpecification) occ1);
				Point endPoint = SequenceUtil.findLocationOfEvent(lifelinePart, (OccurrenceSpecification) occ2);
				if (startPoint != null && endPoint != null) {
					int height = endPoint.y - startPoint.y;
					Rectangle newBounds = null;
					if (height < 0) {
						newBounds = new Rectangle(endPoint, new Dimension(-1, -height));
					} else {
						newBounds = new Rectangle(startPoint, new Dimension(-1, height));
					}
					lifelinePart.getFigure().translateToRelative(newBounds);
					Point parentLoc = lifelinePart.getLocation();
					newBounds.translate(parentLoc.getNegated());
					SetBoundsCommand setBoundsCommand = new SetBoundsCommand(getEditingDomain(), "move", descriptor, newBounds);
					cc.add(new ICommandProxy(setBoundsCommand));
					return cc;
				}
			}
		} else if (lifelinePart != null && constraint.getConstrainedElements().size() == 1) {
			Element occ1 = constraint.getConstrainedElements().get(0);
			if (occ1 instanceof OccurrenceSpecification) {
				Point middlePoint = SequenceUtil.findLocationOfEvent(lifelinePart, (OccurrenceSpecification) occ1);
				if (middlePoint != null) {
					int height = getDefaultDropHeight(nodeVISUALID);
					Point startPoint = middlePoint.getCopy();
					if (height > 0) {
						startPoint.translate(0, -height / 2);
					}
					Rectangle newBounds = new Rectangle(startPoint, new Dimension(-1, height));
					lifelinePart.getFigure().translateToRelative(newBounds);
					Point parentLoc = lifelinePart.getLocation();
					newBounds.translate(parentLoc.getNegated());
					SetBoundsCommand setBoundsCommand = new SetBoundsCommand(getEditingDomain(), "move", descriptor, newBounds);
					cc.add(new ICommandProxy(setBoundsCommand));
					return cc;
				}
			}
		}
		return UnexecutableCommand.INSTANCE;
	}

	/**
	 * Get the default height to set to a drop object. This method is useful for dropped objects which must be positioned relatively to their center.
	 *
	 * @param nodeVISUALID
	 *            the node visual id
	 * @return arbitrary default height for the node visual id (eventually -1)
	 */
	private int getDefaultDropHeight(String nodeVISUALID) {
		if (TimeConstraintEditPart.VISUAL_ID.equals(nodeVISUALID) || TimeObservationEditPart.VISUAL_ID.equals(nodeVISUALID)) {
			return 2;
		}
		return -1;
	}

	private Command dropStateInvariant(StateInvariant stateInvariant, String nodeVISUALID, Point location) {
		// an StateInvariant covereds systematically a unique lifeline
		Lifeline lifeline = stateInvariant.getCovereds().get(0);
		// Check that the container view is the view of the lifeline
		if (lifeline.equals(getHostObject())) {
			return new ICommandProxy(getDefaultDropNodeCommand(nodeVISUALID, location, stateInvariant));
		}
		return UnexecutableCommand.INSTANCE;
	}

	/**
	 * Get lifelines element which contains these existingViews
	 *
	 * @param existingViews
	 *            the existing views.
	 * @return the list of lifeline.
	 */
	private List<Lifeline> getLifelines(List<View> existingViews) {
		List<Lifeline> lifelines = new ArrayList<Lifeline>();
		for (View view : existingViews) {
			EObject eObject = ViewUtil.resolveSemanticElement((View) view.eContainer());
			if (eObject instanceof Lifeline) {
				lifelines.add((Lifeline) eObject);
			}
		}
		return lifelines;
	}

	/**
	 * Create command for drop a Gate.
	 *
	 * @param gate
	 * @param location
	 * @return
	 */
	private Command dropGate(Gate gate, Point location) {
		return new ICommandProxy(getDropGateCommand(gate, location));
	}

	/**
	 * Create command for drop a Gate.
	 *
	 * @param gate
	 * @param location
	 * @return
	 */
	private ICommand getDropGateCommand(Gate gate, Point location) {
		List<View> existingViews = DiagramEditPartsUtil.findViews(gate, getViewer());
		if (existingViews.isEmpty()) {
			EObject owner = gate.eContainer();
			if (owner != null) {
				org.eclipse.gef.GraphicalEditPart parent = (org.eclipse.gef.GraphicalEditPart) lookForEditPart(owner);
				if (parent != null) {
					Point gateLocation = GateHelper.computeGateLocation(location, parent.getFigure(), null);
					return new OLDCreateGateViewCommand(getEditingDomain(), parent, gateLocation, new EObjectAdapter(gate));
				}
			}
		}
		return org.eclipse.gmf.runtime.common.core.command.UnexecutableCommand.INSTANCE;
	}

	/**
	 * Drop a destructionEvent on a lifeline
	 *
	 * @param destructionOccurence
	 *            the destructionEvent to drop
	 * @param nodeVISUALID
	 *            the node visualID
	 * @return the command to drop the destructionEvent on a lifeline if allowed.
	 */
	private Command dropDestructionOccurrence(DestructionOccurrenceSpecification destructionOccurence, String nodeVISUALID, Point location) {
		// Get all the view of this destructionEvent.
		List<View> existingViews = DiagramEditPartsUtil.findViews(destructionOccurence, getViewer());
		// Get the lifelines containing the graphical destructionEvent
		List<Lifeline> lifelines = getLifelines(existingViews);
		// If the list of lifeline already containing the destructionEvent doesn't contain the lifeline targeted.
		//		if (!lifelines.contains(getHostObject())) {
		//			Lifeline lifeline = (Lifeline) getHostObject();
		//			for (InteractionFragment ift : lifeline.getCoveredBys()) {
		//				if (ift instanceof DestructionOccurrenceSpecification) {
		//					DestructionOccurrenceSpecification occurrenceSpecification = (DestructionOccurrenceSpecification) ift;
		//					// if the event of the occurrenceSpecification is the DestructionEvent, create the command
		//					if (destructionOccurence.equals(occurrenceSpecification)) {
		return new ICommandProxy(getDefaultDropNodeCommand(nodeVISUALID, location, destructionOccurence));
		//					}
		//				}
		//			}
		//		}
		//		return UnexecutableCommand.INSTANCE;
	}

	/**
	 * Get the command to drop an execution specification node
	 *
	 * @param es
	 *            execution specification
	 * @param nodeVISUALID
	 *            the execution specification's visual id
	 * @param location
	 *            the location of the drop request
	 * @return the drop command
	 */
	private Command dropExecutionSpecification(ExecutionSpecification es, String nodeVISUALID, Point location) {
		List<View> existingViews = DiagramEditPartsUtil.findViews(es, getViewer());
		// only allow one view instance of a single element by diagram
		if (existingViews.isEmpty()) {
			// Find the lifeline of the ES
			if (es.getStart() != null && !es.getStart().getCovereds().isEmpty()) {
				// an Occurrence Specification covers systematically a unique lifeline
				Lifeline lifeline = es.getStart().getCovereds().get(0);
				// Check that the container view is the view of the lifeline
				if (lifeline.equals(getHostObject())) {
					// return new ICommandProxy(getDefaultDropNodeCommand(nodeVISUALID, location, es));
					IHintedType type = ((IHintedType) getUMLElementType(nodeVISUALID));
					String semanticHint = null;
					if (type != null) {
						semanticHint = type.getSemanticHint();
					}
					IAdaptable elementAdapter = new EObjectAdapter(es);
					ViewDescriptor descriptor = new ViewDescriptor(elementAdapter, Node.class, semanticHint, ViewUtil.APPEND, true, getDiagramPreferencesHint());
					CreateViewRequest createViewRequest = new CreateViewRequest(descriptor);
					// find best bounds
					Rectangle bounds = getExecutionSpecificationBounds(es);
					if (bounds != null) {
						createViewRequest.setLocation(bounds.getLocation());
						createViewRequest.setSize(bounds.getSize());
					} else {
						createViewRequest.setLocation(location);
					}
					// "ask" the host for a command associated with the CreateViewRequest
					Command command = getHost().getCommand(createViewRequest);
					// set the viewdescriptor as result
					// it then can be used as an adaptable to retrieve the View
					return new ICommandProxy(new CommandProxyWithResult(command, descriptor));
				}
			}
		}
		return UnexecutableCommand.INSTANCE;
	}


	/**
	 * Get the advised bounds to drop an execution specification
	 *
	 * @param es
	 *            the dropped execution specification
	 * @return bounds of the es in absolute coordinates or null
	 */
	private Rectangle getExecutionSpecificationBounds(ExecutionSpecification es) {
		Point startLocation = null;
		Point finishLocation = null;
		Rectangle possibleStartLocations = null;
		Rectangle possibleFinishLocations = null;
		// end events of the link
		OccurrenceSpecification startEvent = es.getStart();
		OccurrenceSpecification finishEvent = es.getFinish();
		if (startEvent != null && finishEvent != null && getHost() instanceof LifelineEditPart) {
			LifelineEditPart hostLifeline = (LifelineEditPart) getHost();
			// find location constraints for source
			startLocation = SequenceUtil.findLocationOfEvent((LifelineEditPart) getHost(), startEvent);
			if (startLocation == null) {
				possibleStartLocations = SequenceUtil.findPossibleLocationsForEvent(hostLifeline, startEvent);
			}
			// find location constraints for target
			finishLocation = SequenceUtil.findLocationOfEvent(hostLifeline, finishEvent);
			if (finishLocation == null) {
				possibleFinishLocations = SequenceUtil.findPossibleLocationsForEvent(hostLifeline, finishEvent);

				// Minimum shape (not just an horizontal line)
				if (possibleFinishLocations.height <= 0) {
					possibleFinishLocations.height = OLDLifelineXYLayoutEditPolicy.EXECUTION_INIT_HEIGHT;
				}
			}
			// find start and finish locations with correct y (start.y < finish.y) and proportions
			if (startLocation == null) {
				if (finishLocation != null) {
					int top = possibleStartLocations.x;
					int bottom = possibleStartLocations.bottom();
					if (top > finishLocation.y) {
						return null;
					} else {
						startLocation = possibleStartLocations.getTop();
						startLocation.y = (top + Math.min(bottom, finishLocation.y)) / 2;
					}
				} else {
					int topS = possibleStartLocations.y;
					int bottomS = possibleStartLocations.bottom();
					int topF = possibleFinishLocations.y;
					int bottomF = possibleFinishLocations.bottom();
					if (topS > bottomF) {
						return null;
					} else {
						startLocation = possibleStartLocations.getTop();
						finishLocation = possibleFinishLocations.getBottom();
						if (bottomS < topF) {
							startLocation.y = (topS + bottomS) / 2;
							finishLocation.y = (topF + bottomF) / 2;
						} else {
							startLocation.y = (topS + bottomS + topS + topF) / 4;
							finishLocation.y = (bottomF + topF + bottomF + bottomS) / 4;
						}
					}
				}
			}
			if (finishLocation == null) {
				// startLocation != null
				int top = possibleFinishLocations.y;
				int bottom = possibleFinishLocations.bottom();
				if (bottom < startLocation.y) {
					return null;
				} else {
					finishLocation = possibleFinishLocations.getBottom();
					finishLocation.y = (bottom + Math.max(top, startLocation.y)) / 2;
				}
			}
			// deduce bounds
			Rectangle result = new Rectangle(startLocation, finishLocation);
			result.width = OLDLifelineXYLayoutEditPolicy.EXECUTION_INIT_WIDTH;
			return result;
		}
		return null;
	}

	/**
	 * Get the command to drop a message link
	 *
	 * @param dropRequest
	 *            request to drop
	 * @param semanticLink
	 *            message link
	 * @param linkVISUALID
	 *            the message's visual id
	 * @return the drop command
	 */
	private Command dropMessage(DropObjectsRequest dropRequest, Element semanticLink, String linkVISUALID) {
		// Do NOT drop again if existed.
		List<View> existingViews = DiagramEditPartsUtil.findViews(semanticLink, getViewer());
		if (!existingViews.isEmpty()) {
			return UnexecutableCommand.INSTANCE;
		}
		Collection<?> sources = SequenceLinkMappingHelper.getInstance().getSource(semanticLink);
		Collection<?> targets = SequenceLinkMappingHelper.getInstance().getTarget(semanticLink);
		if (!sources.isEmpty() && !targets.isEmpty()) {
			Element source = (Element) sources.toArray()[0];
			Element target = (Element) targets.toArray()[0];
			return getDropLocatedLinkCommand(dropRequest, source, target, linkVISUALID, semanticLink);
		} else {
			return UnexecutableCommand.INSTANCE;
		}
	}

	/**
	 * The method provides command to create the binary link into the diagram. If the source and the
	 * target views do not exist, these views will be created.
	 *
	 * This implementation is very similar to {@link CommonDiagramDragDropEditPolicy#dropBinaryLink(CompositeCommand, Element, Element, int, Point, Element)}.
	 *
	 * @param dropRequest
	 *            the drop request
	 * @param cc
	 *            the composite command that will contain the set of command to create the binary
	 *            link
	 * @param source
	 *            the element source of the link
	 * @param target
	 *            the element target of the link
	 * @param linkVISUALID
	 *            the link VISUALID used to create the view
	 * @param location
	 *            the location the location where the view will be be created
	 * @param semanticLink
	 *            the semantic link that will be attached to the view
	 *
	 * @return the composite command
	 */
	protected Command getDropLocatedLinkCommand(DropObjectsRequest dropRequest, Element source, Element target, String linkVISUALID, Element semanticLink) {
		// look for editpart
		GraphicalEditPart sourceEditPart = (GraphicalEditPart) lookForEditPart(source);
		GraphicalEditPart targetEditPart = (GraphicalEditPart) lookForEditPart(target);
		CompositeCommand cc = new CompositeCommand("Drop");
		// descriptor of the link
		CreateConnectionViewRequest.ConnectionViewDescriptor linkdescriptor = new CreateConnectionViewRequest.ConnectionViewDescriptor(getUMLElementType(linkVISUALID), ((IHintedType) getUMLElementType(linkVISUALID)).getSemanticHint(),
				getDiagramPreferencesHint());
		// get source and target adapters, creating the add commands if necessary
		IAdaptable sourceAdapter = null;
		IAdaptable targetAdapter = null;
		if (sourceEditPart == null) {
			ICommand createCommand = getDefaultDropNodeCommand(getLinkSourceDropLocation(dropRequest.getLocation(), source, target), source);
			cc.add(createCommand);
			sourceAdapter = (IAdaptable) createCommand.getCommandResult().getReturnValue();
		} else {
			sourceAdapter = new SemanticAdapter(null, sourceEditPart.getModel());
		}
		if (targetEditPart == null) {
			ICommand createCommand = getDefaultDropNodeCommand(getLinkTargetDropLocation(dropRequest.getLocation(), source, target), target);
			cc.add(createCommand);
			targetAdapter = (IAdaptable) createCommand.getCommandResult().getReturnValue();
		} else {
			targetAdapter = new SemanticAdapter(null, targetEditPart.getModel());
		}
		CreateLocatedConnectionViewCommand aLinkCommand = new CreateLocatedConnectionViewCommand(getEditingDomain(), ((IHintedType) getUMLElementType(linkVISUALID)).getSemanticHint(), sourceAdapter, targetAdapter, getViewer(), getDiagramPreferencesHint(),
				linkdescriptor, null);
		aLinkCommand.setElement(semanticLink);
		Point[] sourceAndTarget = getLinkSourceAndTargetLocations(semanticLink, sourceEditPart, targetEditPart, dropRequest.getLocation().getCopy());// add default location support.
		aLinkCommand.setLocations(sourceAndTarget[0], sourceAndTarget[1]);
		cc.compose(aLinkCommand);
		return new ICommandProxy(cc);
	}

	@Override
	protected ICommand getDefaultDropNodeCommand(Point absoluteLocation, EObject droppedObject) {
		// Custom command for dropping a gate.
		if (droppedObject instanceof Gate) {
			return getDropGateCommand((Gate) droppedObject, absoluteLocation);
		}
		return super.getDefaultDropNodeCommand(absoluteLocation, droppedObject);
	}

	/**
	 * Get the source and target recommended points for creating the link
	 *
	 * @param semanticLink
	 *            link to create
	 * @param sourceEditPart
	 *            edit part source of the link
	 * @param targetEditPart
	 *            edit part target of the link
	 * @param dropLocation
	 *            default location if NOT found.
	 * @return a point array of size 2, with eventually null values (when no point constraint). Index 0 : source location, 1 : target location
	 */
	private Point[] getLinkSourceAndTargetLocations(Element semanticLink, GraphicalEditPart sourceEditPart, GraphicalEditPart targetEditPart, Point dropLocation) {
		// index 0 : source location, 1 : target location
		Point[] sourceAndTarget = new Point[] { null, null };
		// end events of the link
		OccurrenceSpecification sourceEvent = null;
		OccurrenceSpecification targetEvent = null;
		if (semanticLink instanceof Message) {
			MessageEnd sendEvent = ((Message) semanticLink).getSendEvent();
			if (sendEvent instanceof OccurrenceSpecification) {
				sourceEvent = (OccurrenceSpecification) sendEvent;
			} else if (sendEvent instanceof Gate) {// for gate
				if (sourceEditPart != null) {
					sourceAndTarget[0] = SequenceUtil.getAbsoluteBounds(sourceEditPart).getCenter();
				} else {
					sourceAndTarget[0] = dropLocation;
				}
			} else if (sendEvent == null) {// found message
				sourceAndTarget[0] = dropLocation;
			}
			MessageEnd rcvEvent = ((Message) semanticLink).getReceiveEvent();
			if (rcvEvent instanceof OccurrenceSpecification) {
				targetEvent = (OccurrenceSpecification) rcvEvent;
			} else if (rcvEvent instanceof Gate) {// for gate
				if (targetEditPart != null) {
					sourceAndTarget[1] = SequenceUtil.getAbsoluteBounds(targetEditPart).getCenter();
				} else {
					sourceAndTarget[1] = dropLocation;
				}
			} else if (rcvEvent == null) {// lost message
				sourceAndTarget[1] = dropLocation;
			}
		} else if (semanticLink instanceof GeneralOrdering) {
			sourceEvent = ((GeneralOrdering) semanticLink).getBefore();
			targetEvent = ((GeneralOrdering) semanticLink).getAfter();
		}
		if (sourceEvent != null || targetEvent != null) {
			Rectangle possibleSourceLocations = null;
			Rectangle possibleTargetLocations = null;
			// find location constraints for source
			if (sourceEvent != null && sourceEditPart instanceof LifelineEditPart) {
				sourceAndTarget[0] = SequenceUtil.findLocationOfEvent((LifelineEditPart) sourceEditPart, sourceEvent);
				if (sourceAndTarget[0] == null) {
					possibleSourceLocations = SequenceUtil.findPossibleLocationsForEvent((LifelineEditPart) sourceEditPart, sourceEvent);
				}
			}
			// find possible source location on Gate
			else if (sourceEditPart instanceof OLDGateEditPart) {
				possibleSourceLocations = SequenceUtil.getAbsoluteBounds(sourceEditPart);
			}
			// find location constraints for target
			if (targetEvent != null && targetEditPart instanceof LifelineEditPart) {
				sourceAndTarget[1] = SequenceUtil.findLocationOfEvent((LifelineEditPart) targetEditPart, targetEvent);
				if (sourceAndTarget[1] == null) {
					possibleTargetLocations = SequenceUtil.findPossibleLocationsForEvent((LifelineEditPart) targetEditPart, targetEvent);
				}
			}
			// find possible target location on Gate
			else if (targetEditPart instanceof OLDGateEditPart) {
				possibleTargetLocations = SequenceUtil.getAbsoluteBounds(sourceEditPart);
			}
			// deduce a possibility
			if (sourceAndTarget[0] == null && possibleSourceLocations != null) {
				// we must fix the source
				if (sourceAndTarget[1] == null && possibleTargetLocations == null) {
					// no target constraint, take center for source
					sourceAndTarget[0] = possibleSourceLocations.getCenter();
				} else if (sourceAndTarget[1] != null) {
					// target is fixed, find arranging source
					int topSource = possibleSourceLocations.y;
					int centerSource = possibleSourceLocations.getCenter().y;
					if (sourceAndTarget[1].y < topSource) {
						// we would draw an uphill message (forbidden).
						// return best locations (command will not execute correctly and handle error report)
						sourceAndTarget[0] = possibleSourceLocations.getTop();
					} else if (centerSource <= sourceAndTarget[1].y) {
						// simply fix to the center of constraint
						sourceAndTarget[0] = possibleSourceLocations.getCenter();
					} else {
						// horizontal message makes source as near as possible to the center
						sourceAndTarget[0] = possibleSourceLocations.getCenter();
						sourceAndTarget[0].y = sourceAndTarget[1].y;
					}
				} else {
					// possibleTargetLocations !=null
					// find arranging target and source
					int centerTarget = possibleTargetLocations.getCenter().y;
					int bottomTarget = possibleTargetLocations.bottom();
					int topSource = possibleSourceLocations.y;
					int centerSource = possibleSourceLocations.getCenter().y;
					if (bottomTarget < topSource) {
						// we would draw an uphill message (forbidden).
						// return best locations (command will not execute correctly and handle error report)
						sourceAndTarget[0] = possibleSourceLocations.getTop();
						sourceAndTarget[1] = possibleTargetLocations.getBottom();
					} else if (centerSource <= centerTarget) {
						// simply fix to centers
						sourceAndTarget[0] = possibleSourceLocations.getCenter();
						sourceAndTarget[1] = possibleTargetLocations.getCenter();
					} else {
						// horizontal message makes source and target as near as possible to the centers
						sourceAndTarget[0] = possibleSourceLocations.getCenter();
						sourceAndTarget[0].y = (topSource + bottomTarget) / 2;
						sourceAndTarget[1] = possibleTargetLocations.getCenter();
						sourceAndTarget[1].y = (topSource + bottomTarget) / 2;
					}
				}
			}
			if (sourceAndTarget[1] == null && possibleTargetLocations != null) {
				// we must fix the target
				// fixedSourceLocation == null => possibleSourceLocations == null
				// source is fixed, find arranging target
				int centerTarget = possibleTargetLocations.getCenter().y;
				int bottomTarget = possibleTargetLocations.bottom();
				if (sourceAndTarget[0] == null) {
					// simply fix to the center of constraint
					sourceAndTarget[1] = possibleTargetLocations.getCenter();
				} else if (bottomTarget < sourceAndTarget[0].y) {
					// we would draw an uphill message (forbidden).
					// return best locations (command will not execute correctly and handle error report)
					sourceAndTarget[1] = possibleTargetLocations.getBottom();
				} else if (sourceAndTarget[0].y <= centerTarget) {
					// simply fix to the center of constraint
					sourceAndTarget[1] = possibleTargetLocations.getCenter();
				} else {
					// horizontal message makes target as near as possible to the center
					sourceAndTarget[1] = possibleTargetLocations.getCenter();
					sourceAndTarget[1].y = sourceAndTarget[0].y;
				}
			}
		}
		// repair location for lost/found.
		if (semanticLink instanceof Message) {
			MessageEnd sendEvent = ((Message) semanticLink).getSendEvent();
			if (sendEvent == null && sourceAndTarget[1] != null) {// found message
				sourceAndTarget[0] = new Point(dropLocation.x, sourceAndTarget[1].y);
			}
			MessageEnd rcvEvent = ((Message) semanticLink).getReceiveEvent();
			if (rcvEvent == null && sourceAndTarget[0] != null) {// lost message
				sourceAndTarget[1] = new Point(dropLocation.x, sourceAndTarget[0].y);
			}
		}
		return sourceAndTarget;
	}

	/**
	 * Get the command to drop a general ordering link
	 *
	 * @param dropRequest
	 *            request to drop
	 * @param semanticLink
	 *            general ordering link
	 * @param linkVISUALID
	 *            the link's visual id
	 * @return the drop command
	 */
	private Command dropGeneralOrdering(DropObjectsRequest dropRequest, Element semanticLink, String linkVISUALID) {
		Collection<?> sources = SequenceLinkMappingHelper.getInstance().getSource(semanticLink);
		Collection<?> targets = SequenceLinkMappingHelper.getInstance().getTarget(semanticLink);
		if (!sources.isEmpty() && !targets.isEmpty()) {
			Element source = (Element) sources.toArray()[0];
			Element target = (Element) targets.toArray()[0];
			return getDropLocatedLinkCommand(dropRequest, source, target, linkVISUALID, semanticLink);
		} else {
			return UnexecutableCommand.INSTANCE;
		}
	}
}
